/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.deployers.mc;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.List;

import org.jboss.deployers.plugins.main.MainDeployerImpl;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.deployers.vfs.spi.client.VFSDeploymentFactory;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.test.kernel.junit.MicrocontainerTest;
import org.jboss.virtual.MemoryFileFactory;
import org.jboss.virtual.VFS;
import org.jboss.virtual.VirtualFile;

/**
 * Unit test for {@link EsbWsdlDeployer}
 * <p/>
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 */
public class EsbWsdlDeployerUnitTest extends MicrocontainerTest
{
    private URL dir = getClass().getResource("/org/jboss/soa/esb/listeners/deployers/mc/test_esb_archives");
    private VFSDeployment deployment;
    private MainDeployerImpl mainDeployer;

    public EsbWsdlDeployerUnitTest(String name)
    {
        super(name);
    }
    
    @Override
    public void afterSetUp() throws Exception
    {
        super.afterSetUp();
        
        final String archiveName = "exploded-esb-archive.esb";
        final VirtualFile archiveFile = VFS.getVirtualFile(dir, archiveName);
        deployment = VFSDeploymentFactory.getInstance().createVFSDeployment(archiveFile);
        mainDeployer = (MainDeployerImpl) getBean("MainDeployer");
        mainDeployer.deploy(deployment);
    }
    
    public void _testGetAllSchemas() throws Throwable
    {
        final VirtualFile virtualFile = VFS.getVirtualFile(dir, "exploded-esb-archive.esb");
        final EsbWsdlDeployer esbWsdlDeployer = new EsbWsdlDeployer();
        final List<VirtualFile> schemas = esbWsdlDeployer.getSchemas(virtualFile);
        
        assertNotNull(schemas);
        assertEquals(2, schemas.size());
    }
    
    public void _testAddSchemasToVFS() throws IOException, URISyntaxException
    {
        final String archiveName = "exploded-esb-archive.esb";
        final VirtualFile virtualFile = VFS.getVirtualFile(dir, archiveName);
        final EsbWsdlDeployer esbWsdlDeployer = new EsbWsdlDeployer();
        final List<VirtualFile> schemas = esbWsdlDeployer.getSchemas(virtualFile);
        
        final URL dynamicClassRootUrl = new URL("vfsmemory", "junit", "");
        final VirtualFile dynamicClasspathDir = MemoryFileFactory.createRoot(dynamicClassRootUrl).getRoot();
        final URL wsdlUrl = new URL(dynamicClassRootUrl + "/somefile");
        VirtualFile putFile = MemoryFileFactory.putFile(wsdlUrl, "dummy".getBytes());
        
        esbWsdlDeployer.addSchemasToVfs(schemas, putFile.getParent(), archiveName);
        
        final List<VirtualFile> addedSchemas = esbWsdlDeployer.getSchemas(dynamicClasspathDir);
        assertNotNull(addedSchemas);
        assertEquals(2, addedSchemas.size());
        
        for (final VirtualFile schemaFile : addedSchemas)
        {
            final String fileName = schemaFile.getName();
            if (fileName.equals("request.xsd"))
            {
                assertEquals("vfsmemory://junit/WEB-INF/wsdl/request.xsd", schemaFile.toURL().toString());
            }
            else
            {
                assertEquals("vfsmemory://junit/WEB-INF/wsdl/types/custom-request-type.xsd", schemaFile.toURL().toString());
            }
        }
    }
    
    public void testUndeployCleanup() throws DeploymentException, MalformedURLException, IOException
    {
        final String wsdl = "WEB-INF/wsdl/FirstServiceESB/SimpleListener.wsdl";
        final EsbWsdlDeployer wsdlDeployer = (EsbWsdlDeployer) getBean("EsbWsdlDeployer");
        
        final VFSDeploymentUnit unit = (VFSDeploymentUnit) mainDeployer.getDeploymentUnit(deployment.getName());
        assertNotNull(unit.getMetaDataFile(wsdl));
        
        wsdlDeployer.undeploy(unit, null);
        
        assertNull(unit.getMetaDataFile(wsdl));
        assertNull(unit.getAttachment(EsbWsdlDeployer.WSDL_LOCATIONS_KEY));
        assertNull(unit.getAttachment(EsbWsdlDeployer.DYNAMIC_ROOT_URL_KEY));
    }
    
    public void tearDown() throws Exception
    {
        mainDeployer.undeploy(deployment);
        super.tearDown();
    }
}
