/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions.soap.adapter;

import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletResponse;

/**
 * This is an adapter used when integrating with JBossWS.
 *  
 * @author <a href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
 */
public class SOAPProcessorHttpServletResponse implements HttpServletResponse
{
    private static final String CHARSET = "charset=" ;
    
    private Map<String, List<String>> headers = new HashMap<String, List<String>>() ;
    
    private int status ;
    
    private String statusMessage ;
    
    private ByteArrayOutputStream baos ;
    
    private boolean streamReturned ;
    
    public SOAPProcessorHttpServletResponse()
    {
        baos = new ByteArrayOutputStream() ;
    }

    public void addHeader(final String name, final String value)
    {
        final String normalisedName = name.toLowerCase() ;
        final List<String> values = headers.get(normalisedName) ;
        if (values != null)
        {
            values.add(value) ;
        }
        else
        {
            final List<String> newValues = new ArrayList<String>() ;
            newValues.add(value) ;
            headers.put(normalisedName, newValues) ;
        }
    }

    public void setHeader(final String name, final String value)
    {
        final List<String> values = new ArrayList<String>() ;
        values.add(value) ;
        headers.put(name.toLowerCase(), values) ;
    }

    public void setStatus(final int status)
    {
    	setStatus(status, null) ;
    }

    public void setStatus(final int status, final String statusMessage)
    {
        this.status = status ;
        this.statusMessage = statusMessage ;
    }

    public ServletOutputStream getOutputStream() throws IOException
    {
        if (streamReturned)
        {
            throw new IOException("Output stream already accessed") ;
        }
        streamReturned = true ;
        return new SOAPProcessorServletOutputStream(baos) ;
    }

    public PrintWriter getWriter() throws IOException
    {
        if (streamReturned)
        {
            throw new IOException("Output stream already accessed") ;
        }
        streamReturned = true ;
        final String charset = getCharset() ;
        final OutputStreamWriter osw = (charset != null ? new OutputStreamWriter(baos, charset) : new OutputStreamWriter(baos)) ;
        return new PrintWriter(new BufferedWriter(osw)) ;
    }

    public void setContentType(final String contentType)
    {
        setHeader("content-type", contentType) ;
    }
    
    public int getStatus()
    {
        return status ;
    }
    
    public String getStatusMessage()
    {
        return statusMessage ;
    }
    
    public Map<String, List<String>> getHeaders()
    {
        return headers ;
    }
    
    public byte[] getContent()
    {
        return baos.toByteArray() ;
    }
    
    public String getContentType()
    {
        final List<String> values = headers.get("content-type") ;
        if (values != null)
        {
            return values.get(0) ;
        }
        return null ;
    }
    
    public String getCharset()
    {
        final String contentType = getContentType() ;
        if (contentType != null)
        {
            final int index = contentType.indexOf(CHARSET) ;
            if (index >= 0)
            {
                final int beginIndex = index + CHARSET.length() ;
                if (beginIndex < contentType.length())
                {
                    final int endIndex = contentType.indexOf(';', beginIndex) ;
                    if (endIndex >= 0)
                    {
                        return contentType.substring(beginIndex, endIndex).trim() ;
                    }
                    else
                    {
                        return contentType.substring(beginIndex).trim();
                    }
                }
            }
        }
        return null ;
    }
    
    public void setCharacterEncoding(final String charset)
    {
        final String contentType = getContentType() ;
        if (contentType != null)
        {
            final int charsetIndex = contentType.indexOf(CHARSET) ;
            if (charsetIndex == -1)
            {
                setContentType(contentType + ';' + CHARSET + charset) ;
            }
            else
            {
                final StringBuilder sb = new StringBuilder() ;
                sb.append(contentType, 0, charsetIndex) ;
                sb.append(CHARSET) ;
                sb.append(charset) ;
                final int endIndex = contentType.indexOf(';', charsetIndex) ;
                if (endIndex != -1)
                {
                    sb.append(contentType, endIndex, contentType.length()) ;
                }
                setContentType(sb.toString()) ;
            }
        }
    }
    
    /*
     * The following methods are not supported by this adapter.
     */
    
    public void addCookie(Cookie arg0)
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public void addDateHeader(String arg0, long arg1)
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public void addIntHeader(String arg0, int arg1)
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public boolean containsHeader(String arg0)
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public String encodeRedirectURL(String arg0)
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public String encodeRedirectUrl(String arg0)
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public String encodeURL(String arg0)
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public String encodeUrl(String arg0)
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public void sendError(int arg0) throws IOException
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public void sendError(int arg0, String arg1) throws IOException
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public void sendRedirect(String arg0) throws IOException
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public void setDateHeader(String arg0, long arg1)
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public void setIntHeader(String arg0, int arg1)
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public void flushBuffer() throws IOException
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public int getBufferSize()
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public String getCharacterEncoding()
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public Locale getLocale()
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public boolean isCommitted()
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public void reset()
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public void resetBuffer()
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public void setBufferSize(int arg0)
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public void setContentLength(int arg0)
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }

    public void setLocale(Locale arg0)
    {
        throw new UnsupportedOperationException("Not yet supported") ;
    }
}
