/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.security.opensso;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.security.Principal;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.login.LoginException;
import javax.security.auth.spi.LoginModule;

import org.apache.log4j.Logger;
import org.jboss.security.SecurityAssociation;

import com.iplanet.am.util.SystemProperties;
import com.iplanet.sso.SSOException;
import com.iplanet.sso.SSOToken;
import com.iplanet.sso.SSOTokenManager;
import com.sun.identity.authentication.AuthContext;

/**
 * OpenSSOLoginModule is a JAAS Login module implementation for OpenSSO.
 * <p/>
 * This implemenation will check if the calling Subject has an existing 
 * SSOPrincipal and check if that principal has an existing OpenSSO session.
 * A SSOPrincipal is just a Principal with an OpenSSO TokenID, which identifies
 * an OpenSSO session.
 * <p/>
 * 
 * <br>
 * <pre>
 * {@literal
 * Example of standalone JAAS login configuration:
 * OpenSSOLogin {
 *     org.jboss.security.opensso.OpenSSOLoginModule required orgName=opensso moduleName=DataStore amPropertiesFile=AMConfig.properties;
 * }; 
 * }</pre>
 * <br>
 * <pre>
 * {@literal
 * Example of JBossAS login-config.xml:
 *  <application-policy name="OpenSSO">
 *      <authentication>
 *          <login-module code="org.jboss.security.opensso.OpenSSOLoginModule" flag="required">
 *              <module-option name="orgName">opensso</module-option>
 *              <module-option name="moduleName">DataStore</module-option>
 *              <module-option name="amPropertiesFile">/AMConfig.properties</module-option>
 *          </login-module>
 *      </authentication>
 * </application-policy>
 *          
 * }</pre>
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 * @author jeffyu
 *
 */
public final class OpenSSOLoginModule implements LoginModule 
{
	// option property names
	private static final String MODULE_NAME = "moduleName";
	private static final String AM_PROPERTIES_FILE = "amPropertiesFile";
	private static final String ORG_NAME = "orgName";

	//	the subject to be authenticated
	private Subject subject;
	
	//	callback handler to be used
	private CallbackHandler callbackHandler;
	
	//	options from the login module configuration
	private Map<String, ?> options;
	
	// OpenSSO implementation for authenticating a user
	private AuthContext authContext;

	//	the opensso organization name passed to AuthContext upon creation
	private String orgName;

	//	the jaas configuration module index name 
	private String moduleName;
	
	// the authentication status
	private boolean succeeded = false;
	
	// the commit phase status
	private boolean commitSucceeded = false;
	
	//	flag which indicates if the subject as a pre-existing opensso session
	private boolean hasExistingSSOSession;
	
	private Logger log = Logger.getLogger(OpenSSOLoginModule.class);
	
	/**
	 *  Initializes the login module.
	 */
	public void initialize(
			final Subject subject, 
			final CallbackHandler callbackHandler, 
			final Map<String, ?> sharedState, 
			final Map<String, ?> options) 
	{
		this.subject = subject;
		this.callbackHandler = callbackHandler;
		
		if ( options == null )
			throw new NullPointerException("options map cannot be null");
		this.options = options;
		
		orgName = (String)this.options.get(ORG_NAME);
		assertOptionNotNull(orgName, ORG_NAME);
		
		moduleName = (String)this.options.get(MODULE_NAME);
		assertOptionNotNull(moduleName, MODULE_NAME);
		
		final String configFileName = (String) options.get(AM_PROPERTIES_FILE);
		assertOptionNotNull(configFileName, AM_PROPERTIES_FILE);
		configure(configFileName);
	}
	
	/**
	 *  Peforms authentication of the Subject.
	 *  <p/>
	 *  This method will check in the Subject contains an SSOPrincipal, and if so, use
	 *  that principals SSOTokenID to check if the Subject has a valid session within the 
	 *  OpenSSO system.<br>
	 *  If the Subject does not have an existing session a normal login process will occur and
	 *  an SSOPrincipal will be created and added to the Subject principals.
	 *  
	 *  @return true if the authentication succeeded, or false if this LoginModule should be ignored. 
	 *  @throws LoginException if the authentication fails
	 */
	public boolean login() throws LoginException 
	{
		subject = chooseSubject(SecurityAssociation.getSubject());
		log.debug("Try to authenticate subject : " + subject);
		
		hasExistingSSOSession = checkValidSSOSession( subject.getPrincipals(SSOPrincipal.class) ) ;
		
		if ( hasExistingSSOSession )
		{
			succeeded = true;
			return succeeded;
		}
		
		authContext = new AuthContext(orgName);
		
		// login using the module authentication type
		authContext.login(AuthContext.IndexType.MODULE_INSTANCE, moduleName);
		
		//	get the callbacks that need to be populated the authentication plugin.
		Callback[] callbacks = authContext.getRequirements();
		
		//	populate the callbacks.
		handleCallbacks(callbacks);
		
		//	now submit the populated callbacks to plugin-modules.
		authContext.submitRequirements(callbacks);
	
		//	check the retured status
		if (authContext.getStatus() == AuthContext.Status.SUCCESS)
		{
			log.info("Login succeeded.");
			succeeded = true;
		} 
		return succeeded;
	}
	
	Subject chooseSubject(final Subject subjectFromAs)
	{
		if ( subject.getPrincipals(SSOPrincipal.class).isEmpty() && subjectFromAs != null)
		{
			if ( !subjectFromAs.getPrincipals(SSOPrincipal.class).isEmpty() )
			{
				return subjectFromAs;
			}
		}
		return subject;
	}

	public boolean commit() throws LoginException 
	{
		if (succeeded == false) 
		{
			return false;
		}
		
		try 
		{
			if ( !hasExistingSSOSession )
			{
    			//	this means that this was a new authentication so create a new SSOPrincipal
				SSOToken ssoToken = authContext.getSSOToken();
				Principal principal = new SSOPrincipal(ssoToken.getTokenID().toString());
				subject.getPrincipals().add(principal);
			}
			commitSucceeded = true;
		} 
		catch (final Exception ignore) 
		{
			log.error("Exception in commit: ", ignore);
			commitSucceeded = false;
		}
				
		return commitSucceeded;
	}

	/**
	 * 
	 */
	public boolean abort() throws LoginException 
	{
		if (succeeded == false) 
		{
			return true;
		}
		succeeded = false;
		commitSucceeded = false;
		authContext.logout();
		return true;
	}

	/**
	 * Perform clean up operations.
	 * Will clear all principals, logout from the AuthenticationContext, 
	 * and reset all internal flags.
	 */
	public boolean logout() throws LoginException 
	{
		subject.getPrincipals().clear();
		succeeded = false;
		commitSucceeded = false;
		authContext.logout();
		return true;
	}
	
	private boolean checkValidSSOSession(final Set<SSOPrincipal> principals )
	{
		boolean hasSession = false;
		if ( !principals.isEmpty() )
		{
			SSOPrincipal ssoPrincipal = principals.iterator().next();
			try 
			{
				SSOTokenManager tokenMgr = SSOTokenManager.getInstance();
				SSOToken ssoToken = tokenMgr.createSSOToken(ssoPrincipal.getToken());
				hasSession = tokenMgr.isValidToken(ssoToken);
			} 
			catch (final SSOException ignore) 
			{
				hasSession = false;
			}
		}
		log.info("Has valid OpenSSO session : " +  hasSession);
		return hasSession;
	}

	private void handleCallbacks(Callback[] requirements) throws LoginException
	{
		try
		{
			callbackHandler.handle(requirements);
		} 
		catch (IOException e)
		{
			log.error("IOException while handling callbacks : ", e);
			throw new LoginException(e.getMessage());
		} 
		catch (UnsupportedCallbackException e)
		{
			log.error("UnsupportedCallbackException while handling callbacks : ", e);
			throw new LoginException(e.getMessage());
		}
	}

	/**
	 * Will configure OpenSSO.
	 */
	private void configure(final String amProperties) 
	{
		log.debug("Access Manager(AM) configuration properties file : " + amProperties);
		if ( amProperties != null )
		{
    		Properties props = new Properties();
    		try
			{
        		InputStream inputStream = getResourceAsStream(amProperties, getClass());
        		if ( inputStream != null )
        		{
    				props.load(inputStream);
        		}
        		else
            		throw new IllegalStateException("Could not locate Access Manager(AM) configuration properties file: " + amProperties);
			} 
    		catch (FileNotFoundException e)
			{
        		throw new IllegalStateException("Could not locate Access Manager(AM) configuration properties file: " + amProperties, e );
			} 
    		catch (IOException e)
			{
        		throw new IllegalStateException("Could not locate Access Manager(AM) configuration properties file: " + amProperties, e );
			}
    		SystemProperties.initializeProperties(props);
		}
	}
	
	private InputStream getResourceAsStream(final String resourceName, final Class<?> caller)
    {
        final String resource ;
        if (resourceName.startsWith("/"))
        {
            resource = resourceName.substring(1) ;
        }
        else
        {
            final Package callerPackage = caller.getPackage() ;
            if (callerPackage != null)
            {
                resource = callerPackage.getName().replace('.', '/') + '/' + resourceName ;
            }
            else
            {
                resource = resourceName ;
            }
        }
        final ClassLoader threadClassLoader = Thread.currentThread().getContextClassLoader() ;
        if (threadClassLoader != null)
        {
            final InputStream is = threadClassLoader.getResourceAsStream(resource) ;
            if (is != null)
            {
                    return is ;
            }
        }
        
        final ClassLoader classLoader = caller.getClassLoader() ;
        if (classLoader != null)
        {
            final InputStream is = classLoader.getResourceAsStream(resource) ;
            if (is != null)
            {
                return is ;
            }
        }
        return ClassLoader.getSystemResourceAsStream(resource) ;
    }

	private void assertOptionNotNull(final Object variable, final String variableName)
	{
		if ( variable == null )
			throw new NullPointerException("options map must contain the required property '" + variableName + "'.");
	}
}
