/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.listeners.jca;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.filter.FilterManager;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.ListenerUtil;
import org.jboss.soa.esb.listeners.RegistryUtil;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.util.Util;

/**
 * @author <a href="kevin.conner@jboss.com">Kevin Conner</a>
 */
public class JcaGatewayListener extends BaseJcaInflow<InflowMessageProcessorAdapter> implements InflowMessageProcessor
{
    private final String targetServiceCategory ;
    private final String targetServiceName ;
    private final String serviceCategory ;
    private final String serviceName ;
    private final EPR serviceEPR ;
    
    private final Object composer ;
    private final Method processMethod ;
    
    private ServiceInvoker serviceInvoker ;

    public JcaGatewayListener(final ConfigTree config)
        throws ConfigurationException
    {
        super(config, InflowMessageProcessorAdapter.class);
        
        targetServiceCategory = ListenerUtil.getValue(config, ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG) ;
        if (Util.isNullString(targetServiceCategory))
        {
            throw new ConfigurationException("No service category defined!") ;
        }
        targetServiceName = ListenerUtil.getValue(config, ListenerTagNames.TARGET_SERVICE_NAME_TAG) ;
        if (Util.isNullString(targetServiceName))
        {
            throw new ConfigurationException("No service name defined!") ;
        }
        
        serviceCategory = ListenerUtil.getValue(config, ListenerTagNames.SERVICE_CATEGORY_NAME_TAG) ;
        serviceName = ListenerUtil.getValue(config, ListenerTagNames.SERVICE_NAME_TAG) ;
        
        if (serviceName != null)
        {
            serviceEPR = ListenerUtil.assembleEpr(config) ;
        }
        else
        {
            serviceEPR = null ;
        }
        
        final String composerName = config.getAttribute(ListenerTagNames.GATEWAY_COMPOSER_CLASS_TAG) ;
        if (Util.isNullString(composerName))
        {
            throw new ConfigurationException("No composer class defined") ;
        }
        final Class<?> composerClass ;
        try
        {
            composerClass = ClassUtil.forName(composerName, getClass()) ;
        }
        catch (final ClassNotFoundException cnfe)
        {
            throw new ConfigurationException("Could not load composer class: " + composerName, cnfe) ;
        }
        
        try
        {
            Object composer ;
            try
            {
                final Constructor<?> configConstructor = composerClass.getConstructor(ConfigTree.class) ;
                composer = configConstructor.newInstance(config) ;
            }
            catch (final NoSuchMethodException nsme)
            {
                composer = composerClass.newInstance() ;
            }
            this.composer = composer ;
        }
        catch (final Throwable th)
        {
            throw new ConfigurationException("Unexpected error instantiating composer: " + composerName, th) ;
        }
        
        final String processMethodName = config.getAttribute(ListenerTagNames.GATEWAY_COMPOSER_METHOD_TAG) ;
        try
        {
            processMethod = composerClass.getMethod(processMethodName, Object.class) ;
        }
        catch (final NoSuchMethodException nsme)
        {
            throw new ConfigurationException("Could not locate process method: " + processMethodName, nsme) ;
        }
    }
    
    @Override
    protected String getDescription()
    {
        return "targetCategory: " + targetServiceCategory + " targetService: " + targetServiceName ;
    }
    
    @Override
    protected void doInitialise() throws ManagedLifecycleException
    {
        super.doInitialise();
        bean.setInflowMessageProcessor(this);
        
        try
        {
            serviceInvoker = new ServiceInvoker(targetServiceCategory, targetServiceName) ;
        }
        catch (final MessageDeliverException mde)
        {
            throw new ManagedLifecycleException("Unexpected exception creating service invoker", mde) ;
        }
        
        if (serviceEPR != null)
        {
            try
            {
                RegistryUtil.register(getConfig(), serviceEPR) ;
            }
            catch (final RegistryException re)
            {
                throw new ManagedLifecycleException("Unexpected exception while registering service", re) ;
            }
        }
    }
    
    @Override
    protected void doDestroy() throws ManagedLifecycleException
    {
        super.doDestroy();
        if (serviceEPR != null)
        {
            RegistryUtil.unregister(serviceCategory, serviceName, serviceEPR) ;
        }
    }

    public void process(final Object messageIn)
    {
        try
        {
            Message messageOut = (Message)processMethod.invoke(composer, messageIn) ;
            
            final Map<String, Object> params = new HashMap<String, Object>() ;
            params.put(Environment.GATEWAY_CONFIG, getConfig()) ;
            
            messageOut = FilterManager.getInstance().doOutputWork(messageOut, params) ;
            
            serviceInvoker.deliverAsync(messageOut) ;
        }
        catch (final RuntimeException re)
        {
            throw re ;
        }
        catch (final Error err)
        {
            throw err ;
        }
        catch (final Throwable th)
        {
            throw new JcaGatewayException("Unexpected exception caught while processing JCA message", th) ;
        }
    }
}
