/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.services.registry;

import java.util.List;

import org.jboss.soa.esb.addressing.EPR;
/**
 * Registry interface for the ESB.
 *
 * @author Kurt Stam
 */
public interface Registry 
{
	/**
	 * Removes a service from the Registry along with all the ServiceBindings underneath it.
	 *
	 * @param category           - name of the service category, for example 'transformation'.
	 * @param serviceName        - name of the service, for example 'smooks'.
	 * @throws RegistryException
	 */
	public void unRegisterService(String category, String serviceName) throws RegistryException, ServiceNotFoundException;
	/**
	 * Registers an EPR under the specified category and service. If the specified service does
	 * not exist, it will be created at the same time.
	 * 
	 * @param serviceCategoryName - name of the category to which the service belongs.
	 * @param serviceName         - name of the service to which the EPS belongs.
	 * @param serviceDescription  - human readable description of the service, 
	 * 							   only used when it the service does not yet exist.
	 * @param epr				  - the EndPointReference (EPR) that needs to be registered.
	 * @param eprDescription	  - human readable description of the EPR
	 * @throws RegistryException
	 */
	public void registerEPR(String serviceCategoryName, String serviceName, String serviceDescription, EPR epr, String eprDescription) 
	throws RegistryException;
	/**
	 * Removes an EPR from the Registry. 
	 * @param serviceCategoryName - name of the category to which the service belongs.
	 * @param serviceName         - name of the service to which the EPS belongs.
	 * @param epr                 - the EndPointReference (EPR) that needs to be unregistered.
	 * @throws RegistryException
	 */
	public void unRegisterEPR(String serviceCategoryName, String serviceName, EPR epr) throws RegistryException, ServiceNotFoundException;
	/** 
	 * Find all Services assigned to the Red Hat/JBossESB organization.
	 * @return Collection of Strings containing the service names.
	 * @throws RegistryException
	 */
	public List<String> findAllServices() throws RegistryException;
	/**
	 * Find all services that belong to the supplied category. 
	 * 
	 * @param serviceCategoryName - name of the category to which the service belongs.
	 * @return Collection of Strings containing the service names
	 * @throws RegistryException
	 */
	public List<String> findServices(String serviceCategoryName) throws RegistryException;
	/**
	 * Finds all the EPRs that belong to a specific category and service combination.
	 * 
	 * @param serviceCategoryName - name of the category to which the service belongs.
	 * @param serviceName         - name of the service to which the EPS belongs.
	 * @return Collection of EPRs.
	 * @throws RegistryException
	 */
	public List<EPR> findEPRs(String serviceCategoryName, String serviceName) throws RegistryException, ServiceNotFoundException;
	/**
	 * Returns the first EPR in the list that belong to a specific category and service combination.
	 * 
	 * @param serviceCategoryName - name of the category to which the service belongs.
	 * @param serviceName         - name of the service to which the EPS belongs.
	 * @return EPR.
	 * @throws RegistryException
	 */
	public EPR findEPR(String serviceCategoryName, String serviceName) throws RegistryException, ServiceNotFoundException;
}
