package org.jboss.internal.soa.esb.couriers.transport;

import java.net.URI;

import javax.transaction.Synchronization;
import javax.transaction.xa.XAException;
import javax.transaction.xa.XAResource;

import static junit.framework.Assert.assertEquals;
import static junit.framework.Assert.assertNotNull;
import static junit.framework.Assert.assertNull;
import static junit.framework.Assert.assertTrue;
import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.addressing.eprs.InVMEpr;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

/**
 * InVMResourceManager unit tests.
 * 
 * @author <a href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
 */
public class InVMResourceManagerUnitTest
{
    private static final String TEST_CATEGORY1 = "test category1" ;
    private static final String TEST_CATEGORY2 = "test category2" ;
    private static final String TEST_NAME1 = "service name 1" ;
    private static final String TEST_NAME2 = "service name 2" ;
    
    private InVMEpr epr1 ;
    private InVMEpr epr2 ;
    private InVMEpr tempEPR1 ;
    private InVMEpr tempEPR2 ;
    private TransactionStrategy transactionStrategy ;
    private TestTransactionStrategy testTransactionStrategy ;

    @Before
    public void setup()
        throws Exception
    {
        epr1 = new InVMEpr(URI.create(InVMEpr.INVM_PROTOCOL + "://" + InVMEpr.createEncodedServiceId(TEST_CATEGORY1, TEST_NAME1))) ;
        epr2 = new InVMEpr(URI.create(InVMEpr.INVM_PROTOCOL + "://" + InVMEpr.createEncodedServiceId(TEST_CATEGORY2, TEST_NAME2))) ;
        tempEPR1 = new InVMEpr(URI.create(InVMEpr.INVM_PROTOCOL + "://" + InVMEpr.createEncodedServiceId(TEST_CATEGORY1, TEST_NAME1))) ;
        tempEPR1.setTemporaryEPR(true) ;
        tempEPR2 = new InVMEpr(URI.create(InVMEpr.INVM_PROTOCOL + "://" + InVMEpr.createEncodedServiceId(TEST_CATEGORY2, TEST_NAME2))) ;
        tempEPR2.setTemporaryEPR(true) ;
        
        final InVMTransport transport = InVMTransport.getInstance() ;
        transport.registerEPR(TEST_CATEGORY1, TEST_NAME1, epr1) ;
        transport.registerEPR(TEST_CATEGORY2, TEST_NAME2, epr2) ;
        
        transactionStrategy = TransactionStrategy.getTransactionStrategy(true) ;
        testTransactionStrategy = new TestTransactionStrategy() ;
        TransactionStrategy.setTransactionStrategy(testTransactionStrategy) ;
    }

    @After
    public void teardown()
        throws Exception
    {
        
        final InVMTransport transport = InVMTransport.getInstance() ;
        transport.unRegisterEPR(TEST_CATEGORY1, TEST_NAME1, epr1) ;
        transport.unRegisterEPR(TEST_CATEGORY2, TEST_NAME2, epr2) ;

        final InVMTemporaryTransport tempTransport = InVMTemporaryTransport.getInstance() ;
        while(tempTransport.pickup(tempEPR1, 0) != null) ;
        while(tempTransport.pickup(tempEPR2, 0) != null) ;
        
        TransactionStrategy.setTransactionStrategy(transactionStrategy) ;
    }
    
    @Test
    public void testInsertCommit()
        throws Exception
    {
        final Message message1 = MessageFactory.getInstance().getMessage() ;
        final Message message2 = MessageFactory.getInstance().getMessage() ;
        
        final InVMTransport transport = InVMTransport.getInstance() ;
        
        testTransactionStrategy.begin();
        
        transport.deliver(epr1, message1) ;
        transport.deliver(epr2, message2) ;
        
        assertTrue("Enlisted", testTransactionStrategy.isResourceEnlisted()) ;
        testTransactionStrategy.terminate() ;
        
        assertNotNull("Service 1 delivery", transport.pickup(epr1, 0)) ;
        assertNotNull("Service 2 delivery", transport.pickup(epr2, 0)) ;
    }

    @Test
    public void testInsertRollback()
        throws Exception
    {
        final Message message1 = MessageFactory.getInstance().getMessage() ;
        final Message message2 = MessageFactory.getInstance().getMessage() ;
        
        final InVMTransport transport = InVMTransport.getInstance() ;
        
        testTransactionStrategy.begin();
        
        transport.deliver(epr1, message1) ;
        transport.deliver(epr2, message2) ;
        
        assertTrue("Enlisted", testTransactionStrategy.isResourceEnlisted()) ;
        testTransactionStrategy.rollbackOnly() ;
        testTransactionStrategy.terminate() ;
        
        assertNull("Service 1 delivery", transport.pickup(epr1, 0)) ;
        assertNull("Service 2 delivery", transport.pickup(epr2, 0)) ;
    }
    
    @Test
    public void testRemoveCommit()
        throws Exception
    {
        final Message message1 = MessageFactory.getInstance().getMessage() ;
        final Message message2 = MessageFactory.getInstance().getMessage() ;
        
        final InVMTransport transport = InVMTransport.getInstance() ;
        
        transport.deliver(epr1, message1) ;
        transport.deliver(epr2, message2) ;
        
        testTransactionStrategy.begin();
        
        assertNotNull("Service 1 during transaction pickup", transport.pickup(epr1, 0)) ;
        assertNotNull("Service 2 during transaction pickup", transport.pickup(epr2, 0)) ;
        
        assertTrue("Enlisted", testTransactionStrategy.isResourceEnlisted()) ;
        testTransactionStrategy.terminate() ;
        
        assertNull("Service 1 pickup", transport.pickup(epr1, 0)) ;
        assertNull("Service 2 pickup", transport.pickup(epr2, 0)) ;
    }

    @Test
    public void testRemoveRollback()
        throws Exception
    {
        final Message message1 = MessageFactory.getInstance().getMessage() ;
        final Message message2 = MessageFactory.getInstance().getMessage() ;
        
        final InVMTransport transport = InVMTransport.getInstance() ;
        
        transport.deliver(epr1, message1) ;
        transport.deliver(epr2, message2) ;
        
        testTransactionStrategy.begin();
        
        assertNotNull("Service 1 during transaction pickup", transport.pickup(epr1, 0)) ;
        assertNotNull("Service 2 during transaction pickup", transport.pickup(epr2, 0)) ;
        
        assertTrue("Enlisted", testTransactionStrategy.isResourceEnlisted()) ;
        testTransactionStrategy.rollbackOnly() ;
        testTransactionStrategy.terminate() ;
        
        assertNotNull("Service 1 pickup", transport.pickup(epr1, 0)) ;
        assertNotNull("Service 2 pickup", transport.pickup(epr2, 0)) ;
    }
    
    @Test
    public void testTemporaryInsertCommit()
        throws Exception
    {
        final Message message1 = MessageFactory.getInstance().getMessage() ;
        final Message message2 = MessageFactory.getInstance().getMessage() ;
        
        final InVMTemporaryTransport transport = InVMTemporaryTransport.getInstance() ;
        
        testTransactionStrategy.begin();
        
        transport.deliver(tempEPR1, message1) ;
        transport.deliver(tempEPR2, message2) ;
        
        assertTrue("Enlisted", testTransactionStrategy.isResourceEnlisted()) ;
        testTransactionStrategy.terminate() ;
        
        assertNotNull("Service 1 delivery", transport.pickup(tempEPR1, 0)) ;
        assertNotNull("Service 2 delivery", transport.pickup(tempEPR2, 0)) ;
    }

    @Test
    public void testTemporaryInsertRollback()
        throws Exception
    {
        final Message message1 = MessageFactory.getInstance().getMessage() ;
        final Message message2 = MessageFactory.getInstance().getMessage() ;
        
        final InVMTemporaryTransport transport = InVMTemporaryTransport.getInstance() ;
        
        testTransactionStrategy.begin();
        
        transport.deliver(tempEPR1, message1) ;
        transport.deliver(tempEPR2, message2) ;
        
        assertTrue("Enlisted", testTransactionStrategy.isResourceEnlisted()) ;
        testTransactionStrategy.rollbackOnly() ;
        testTransactionStrategy.terminate() ;
        
        assertNull("Service 1 delivery", transport.pickup(tempEPR1, 0)) ;
        assertNull("Service 2 delivery", transport.pickup(tempEPR2, 0)) ;
    }
    
    @Test
    public void testTemporaryRemoveCommit()
        throws Exception
    {
        final Message message1 = MessageFactory.getInstance().getMessage() ;
        final Message message2 = MessageFactory.getInstance().getMessage() ;
        
        final InVMTemporaryTransport transport = InVMTemporaryTransport.getInstance() ;
        
        transport.deliver(tempEPR1, message1) ;
        transport.deliver(tempEPR2, message2) ;
        
        testTransactionStrategy.begin();
        
        assertNotNull("Service 1 during transaction pickup", transport.pickup(tempEPR1, 0)) ;
        assertNotNull("Service 2 during transaction pickup", transport.pickup(tempEPR2, 0)) ;
        
        assertTrue("Enlisted", testTransactionStrategy.isResourceEnlisted()) ;
        testTransactionStrategy.terminate() ;
        
        assertNull("Service 1 pickup", transport.pickup(tempEPR1, 0)) ;
        assertNull("Service 2 pickup", transport.pickup(tempEPR2, 0)) ;
    }

    @Test
    public void testTemporaryRemoveRollback()
        throws Exception
    {
        final Message message1 = MessageFactory.getInstance().getMessage() ;
        final Message message2 = MessageFactory.getInstance().getMessage() ;
        
        final InVMTemporaryTransport transport = InVMTemporaryTransport.getInstance() ;
        
        transport.deliver(tempEPR1, message1) ;
        transport.deliver(tempEPR2, message2) ;
        
        testTransactionStrategy.begin();
        
        assertNotNull("Service 1 during transaction pickup", transport.pickup(tempEPR1, 0)) ;
        assertNotNull("Service 2 during transaction pickup", transport.pickup(tempEPR2, 0)) ;
        
        assertTrue("Enlisted", testTransactionStrategy.isResourceEnlisted()) ;
        testTransactionStrategy.rollbackOnly() ;
        testTransactionStrategy.terminate() ;
        
        assertNotNull("Service 1 pickup", transport.pickup(tempEPR1, 0)) ;
        assertNotNull("Service 2 pickup", transport.pickup(tempEPR2, 0)) ;
    }

    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(InVMResourceManagerUnitTest.class);
    }
    
    private static class TestTransactionStrategy extends TransactionStrategy
    {
        private XAResource resource ;
        private boolean rollback ;
        private boolean active ;
        
        @Override
        public void begin() throws TransactionStrategyException
        {
            active = true ;
        }

        @Override
        public void terminate() throws TransactionStrategyException
        {
            try
            {
                if (resource != null)
                {
                    if (rollback)
                    {
                        resource.rollback(null) ;
                    }
                    else
                    {
                        resource.commit(null, true) ;
                    }
                }
            }
            catch (final XAException xae)
            {
                throw new TransactionStrategyException("Unexpected exception during commit", xae) ;
            }
            finally
            {
                resource = null ;
                rollback = false ;
                active = false ;
            }
        }

        @Override
        public void rollbackOnly() throws TransactionStrategyException
        {
            rollback = true ;
        }

        @Override
        public Object getTransaction() throws TransactionStrategyException
        {
            return (active ? this : null) ;
        }

        @Override
        public Object suspend() throws TransactionStrategyException
        {
            return null ;
        }

        @Override
        public boolean isActive() throws TransactionStrategyException
        {
            return active ;
        }

        @Override
        public void resume(Object tx) throws TransactionStrategyException
        {
        }

        @Override
        public void registerSynchronization(Synchronization sync)
                throws TransactionStrategyException
        {
        }

        @Override
        public void enlistResource(XAResource resource)
                throws TransactionStrategyException
        {
            if (this.resource != null)
            {
                throw new TransactionStrategyException("resource enlisted multiple times") ;
            }
            this.resource = resource ;
        }
        
        public boolean isResourceEnlisted()
        {
            return resource != null ;
        }
    }
}
