/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.internal.soa.esb.services.registry;

import static org.junit.Assert.assertEquals;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import javax.xml.registry.BulkResponse;
import javax.xml.registry.BusinessLifeCycleManager;
import javax.xml.registry.BusinessQueryManager;
import javax.xml.registry.Connection;
import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.Classification;
import javax.xml.registry.infomodel.ClassificationScheme;
import javax.xml.registry.infomodel.Organization;
import javax.xml.registry.infomodel.Service;
import javax.xml.registry.infomodel.ServiceBinding;
import javax.xml.registry.infomodel.User;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.addressing.helpers.EPRHelper;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.ServiceNotFoundException;
import org.jboss.soa.esb.testutils.TestEnvironmentUtil;
import org.jboss.soa.esb.ConfigurationException;

import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Testing the registry.
 * 
 * @author kstam
 *
 */
public class JAXRRegistryUnitTest
{
	private static Logger logger = Logger.getLogger(JAXRRegistryUnitTest.class);
	/**
	 * Tests the successful creation of the RED HAT/JBossESB Organization.
	 */
	@Test
	public void publishOrganization() throws ConfigurationException, JAXRException {
        JAXRConnectionFactory jaxrConnectionFactory = new JAXRConnectionFactory();
		Organization org = JAXRRegistryImpl.createJBossESBOrganization(jaxrConnectionFactory);
		logger.debug("Succesfully created organization: " + org.getName().getValue());
		assertEquals("Red Hat/JBossESB", org.getName().getValue());
	}
	@Test
	public void findOrganization() throws ConfigurationException, JAXRException {
        JAXRConnectionFactory jaxrConnectionFactory = new JAXRConnectionFactory();
		Organization org = JAXRRegistryImpl.findOrganization("Red Hat/JBossESB", jaxrConnectionFactory);
		logger.debug("Succesfully created organization: " + org.getName().getValue());
		assertEquals("Red Hat/JBossESB", org.getName().getValue());

		Organization nullOrg = JAXRRegistryImpl.findOrganization("Not Existing Org", jaxrConnectionFactory);
		logger.debug("Could not find non-existing organization.");
		assertEquals(null, nullOrg);
	}
	/**
	 * Tests the successful registration of a Service.
	 */
	@Test
	public void publishService() throws ConfigurationException, JAXRException {
		JAXRRegistryImpl registry = new JAXRRegistryImpl();
		registry.registerService("registry", "Registry Test ServiceName", "Registry Test Service Description");
	}
	@Test
	public void publishServiceBinding() throws ConfigurationException, RegistryException {
		EPR epr = new EPR();
		JAXRRegistryImpl registry = new JAXRRegistryImpl();
		registry.registerEPR("registry", "Registry Test ServiceName", "Registry Test Service Description",
				epr, "EPR description");
		registry.registerEPR("registry", "Registry Test ServiceName", "Registry Test Service Description",
				epr, "EPR description");
        try {
            registry.unRegisterEPR("registry", "Registry Test ServiceName", epr);
        } catch (ServiceNotFoundException snfe) {
            logger.warn("Failed to find Service for unregistering.", snfe);
        }
	}
	/**
	 * Queries the newly added information
	 * @throws Exception
	 */
	@Test
	public void findServicesForAnOrganization() throws Exception {
        JAXRConnectionFactory jaxrConnectionFactory = new JAXRConnectionFactory();
		Organization org = JAXRRegistryImpl.findOrganization("Red Hat/JBossESB", jaxrConnectionFactory);
		//Listing out the services and their Bindings
		logger.debug("-------------------------------------------------");
		logger.debug("Organization name: " + org.getName().getValue());
		logger.debug("Description: " + org.getDescription().getValue());
		logger.debug("Key id: " + org.getKey().getId());
		User primaryContact = org.getPrimaryContact();
		logger.debug("Primary Contact: " + primaryContact.getPersonName().getFullName());
		Collection services = org.getServices();
		for (Iterator serviceIter = services.iterator();serviceIter.hasNext();) {
			Service service = (Service) serviceIter.next();
			logger.debug("- Service Name: " + service.getName().getValue());
			logger.debug("  Service Key : " + service.getKey().getId());
			Collection serviceBindings = service.getServiceBindings();
			for (Iterator serviceBindingIter = serviceBindings.iterator();serviceBindingIter.hasNext();){
				ServiceBinding serviceBinding = (ServiceBinding) serviceBindingIter.next();
				logger.debug("  ServiceBinding Description: " + serviceBinding.getDescription().getValue());
				String xml = serviceBinding.getAccessURI();
				logger.debug("  ServiceBinding URI: " + xml);
				assertEquals(EPRHelper.toXMLString(new EPR()).trim(),xml.trim());
			}
		}
		logger.debug("-------------------------------------------------");
    }
	/**
	 * This doesn't work because scout drops the classifications on the floor.
	 * We're ignoring this test until I come up with a patch.
	 *
	 */
	public void findServicesByClassification() throws ConfigurationException {
        JAXRConnectionFactory jaxrConnectionFactory = new JAXRConnectionFactory();
        Connection connection = jaxrConnectionFactory.getConnection();
		try {
			// Get registry service and business query manager
			RegistryService rs = connection.getRegistryService();
			BusinessQueryManager bqm = rs.getBusinessQueryManager();
			BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
			ClassificationScheme classificationScheme = bqm.findClassificationSchemeByName(null, "uddi-org:general_keywords");
			Classification classification = blm.createClassification(classificationScheme,
					"Test transformation service", "transformation");
			Collection<Classification> classifications = new ArrayList<Classification>();
			classifications.add(classification);
			//Here I'd like to test filtering by this classification, but scout ignored the classification
			String name=classificationScheme.getName().getValue();
			logger.debug("Name=" + name);
			Collection<String> nameParams = new ArrayList<String>();
			//The name of the service is wild
			nameParams.add("%");
			BulkResponse bs = bqm.findServices(null, null,nameParams,classifications, null);
			int status = bs.getStatus();
			logger.debug("status=" + status);
		} catch (JAXRException je) {
			logger.error(je);
		}
		finally{
            jaxrConnectionFactory.closeConnection(connection);
        }
	}
	/**
	 * Setup the database.
	 * @throws Exception
	 */
	@BeforeClass
	public static void runBeforeAllTests() throws Exception {
		TestEnvironmentUtil.setESBPropertiesFileToUse();

		TestEnvironmentUtil.startJAXRDatabase();
	}

    /**
	 * Shutdown the database
	 * @throws Exception
	 */
	@AfterClass
	public static void runAfterAllTests() throws Exception {
		TestEnvironmentUtil.stopJAXRDatabase();
	}

    public static junit.framework.Test suite() {
		return new JUnit4TestAdapter(JAXRRegistryUnitTest.class);
	}
}
