/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.message;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.net.URI;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

/**
 * Tests for the ServiceInvoker and UncomposedMessageDeliveryAdapter
 * classes.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class MessageDeliveryAdapterUnitTest {
	private Logger log = Logger
			.getLogger( MessageDeliveryAdapterUnitTest.class );

    private static EPR epr1;
    private static EPR epr2;
    private static EPR epr3;
    private static EPR epr4;
    private static MockCourier courier1;
    private static MockCourier courier2;
    private static MockCourier courier3;
    private static MockCourier courier4;
    private static UncomposedMessageDeliveryAdapter deliveryAdapter;
    private String payload = "*XX*";
    private BasicMessageComposer composer;

    @Before
    public void setUp() throws Exception {
        MockCourierFactory.install();
        MockRegistry.install();

        epr1 = new EPR(new URI("1"));
        epr2 = new EPR(new URI("2"));
        epr3 = new EPR(new URI("3"));
        epr4 = new EPR(new URI("4"));
        courier1 = new MockCourier(false);
        courier2 = new MockCourier(new CourierException(""));
        courier3 = new MockCourier(new MalformedEPRException(""));
        courier4 = new MockCourier(true);

        MockRegistry.register("cat", "service", epr1, courier1);
        MockRegistry.register("cat", "service", epr2, courier2);
        MockRegistry.register("cat", "service", epr3, courier3);
        MockRegistry.register("cat", "service", epr4, courier4);

        composer = new BasicMessageComposer();
        composer.setConfiguration(new ConfigTree("config"));
        deliveryAdapter = new UncomposedMessageDeliveryAdapter("cat", "service", composer, null);
    }
    @After
    public void tearDown() throws Exception {
        MockRegistry.uninstall();
        MockCourierFactory.uninstall();
    }
    @Test
    public void test_getCourier_CourierException() throws MessageDeliverException {
        // Get the courier factory to throw a CourierException
        MockCourierFactory.courierException = new CourierException("");
        try {
        	log.warn( "The following warings are intentional" );
            deliveryAdapter.deliverAsync(payload);
            fail("Expected MessageDeliverException");
        } catch(MessageDeliverException e) {
            // Expected.
        } catch(RegistryException re) {
            fail("RegistryException");
        }
        assertNoDeliveryAttempted();
    }
    @Test
    public void test_getCourier_MalformedEPRException() throws MessageDeliverException {
        // Get the courier factory to throw a MalformedEPRException
        MockCourierFactory.malformedEPRException = new MalformedEPRException("");
        try {
        	log.warn( "The following warings are intentional" );
            deliveryAdapter.deliverAsync(payload);
            fail("Expected MessageDeliverException");
        } catch(MessageDeliverException e) {
            // Expected.
        } catch(RegistryException re) {
            fail("RegistryException");
        }
        assertNoDeliveryAttempted();
    }
    @Test
    public void test_No_EPRs() throws RegistryException, MessageDeliverException {
        // Make sure there's no attempt to make a delivery when there's no
        // EPRs for the service.
        deliveryAdapter = new UncomposedMessageDeliveryAdapter("x", "y", composer, null);
        try {
            deliveryAdapter.deliverAsync(payload);
            fail("Expected MessageDeliverException");
        } catch(MessageDeliverException e) {
            // Expected.
        } catch(RegistryException re) {
            fail("RegistryException");
        }
        assertNoDeliveryAttempted();
    }
    @Test
    public void test_deliver() throws MessageDeliverException {
        // Make sure the delivery happens as expected...
        try {
            deliveryAdapter.deliverAsync(payload);
            assertEquals(null, courier1.message);
            assertEquals(null, courier2.message);
            assertEquals(null, courier3.message);
            assertEquals(payload, courier4.message.getBody().get());
    
            String payload2 = "*YYY*";
            deliveryAdapter.deliverAsync(payload2);
            assertEquals(payload2, courier4.message.getBody().get());
        } catch(RegistryException re) {
            fail("RegistryException");
        }
    }
    

    private void assertNoDeliveryAttempted() {
        if(courier1.deliveryAttempted ||
           courier2.deliveryAttempted ||
           courier3.deliveryAttempted ||
           courier4.deliveryAttempted) {
            fail("A deliverAsync attempt was made on one of the couriers.");
        }
    }
    
    public static junit.framework.Test suite() {
        return new JUnit4TestAdapter(MessageDeliveryAdapterUnitTest.class);
    }
}
