/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.notification;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.Test;

/**
 * Unit test for NotifyFTP
 * 
 * @author <a href="mailto:daniel.bevenius@gmail.com">Daniel Bevenius</a>			
 *
 */
public class NotifyFTPListUnitTest
{
	private static final String LINE_SEP = System.getProperty( "line.separator" );
	
	@Test
	public void sendNotificationSingleListFile() throws NotificationException, IOException
	{
		File listFile = createListFile( "notifyFtpUnitTest", 3 );
		final boolean listFiles = true;
		ConfigTree config = createConfig( listFiles, true );
		config.setAttribute( "listFiles", String.valueOf( listFiles) );
		MockNotfiyFTPList notifyFTP = new MockNotfiyFTPList( config );
		
		Message message = MessageFactory.getInstance().getMessage();
		message.getBody().add( listFile.getAbsolutePath() );
		
		notifyFTP.sendNotification( message );
		assertEquals( 3, notifyFTP.getSendCounter() );
		
		assertFalse( listFile.exists() );
	}
	
	@Test
	public void sendNotificationList() throws NotificationException, IOException
	{
		File listFile = createListFile( "notifyFtpUnitTest1", 2 );
		File listFile2 = createListFile( "notifyFtpUnitTest2", 2 );
		ArrayList<String> listFiles = new ArrayList<String>();
		listFiles.add( listFile.getAbsolutePath() );
		listFiles.add( listFile2.getAbsolutePath() );
		
		final boolean listFilesProperty = true;
		ConfigTree config = createConfig( listFilesProperty, false );
		config.setAttribute( "listFiles", String.valueOf( listFilesProperty ) );
		MockNotfiyFTPList notifyFTP = new MockNotfiyFTPList( config );
		
		Message message = MessageFactory.getInstance().getMessage();
		message.getBody().add( listFiles );
		
		notifyFTP.sendNotification( message );
		assertEquals( 4, notifyFTP.getSendCounter() );
	}
	
	@Test
	public void sendNotificationSingleFile() throws NotificationException, IOException
	{
		File singleFile = File.createTempFile( "notifyFtpSingleFile", null );
		
		ConfigTree config = createConfig( false, false );
		MockNotfiyFTPList notifyFTP = new MockNotfiyFTPList( config );
		
		Message message = MessageFactory.getInstance().getMessage();
		message.getBody().add( singleFile.getAbsolutePath() );
		
		notifyFTP.sendNotification( message );
		assertEquals( 1, notifyFTP.getSendCounter() );
	}
	
	@Test
	public void sendNotificationSimpleListOfFileNames() throws NotificationException, IOException
	{
		File singleFile = File.createTempFile( "notifyFtpSingleFileInList", null );
		File singleFile2 = File.createTempFile( "notifyFtpSingleFileInList", null );
		ArrayList<String> singleFiles = new ArrayList<String>();
		singleFiles.add( singleFile.getAbsolutePath() );
		singleFiles.add( singleFile2.getAbsolutePath() );
		
		ConfigTree config = createConfig( false, false );
		MockNotfiyFTPList notifyFTP = new MockNotfiyFTPList( config );
		
		Message message = MessageFactory.getInstance().getMessage();
		message.getBody().add( singleFiles );
		
		notifyFTP.sendNotification( message );
		assertEquals( 2, notifyFTP.getSendCounter() );
	}
	
	@Test
	public void getFilesToSend() throws IOException, NotificationException
	{
		final String fileName = "dummy.txt";
		ConfigTree config = createConfig( false, false );
		MockNotfiyFTPList notifyFTP = new MockNotfiyFTPList( config );
		Message message = MessageFactory.getInstance().getMessage();
		message.getBody().add( NotifyFTPList.FILE_LOCATION, new File ( fileName)  );
		File fileToSend = notifyFTP.getFileToSend( message );
		assertEquals( fileName, fileToSend.getName() );
	}
	
	private File createListFile( final String name, final int nrOfFiles ) throws IOException
	{
		File listFile = File.createTempFile( name, ".lst" );
		listFile.deleteOnExit();
		
		BufferedWriter writer = null;
		try
		{
    		writer = new BufferedWriter( new FileWriter ( listFile ) );
    		
			for ( int i = 0 ; i < nrOfFiles ; i++ )
			{
	    		File file = File.createTempFile( "notifyFtpTest-" + i, ".xml" );
    			file.deleteOnExit();
        		writer.write( file.getAbsolutePath() + LINE_SEP );
			}
		}
		finally
		{
			if ( writer != null )
				writer.close();
		}
		
		return listFile;
	}
	
	private static class MockNotfiyFTPList extends NotifyFTPList
	{
		int sendCounter;
		
		public MockNotfiyFTPList(ConfigTree p_op) throws NotificationException
		{
			super( p_op );
		}

		@Override
		protected void send( Message message ) throws NotificationException
		{
			sendCounter++;
		}

		public int getSendCounter()
		{
			return sendCounter;
		}
	}
	
	private ConfigTree createConfig( final boolean listFiles, final boolean deleteListFile )
	{
		ConfigTree config = new ConfigTree( "target" );
		config.setAttribute( "class", "NotifyFTPList" );
		ConfigTree ftp = new ConfigTree( "ftp", config );
		ftp.setAttribute( "URL", "ftp://username:pwd@server.com/remote/dir" );
		if ( listFiles == true )
    		ftp.setAttribute( "listFiles", "true" );
		if ( deleteListFile )
    		ftp.setAttribute( "deleteListFile", "true" );
		return config;
	}
	
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( NotifyFTPListUnitTest.class );
	}
}
