/*
 * RHQ Management Platform
 * Copyright (C) 2005-2008 Red Hat, Inc.
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package org.jbosson.plugins.jbossesb;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mc4j.ems.connection.bean.EmsBean;
import org.mc4j.ems.connection.bean.attribute.EmsAttribute;
import org.rhq.core.domain.configuration.Configuration;
import org.rhq.core.domain.configuration.Property;
import org.rhq.core.domain.measurement.AvailabilityType;
import org.rhq.core.pluginapi.inventory.DiscoveredResourceDetails;
import org.rhq.core.pluginapi.inventory.ResourceContext;
import org.rhq.core.pluginapi.inventory.ResourceDiscoveryContext;
import org.rhq.plugins.jmx.JMXComponent;
import org.rhq.plugins.jmx.MBeanResourceComponent;

/**
 * Discovers ESB actions from a JMX bean
 *
 * @author Tom Cunningham
 * @param <T>
 */
public class ActionDiscoveryComponent<T extends JMXComponent> extends SOADiscoveryComponent {
	
	private static final String SERVICE_NAME = "service-name";
	private static final String MESSAGES_SUCCESSFULLY_PROCESSED = "messages successfully processed count";
	private static final String MESSAGES_FAILED_COUNT = "messages failed count";
	private static final String MESSAGES_PROCESSED_TIME = "processing time";
	private static final String OVERALL_BYTES_PROCESSED = "overall bytes processed";
	private static Log log = LogFactory.getLog(ActionDiscoveryComponent.class);
	
	protected ResourceContext<T> resourceContext;
	
	public void start(ResourceContext<T> context) {
		this.resourceContext = context;
	}
	
	public AvailabilityType getAvailability() {
		JMXComponent parent = resourceContext.getParentResourceComponent();
		
		try {
			return AvailabilityType.UP;
		} catch (NullPointerException npe) {
			log.warn("Could not determine availability of unknown ems bean");
		}
		return AvailabilityType.DOWN;
	}
	
    public Set<DiscoveredResourceDetails> discoverResources(ResourceDiscoveryContext<MBeanResourceComponent> context) {
        Set<DiscoveredResourceDetails> entities = new HashSet<DiscoveredResourceDetails>();

        EmsBean actionBean = context.getParentResourceComponent().getEmsBean();
        String serviceName = actionBean.getBeanName().getKeyProperty(SERVICE_NAME);        
        
        String versionNumber;
        try {
        	Hashtable ht = getVersionAttribute(context);
        	versionNumber = (String) ht.get(VERSION_NUMBER);
        } catch (Exception e) {
        	versionNumber = "";
        }
        
        // Do two passes - one to grab the entity names (ex. "data fileraction"
        // and then a second to build up the metrics for the entity
        Set<EmsAttribute> attributeSet = actionBean.getAttributes();
        for (Iterator i = attributeSet.iterator(); i.hasNext();) {
        	EmsAttribute attr = (EmsAttribute) i.next();
        	if (attr.getName().endsWith(MESSAGES_SUCCESSFULLY_PROCESSED)) {
        		String actionName = attr.getName().replaceAll(MESSAGES_SUCCESSFULLY_PROCESSED, "");
                DiscoveredResourceDetails detail = new DiscoveredResourceDetails(context.getResourceType(), actionName,
                        actionName, null, "Action Entity", null, null);
                
                Configuration pc = detail.getPluginConfiguration();
                ArrayList<Property> props = new ArrayList<Property>();
                detail.setResourceVersion(versionNumber);
                entities.add(detail);
        	}
        }
        
        return entities;
    }
}