/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.qa.teargas.qs;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Map.Entry;
import java.util.Vector;
import org.jboss.soa.esb.qa.teargas.message.generator.AbstractMessageGenerator;
import org.jboss.soa.esb.qa.teargas.message.generator.QSMessageGenerator;
import org.jboss.soa.esb.qa.teargas.message.sender.HTTPSender;
import org.jboss.soa.esb.qa.teargas.message.sender.MessageSender;

/**
 *
 * @author mvecera
 */
public class QSRunner {
   private Vector<String> params = new Vector<String>();
   private Hashtable<String, String> options = new Hashtable<String, String>();
   private Hashtable<String, String> properties = new Hashtable<String, String>();

   private int count;
   private StringBuilder sbMessage = new StringBuilder();
   private String address;
   private MessageSender sender = null;

   public void parseArguments(String[] args) {
      int paramIndex = 0;
      for (int i = 0; i < args.length; i++) {
         if (args[i].startsWith("-")) {
            if (args[i].startsWith("-D")) {
               int loc = args[i].indexOf("=");
               String key = (loc > 0) ? args[i].substring(2, loc) : args[i].substring(2);
               String value = (loc > 0) ? args[i].substring(loc + 1) : "";
               properties.put(key.toLowerCase(), value);
            } else {
               int loc = args[i].indexOf("=");
               String key = (loc > 0) ? args[i].substring(1, loc) : args[i].substring(1);
               String value = (loc > 0) ? args[i].substring(loc + 1) : "";
               options.put(key.toLowerCase(), value);
            }
         } else {
            params.addElement(args[i]);
         }
      }
   }

   private boolean hasOption(String opt) {
      return options.containsKey(opt.toLowerCase());
   }

   private String getOption(String opt) {
      return options.get(opt.toLowerCase());
   }

   public void runTest() throws Exception {
      AbstractMessageGenerator generator = new QSMessageGenerator();
      if (sender == null) {
         sender = new HTTPSender();
      }

      for (Entry<String, String> e: properties.entrySet()) {
         sender.setProperty(e.getKey(), e.getValue());
         generator.setProperty(e.getKey(), e.getValue());
      }

      generator.init(address, sender);
      generator.generate(sbMessage.toString(), count);
      generator.close();
   }

   private void loadMessage(String fileName) throws IOException {
      FileReader fr = new FileReader(fileName);
      BufferedReader br = new BufferedReader(fr);

      String line = "";
      while ((line = br.readLine()) != null) {
         sbMessage.append(line);
      }

      br.close();
      fr.close();
   }

   public boolean validArguments() {
      if (hasOption("message") && hasOption("address") && hasOption("count")) {
         try {
            count = Integer.parseInt(getOption("count"));
         } catch (NumberFormatException e) {
            System.err.println("Option count is not a valid number.");
            return false;
         }

         try {
            loadMessage(getOption("message"));
         } catch (IOException e) {
            System.err.println("Unable to read meesage file.");
            return false;
         }

         address = getOption("address");
         if (address == null || "".equals(address)) {
            System.err.println("Invalid address.");
            return false;
         }
      }

      if (hasOption("sender")) {
         String senderClass = "org.jboss.soa.esb.qa.teargas.message.sender." + getOption("sender");

         try {
            sender = (MessageSender) Class.forName(senderClass, false, this.getClass().getClassLoader()).newInstance();
         } catch (ClassNotFoundException e) {
            System.err.println("Unable to instantiate sender class - class not found: " + senderClass);
            return false;
         } catch (InstantiationException e) {
            System.err.println("Unable to instantiate sender class.");
            return false;
         } catch (IllegalAccessException e) {
            System.err.println("Unable to instantiate sender class.");
            return false;
         }
      }

      return true;
   }

   public static void main(String[] args) throws Exception {
      System.out.println("Teargas testing framework - QS Performance measurement");

      QSRunner qs = new QSRunner();
      qs.parseArguments(args);
      if (!qs.validArguments()) {
         System.out.println("Usage: QSRunTest -address=<WS URL> -message=<message file> -count=<number of messages>");
         System.out.println("Additional properties for message sender can be specified using -D<property_name>=<value>");
      } else {
         qs.runTest();
      }
   }

}
