/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.couriers;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.transport.InVMTemporaryTransport;
import org.jboss.internal.soa.esb.couriers.transport.InVMTransport;
import org.jboss.soa.esb.addressing.eprs.InVMEpr;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.message.Message;


public class InVMCourier implements PickUpOnlyCourier, DeliverOnlyCourier {

    private static Logger logger = Logger.getLogger(InVMCourier.class);

    private final InVMEpr epr;

    /**
     * Objects of this class should only be instantiated by internal
     * implementations
     *
     * @param epr
     */

    public InVMCourier(InVMEpr epr) {
        this.epr = epr;
    }

    /**
     * package the ESB message into the queue. If this is a transactional interaction
     * then the deliver will return immediately, but the message may not go into
     * the queue if the transaction subsequently rolls back. The caller must monitor
     * the transaction outcome to determine the fate of the message. For example, register
     * a Synchronization.
     *
     * @param  message - the message to deliverAsync
     * @return boolean - the result of the delivery
     * @throws CourierException -
     *                          if problems were encountered
     */
    
    public boolean deliver(Message message) throws CourierException {
        if (message == null) {
            return false;
        }

        try
        {
            if (epr.isTemporaryEPR()) {
                InVMTemporaryTransport.getInstance().deliver(epr, message) ;
            } else {
                InVMTransport.getInstance().deliver(epr, message) ;
            }
            
            return true;
        }
        catch (final Throwable ex)
        {
            logger.warn("InVMCourier delivery caught: "+ex);
            logger.debug("InVMCourier delivery caught", ex);
            
            return false;
        }
    }

    /**
     * Get a mesage from the queue or wait for the specified time in case one
     * arrives.
     * 
     * If this is a transactional interaction then the message will be placed back
     * on the queue if the enclosing transaction rolls back. Note that for performance
     * reasons it is not guaranteed that the message will go back at the same relative
     * position.
     * 
     * @param millis The time to wait if the queue is empty.
     * @return a Message or <code>null</code> if there is nothing on the queue.
     */
    
    // see associated test
    
    public Message pickup(long millis) throws CourierException {
        try
        {
            if (epr.isTemporaryEPR()) {
                return InVMTemporaryTransport.getInstance().pickup(epr, millis) ;
            } else {
                return InVMTransport.getInstance().pickup(epr, millis) ;
            }
        }
        catch (final Throwable ex)
        {
            logger.warn("InVMCourier pickup caught: "+ex);
            logger.debug("InVMCourier pickup caught", ex);
            
            return null;
        }
    }

    public void cleanup() {
    }
}
