/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.internal.soa.esb.dependencies;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.URL;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.registry.server.ESBInVMServerTransport;
import org.jboss.internal.soa.esb.util.PropertiesHelper;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.system.ServiceMBeanSupport;
import org.apache.juddi.Registry;

public class JuddiRMIService extends ServiceMBeanSupport implements JuddiRMIServiceMBean
{
   private Logger logger = Logger.getLogger(this.getClass());
   
   private static final String DEFAULT_PROPERTIES_RESOURCE_FILE = "esb.juddi.xml";
   private static final String JUDDI_PROPERTIES_RESOURCE_FILE = "esb.juddi.properties";
   private static final String JUDDI_PROPERTY_FILE_COMMENTS = "Auto generated property file, do not edit" ;

   private String propertiesResourceFile ;
   private String propertiesFileDir;
   private int numThreads = 20 ;
   private long timeout = 20 ;
   
   private ESBInVMServerTransport inVMServerTransport ;

   public String getPropertiesResourceFile()
   {
      return propertiesResourceFile;
   }

   public void setPropertiesResourceFile(String propertiesResourceFile)
   {
      this.propertiesResourceFile = propertiesResourceFile;
   }
   
   public int getNumThreads()
   {
      return numThreads;
   }

   public void setNumThreads(final int numThreads)
   {
      this.numThreads = numThreads;
   }
   
   public long getTimeout()
   {
      return timeout;
   }

   public void setTimeout(final long timeout)
   {
      this.timeout = timeout;
   }
   
   protected void startService() throws Exception
	{
		logger.info("starting juddi RMI service");
		final String propertiesResourceFileVal ;
        if (propertiesResourceFile == null)
        {
            propertiesResourceFileVal = DEFAULT_PROPERTIES_RESOURCE_FILE ;
        }
        else
        {
            propertiesResourceFileVal = propertiesResourceFile ;
        }
        final File baseFile = new File(propertiesResourceFileVal) ;

        final InputStream xmlPropertyIS ;
        if (baseFile.isAbsolute())
        {
            xmlPropertyIS = new FileInputStream(baseFile);
        }
        else
        {
            final URL url = Thread.currentThread().getContextClassLoader().getResource(propertiesResourceFileVal);
            xmlPropertyIS = url.openStream() ;
        }
        final byte[] propertyFileContents ;
        try {
            final ByteArrayOutputStream baos = new ByteArrayOutputStream() ;
            XMLHelper.replaceSystemProperties(XMLHelper.getXMLStreamReader(xmlPropertyIS),
                XMLHelper.getXMLStreamWriter(baos)) ;
            propertyFileContents = baos.toByteArray() ;
        } finally {
            xmlPropertyIS.close() ;
        }
        
        if (propertiesFileDir == null)
        {
            final String errorMsg = String.format("No property named '%s' was configured in jbossesb.sar/META-INF/jboss-service.xml for %s", "propertiesFileDir", getClass().getName());
            throw new IllegalArgumentException(errorMsg);
        }
        
        File dataDir = new File(propertiesFileDir);
        if (!dataDir.exists())
        {
            final String errorMsg = String.format("The directory configured for %s='%s' does not exist.", "warFilesDir", dataDir);
            throw new FileNotFoundException(errorMsg);
        }
        final File juddiPropertyFile = new File(dataDir, JUDDI_PROPERTIES_RESOURCE_FILE) ;
        
        final ByteArrayInputStream bais = new ByteArrayInputStream(propertyFileContents) ;
        final FileOutputStream juddiPropertyOS = new FileOutputStream(juddiPropertyFile) ;
        try {
            PropertiesHelper.translateXMLToText(bais, juddiPropertyOS, JUDDI_PROPERTY_FILE_COMMENTS) ;
        } finally {
            juddiPropertyOS.close() ;
        }
        
        System.setProperty("juddi.propertiesFile", juddiPropertyFile.getAbsoluteFile().toURI().toString());
        
        Registry.start();
        
        try
        {
            final ESBInVMServerTransport inVMServerTransport = new ESBInVMServerTransport() ;
            inVMServerTransport.start(numThreads) ;
            this.inVMServerTransport = inVMServerTransport ;
        }
        finally
        {
            if (inVMServerTransport == null)
            {
                Registry.stop();
            }
        }
	}

	protected void stopService() throws Exception
	{
		try
		{
			inVMServerTransport.stop(timeout) ;
		}
		finally
		{
			inVMServerTransport = null ;
			logger.info("Unbinding juddi services");
			Registry.stop();
		}
	}

    public void setPropertiesFileDir(final String directory)
    {
        this.propertiesFileDir = directory;
    }
}
