/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.listeners.war;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.internal.soa.esb.assertion.AssertArgument;

import java.util.*;

/**
 * JEE WAR deployment model.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class WebModel {

    private String esbName;
    private String warName;
    private List<Filter> filters = new ArrayList<Filter>();
    private List<Servlet> servlets = new ArrayList<Servlet>();
    private Set<String> securityRoles = new LinkedHashSet<String>();
    private String authMethod;
    private String authDomain;

    public WebModel(String esbName) {
        AssertArgument.isNotNullAndNotEmpty(esbName, "esbName");
        this.esbName = getESBDeploymentName(esbName);
        this.warName = esbName;
    }

    public WebModel(String esbName, String warName) {
        AssertArgument.isNotNullAndNotEmpty(esbName, "esbName");
        AssertArgument.isNotNullAndNotEmpty(warName, "warName");
        this.esbName = esbName;
        this.warName = warName;
    }
    
    private String getESBDeploymentName(String deploymentName)
    {
        final String esbName = deploymentName;
        final int lastSeparator = esbName.lastIndexOf('.');
        return ((lastSeparator >= 0) ? esbName.substring(0, lastSeparator) : esbName);
    }

    
    public String getEsbName() {
        return esbName;
    }

    public String getWarName() {
        return warName;
    }
    
    public List<Filter> getFilters() {
        return filters;
    }

    public List<Servlet> getServlets() {
        return servlets;
    }

    public Set<String> getSecurityRoles() {
        return securityRoles;
    }

    public String getAuthMethod() {
        if(authMethod == null && (authDomain != null || !securityRoles.isEmpty())) {
            // Default the method to "BASIC" if there's an authDomain,
            // or role set, specified....
            return "BASIC";
        }

        return authMethod;
    }

    public void setAuthMethod(String authMethod) throws ConfigurationException {
        if(authMethod == null) {
            // Just ignore it...
            return;
        }

        // If it has already been set, create an error if the incoming value is something
        // other than the current value?
        if(this.authMethod != null && !this.authMethod.equals(authMethod)) {
            throw new ConfigurationException("The authentication method for the web deployment associated with this deployment has already been set to '" + this.authMethod + "'.  Cannot reset to '" + authMethod + "'.");
        }

        this.authMethod = authMethod;
    }

    public String getAuthDomain() {
        return authDomain;
    }

    public void setAuthDomain(String authDomain) throws ConfigurationException {
        if(authDomain == null) {
            // Just ignore it...
            return;
        }

        if(!authDomain.startsWith("java:/jaas/")) {
            authDomain = "java:/jaas/" + authDomain;
        }

        // If it has already been set, create an error if the incoming value is something
        // other than the current value?
        if(this.authDomain != null && !this.authDomain.equals(authDomain)) {
            throw new ConfigurationException("The authentication domain for the web deployment associated with this deployment has already been set to '" + this.authDomain + "'.  Cannot reset to '" + authDomain + "'.");
        }

        this.authDomain = authDomain;
    }
}
