/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2009
 */
package org.jboss.soa.esb.actions.routing.http;

import java.io.IOException;
import java.net.SocketException;
import java.net.URL;

import org.apache.commons.httpclient.DefaultHttpMethodRetryHandler;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpMethodRetryHandler;
import org.apache.commons.httpclient.params.HttpMethodParams;
import org.jboss.soa.esb.Configurable;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * AbstractHttpMethodFactory.
 * 
 * @author dward at jboss.org
 */
public abstract class AbstractHttpMethodFactory implements HttpMethodFactory
{
    public static final String RETRY_HANDLER = "org.jboss.soa.esb.actions.routing.http.retryHandler";
	
	private URL endpoint;
	private Class<?> retryHandlerClass ;
	private boolean isConfigurable ;
	private ConfigTree config ;

    public void setEndpoint(URL endpoint)
    {
    	this.endpoint = endpoint;
    }
    
    protected String getEndpointPathAndQuery()
    {
    	StringBuilder sb = new StringBuilder();
    	String path = endpoint.getPath();
    	sb.append("".equals(path) ? "/" : path);
    	String query = endpoint.getQuery();
    	if (query != null)
    	{
    		sb.append("?");
    		sb.append(query);
    	}
    	return sb.toString();
    }
    
    public void setConfiguration(ConfigTree config)
        throws ConfigurationException
    {
        this.config = config ;
        final String retryHandler = config.getAttribute(RETRY_HANDLER) ;
        if (retryHandler != null)
        {
            final Class<?> retryHandlerClass ;
            try
            {
                retryHandlerClass = ClassUtil.forName(retryHandler, getClass()) ;
            }
            catch (final ClassNotFoundException cnfe)
            {
                throw new ConfigurationException("Could not locate Retry Handler: " + retryHandler, cnfe) ;
            }
            if (!HttpMethodRetryHandler.class.isAssignableFrom(retryHandlerClass))
            {
                throw new ConfigurationException("Retry Handler does not implement HttpMethodRetryHandler: " + retryHandler) ;
            }
            try
            {
                retryHandlerClass.getConstructor() ;
            }
            catch (final NoSuchMethodException nsme)
            {
                throw new ConfigurationException("Retry Handler does not have a default constructor: " + retryHandler, nsme) ;
            }
            this.retryHandlerClass = retryHandlerClass ;
            isConfigurable = Configurable.class.isAssignableFrom(retryHandlerClass) ;
        }
    }
    
    protected void initialiseRetryHandler(HttpMethod method) throws IOException {
    	HttpMethodParams params = method.getParams();
    	if (retryHandlerClass != null) {
    	    final Object handler ;
    	    try {
    	        handler = retryHandlerClass.newInstance() ;
    	    } catch (final Throwable th) {
    	        final IOException ioe = new IOException("Failed to instantiate Retry Handler") ;
    	        ioe.initCause(th) ;
    	        throw ioe ;
    	    }
    	    if (isConfigurable) {
    	        final Configurable configurable = Configurable.class.cast(handler) ;
    	        try {
    	            configurable.setConfiguration(config) ;
    	        } catch (final ConfigurationException ce) {
    	            final IOException ioe = new IOException("Failed to configure Retry Handler") ;
    	            ioe.initCause(ce) ;
    	            throw ioe ;
    	        }
    	    }
    	    params.setParameter(HttpMethodParams.RETRY_HANDLER, handler);
    	} else {
    	    HttpMethodRetryHandler other = (HttpMethodRetryHandler)params.getParameter(HttpMethodParams.RETRY_HANDLER);
    	    params.setParameter(HttpMethodParams.RETRY_HANDLER, new HttpMethodRetryHandlerWrapper(other));
    	}
    }
    
    static class HttpMethodRetryHandlerWrapper implements HttpMethodRetryHandler {
    	
    	private HttpMethodRetryHandler other;
    	
    	private HttpMethodRetryHandlerWrapper(HttpMethodRetryHandler other) {
    		this.other = (other != null ? other : new DefaultHttpMethodRetryHandler());
    	}
    	
    	public boolean retryMethod(HttpMethod method, IOException exception, int executionCount) {
    		boolean retry = other.retryMethod(method, exception, executionCount);
    		if (retry && exception instanceof SocketException) {
    			retry = false;
    		}
    		return retry;
    	}
    	
    }
}
