/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.http.protocol;

import org.apache.commons.httpclient.ConnectTimeoutException;
import org.apache.commons.httpclient.params.HttpConnectionParams;
import org.apache.commons.httpclient.protocol.ProtocolSocketFactory;
import org.apache.commons.httpclient.protocol.SecureProtocolSocketFactory;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.util.ssl.SSLUtil;
import org.jboss.soa.esb.util.ssl.SelfSignedTrustManager;
import org.jboss.internal.soa.esb.assertion.AssertArgument;

import javax.net.SocketFactory;
import javax.net.ssl.KeyManager;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import java.io.IOException;
import java.net.*;
import java.security.*;

/**
 * Builder for creating a ProtocolSocketFactory that supports Self-Signed Server
 * certificates.
 * <p/>
 * <b>For test/demo purposes only.</b>
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class SelfSignedSSLProtocolSocketFactoryBuilder extends AbstractProtocolSocketFactoryBuilder {

    public ProtocolSocketFactory newInstance() throws ConfigurationException {
        try {
            return new SelfSignedSSLProtocolSocketFactory(getKeystore(), getPasswordFromFile(getKeystorePassword()), getTruststore());
        } catch (NoSuchAlgorithmException e) {
            throw new ConfigurationException("Failed to create SelfSignedSSLProtocolSocketFactory.", e);
        } catch (KeyStoreException e) {
            throw new ConfigurationException("Failed to create SelfSignedSSLProtocolSocketFactory.", e);
        } catch (UnrecoverableKeyException e) {
            throw new ConfigurationException("Failed to create SelfSignedSSLProtocolSocketFactory.", e);
        }
    }

    private static class SelfSignedSSLProtocolSocketFactory implements SecureProtocolSocketFactory {

        private SSLContext sslContext;

        public SelfSignedSSLProtocolSocketFactory(KeyStore keystore, String keystorePassword, KeyStore truststore) throws NoSuchAlgorithmException, KeyStoreException, ConfigurationException, UnrecoverableKeyException {
            KeyManager[] keyManagers = SSLUtil.getKeyManagers(keystore, keystorePassword);
            SelfSignedTrustManager trustManager = new SelfSignedTrustManager(truststore);

            sslContext = SSLContext.getInstance("SSL");
            try {
                sslContext.init(keyManagers, new TrustManager[]{trustManager}, null);
            } catch (KeyManagementException e) {
                throw new ConfigurationException("Failed to initialize SSL Context.", e);
            }
        }

        public Socket createSocket(Socket socket, String host, int port, boolean autoClose) throws IOException, UnknownHostException {
            return sslContext.getSocketFactory().createSocket(socket, host, port, autoClose);
        }

        public Socket createSocket(String host, int port, InetAddress clientHost, int clientPort) throws IOException {
            return sslContext.getSocketFactory().createSocket(host, port, clientHost, clientPort);
        }

        public Socket createSocket(String host, int port, InetAddress localAddress, int localPort, HttpConnectionParams params) throws IOException, ConnectTimeoutException {
            AssertArgument.isNotNull(params, "params");

            SocketFactory socketfactory = sslContext.getSocketFactory();
            int connectTimeout = params.getConnectionTimeout();

            if (connectTimeout == 0) {
                return socketfactory.createSocket(host, port, localAddress, localPort);
            } else {
                Socket socket = socketfactory.createSocket();
                SocketAddress localSocketAddress = new InetSocketAddress(localAddress, localPort);
                SocketAddress remoteSocketAddress = new InetSocketAddress(host, port);
                socket.bind(localSocketAddress);
                socket.connect(remoteSocketAddress, connectTimeout);

                return socket;
            }
        }

        public Socket createSocket(String host, int port) throws IOException {
            return sslContext.getSocketFactory().createSocket(host, port);
        }
    }
}