/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners.gateway;

import java.io.File;
import java.io.IOException;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.FTPEpr;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.ListenerUtil;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.util.FtpClientUtil;
import org.jboss.soa.esb.util.RemoteFileSystem;
import org.jboss.soa.esb.util.RemoteFileSystemException;
import org.jboss.soa.esb.util.RemoteFileSystemFactory;

/**
 * Sample Listener Configuration:
 * <pre>
 * &lt;ftp-listener name="FtpGateway"
 *  busidref="helloFTPChannel"
 *   maxThreads="1"
 *    is-gateway="true"&gt;
 *     &lt;property name="pollLatencySeconds" value="5"/&gt; &lt;!-- Interval to poll the remote file system --&gt;
 *&lt;/ftp-listener&gt;
 * </pre>
 * 
 * @author John Doe				
 * @author Daniel Bevenius				
 *
 */
public class RemoteGatewayListener extends FileGatewayListener
{
    public RemoteGatewayListener(ConfigTree config) throws ConfigurationException, RegistryException, GatewayException {
		super(config);
        checkMyParms() ;
	}

	protected void checkMyParms() throws ConfigurationException, RegistryException, GatewayException
	{
		try
		{
            EPR epr = ListenerUtil.assembleEpr(config);
			if (epr instanceof FTPEpr)  {
				FTPEpr ftpEpr = (FTPEpr) epr;
				RemoteFileSystem rfs = RemoteFileSystemFactory.getRemoteFileSystem(ftpEpr, true);
				rfs.quit();
			} else {
				throw new GatewayException("This Gateway only accepts FTP and SFTP.");
			}
		}
		catch (RemoteFileSystemException ex)
		{
			throw new ConfigurationException(ex);
		}
		
	}

	@Override
	protected void seeIfOkToWorkOnDir(File p_oDir) {
		// TODO: Implement. Very expensive though.
		// It is possible to check for existence and permission by connecting,
		// cd to target, put dummy file, ls to see list it, rm to clean it
		// p_oDir exists 
		// p_oDir writable 
		// p_oDir readable 
	}

	@Override
	public boolean deleteFile(File file) throws GatewayException
	{
		RemoteFileSystem rfs = null;
		try
		{
			EPR epr = ListenerUtil.assembleEpr(config);
			if (epr instanceof FTPEpr)  {
				FTPEpr ftpEpr = (FTPEpr) epr;
				rfs = RemoteFileSystemFactory.getRemoteFileSystem(ftpEpr, true);
				rfs.setRemoteDir(FtpClientUtil.fileToFtpString(_inputDirectory));
				rfs.deleteRemoteFile(file.toString());
				return true;
			} else {
				throw new GatewayException("This Gateway only accepts FTP and SFTP.");
			}
			
		} catch (Exception e)
		{
			throw new GatewayException(e);
		} finally
		{
			if (null != rfs)
				rfs.quit();
		}
	}

    @Override
	byte[] getFileContents(File file) throws GatewayException
	{
		RemoteFileSystem rfs = null;
		try
		{
			File temp = File.createTempFile("FTPdown", ".tmp");
			EPR epr = ListenerUtil.assembleEpr(config);
			if (epr instanceof FTPEpr)  {
				FTPEpr ftpEpr = (FTPEpr) epr;
				rfs = RemoteFileSystemFactory.getRemoteFileSystem(ftpEpr, true);
				rfs.setRemoteDir(FtpClientUtil.fileToFtpString(_inputDirectory));
				rfs.downloadFile(file.toString(), temp.getAbsolutePath());
				final byte[] contents = super.getFileContents(temp);
				temp.delete() ;
				return contents ;
			} else {
				throw new GatewayException("This Gateway only accepts FTP and SFTP.");
			}
			
		} catch (Exception e)
		{
			throw new GatewayException(e);
		} finally
		{
			if (null != rfs)
				rfs.quit();
		}
	}

    @Override
    String getDefaultComposer() {
        return RemoteFileMessageComposer.class.getName();
    }

    @Override
	File[] getFileList() throws GatewayException
	{
		RemoteFileSystem rfs = null;
		File[] oaRet = null;
		try
		{
			EPR epr = ListenerUtil.assembleEpr(config);
			if (epr instanceof FTPEpr)  {
				FTPEpr ftpEpr = (FTPEpr) epr;
				rfs = RemoteFileSystemFactory.getRemoteFileSystem(ftpEpr, true);
				rfs.setRemoteDir(FtpClientUtil.fileToFtpString(_inputDirectory));
				String[] sa = rfs.getFileListFromRemoteDir(_inputSuffix);
				oaRet = new File[(null == sa) ? 0 : sa.length];
				int i1 = 0;
				if (null != sa)
					for (String sCurr : sa)
						oaRet[i1++] = new File(sCurr);
			} else {
				throw new GatewayException("This Gateway only accepts FTP and SFTP.");
			}
		} catch (Exception e){
			throw new GatewayException(e);
		} finally {
			if (null != rfs)
				rfs.quit();
		}
		return oaRet;
	}

	@Override
	boolean renameFile(File from, File to) throws GatewayException
	{
		RemoteFileSystem rfs = null;
		try
		{
			EPR epr = ListenerUtil.assembleEpr(config);
			if (epr instanceof FTPEpr)  {
				FTPEpr ftpEpr = (FTPEpr) epr;
				rfs = RemoteFileSystemFactory.getRemoteFileSystem(ftpEpr, true);
				rfs.setRemoteDir(FtpClientUtil.fileToFtpString(_inputDirectory));
				rfs.remoteRename( from, to );
				return true;
			} else {
				throw new GatewayException("This Gateway only accepts FTP and SFTP.");
			}
		} catch (Exception e) {
			throw new GatewayException(e);
		} finally {
			if (null != rfs)
				rfs.quit();
		}
	}
	
	@Override
	protected File getWorkFileName(File file, String suffix )
	{
		return new File( file.toString() + suffix);
	}
	
	@Override
	void bytesToFile(byte[] bytes, File file) throws GatewayException 
	{
		RemoteFileSystem rfs = null;
		try
		{
			EPR epr = ListenerUtil.assembleEpr(config);
			if (!(epr instanceof FTPEpr))  
				throw new Exception("This Gateway only accepts FTP and SFTP.");

			FTPEpr ftpEpr = (FTPEpr) epr;
			rfs = RemoteFileSystemFactory.getRemoteFileSystem(ftpEpr, true);
			rfs.setRemoteDir(FtpClientUtil.fileToFtpString(_inputDirectory));

			File tmpFile = File.createTempFile("RemoteGW", ".forUpload");
			super.bytesToFile(bytes, tmpFile);
			rfs.uploadFile(tmpFile, file.getName());
			tmpFile.delete();

		} catch (Exception e) {
			throw new GatewayException(e);
		} finally {
			if (null != rfs)
				rfs.quit();
		}
	}
	
}
