/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.lifecycle;

import java.util.List;
import java.util.SortedSet;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanParameterInfo;
import javax.management.ReflectionException;

/**
 * ThreadedLifecycleController is an MBean implementation that
 * extends the base LifecycleController to enable alteration of the
 * thread pool. 
 *
 * @author <a href="Kevin.Conner@jboss.com">Kevin Conner</a>
 */
public class ThreadedLifecycleController extends LifecycleController
{
    private ThreadedManagedLifecycleAdapter m_threadedLifecycle ;
    
    public static final String SET_MINIMUM_THREAD_POOL_COUNT_ACTION = "setMinimumThreadPoolCount" ;
    public static final String SET_MAXIMUM_THREAD_POOL_COUNT_ACTION = "setMaximumThreadPoolCount" ;
    
    public static final String MINIMUM_THREAD_POOL_COUNT_ATTRIB = "minimumThreadPoolCount" ;
    public static final String MAXIMUM_THREAD_POOL_COUNT_ATTRIB = "maximumThreadPoolCount" ;
        
    /**
     * Constructor using lifecycle and config tree.
     * @param f_lifecycle lifecycle
     * @param f_configtree config tree
     */
    public ThreadedLifecycleController(final ThreadedManagedLifecycleAdapter f_threadedLifecycle)
    {
        super(f_threadedLifecycle) ;
        m_threadedLifecycle = f_threadedLifecycle ;
    }
    
    /**
     * Lifecycle mutator.
     * @param f_mla lifecycle
     */
    public void setLifecycle(final ManagedLifecycleAdapter f_mla)
    {
        if (f_mla instanceof ThreadedManagedLifecycleAdapter)
        {
            m_threadedLifecycle = (ThreadedManagedLifecycleAdapter)f_mla ;
            super.setLifecycle(f_mla) ;
        }
        else
        {
            throw new IllegalArgumentException("Invalid managed lifecycle adapter : " + f_mla) ;
        }
    }

    /**
     * Get the list of attributes.
     */
    public AttributeList getAttributes(final String[] arg0)
    {
        final AttributeList attributeList = super.getAttributes(arg0) ;
        
        final Attribute minimumThreadPoolCount = new Attribute(MINIMUM_THREAD_POOL_COUNT_ATTRIB, m_threadedLifecycle.getMinimumThreadPoolCount()) ;
        attributeList.add(minimumThreadPoolCount) ;

        final Attribute maximumThreadPoolCount = new Attribute(MAXIMUM_THREAD_POOL_COUNT_ATTRIB, m_threadedLifecycle.getMaximumThreadPoolCount()) ;
        attributeList.add(maximumThreadPoolCount) ;

        return attributeList ;
    }

    /**
     * Gets the attribute value.   
     */
    public synchronized String getAttribute(final String name)
        throws AttributeNotFoundException
    {
        final String value ;
        if (MINIMUM_THREAD_POOL_COUNT_ATTRIB.equals(name))
        {
            value = Integer.toString(m_threadedLifecycle.getMinimumThreadPoolCount()) ;
        }
        else if (MAXIMUM_THREAD_POOL_COUNT_ATTRIB.equals(name))
        {
            value = Integer.toString(m_threadedLifecycle.getMaximumThreadPoolCount()) ;
        }
        else
        {
            value = super.getAttribute(name) ;
        }
        return value ;
    }
    /**
     * Get the list of attribute names
     * @return The list of attribute names
     */
    protected SortedSet<String> getAttributeNames()
    {
        final SortedSet<String> names = super.getAttributeNames() ;
        names.add(MINIMUM_THREAD_POOL_COUNT_ATTRIB) ;
        names.add(MAXIMUM_THREAD_POOL_COUNT_ATTRIB) ;
        return names ;
    }
    
    /**
     * Get the list of operations
     * @return The list of operations
     */
    protected List<MBeanOperationInfo> getOperations()
    {
        final List<MBeanOperationInfo> opers = super.getOperations() ;
        opers.add(new MBeanOperationInfo(
            SET_MINIMUM_THREAD_POOL_COUNT_ACTION,
            "Set the minimum thread pool size",
            new MBeanParameterInfo[] { new MBeanParameterInfo(
                MINIMUM_THREAD_POOL_COUNT_ATTRIB, "int", "Minimum thread pool count") },
            "void", MBeanOperationInfo.ACTION)) ;
        opers.add(new MBeanOperationInfo(
            SET_MAXIMUM_THREAD_POOL_COUNT_ACTION,
            "Set the maximum thread pool size",
            new MBeanParameterInfo[] { new MBeanParameterInfo(
                MAXIMUM_THREAD_POOL_COUNT_ATTRIB, "int", "Maximum thread pool count") },
            "void", MBeanOperationInfo.ACTION)) ;
        return opers ;
    }

    /**
     * Invoke the operations on the controller
     */
    protected Object invokeOperation(final String method, final Object[] arg1, final Object[] arg2)
        throws ReflectionException
    {
        if (SET_MINIMUM_THREAD_POOL_COUNT_ACTION.equals(method))
        {
            if ((arg1 == null) || (arg1.length != 1) || !(arg1[0] instanceof Integer))
            {
                return "Invalid parameter to " + SET_MINIMUM_THREAD_POOL_COUNT_ACTION + " action" ;
            }
            final Integer intVal = (Integer) arg1[0] ;
            m_threadedLifecycle.setMinimumThreadPoolCount(intVal.intValue()) ;
            return "Invoking the " + method + " on the lifecycle." ;
        }
        else if (SET_MAXIMUM_THREAD_POOL_COUNT_ACTION.equals(method))
        {
            if ((arg1 == null) || (arg1.length != 1) || !(arg1[0] instanceof Integer))
            {
                return "Invalid parameter to " + SET_MAXIMUM_THREAD_POOL_COUNT_ACTION + " action" ;
            }
            final Integer intVal = (Integer) arg1[0] ;
            m_threadedLifecycle.setMaximumThreadPoolCount(intVal.intValue()) ;
            return "Invoking the " + method + " on the lifecycle." ;
        }
        else
        {
            return super.invokeOperation(method, arg1, arg2) ;
        }
    }
}
