/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2007
 */
package org.jboss.soa.esb.message.mapping;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.message.Body;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.mvel2.MVEL;

/**
 * Extracts objects from an ESB message and puts them into a Map, which can be
 * used for later processsing, based on expressions. This class is used internally
 * within the jBPM and Rules integrations. It may be useful elsewhere.
 * 
 * Likewise, it can be used to take a list of Objects and place them into a Message
 * based on the same type of expressions.
 * 
 * It is based on the notion of an ESB Message Object Path (MOP). The path
 * should follow the syntax:
 * 
 * location.objectname.<bean>...
 * 
 * location : one of [body, property, attachment]
 * objectname: name of the object. Since attachments can be named or numbered, this can be a
 * number too and will be converted to an integer value. If a stringified number is used elsewhere
 * to name an object then it will not be interpreted as an integer, i.e., it will remain as a string.
 * The objectname can be hierarchical if required, but should be quoted in that case, e.g., 'foo.bar.a.b.c'.
 * <bean> : optionally you traverse a bean graph by specifying bean names.
 * 
 * examples : 
 * properties.Order, gets the property object named "Order"
 * attachment.1, gets the first attachment Object 
 * attachment.FirstAttachment, gets the attachment named 'FirstAttachment'
 * body.foo, gets the named Object foo
 * body.1 gets the named Object 1
 * body.'foo.bar.a.b.c' looks for the Object named foo.bar.a.b.c in the body
 * 
 * @author kurt.stam@jboss.com
 * 
 */

public class ObjectMapper 
{
    /** Name to get the byte[] content */
    public static final String BODY_CONTENT = "BODY_CONTENT";
    private Logger logger = Logger.getLogger(this.getClass());
    
    /**
     * The quote used to delimit names.
     */
    private static final char QUOTE = '\'' ;
    /**
     * The escape character used to within names.
     */
    private static final char ESCAPE = '\\' ;
    /**
     * The expression separator character.
     */
    private static final char EXPRESSION_SEPARATOR = '.' ;

    private MessagePayloadProxy payloadProxy;
    private static final String[] legacyGetPayloadLocations = new String[] {BytesBody.BYTES_LOCATION};
    private static final String[] legacySetPayloadLocations = new String[] {BytesBody.BYTES_LOCATION};

    public ObjectMapper() {
        // Unconfigurable payload proxy...
        this.payloadProxy = new MessagePayloadProxy(new ConfigTree("null-config"),
                                                    legacyGetPayloadLocations,
                                                    legacySetPayloadLocations);
        payloadProxy.setNullGetPayloadHandling(MessagePayloadProxy.NullPayloadHandling.NONE);
    }

    public ObjectMapper(ConfigTree config) {
        this.payloadProxy = new MessagePayloadProxy(config,
                                                    legacyGetPayloadLocations,
                                                    legacySetPayloadLocations);
    }

    /**
     * Given a message and a series of MOPs, traverse the content and create
     * a list of Objects.
     * 
     * For example:
     * 
     * body.myObject MOP will add 'myObject', while
     * body.myObject.mySubObject MOP will create an entry of 'mySubObject'. 
     * body.'myObject.mySubObject' MOP will create an entry of 'myObject.mySubject' in body.
     * 
     * in the returned List.
     * 
     * @param message - the message 
     * @param messageObjectPathList - the key represents the query to get the object from
     *                      from the message.
     *                      
     * @return List of Objects pulled from the input Message. Will never be null, but could be
     * of zero length.
     * 
     * @throws ObjectMappingException
     */
    public List<Object> createObjectList (Message message,List<String> messageObjectPathList)
        throws ObjectMappingException
    {
        List<Object> objectList = new ArrayList<Object>();
        if (messageObjectPathList!=null) {
            for (String messageObjectPath: messageObjectPathList)
            {   
                Object value = getObjectFromMessage(message, messageObjectPath);
                if (value==null) {
                    logger.warn("The value of " + messageObjectPath + " is null");
                } else {
                    if (value instanceof Collection) {
                        Collection valuesList = (Collection) value;
                        for (Object object : valuesList) {
                            objectList.add(object);
                        }
                    } else if (value instanceof Map) {
                        Map valuesMap = (Map) value;
                        for (Object object : valuesMap.entrySet()) {
                            objectList.add(object);
                        }
                    } else {
                        objectList.add(value);
                    }
                }
            }
        }
        return objectList;
    }
    /**
     * Set objects on the message using a OGNL expression syntax to describe the position
     * in the message where the object needs to be attached.
     * 
     * Each Object within the supplied Map should be associated with an expression that
     * will be used to determine where in the Message it should be placed.
     * 
     * location.objectname...
     * 
     * location  : one of [body, property, attachment, header]
     * objectname: name of the object name, attachments can be named or numbered, so for
     *             attachments this can be a number too.
     * 
     * @param message - the message on which the objects will be placed
     * @param expressionAndObject map containing objects with their expression
     * @return the message with the objects attached.
     * 
     * @throws ObjectMappingException
     */
    public Message setObjectsOnMessage (Message message,Map<Object, String> expressionAndObject)
        throws ObjectMappingException
    {
        if (expressionAndObject!=null) {
            for (Map.Entry<Object, String> entry: expressionAndObject.entrySet())
            {
                setObjectOnMessage(message, entry.getValue(), entry.getKey());
            }
        }
        return message;
    }   
    
    public int foundExpression(String value) {
		int startIndex = value.indexOf('{');
		if (startIndex == -1) {
			return -2;
		}
		int endIndex = value.indexOf('}');
		if (endIndex == -1) {
			return -2;
		}
		if (startIndex >= 0 && endIndex > 0) {
			return startIndex;
		} else {
			return -2;
		}
    }
    
    public String parseExpression (String value) {
		int startIndex = value.indexOf('{');
		if (startIndex == -1) {
			return value;
		}
		int endIndex = value.indexOf('}');
		if (endIndex == -1) {
			return value;
		}
		String propName = value.substring(startIndex + 1, endIndex);
		return propName;
    }
    
    /**
     * Extracts objects from the message, using a ESB Message Object Path. The
     * path should follow the syntax:
     * 
     * location.objectname.<bean>...
     * 
     * location  : one of [body, property, attachment]
     * objectname: name of the object name, attachments can be named or numbered, so for
     *             attachments this can be a number too.
     * <bean>    : optionally you traverse a bean graph by specifying bean names;
     * 
     *  examples : 
     *  properties.Order, gets the property object named "Order"
     *  attachment.1, gets the first attachment Object
     *  attachment.FirstAttachment, gets the attachment named 'FirstAttachment'
     *  attachment.1.Order, calls getOrder() on the attached Object.
     *  body.BODY_CONTENT, gets the byte[] of the body.
     *  body.Order1.lineitem, obtains the object named "Order1"
     *  from the body of the message. Next it will call getLineitem() on this object.
     *  
     *  More <bean> elements can be added to the query to traverse the bean graph.
     * 
     * @param message - an ESB Message.
     * @param expression - path to the object.
     * @return Object obtained from the message.
     * 
     * @throws ObjectMappingException
     */    
    public Object getObjectFromMessage(Message message, String expression) 
        throws ObjectMappingException
    {    	
        Object object=null;
        final String[] path = getExpressionPath(expression) ;
        if (path.length == 0)
        {
            throw new ObjectMappingException(expression + " should start with [<location>.]<name>") ;
        }
        String location = path[0];
        String name     = path[1];
        if ("body".equalsIgnoreCase(location)) {
            Body body   = message.getBody();
            try {
                object =
                    BODY_CONTENT.equals(name)
                    ? payloadProxy.getPayload(message)
                    : body.get(name);
            } catch (MessageDeliverException e) {
                throw new ObjectMappingException("Unable to get payload '" + name + "' from message.", e);
            }
        } else if ("properties".equalsIgnoreCase(location)) {
            object = message.getProperties().getProperty(name);
        } else if ("attachment".equalsIgnoreCase(location)) {
            if (isNumeric(name)) {
                int index = Integer.valueOf(name);
                object = message.getAttachment().itemAt(index);
            } else {
                object = message.getAttachment().get(name);
            }
        } else if ("header".equalsIgnoreCase(location)) {
            object = message.getHeader().getCall();
        } else {
            throw new ObjectMappingException(expression + " should start with one of [header,body,properties,attachment]");
        }
        //If needed use MVEL for evaluation of the rest of the path
        if (path.length==3) {
            object = MVEL.getProperty(path[2], object);
        }
        logger.debug("expression=" + expression + " value=" + object);
        return object;
    }
    /**
     * Sets an object somewhere on the Message, based on the value of the expression (MOP).
     * The expression is OGNL based. MVEL is used to do the actual mapping.
     * 
     * @param message - on which the object will be placed.
     * @param expression - desribes the place in the Message where the object should be placed
     * @param object - the object which will be attached to the message.
     * @throws ObjectMappingException
     */
    public void setObjectOnMessage(Message message, String expression, Object object)
        throws ObjectMappingException
    {
    	
        String[] path = getExpressionPath(expression) ;
        if(path.length == 0)
        {
            throw new ObjectMappingException(expression + " should start with [<location>.]<name>") ;
        }
        else if ((path.length > 2) && !("header".equals(path[0]) || "body".equals(path[0])))
        {
            throw new ObjectMappingException("Only 'header' and 'body' can contain hierarchical names: " + expression) ;
        }
        String location = path[0];
        String name     = path[1];
        if ("body".equalsIgnoreCase(location)) {
            Body body   = message.getBody();
            if (path.length == 2) {
                if (BODY_CONTENT.equals(name)) {
                    try {
                        payloadProxy.setPayload(message, object);
                    } catch (MessageDeliverException e) {
                        throw new ObjectMappingException("Unable to set payload on message.", e) ;
                    }
                } else {
                    body.add(name, object);
                }
            } else {
                final Object bodyObject ;
                if (BODY_CONTENT.equals(name)) {
                    try {
                        bodyObject = payloadProxy.getPayload(message) ;
                    } catch (final MessageDeliverException mde) {
                        throw new ObjectMappingException("Unable to get payload from message.", mde) ;
                    }
                } else {
                    bodyObject = body.get(name) ;
                }
                
                if (bodyObject == null) {
                    throw new ObjectMappingException("Unable to set property on named object: " + name + ", object does not exist in message") ;
                }
                MVEL.setProperty(bodyObject, path[2], object) ;
            }
        } else if ("properties".equalsIgnoreCase(location)) {
            message.getProperties().setProperty(name, object);
        } else if ("attachment".equalsIgnoreCase(location)) {
            if (isNumeric(name)) {
                int index = Integer.valueOf(name);
                message.getAttachment().addItemAt(index, object);
            } else {
                message.getAttachment().put(name, object);
            }
        } else if ("header".equalsIgnoreCase(location)) {
            if (path.length == 2)
            {
                expression = location + '.' + name ;
            }
            else
            {
                expression = location + '.' + name + '.' + path[2] ;
            }
            MVEL.setProperty(message, expression, object);
        } else {
            throw new ObjectMappingException(expression + " should start with one of [header,body,properties,attachment]");
        }
    }
    
    /**
     * Checks to see if parameter name is number.
     * 
     * @param name - parameter to be analyzed
     * @return true if name is numeric, false in all other cases.
     */
    private boolean isNumeric(String name) {
        for (int i=0; i<name.length(); i++) {
            if (!Character.isDigit(name.charAt(i))) return false;
        }
        return true;
    }
    /**
     * Turns an object into a byte[].
     * 
     * @param object to be serialized to bytes.
     * @return byte[] representation of the object passed in
     * @throws ObjectMappingException
     */

    public byte[] getBytes(Serializable object) throws ObjectMappingException
    {
        if (object instanceof byte[]) {
            return (byte[])object;
        } else {
            return object.toString().getBytes();
        }
    }
    
    /**
     * Get the expression path from the expression.
     * @param expression The expression to split up.
     * @return The expression path.
     * @throws ObjectMappingException for errors during parsing
     * 
     * The expression path returned by this method will be
     * <code>
     *   - [&lt;location&gt;]
     *   - &lt;name&gt; 
     *   - &lt;property path&gt;
     * </code>
     * <p/>
     * The name can be fully qualified by wrapping it within single quote (') characters.
     * 
     * If the path evaluates to a single entry then it is assumed that this is the name of a property
     * location in the body.  In this case the result of this method will be "body", &lt;name&gt;.
     */
    private String[] getExpressionPath(final String expression)
        throws ObjectMappingException
    {
        if (expression != null)
        {
            final int length =  expression.length() ;
            if (length > 0)
            {
                final ObjectMapperState state = new ObjectMapperState(expression) ;
                skipWhitespace(state) ;
                
                if (state.getIndex() < length)
                {
                    final String first = getPathElement(state) ;
                    if (state.getIndex() >= length)
                    {
                        return new String[] { "body", first} ;
                    }
                    
                    final String second = getPathElement(state) ;
                    final int index = state.getIndex() ;
                    if (index >= length)
                    {
                        return new String[] { first, second } ;
                    }
                    else
                    {
                        return new String[] {first, second, expression.substring(index)} ;
                    }
                }
            }
        }
        
        return new String[0] ;
    }
    
    /**
     * Return the next path element from the expression.
     * @param state The ObjectMapper state.
     * @return The next expression path element.
     * @throws ObjectMappingException for errors during parsing
     */
    private String getPathElement(final ObjectMapperState state)
        throws ObjectMappingException
    {
        final String expression = state.getExpression() ;
        final int length = expression.length() ;
        final int startIndex = state.getIndex() ;
        
        if (expression.charAt(startIndex) == QUOTE)
        {
            state.setIndex(startIndex+1) ;
            final String element = getElement(state, QUOTE, true) ;
            final int index = state.getIndex() ;
            
            if ((index < length) && (expression.charAt(index) != EXPRESSION_SEPARATOR))
            {
                throw new ObjectMappingException("Quoted path element terminated at index: " + index + " before separator character reached: " + expression) ;
            }
            state.setIndex(index+1) ;
            return element ;
        }
        else
        {
            return getElement(state, EXPRESSION_SEPARATOR, false) ;
        }
    }
    
    /**
     * Get an element from the specified location.
     * @param state The ObjectMapper state.
     * @param endChar The character which ends the element.
     * @param endCharPresent true if the endChar must be present.
     * @return The element.
     * @throws ObjectMappingException for errors during parsing
     */
    private String getElement(final ObjectMapperState state, final char endChar, final boolean endCharPresent)
        throws ObjectMappingException
    {
        final String expression = state.getExpression() ;
        final int length = expression.length() ;
        int startIndex = state.getIndex() ;
        int index = startIndex ;
        
        StringBuilder sb = null ;
        
        while(index < length)
        {
            final char current = expression.charAt(index) ;
            if (current == ESCAPE)
            {
                final String currentExpression = expression.substring(startIndex, index) ;
                if (sb == null)
                {
                    sb = new StringBuilder(currentExpression) ;
                }
                else
                {
                    sb.append(currentExpression) ;
                }
                index++ ;
                if (index >= length)
                {
                    throw new ObjectMappingException("Unexpected end of expression reached while escaping: " + expression) ;
                }
                sb.append(expression.charAt(index)) ;
                startIndex = index+1 ;
                index = startIndex ;
            }
            else if (current == endChar)
            {
                break ;
            }
            else
            {
                index ++ ;
            }
        }
        
        final String remainder = expression.substring(startIndex, index) ;
        final String result ;
        if (sb == null)
        {
            result = remainder ;
        }
        else
        {
            sb.append(remainder) ;
            result = sb.toString() ;
        }
        
        if (endCharPresent && ((index >= length) || (endChar != expression.charAt(index))))
        {
            throw new ObjectMappingException("Expected element termination with character \"" + endChar + "\" at index: " + index + " of expression: " + expression) ;
        }
        state.setIndex(index+1) ;
        return result ;
    }

    public static MessagePayloadProxy createPayloadProxy(ConfigTree config) {
        return  new MessagePayloadProxy(config,
                new String[] {BytesBody.BYTES_LOCATION},
                new String[] {Body.DEFAULT_LOCATION, BytesBody.BYTES_LOCATION});
    }

    /**
     * Skip whitespace from the specified location.
     * @param state The ObjectMapper state.
     */
    private void skipWhitespace(final ObjectMapperState state)
    {
        final String expression = state.getExpression() ;
        final int length = expression.length() ;
        int index = state.getIndex() ;
        while((index < length) && Character.isWhitespace(expression.charAt(index)))
        {
            index++ ;
        }
        state.setIndex(index) ;
    }
    
    /**
     * Current object mapper state
     * @author kevin
     *
     */
    private static final class ObjectMapperState
    {
        /**
         * Expression being parsed.
         */
        private final String expression ;
        /**
         * Current index.
         */
        private int index ;
        
        /**
         * Construct the mapper state.
         * @param expression The expression to parse.
         */
        public ObjectMapperState(final String expression)
        {
            this.expression = expression ;
        }
        
        /**
         * Get the expression being parsed.
         * @return the expression.
         */
        public String getExpression()
        {
            return expression ;
        }
        
        /**
         * Set the current index.
         * @param index The current index.
         */
        public void setIndex(final int index)
        {
            this.index = index ;
        }
        
        /**
         * Get the current index.
         * @return The current index.
         */
        public int getIndex()
        {
            return index ;
        }
    }
}
