/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.converters;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;

import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.ActionProcessingConstants;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.testutils.AbstractTestRunner;

import junit.framework.TestCase;

/**
 * @author <a href="mailto:tom.fennelly@gmail.com">tom.fennelly@gmail.com</a>
 */
public class BOMFilterUnitTest extends TestCase {

	public void test_bytearray_with_BOM() throws ActionProcessingException, IOException {
		BOMFilter bomFilter = createBOMFilter("UTF-8");
		Message message = MessageFactory.getInstance().getMessage();
		byte[] byteArray = createBuffer("<x/>", "UTF-8");
		
		message.getBody().add(byteArray);
		assertNotSame("<x/>", new String((byte[])message.getBody().get(), "UTF-8"));
		message = bomFilter.process(message);
		assertEquals("<x/>", new String((byte[])message.getBody().get(), "UTF-8"));
	}

	public void test_bytearray_without_BOM() throws ActionProcessingException, IOException {
		BOMFilter bomFilter = createBOMFilter("UTF-8");
		Message message = MessageFactory.getInstance().getMessage();
		byte[] byteArray = "<x/>".getBytes("UTF-8");
		
		message.getBody().add(byteArray);
		assertEquals("<x/>", new String((byte[])message.getBody().get(), "UTF-8"));
		message = bomFilter.process(message);
		assertEquals("<x/>", new String((byte[])message.getBody().get(), "UTF-8"));
	}

	public void test_bytearray_with_BOM_message_defined_encoding() throws ActionProcessingException, IOException {
		BOMFilter bomFilter = createBOMFilter(null);
		Message message = MessageFactory.getInstance().getMessage();
		byte[] byteArray = createBuffer("<x/>", "UTF-8");
		
		message.getBody().add(byteArray);
		message.getProperties().setProperty(ActionProcessingConstants.PROPERTY_PAYLOAD_CONTENT_ENCODING, "UTF-8");
		
		assertNotSame("<x/>", new String((byte[])message.getBody().get(), "UTF-8"));
		message = bomFilter.process(message);
		assertEquals("<x/>", new String((byte[])message.getBody().get(), "UTF-8"));
	}

	public void test_InputStream_with_BOM() throws ActionProcessingException, IOException {
		BOMFilter bomFilter = createBOMFilter("UTF-8");
		Message message = MessageFactory.getInstance().getMessage();
		byte[] byteArray = createBuffer("<x/>", "UTF-8");
		
		message.getBody().add(new ByteArrayInputStream(byteArray));
		message = bomFilter.process(message);
		InputStream bomFilteredInputStream = (InputStream) message.getBody().get();
		
		String dataString = StreamUtils.readStreamString(bomFilteredInputStream, "UTF-8");
		
		assertEquals("<x/>", dataString);
	}

	public void test_InputStream_without_BOM() throws ActionProcessingException, IOException {
		BOMFilter bomFilter = createBOMFilter("utf8");
		Message message = MessageFactory.getInstance().getMessage();
		byte[] byteArray = "<x/>".getBytes("UTF-8");
		
		message.getBody().add(new ByteArrayInputStream(byteArray));
		message = bomFilter.process(message);
		InputStream bomFilteredInputStream = (InputStream) message.getBody().get();
		
		String dataString = StreamUtils.readStreamString(bomFilteredInputStream, "UTF-8");
		
		assertEquals("<x/>", dataString);
	}
	
    public void test_in_container() throws Exception {
        AbstractTestRunner testRunner = new AbstractTestRunner() {
            public void test() throws Exception {
                ServiceInvoker invoker = new ServiceInvoker("Services", "ServiceA");
        		Message message = MessageFactory.getInstance().getMessage();
        		byte[] byteArray = createBuffer("<x/>", "UTF-8");

        		message.getBody().add(byteArray);
        		assertNotSame("<x/>", new String((byte[])message.getBody().get(), "UTF-8"));
                message = invoker.deliverSync(message, 10000);
        		assertEquals("<x/>", new String((byte[])message.getBody().get(), "UTF-8"));
            }
        }.setServiceConfig("BOMFilterTest-config01.xml");

        testRunner.run();
    }	

	private BOMFilter createBOMFilter(String contentEncoding) {
		ConfigTree config = new ConfigTree("x");
		
		if (contentEncoding != null) {
			config.setAttribute("contentEncoding", contentEncoding);
		}
		
		return new BOMFilter(config);
	}

	private byte[] createBuffer(String data, String encoding) throws ActionProcessingException, IOException {
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		os.write(BOMFilter.getBOM(encoding));
		os.write(data.getBytes(encoding));
		return os.toByteArray();
	}
}
