/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * (C) 2005-2008, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.message.errors;

import static org.junit.Assert.*;

import java.net.URI;
import java.net.URISyntaxException;

import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.couriers.FaultMessageException;
import org.jboss.soa.esb.message.Fault;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.Test;

public class FactoryUnitTest {

	@Test(expected = FaultMessageException.class)
	public void testCreateExceptionFromFault() throws FaultMessageException, URISyntaxException {
		Message faultMessage = MessageFactory.getInstance().getMessage();
		faultMessage.getFault().setCode(new URI("test.code"));
		faultMessage.getFault().setReason("test.reason");
		faultMessage.getFault().setCause(new Exception());
		Factory.createExceptionFromFault(faultMessage);
	}

	@Test
	public void testCreateErrorMessage_ok() throws URISyntaxException {
		Message input = MessageFactory.getInstance().getMessage();
		setHeader(input, "test.message.id", "test.from", "test.to", "test.reply.to", "test.fault.to");
		Exception exception = new Exception("dummy exception");
		Message errorMessage = Factory.createErrorMessage(Factory.PROCESSING_ERROR, input, exception);

		assertNotNull(errorMessage);
		assertNotSame(input, errorMessage);
		assertEquals(input.getType(), errorMessage.getType());
		assertCall(errorMessage.getHeader().getCall(), "test.fault.to", "test.message.id");
		assertFault(errorMessage, exception, Factory.PROCESSING_ERROR);
	}

	@Test
	public void testCreateErrorMessage_noDestination() {
		Message noDestination = MessageFactory.getInstance().getMessage(); // message without destination
		Message errorMessage = Factory.createErrorMessage(Factory.PROCESSING_ERROR, noDestination, new Exception());

		assertSame(noDestination, errorMessage);
	}

	@Test
	public void testCreateErrorMessage_fromCall_ok() throws URISyntaxException {
		Message input = MessageFactory.getInstance().getMessage();
		setHeader(input, "test.message.id", "test.from", "test.to", "test.reply.to", "test.fault.to");
		Exception exception = new Exception("dummy exception");
		Message defaultError = MessageFactory.getInstance().getMessage();
		Message errorMessage = Factory.createErrorMessage(Factory.PROCESSING_ERROR, input.getType(), input.getHeader()
				.getCall(), exception, defaultError);

		assertNotNull(errorMessage);
		assertNotSame(defaultError, errorMessage);
		assertEquals(input.getType(), errorMessage.getType());
		assertCall(errorMessage.getHeader().getCall(), "test.fault.to", "test.message.id");
		assertFault(errorMessage, exception, Factory.PROCESSING_ERROR);
	}

	@Test
	public void testCreateErrorMessage_fromCall_noDestination() {
		Message noDestination = MessageFactory.getInstance().getMessage(); // message without destination
		Message defaultError = MessageFactory.getInstance().getMessage();
		Message errorMessage = Factory.createErrorMessage(Factory.PROCESSING_ERROR, noDestination.getType(),
				noDestination.getHeader().getCall(), new Exception(), defaultError);

		assertSame(defaultError, errorMessage);
	}

	private static void setHeader(Message input, String messageID, String from, String to, String replyTo,
			String faultTo) throws URISyntaxException {
		Call call = input.getHeader().getCall();
		call.setMessageID(new URI(messageID));
		call.setFrom(new EPR(new URI(from)));
		call.setTo(new EPR(new URI(to)));
		call.setReplyTo(new EPR(new URI(replyTo)));
		call.setFaultTo(new EPR(new URI(faultTo)));
	}

	private static void assertCall(Call actualCall, String expectedTo, String expectedRelatesTo)
			throws URISyntaxException {
		assertEquals(new URI(expectedTo), actualCall.getTo().getURI());
		assertEquals(new URI(expectedRelatesTo), actualCall.getRelatesTo());
	}

	private static void assertFault(Message actualMessage, Throwable expectedThrowable, String expectedFaultCode)
			throws URISyntaxException {
		assertEquals(expectedThrowable, actualMessage.getBody().get(Fault.THROWABLE_CONTENT));
		assertEquals(new URI(expectedFaultCode), actualMessage.getFault().getCode());
		assertEquals(expectedThrowable.toString(), actualMessage.getFault().getReason());
	}
}
