/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.jbpm.cmd;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.net.URI;
import java.util.Map;

import junit.framework.JUnit4TestAdapter;

import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.PortReference;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.services.jbpm.Constants;
import org.jboss.soa.esb.services.jbpm.Mapping;
import org.jbpm.JbpmConfiguration;
import org.jbpm.JbpmContext;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.exe.ProcessInstance;
import org.jbpm.graph.exe.Token;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Unit test for {@link CallbackCommand}.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 *
 */
public class CallbackCommandUnitTest
{
    @Test
    public void getVariablesMap() throws CallbackException
    {
        final String payload = "sample payload";
        
        final ProcessDefinition processDef = ProcessDefinition.parseXmlResource("callbackCommandUnitTest.xml");
        final ProcessInstance instance = new ProcessInstance(processDef);
        
        Token token = instance.getRootToken();
        Message message = MessageFactory.getInstance().getMessage();
        message.getBody().add(payload);
        
        instance.signal();
        assertEquals("node1", token.getNode().getName());
        
        final CallbackCommand callback = new CallbackCommand();
        Map<Mapping, Object> variablesMap = callback.getVariablesMapFromMessage(token.getNode(), message);
        
        Mapping expectedMappning = expectedMapping(Constants.BODY_CONTENT_VARIABLE_NAME, "theBody", false, null);
        assertEquals(payload, variablesMap.get(expectedMappning));
    }
    
    @Test
    public void execute()
    {
        final String jbpmBodyContentKey = "theBody";
        final String esbBodyContent = "sample payload";
        
        final String esbXmlContentKey = "xml";
        final String jbpmXmlContentKey = "xmlContent";
        final String esbXmlContent = "<somexml/>";
        
        final JbpmContext jbpmContext = JbpmConfiguration.getInstance().createJbpmContext();
        jbpmContext.deployProcessDefinition(ProcessDefinition.parseXmlResource("callbackCommandUnitTest.xml"));
        final ProcessDefinition processDefinition = jbpmContext.getGraphSession().findLatestProcessDefinition("callback");
        final ProcessInstance instance = new ProcessInstance(processDefinition);
        
        Token token = instance.getRootToken();
        instance.signal();
        assertEquals("node1", token.getNode().getName());
        assertEquals(2L, token.getNode().getId());
        
        assertNotNull(jbpmContext.getToken(instance.getProcessDefinition().getVersion()));
        
        final CallbackCommand callback = new CallbackCommand();
        callback.setCallbackEpr(createEpr(token));
        
        Message message = MessageFactory.getInstance().getMessage();
        message.getBody().add(esbBodyContent);
        message.getBody().add(esbXmlContentKey, esbXmlContent);
        callback.setMessage(message);
        
        callback.execute(jbpmContext);
        
        assertEquals(esbBodyContent, token.getProcessInstance().getContextInstance().getVariable(jbpmBodyContentKey));
        assertEquals(esbXmlContent, token.getProcessInstance().getContextInstance().getVariable(jbpmXmlContentKey));
    }
    
    private Mapping expectedMapping(final String esb, final String bpm, final boolean processScope, final String defaultVal)
    {
        Mapping expectedMappning = new Mapping();
        expectedMappning.setBpm("theBody");
        expectedMappning.setEsb("BODY_CONTENT");
        expectedMappning.setIsProcessScope(false);
        expectedMappning.setDefaultValue(null);
        return expectedMappning;
    }
    
    private EPR createEpr(final Token token)
    {
        ProcessInstance instance = token.getProcessInstance();
        
        final EPR epr = new EPR();
        final PortReference portRef = epr.getAddr();
        final long nodeId = token.getNode().getId();
        final long tokenId = token.getId();
        final long processVersion = instance.getProcessDefinition().getVersion();
        
        portRef.addExtension(Constants.NODE_ID, String.valueOf(nodeId));
        portRef.addExtension(Constants.TOKEN_ID, String.valueOf(tokenId));
        portRef.addExtension(Constants.PROCESS_INSTANCE_ID, String.valueOf(instance.getId()));
        
        // Set the counter
        String counterName = Constants.PROCESS_NODE_VERSION_COUNTER  + nodeId + '_' + tokenId;
        portRef.addExtension(counterName, String.valueOf(processVersion));
        
        // The counterName variable is expected to be found in the process context.
        instance.getContextInstance().setVariableLocally(counterName, processVersion);
        return epr;
    }
    
    @BeforeClass
    public static void setup() throws Exception
    {
        MockCourierFactory.install();
        MockRegistry.install();
        EPR epr1 = new EPR(new URI("test1"));
        MockCourier courier1 = new MockCourier(true);
        MockRegistry.register("MockCategory", "MockService", epr1, courier1);
    }

    @AfterClass
    public static void tearDown()
    {
        MockCourierFactory.uninstall();
        MockRegistry.uninstall();
    }

    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(CallbackCommandUnitTest.class);
    }

}
