/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * (C) 2005-2008, JBoss Inc.
 */
package org.jboss.soa.esb.actions.soap;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.management.JMException;
import javax.management.MBeanServer;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.jboss.mx.util.MBeanServerLocator;

public class SchemaDeployerImpl implements SchemaDeployer {
	private static final String XSD_PATTERN = "\\w+[.]xsd";
	private Logger logger = Logger.getLogger(SchemaDeployerImpl.class);
    private Map<String,String> resources = new HashMap<String,String>();
	
	private String esbContext;
	private Pattern xsdPattern = Pattern.compile(XSD_PATTERN);

	/* (non-Javadoc)
	 * @see org.jboss.soa.esb.actions.soap.SchemaDeployer#getResources()
	 */
	public Map<String,String> getResources() {
		return resources;
	}
	
	/* (non-Javadoc)
	 * @see org.jboss.soa.esb.actions.soap.SchemaDeployer#getEsbContext()
	 */
	public String getEsbContext() {
		return esbContext;
	}

	/* (non-Javadoc)
	 * @see org.jboss.soa.esb.actions.soap.SchemaDeployer#setEsbContext(java.lang.String)
	 */
	public void setEsbContext(String esbContext) {
		this.esbContext = esbContext;
	}

	/**
	 * Find the temp directory from the ServerConfig bean
	 * @return
	 */
	public String getTempDirectory() {
		File tempDir = null;
		MBeanServer mbeanServer = MBeanServerLocator.locateJBoss();
		try {
			tempDir = (File) mbeanServer.getAttribute(new ObjectName("jboss.system:type=ServerConfig"), "ServerTempDir");
		} catch (JMException ignored) {
		}
		return tempDir.getAbsolutePath();
	}
		
	/**
	 * Get the XSD content from an URL and store it in a String.
	 * @param url
	 * @param tempFile
	 * @return
	 * @throws IOException
	 */
	protected String getXSDString(URL url, File tempFile) throws IOException {
		BufferedReader bufferedIn = new BufferedReader(new InputStreamReader(url.openStream()));
	    BufferedWriter bufferedOut = new BufferedWriter(new FileWriter(tempFile));
	         
	    StringBuffer buffer = new StringBuffer(100);
	    String inputLine;
	    while ((inputLine = bufferedIn.readLine()) != null) {
	    	bufferedOut.write(inputLine);
	    	buffer.append(inputLine);
	    }
	    
	    bufferedIn.close();
	    bufferedOut.close();		
	    
	    return buffer.toString();
	}

	protected String extractXSDFile(URL url) {
		// try external XSD first
		Matcher pathMatcher = xsdPattern.matcher(url.getPath());
		if (pathMatcher.find()) {
			return pathMatcher.group();
		}
		// try internal XSD
		Matcher queryMatcher = xsdPattern.matcher(url.getQuery());
		if (queryMatcher.find()) {
			return queryMatcher.group();
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see org.jboss.soa.esb.actions.soap.SchemaDeployer#deploy(java.lang.String, java.lang.String)
	 */
	public String deploy(String location, String soapAddress) {
		File tmpXSD = null;
		URL xsdUrl = null;
		String xsdFile = null;
		String xsdString = null;

		try {
			xsdUrl = new URL(location);
			xsdFile = extractXSDFile(xsdUrl);
			if (xsdFile == null) {
				throw new IllegalArgumentException("Could not find match for pattern: " + XSD_PATTERN + 
						", in location: " + location);
			}
			
			tmpXSD = File.createTempFile(xsdFile, ".tmp");
			xsdString = getXSDString(xsdUrl, tmpXSD);
		} catch (Exception ioe) {
			logger.error("Failed to get an XSD file", ioe);
			return location;
		}

		// If by chance we have a name collision and the duplicate names of xsd files
		// prepend an integer on front of the XSD file name until we find a unique name
		if (resources.containsKey(xsdFile)) {
			int i = 0;
			String tempXsdFileName = xsdFile;

			while (resources.containsKey(tempXsdFileName)) {
				i++;
				tempXsdFileName = i + xsdFile;
			}
			
			xsdFile = tempXsdFileName;
		}
		
		resources.put(xsdFile, xsdString);
		return soapAddress + "?wsdl&resource=" + xsdFile;		
	}
}
