/*
 * Created on Jun 1, 2003
 */
package org.epistem.io;

import java.io.PrintWriter;
import java.io.Writer;

/**
 * A PrintWriter that has added indenting capability.
 * 
 * @author D.N.G. Main
 */
public class IndentingPrintWriter extends PrintWriter {

	private String mIndent;
	private int    mIndentLevel = 0;
	private char   mLineSep;
	private boolean mStartOfLine = true; 
	
	/**
	 * Uses an indent of 4 spaces and initial level of zero (no indent).
	 */
	public IndentingPrintWriter( Writer writer ) {
		this( writer, "    ", 0 );
	}

	/**
	 * Uses a given indent and initial level.
	 */
	public IndentingPrintWriter( Writer writer, String indent, int level ) {
		super( writer );
		mIndent = indent;
		mIndentLevel = level;
		String lineSep = System.getProperty( "line.separator" );
		mLineSep = lineSep.charAt( lineSep.length() - 1 );
	}
	
	public String getIndent() {return mIndent;}
	public int getIndentLevel() {return mIndentLevel;}
	public void setIndent(String indent) {mIndent = indent;}
	public void setIndentLevel(int indentLevel) {mIndentLevel = indentLevel;}

	/**
	 * Increase the indent.
	 */
	public void indent() { mIndentLevel++; }
	
	/**
	 * Decrease the indent.
	 */
	public void unindent() { mIndentLevel = (mIndentLevel > 0) ? (mIndentLevel-1) : 0; }
	
	private void writeIndent() {
		for (int i = 0; i < mIndentLevel; i++) print( mIndent );		
	}
	
	
	/**
	 * @see java.io.Writer#write(char[], int, int)
	 */
	public void write(char[] buf, int off, int len) {
		for (int i = off; i < off + len; i++) {
			write( buf[i] );
		}

	}

	/**
	 * Write text within a fixed width field.  If the text is larger than the
	 * field then it is written in its entirety, if smaller then the field is
	 * padded with spaces.
	 * 
	 * @param text the text to write
	 * @param fieldWidth the width of the field
	 * @param rightAlign true for right alignment, false for left
	 */
	public void writeField( String text, int fieldWidth, boolean rightAlign ) {
	    int size = text.length();
		int spaces = fieldWidth - size;

		if( spaces <= 0 ) {
			write( text );
			return; 
		}  

		if( ! rightAlign ) write( text ); 
		while( spaces-- > 0 ) write( " " );
		if( rightAlign ) write( text ); 
	}

	/**
	 * @see java.io.Writer#write(char[])
	 */
	public void write(char[] buf) {
		write( buf, 0, buf.length );
	}

	/**
	 * @see java.io.Writer#write(int)
	 */
	public void write(int c) {
		
		if( mStartOfLine ) {
			mStartOfLine = false;
			writeIndent();
		}
		
		super.write(c);
		
		if( c == '\n' || c == mLineSep ) {
			mStartOfLine = true;
		}
	}

	/**
	 * @see java.io.Writer#write(java.lang.String, int, int)
	 */
	public void write(String s, int off, int len) {
		write( s.toCharArray(), off, len );
	}

	/**
	 * @see java.io.Writer#write(java.lang.String)
	 */
	public void write(String s) {
		write( s, 0, s.length() );
	}

	/**
	 * @see java.io.PrintWriter#println()
	 */
	public void println() {
		super.println();
		mStartOfLine = true;
	}

}
