package org.infinispan.configuration.global;

import org.infinispan.commons.configuration.Builder;
import org.infinispan.util.logging.Log;
import org.infinispan.util.logging.LogFactory;

/**
 * GlobalStateConfigurationBuilder. Configures filesystem paths where global state is stored.
 *
 * @author Tristan Tarrant
 * @since 8.1
 */
public class GlobalStateConfigurationBuilder extends AbstractGlobalConfigurationBuilder
      implements Builder<GlobalStateConfiguration> {

   private static final Log log = LogFactory.getLog(GlobalStateConfigurationBuilder.class);

   private boolean enabled;
   private String persistentLocation;

   GlobalStateConfigurationBuilder(GlobalConfigurationBuilder globalConfig) {
      super(globalConfig);
   }

   public GlobalStateConfigurationBuilder enable() {
      return enabled(true);
   }

   public GlobalStateConfigurationBuilder disable() {
      return enabled(false);
   }

   /**
    * Enables or disables the storage of global state.
    */
   public GlobalStateConfigurationBuilder enabled(boolean enabled) {
      this.enabled = enabled;
      return this;
   }

   /**
    * Defines the filesystem path where persistent state data which needs to survive container restarts
    * should be stored. The data stored at this location is required for graceful
    shutdown and restore. Defaults to the user.dir system property which usually is where the
    * application was started. This value should be overridden to a more appropriate location.
    */
   public GlobalStateConfigurationBuilder persistentLocation(String location) {
      persistentLocation = location;
      return this;
   }

   @Override
   public void validate() {
      if (enabled && persistentLocation == null) {
         log.missingGlobalStatePersistentLocation();
      }
   }

   @Override
   public GlobalStateConfiguration create() {
      return new GlobalStateConfiguration(enabled, persistentLocation);
   }

   @Override
   public Builder<?> read(GlobalStateConfiguration template) {
      enabled = template.enabled();
      persistentLocation = template.persistentLocation();
      return this;
   }

}
