/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.file;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.nio.charset.Charset;

import org.apache.camel.ContextTestSupport;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.util.IOHelper;

/**
 *
 */
public class FileProducerCharsetUTFtoISOConfiguredTest extends ContextTestSupport {

    private byte[] utf;
    private byte[] iso;

    @Override
    protected void setUp() throws Exception {
        // use utf-8 as original payload with 00e6 which is a danish ae letter
        utf = "ABC\u00e6".getBytes("utf-8");
        iso = "ABC\u00e6".getBytes("iso-8859-1");

        deleteDirectory("target/charset");
        createDirectory("target/charset/input");

        log.debug("utf: {}", new String(utf, Charset.forName("utf-8")));
        log.debug("iso: {}", new String(iso, Charset.forName("iso-8859-1")));

        for (byte b : utf) {
            log.debug("utf byte: {}", b);
        }
        for (byte b : iso) {
            log.debug("iso byte: {}", b);
        }

        // write the byte array to a file using plain API
        FileOutputStream fos = new FileOutputStream("target/charset/input/input.txt");
        fos.write(utf);
        fos.close();

        super.setUp();
    }

    public void testFileProducerCharsetUTFtoISO() throws Exception {
        oneExchangeDone.matchesMockWaitTime();

        File file = new File("target/charset/output.txt");
        assertTrue("File should exist", file.exists());

        InputStream fis = IOHelper.buffered(new FileInputStream(file));
        byte[] buffer = new byte[100];

        int len = fis.read(buffer);
        assertTrue("Should read data: " + len, len != -1);
        byte[] data = new byte[len];
        System.arraycopy(buffer, 0, data, 0, len);
        fis.close();

        // data should be in iso, where the danish ae is -26
        assertEquals(4, data.length);
        assertEquals(65, data[0]);
        assertEquals(66, data[1]);
        assertEquals(67, data[2]);
        assertEquals(-26, data[3]);
    }

    @Override
    protected RouteBuilder createRouteBuilder() throws Exception {
        return new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                from("file:target/charset/input?charset=utf-8&noop=true")
                    .to("file:target/charset/?fileName=output.txt&charset=iso-8859-1");
            }
        };
    }
}
