package org.apache.camel.component.infinispan;

import java.io.IOException;

import org.apache.camel.EndpointInject;
import org.apache.camel.Message;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.infinispan.remote.InfinispanRemoteCustomListener;
import org.apache.camel.component.mock.MockEndpoint;
import org.apache.camel.impl.JndiRegistry;
import org.apache.camel.test.junit4.CamelTestSupport;
import org.infinispan.client.hotrod.RemoteCache;
import org.infinispan.client.hotrod.RemoteCacheManager;
import org.infinispan.client.hotrod.annotation.ClientListener;
import org.infinispan.client.hotrod.configuration.ConfigurationBuilder;
import org.junit.Test;

public class InfinispanRemoteAsyncConsumerFilterFactoryIT extends CamelTestSupport {
    @EndpointInject(uri = "mock:result")
    private MockEndpoint mockResult;

    private RemoteCacheManager manager;

    @ClientListener(filterFactoryName = "static-filter-factory")
    private static class MyCustomListener extends InfinispanRemoteCustomListener {
    }

    @Override
    protected void doPreSetup() throws IOException {
        ConfigurationBuilder builder = new ConfigurationBuilder();
        builder.addServer().host("localhost").port(11222);
        manager = new RemoteCacheManager(builder.build());
    }

    @Override
    protected JndiRegistry createRegistry() throws Exception {
        JndiRegistry registry = super.createRegistry();
        registry.bind("myCustomListener", new MyCustomListener());
        registry.bind("myCustomContainer", manager);
        return registry;
    }

    @Override
    protected RouteBuilder createRouteBuilder() throws Exception {
        return new RouteBuilder() {
            @Override
            public void configure() {
                from("infinispan://?cacheContainer=#myCustomContainer&cacheName=static_filter_factory&customListener=#myCustomListener&sync=false")
                .to("mock:result");
            }
        };
    }

    @Test
    public void customEventConsumed() throws InterruptedException {
        mockResult.expectedMessageCount(2);

        int filteredKey1 = 1;
        int acceptedKey = 2;
        int filteredKey2 = 3;

        getCache().put(filteredKey1, "filteredValue1");
        getCache().put(acceptedKey, "acceptedValue");
        getCache().put(filteredKey2, "filteredValue2");

        getCache().remove(filteredKey1);
        getCache().remove(acceptedKey);
        getCache().remove(filteredKey2);

        mockResult.assertIsSatisfied();

        Message createEventMessage = mockResult.getExchanges().get(0).getOut();
        assertEquals("CLIENT_CACHE_ENTRY_CREATED", createEventMessage.getHeader(InfinispanConstants.EVENT_TYPE));
        assertEquals(acceptedKey, createEventMessage.getHeader(InfinispanConstants.KEY));

        Message removeEventMessage = mockResult.getExchanges().get(1).getOut();
        assertEquals("CLIENT_CACHE_ENTRY_REMOVED", removeEventMessage.getHeader(InfinispanConstants.EVENT_TYPE));
        assertEquals(acceptedKey, removeEventMessage.getHeader(InfinispanConstants.KEY));
    }

    private RemoteCache<Object, Object> getCache() {
        return manager.getCache("static_filter_factory");
    }

}
