﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VdcCommon.BusinessEntities;
using System.Xml;
using System.IO;
using System.Text.RegularExpressions;
using VdcUtils;

namespace VdcCommon.Ovf
{
    public class OvfTemplateWriter : OvfWriter
    {
        protected VmTemplate _vmTemplate;

        public OvfTemplateWriter(out XmlDocument document, VmTemplate vmTemplate, List<DiskImage> images)
            : base(out document, images)
        {
            _vmTemplate = vmTemplate;
        }

        protected override void WriteGeneralData()
        {
            _writer.WriteStartElement("Name");
            _writer.WriteRaw(_vmTemplate.name);
            _writer.WriteEndElement();
            _writer.WriteStartElement("TemplateId");
            _writer.WriteRaw(_vmTemplate.vmt_guid.ToString());
            _writer.WriteEndElement();
            _writer.WriteStartElement("Description");
            _writer.WriteRaw(_vmTemplate.description);
            _writer.WriteEndElement();
            _writer.WriteStartElement("Domain");
            _writer.WriteRaw(_vmTemplate.domain);
            _writer.WriteEndElement();
            _writer.WriteStartElement("CreationDate");
            _writer.WriteRaw(OvfParser.LocalDateToUtcDateString(_vmTemplate.creation_date));
            _writer.WriteEndElement();
            _writer.WriteStartElement("IsAutoSuspend");
            _writer.WriteRaw(_vmTemplate.is_auto_suspend.ToString());
            _writer.WriteEndElement();
            _writer.WriteStartElement("TimeZone");
            _writer.WriteRaw(_vmTemplate.time_zone);
            _writer.WriteEndElement();
            _writer.WriteStartElement("VmType");
            _writer.WriteRaw(((int)_vmTemplate.vm_type).ToString());
            _writer.WriteEndElement();
            _writer.WriteStartElement("default_display_type");
            _writer.WriteRaw(((int)_vmTemplate.default_display_type).ToString());
            _writer.WriteEndElement();
        }

        protected override void WriteAppList()
        {
        }

        protected override void WriteContentItems()
        {
            // os
            _writer.WriteStartElement("Section");
            _writer.WriteAttributeString("ovf", "id", null, _vmTemplate.vmt_guid.ToString());
            _writer.WriteAttributeString("ovf", "required", null, "false");
            _writer.WriteAttributeString("xsi", "type", null, "ovf:OperatingSystemSection_Type");
            _writer.WriteStartElement("Info");
            _writer.WriteRaw("Guest Operating System");
            _writer.WriteEndElement();
            _writer.WriteStartElement("Description");
            _writer.WriteRaw(Enum.GetName(typeof(VmOsType), _vmTemplate.os));
            _writer.WriteEndElement();
            _writer.WriteEndElement();

            // hardware
            _writer.WriteStartElement("Section");
            _writer.WriteAttributeString("xsi", "type", null, "ovf:VirtualHardwareSection_Type");
            _writer.WriteStartElement("Info");
            _writer.WriteRaw(string.Format("{0} CPU, {1} Memeory", _vmTemplate.num_of_cpus, _vmTemplate.mem_size_mb));
            _writer.WriteEndElement();

            _writer.WriteStartElement("System");
            _writer.WriteStartElement("vssd:VirtualSystemType");
            _writer.WriteRaw(String.Format("{0} {1}", Config.GetValue<string>(ConfigValues.OvfVirtualSystemType), Config.GetValue<string>(ConfigValues.VdcVersion)));
            _writer.WriteEndElement();
            _writer.WriteEndElement();

            // item cpu
            _writer.WriteStartElement("Item");
            _writer.WriteStartElement("rasd:Caption");
            _writer.WriteRaw(string.Format("{0} virtual cpu", _vmTemplate.num_of_cpus));
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:Description");
            _writer.WriteRaw("Number of virtual CPU");
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:InstanceId");
            _writer.WriteRaw((++_instanceId).ToString());
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:ResourceType");
            _writer.WriteRaw(OvfHardware.CPU);
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:num_of_sockets");
            _writer.WriteRaw(_vmTemplate.num_of_sockets.ToString());
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:cpu_per_socket");
            _writer.WriteRaw(_vmTemplate.cpu_per_socket.ToString());
            _writer.WriteEndElement();
            _writer.WriteEndElement(); // item

            // item memory
            _writer.WriteStartElement("Item");
            _writer.WriteStartElement("rasd:Caption");
            _writer.WriteRaw(string.Format("{0} MB of memory", _vmTemplate.mem_size_mb));
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:Description");
            _writer.WriteRaw("Memory Size");
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:InstanceId");
            _writer.WriteRaw((++_instanceId).ToString());
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:ResourceType");
            _writer.WriteRaw(OvfHardware.Memory);
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:AllocationUnits");
            _writer.WriteRaw("MegaBytes");
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:VirtualQuantity");
            _writer.WriteRaw(_vmTemplate.mem_size_mb.ToString());
            _writer.WriteEndElement();
            _writer.WriteEndElement(); // item

            // item drive
            foreach (DiskImage image in _images)
            {
                _writer.WriteStartElement("Item");
                _writer.WriteStartElement("rasd:Caption");
                _writer.WriteRaw(string.Format("Drive {0}", image.internal_drive_mapping));
                _writer.WriteEndElement();
                _writer.WriteStartElement("rasd:InstanceId");
                _writer.WriteRaw(image.image_guid.ToString());
                _writer.WriteEndElement();
                _writer.WriteStartElement("rasd:ResourceType");
                _writer.WriteRaw(OvfHardware.DiskImage);
                _writer.WriteEndElement();
				_writer.WriteStartElement("rasd:HostResource");
				_writer.WriteRaw(image.image_group_id + "/" + image.image_guid);
				_writer.WriteEndElement();
                _writer.WriteStartElement("rasd:Parent");
                _writer.WriteRaw(image.ParentId.ToString());
                _writer.WriteEndElement();
                _writer.WriteStartElement("rasd:Template");
                _writer.WriteRaw(image.it_guid.ToString());
                _writer.WriteEndElement();
                _writer.WriteStartElement("rasd:ApplicationList");
                _writer.WriteRaw(image.appList);
                _writer.WriteEndElement();
                if (image.storage_id.HasValue)
                {
                    _writer.WriteStartElement("rasd:StorageId");
                    _writer.WriteRaw(image.storage_id.Value.ToString());
                    _writer.WriteEndElement();
                }
                if (image.storage_pool_id.HasValue)
                {
                    _writer.WriteStartElement("rasd:StoragePoolId");
                    _writer.WriteRaw(image.storage_pool_id.Value.ToString());
                    _writer.WriteEndElement();
                }
                _writer.WriteStartElement("rasd:CreationDate");
                _writer.WriteRaw(OvfParser.LocalDateToUtcDateString(image.creation_date));
                _writer.WriteEndElement();
                _writer.WriteStartElement("rasd:LastModified");
                _writer.WriteRaw(OvfParser.LocalDateToUtcDateString(image.lastModified));
                _writer.WriteEndElement();
                _writer.WriteEndElement(); // item
            }

            // item network
            foreach (Interface iface in _vmTemplate.Interfaces)
            {
                _writer.WriteStartElement("Item");
                _writer.WriteStartElement("rasd:Caption");
                _writer.WriteRaw("Ethernet adapter on " + iface.network_name);
                _writer.WriteEndElement();
                _writer.WriteStartElement("rasd:InstanceId");
                _writer.WriteRaw((++_instanceId).ToString());
                _writer.WriteEndElement();
                _writer.WriteStartElement("rasd:ResourceType");
                _writer.WriteRaw(OvfHardware.Network);
                _writer.WriteEndElement();
                _writer.WriteStartElement("rasd:ResourceSubType");
                if (iface.type.HasValue)
                {
                    _writer.WriteRaw(iface.type.Value.ToString());
                }
                _writer.WriteEndElement();
                _writer.WriteStartElement("rasd:Connection");
                _writer.WriteRaw(iface.network_name);
                _writer.WriteEndElement();
                _writer.WriteStartElement("rasd:Name");
                _writer.WriteRaw(iface.name);
                _writer.WriteEndElement();
                _writer.WriteEndElement(); // item
            }

            // item usb
            _writer.WriteStartElement("Item");
            _writer.WriteStartElement("rasd:Caption");
            _writer.WriteRaw("USB Controller");
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:InstanceId");
            _writer.WriteRaw((++_instanceId).ToString());
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:ResourceType");
            _writer.WriteRaw(OvfHardware.USB);
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:UsbPolicy");
            _writer.WriteRaw(_vmTemplate.usb_policy.ToString());
            _writer.WriteEndElement();
            _writer.WriteEndElement(); // item


            // monitor
            _writer.WriteStartElement("Item");
            _writer.WriteStartElement("rasd:Caption");
            _writer.WriteRaw("Graphical Controller");
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:InstanceId");
            _writer.WriteRaw((++_instanceId).ToString());
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:ResourceType");
            _writer.WriteRaw(OvfHardware.Monitor);
            _writer.WriteEndElement();
            _writer.WriteStartElement("rasd:VirtualQuantity");
            _writer.WriteRaw(_vmTemplate.num_of_monitors.ToString());
            _writer.WriteEndElement();
            _writer.WriteEndElement(); // item
        }

    }
}