using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Text;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon;

namespace org.ovirt.engine.ui.uicommon.models.hosts
{
	public class HostGeneralModel : EntityModel
	{
		#region Events

		public static EventDefinition RequestEditEventDefinition;
		public Event RequestEditEvent { get; private set; }
		public static EventDefinition RequestGOToEventsTabEventDefinition;
		public Event RequestGOToEventsTabEvent { get; private set; }

		#endregion

		#region Commands

		public UICommand SaveNICsConfigCommand { get; private set; }
		public UICommand InstallCommand { get; private set; }
		public UICommand EditHostCommand { get; private set; }
		public UICommand GoToEventsCommand { get; private set; }

		#endregion

		#region Properties

		public new VDS Entity
		{
			get { return (VDS)base.Entity; }
			set { base.Entity = value; }
		}

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		private int? freeMemory;
		public int? FreeMemory
		{
			get { return freeMemory; }
			set
			{
				if (freeMemory == null && value == null)
					return;
				if (freeMemory == null || freeMemory != value)
				{
					freeMemory = value;
					OnPropertyChanged(new PropertyChangedEventArgs("FreeMemory"));
				}
			}
		}

		private int? usedMemory;
		public int? UsedMemory
		{
			get { return usedMemory; }
			set
			{
				if (usedMemory == null && value == null)
					return;
				if (usedMemory == null || usedMemory != value)
				{
					usedMemory = value;
					OnPropertyChanged(new PropertyChangedEventArgs("UsedMemory"));
				}
			}
		}

		private long? usedSwap;
		public long? UsedSwap
		{
			get { return usedSwap; }
			set
			{
				if (usedSwap == null && value == null)
					return;
				if (usedSwap == null || usedSwap != value)
				{
					usedSwap = value;
					OnPropertyChanged(new PropertyChangedEventArgs("UsedSwap"));
				}
			}
		}

		private bool hasAnyAlert;
		public bool HasAnyAlert
		{
			get { return hasAnyAlert; }
			set
			{
				if (hasAnyAlert != value)
				{
					hasAnyAlert = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HasAnyAlert"));
				}
			}
		}

		private bool hasUpgradeAlert;
		public bool HasUpgradeAlert
		{
			get { return hasUpgradeAlert; }
			set
			{
				if (hasUpgradeAlert != value)
				{
					hasUpgradeAlert = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HasUpgradeAlert"));
				}
			}
		}

		private bool hasManualFenceAlert;
		public bool HasManualFenceAlert
		{
			get { return hasManualFenceAlert; }
			set
			{
				if (hasManualFenceAlert != value)
				{
					hasManualFenceAlert = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HasManualFenceAlert"));
				}
			}
		}

		private bool hasNoPowerManagementAlert;
		public bool HasNoPowerManagementAlert
		{
			get { return hasNoPowerManagementAlert; }
			set
			{
				if (hasNoPowerManagementAlert != value)
				{
					hasNoPowerManagementAlert = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HasNoPowerManagementAlert"));
				}
			}
		}

		private bool hasReinstallAlertNonResponsive;
		public bool HasReinstallAlertNonResponsive
		{
			get { return hasReinstallAlertNonResponsive; }
			set
			{
				if (hasReinstallAlertNonResponsive != value)
				{
					hasReinstallAlertNonResponsive = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HasReinstallAlertNonResponsive"));
				}
			}
		}

		private bool hasReinstallAlertInstallFailed;
		public bool HasReinstallAlertInstallFailed
		{
			get { return hasReinstallAlertInstallFailed; }
			set
			{
				if (hasReinstallAlertInstallFailed != value)
				{
					hasReinstallAlertInstallFailed = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HasReinstallAlertInstallFailed"));
				}
			}
		}

		private bool hasReinstallAlertMaintenance;
		public bool HasReinstallAlertMaintenance
		{
			get { return hasReinstallAlertMaintenance; }
			set
			{
				if (hasReinstallAlertMaintenance != value)
				{
					hasReinstallAlertMaintenance = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HasReinstallAlertMaintenance"));
				}
			}
		}

		private bool hasNICsAlert;
		public bool HasNICsAlert
		{
			get { return hasNICsAlert; }
			set
			{
				if (hasNICsAlert != value)
				{
					hasNICsAlert = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HasNICsAlert"));
				}
			}
		}

		private NonOperationalReason? nonOperationalReasonEntity;
		public NonOperationalReason? NonOperationalReasonEntity
		{
			get { return nonOperationalReasonEntity; }
			set
			{
				if (nonOperationalReasonEntity != value)
				{
					nonOperationalReasonEntity = value;
					OnPropertyChanged(new PropertyChangedEventArgs("NonOperationalReasonEntity"));
				}
			}
		}

		#endregion

		static HostGeneralModel()
		{
			RequestEditEventDefinition = new EventDefinition("RequestEditEvent", typeof(HostGeneralModel));
			RequestGOToEventsTabEventDefinition = new EventDefinition("RequestGOToEventsTabEvent", typeof(HostGeneralModel));
		}

		public HostGeneralModel()
		{
			RequestEditEvent = new Event(RequestEditEventDefinition);
			RequestGOToEventsTabEvent = new Event(RequestGOToEventsTabEventDefinition);
			Title = "General";

			SaveNICsConfigCommand = new UICommand("SaveNICsConfig", this);
			InstallCommand = new UICommand("Install", this);
			EditHostCommand = new UICommand("EditHost", this);
			GoToEventsCommand = new UICommand("GoToEvents", this);
		}

		public void SaveNICsConfig()
		{
			Frontend.RunMultipleAction(VdcActionType.CommitNetworkChanges,
				new List<VdcActionParametersBase>
                {
                    new VdsActionParameters(Entity.vds_id)
                },
				result =>
				{
				},
				null
			);
		}

		public void Install()
		{
			if (Window != null)
			{
				return;
			}

			InstallModel model = new InstallModel();
			Window = model;
			model.Title = "Install Host";
			model.HashName = "install_host";
			model.oVirtISO.IsAvailable = false;
			model.RootPassword.IsAvailable = false;
			model.OverrideIpTables.IsAvailable = false;

			if (Entity.vds_type == VDSType.oVirtNode)
			{
				List<string> isos = DataProvider.GetoVirtISOsList();
				model.oVirtISO.Items = isos;
				model.oVirtISO.SelectedItem = Linq.FirstOrDefault(isos);
				model.oVirtISO.IsAvailable = true;
				model.oVirtISO.IsChangable = true;
			}
			else
			{
				model.RootPassword.IsAvailable = true;
				model.RootPassword.IsChangable = true;
				model.OverrideIpTables.IsAvailable = true;
				model.OverrideIpTables.Entity = true;
			}

			model.Commands.Add(
				new UICommand("OnInstall", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void EditHost()
		{
			//Let's the parent model know about request.
			RequestEditEvent.raise(this, EventArgs.Empty);
		}

		public void OnInstall()
		{
			InstallModel model = (InstallModel)Window;
			bool isOVirt = Entity.vds_type == VDSType.oVirtNode;

			if (!model.Validate(isOVirt))
			{
				return;
			}

			UpdateVdsActionParameters param = new UpdateVdsActionParameters();
			param.vds = Entity;
			param.VdsId = Entity.vds_id;
			param.RootPassword = (string)model.RootPassword.Entity;
			param.IsReinstallOrUpgrade = true;
			param.InstallVds = true;
			param.oVirtIsoFile = isOVirt ? (string)model.oVirtISO.SelectedItem : null;
			param.OverrideFirewall = (bool)model.OverrideIpTables.Entity;

			VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.UpdateVds, param);

			if (returnValue != null && returnValue.Succeeded)
			{
				Cancel();
			}
		}

		private VdsVersion GetHostVersion(Guid hostId)
		{
			VDS host = DataProvider.GetHostById(hostId);
			return host != null ? host.Version : new VdsVersion();
		}

		public void Cancel()
		{
			Window = null;
		}

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();

			if (Entity != null)
			{
				UpdateAlerts();
				UpdateMemory();
				UpdateSwapUsed();
			}
		}

		protected override void EntityPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.EntityPropertyChanged(sender, e);

			if (e.PropertyName == "net_config_dirty" || e.PropertyName == "status" || e.PropertyName == "spm_status" || e.PropertyName == "vm_active")
			{
				UpdateAlerts();
			}

			if (e.PropertyName == "usage_mem_percent" || e.PropertyName == "physical_mem_mb")
			{
				UpdateMemory();
			}

			if (e.PropertyName == "swap_total" || e.PropertyName == "swap_free")
			{
				UpdateSwapUsed();
			}
		}

		private void UpdateAlerts()
		{
			HasAnyAlert = false;
			HasUpgradeAlert = false;
			HasManualFenceAlert = false;
			HasNoPowerManagementAlert = false;
			HasReinstallAlertNonResponsive = false;
			HasReinstallAlertInstallFailed = false;
			HasReinstallAlertMaintenance = false;
			HasNICsAlert = false;
			InstallCommand.IsExecutionAllowed = true;
			EditHostCommand.IsExecutionAllowed = VdcActionUtils.CanExecute(new List<VDS>() { Entity }, typeof(VDS), VdcActionType.UpdateVds);
			//Check the network alert presense.
			HasNICsAlert = Entity.net_config_dirty.GetValueOrDefault();


			//Check manual fence alert presense.
			if (Entity.status == VDSStatus.NonResponsive && !Entity.pm_enabled && (Entity.vm_active.GetValueOrDefault() > 0 || Entity.spm_status == VdsSpmStatus.SPM))
			{
				HasManualFenceAlert = true;
			}
			else if (!Entity.pm_enabled)
			{
				HasNoPowerManagementAlert = true;
			}

			//Check the reinstall alert presense.
			if (Entity.status == VDSStatus.NonResponsive)
			{
				HasReinstallAlertNonResponsive = true;
			}
			else if(Entity.status == VDSStatus.InstallFailed)
			{
				HasReinstallAlertInstallFailed = true;
			}
			else if (Entity.status == VDSStatus.Maintenance)
			{
				HasReinstallAlertMaintenance = true;
			}
			
			// TODO: Need to come up with a logic to show the Upgrade action-item.
			// Currently, this action-item will be shown for all oVirts assuming there are
			// available oVirt ISOs that are returned by the backend's GetoVirtISOs query.
			else if (Entity.vds_type == VDSType.oVirtNode && DataProvider.GetoVirtISOsList().Count > 0)
			{
				HasUpgradeAlert = true;
				InstallCommand.IsExecutionAllowed =
					Entity.status != VDSStatus.Up
					&& Entity.status != VDSStatus.Installing
					&& Entity.status != VDSStatus.PreparingForMaintenance
					&& Entity.status != VDSStatus.Reboot
					&& Entity.status != VDSStatus.PendingApproval;

				if (!InstallCommand.IsExecutionAllowed)
				{
					InstallCommand.ExecuteProhibitionReasons.Add("Switch to maintenance mode to enable Upgrade.");
				}
			}

			NonOperationalReasonEntity = (Entity.NonOperationalReason == NonOperationalReason.NONE ? null : (NonOperationalReason?)Entity.NonOperationalReason);

			HasAnyAlert = HasNICsAlert || HasUpgradeAlert || HasManualFenceAlert || HasNoPowerManagementAlert || HasReinstallAlertNonResponsive || HasReinstallAlertInstallFailed || HasReinstallAlertMaintenance;
		}

		private void GoToEvents()
		{
			this.RequestGOToEventsTabEvent.raise(this, null);
		}

		private void UpdateMemory()
		{
			FreeMemory = null;
			UsedMemory = null;
			if (Entity.physical_mem_mb.HasValue && Entity.usage_mem_percent.HasValue)
			{
				FreeMemory = Entity.physical_mem_mb - (Entity.physical_mem_mb / 100 * Entity.usage_mem_percent);
				UsedMemory = Entity.physical_mem_mb - FreeMemory;
			}
		}

		private void UpdateSwapUsed()
		{
			UsedSwap = null;
			if (Entity.swap_total.HasValue && Entity.swap_free.HasValue)
			{
				UsedSwap = Entity.swap_total - Entity.swap_free;
			}
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == SaveNICsConfigCommand)
			{
				SaveNICsConfig();
			}
			else if (command == InstallCommand)
			{
				Install();
			}
			else if (command == EditHostCommand)
			{
				EditHost();
			}
			else if (command == GoToEventsCommand)
			{
				GoToEvents();
			}
			else if (command.Name == "OnInstall")
			{
				OnInstall();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
		}
	}
}
