using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Text;
using System.Windows;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using System;

namespace org.ovirt.engine.ui.uicommon.models.bookmarks
{
	public class BookmarkListModel : SearchableListModel
	{
		#region Events

		public static EventDefinition NavigatedEventDefinition;
		public Event NavigatedEvent { get; private set; }

		#endregion

		#region Commands

		public UICommand NewCommand { get; private set; }
		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		#endregion

		static BookmarkListModel()
		{
			NavigatedEventDefinition = new EventDefinition("Navigated", typeof(BookmarkListModel));
		}

		public BookmarkListModel()
		{
			NavigatedEvent = new Event(NavigatedEventDefinition);

			NewCommand = new UICommand("New", this);
			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);

			SearchCommand.Execute();

			UpdateActionAvailability();
		}

		protected override void SyncSearch()
		{
			base.SyncSearch();

			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAllBookmarks,
				new VdcQueryParametersBase());

			if (returnValue != null && returnValue.Succeeded)
			{
				Items = (List<VdcCommon.BusinessEntities.bookmarks>)returnValue.ReturnValue;
			}
			else
			{
				Items = new List<VdcCommon.BusinessEntities.bookmarks>();
			}
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();
			SyncSearch();
		}

		public void Remove()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Bookmark(s)";
			model.HashName = "remove_bookmark";
			model.Message = "Bookmark(s):";
			
			List<string> list = new List<string>();
			foreach (object item in SelectedItems)
			{
				VdcCommon.BusinessEntities.bookmarks i = (VdcCommon.BusinessEntities.bookmarks)item;
				list.Add(i.bookmark_name);
			}
			model.Items = list;

			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnRemove()
		{
			// 			Frontend.RunMultipleActions(VdcActionType.RemoveBookmark,
			// 				SelectedItems.Cast<bookmarks>()
			// 				.Select(a => (VdcActionParametersBase)new BookmarksParametersBase(a.bookmark_id))
			// 				.ToList()
			// 			);
			//List<VdcActionParametersBase> prms = new List<VdcActionParametersBase>();
			//foreach (object item in SelectedItems)
			//{
			//    VdcCommon.BusinessEntities.bookmarks i = (VdcCommon.BusinessEntities.bookmarks)item;
			//    prms.Add(new BookmarksParametersBase(i.bookmark_id));
			//}
			//Frontend.RunMultipleActions(VdcActionType.RemoveBookmark, prms);

			VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.RemoveBookmark, new BookmarksParametersBase(((VdcCommon.BusinessEntities.bookmarks)SelectedItem).bookmark_id));
			if (returnValue != null && returnValue.Succeeded)
			{
				SearchCommand.Execute();
			}

			Cancel();
		}

		public void Edit()
		{
			VdcCommon.BusinessEntities.bookmarks bookmark = (VdcCommon.BusinessEntities.bookmarks)SelectedItem;

			if (Window != null)
			{
				return;
			}

			BookmarkModel model = new BookmarkModel();
			Window = model;
			model.Title = "Edit Bookmark";
			model.HashName = "edit_bookmark";
			model.IsNew = false;
			model.Name.Entity = bookmark.bookmark_name;
			model.SearchString.Entity = bookmark.bookmark_value;

			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void New()
		{
			if (Window != null)
			{
				return;
			}

			BookmarkModel model = new BookmarkModel();
			Window = model;
			model.Title = "New Bookmark";
			model.HashName = "new_bookmark";
			model.IsNew = true;
			model.SearchString.Entity = SearchString;

			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnSave()
		{
			BookmarkModel model = (BookmarkModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			VdcCommon.BusinessEntities.bookmarks bookmark =
				new VdcCommon.BusinessEntities.bookmarks
				{
					bookmark_id = model.IsNew ? (guid)Guid.Empty : ((VdcCommon.BusinessEntities.bookmarks)SelectedItem).bookmark_id,
					bookmark_name = (string)model.Name.Entity,
					bookmark_value = (string)model.SearchString.Entity
				};


			model.StartProgress(null);

			Frontend.RunAction(model.IsNew ? VdcActionType.AddBookmark : VdcActionType.UpdateBookmark,
				new BookmarksOperationParameters(bookmark),
				result =>
				{
					BookmarkListModel localModel = (BookmarkListModel)result.State;

					localModel.PostOnSave(result.ReturnValue);
				},
				this
			);
		}

		public void PostOnSave(VdcReturnValueBase returnValue)
		{
			BookmarkModel model = (BookmarkModel)Window;

			model.StopProgress();

			if (returnValue != null && returnValue.Succeeded)
			{
				Cancel();
				SearchCommand.Execute();
			}
		}

		public void Cancel()
		{
			Window = null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();

			if (SelectedItem != null)
			{
				NavigatedEvent.raise(this, new BookmarkEventArgs((VdcCommon.BusinessEntities.bookmarks)SelectedItem));
			}
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		private void UpdateActionAvailability()
		{
			EditCommand.IsExecutionAllowed = SelectedItems != null && SelectedItems.Count == 1;
			RemoveCommand.IsExecutionAllowed = SelectedItems != null && SelectedItems.Count > 0;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == NewCommand)
			{
				New();
			}
			else if (command == EditCommand)
			{
				Edit();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}

			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}

			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
		}
	}
}
