using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Text;
using System.Threading;
using System.Transactions;
using org.ovirt.engine.ui.uicommon.models.clusters;
using org.ovirt.engine.ui.uicommon.models.common;
using org.ovirt.engine.ui.uicommon.models.configure;
using org.ovirt.engine.ui.uicommon.models.datacenters;
using org.ovirt.engine.ui.uicommon.models.tags;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using VdcCommon;
using System.Windows;
using System.ComponentModel;

namespace org.ovirt.engine.ui.uicommon.models.hosts
{
	public class HostListModel : ListWithDetailsModel, ITaskTarget, ISupportSystemTreeContext
	{
		#region Commands

		public UICommand NewCommand { get; private set; }
		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }
		public UICommand ActivateCommand { get; private set; }
		public UICommand MaintenanceCommand { get; private set; }
		public UICommand ApproveCommand { get; private set; }
		public UICommand RestartCommand { get; private set; }
		public UICommand StartCommand { get; private set; }
		public UICommand StopCommand { get; private set; }
		public UICommand ManualFenceCommand { get; private set; }
		public UICommand AssignTagsCommand { get; private set; }
		public UICommand ConfigureLocalStorageCommand { get; private set; }

		#endregion

		#region Properties

		private HostEventListModel HostEventListModel { get; set; }


		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}
		private Model confirmWindow;
		public Model ConfirmWindow
		{
			get { return confirmWindow; }
			set
			{
				if (confirmWindow != value)
				{
					confirmWindow = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ConfirmWindow"));
				}
			}
		}

		private bool isPowerManagementEnabled;
		public bool IsPowerManagementEnabled
		{
			get
			{
				return isPowerManagementEnabled;
			}
			set
			{
				if (isPowerManagementEnabled != value)
				{
					isPowerManagementEnabled = value;
					OnPropertyChanged(new PropertyChangedEventArgs("isPowerManagementEnabled"));
				}
			}
		}

		protected object[] SelectedKeys
		{
			get
			{
				if (SelectedItems == null)
				{
					return new object[0];
				}
				else
				{
					object[] keys = new object[SelectedItems.Count];
					for (int i = 0; i < SelectedItems.Count; i++)
					{
						keys[i] = ((VDS)SelectedItems[i]).vds_id;
					}
					return keys;
				}
			}
		}

		#endregion

		public HostListModel()
		{
			Title = "Hosts";

			DefaultSearchString = "Host:";
			SearchString = DefaultSearchString;

			NewCommand = new UICommand("New", this);
			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);
			ActivateCommand = new UICommand("Activate", this);
			MaintenanceCommand = new UICommand("Maintenance", this);
			ApproveCommand = new UICommand("Approve", this);
			RestartCommand = new UICommand("Restart", this);
			StartCommand = new UICommand("Start", this);
			StopCommand = new UICommand("Stop", this);
			ManualFenceCommand = new UICommand("ManualFence", this);
			AssignTagsCommand = new UICommand("AssignTags", this);
			ConfigureLocalStorageCommand = new UICommand("ConfigureLocalStorage", this);

			UpdateActionAvailability();

			SearchNextPageCommand.IsAvailable = true;
			SearchPreviousPageCommand.IsAvailable = true;
		}

		public void AssignTags()
		{
			if (Window != null)
			{
				return;
			}

			TagListModel model = new TagListModel();
			Window = model;
			model.Title = "Assign Tags";
			model.HashName = "assign_tags_hosts";

			model.AttachedTagsToEntities = GetAttachedTagsToSelectedHosts();
			List<TagModel> tags = (List<TagModel>)model.Items;

			model.Commands.Add(
				new UICommand("OnAssignTags", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private IDictionary<Guid, bool> GetAttachedTagsToSelectedHosts()
		{
			Dictionary<Guid, bool> tags = new Dictionary<Guid, bool>();

			//          var hostIds = SelectedItems
			//              .Cast<VDS>()
			//              .Select(a => a.vds_id)
			//              .ToList();

			//          var allAttachedTags = hostIds.SelectMany(a => DataProvider.GetAttachedTagsToHost(a)).ToList();

			//          var attachedTags = allAttachedTags
			//              .Distinct(new TagsEqualityComparer())
			//              .ToList();

			//          attachedTags.Each(a => { tags.Add(a.tag_id, allAttachedTags.Count(b => b.tag_id == a.tag_id) == hostIds.Count() ? true : false ); });

			List<Guid> hostIds = new List<Guid>();

			foreach (object item in SelectedItems)
			{
				VDS vds = (VDS)item;
				hostIds.Add(vds.vds_id);
			}

			List<VdcCommon.BusinessEntities.tags> allAttachedTags = new List<VdcCommon.BusinessEntities.tags>();

			foreach (Guid hostId in hostIds)
			{
				allAttachedTags.AddRange(DataProvider.GetAttachedTagsToHost(hostId));
			}

			List<VdcCommon.BusinessEntities.tags> attachedTags = Linq.Distinct(allAttachedTags, new TagsEqualityComparer());

			foreach (VdcCommon.BusinessEntities.tags tag in attachedTags)
			{
				int count = 0;
				foreach (VdcCommon.BusinessEntities.tags tag2 in allAttachedTags)
				{
					if (tag2.tag_id.Equals(tag.tag_id))
						count++;
				}
				tags.Add(tag.tag_id, count == hostIds.Count ? true : false);
			}

			return tags;
		}

		public void OnAssignTags()
		{
			TagListModel model = (TagListModel)Window;

			//          var hostIds = SelectedItems
			//              .Cast<VDS>()
			//              .Select(a => a.vds_id)
			//              .ToList();
			List<Guid> hostIds = new List<Guid>();

			foreach (object item in SelectedItems)
			{
				VDS vds = (VDS)item;
				hostIds.Add(vds.vds_id);
			}

			//          var attachedTags = GetAttachedTagsToSelectedHosts();
			IDictionary<Guid, bool> attachedTags = GetAttachedTagsToSelectedHosts();

			//prepare attach/detach lists
			List<Guid> tagsToAttach = new List<Guid>();
			List<Guid> tagsToDetach = new List<Guid>();

			//          model.Items
			//              .Cast<TagModel>()
			//              .First()
			//              .EachRecursive(a => a.Children, (a, b) => {
			//                  if(a.Selection == true && (!attachedTags.ContainsKey(a.Id) || attachedTags[a.Id] == false))
			//                  {
			//                      tagsToAttach.Add(a.Id);
			//                  }
			//                  else if (a.Selection == false && attachedTags.ContainsKey(a.Id))
			//                  {
			//                      tagsToDetach.Add(a.Id);
			//                  }
			//              });
			if (model.Items != null && ((List<TagModel>)model.Items).Count > 0)
			{
				List<TagModel> tags = (List<TagModel>)model.Items;
				TagModel rootTag = tags[0];
				TagModel.RecursiveEditAttachDetachLists(rootTag, attachedTags, tagsToAttach, tagsToDetach);
			}
			//Attach tags.

			//          Frontend.RunMultipleActions(VdcActionType.AttachVdsToTag,
			//              tagsToAttach.Select(a =>
			//                  (VdcActionParametersBase)new AttachVdsToTagParameters(a, hostIds)
			//              )
			//              .ToList()
			//          );

			List<VdcActionParametersBase> prmsToAttach = new List<VdcActionParametersBase>();
			foreach (Guid tag_id in tagsToAttach)
			{
				prmsToAttach.Add(new AttachVdsToTagParameters(tag_id, hostIds));
			}
			Frontend.RunMultipleAction(VdcActionType.AttachVdsToTag, prmsToAttach);

			//          Frontend.RunMultipleActions(VdcActionType.DetachVdsFromTag,
			//              tagsToDetach.Select(a =>
			//                  (VdcActionParametersBase)new AttachVdsToTagParameters(a, hostIds)
			//              )
			//              .ToList()
			//);
			List<VdcActionParametersBase> prmsToDetach = new List<VdcActionParametersBase>();
			foreach (Guid tag_id in tagsToDetach)
			{
				prmsToDetach.Add(new AttachVdsToTagParameters(tag_id, hostIds));
			}
			Frontend.RunMultipleAction(VdcActionType.DetachVdsFromTag, prmsToDetach);


			Cancel();
		}

		public void ManualFence()
		{
			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Are you sure?";
			model.HashName = "manual_fence_are_you_sure";
			List<VDS> items = new List<VDS>();
			items.Add((VDS)SelectedItem);
			model.Items = items;

			model.Latch.IsAvailable = true;
			model.Latch.IsChangable = true;


			model.Commands.Add(
				new UICommand("OnManualFence", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnManualFence()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VDS vds = (VDS)item;
				FenceVdsManualyParameters parameters = new FenceVdsManualyParameters(true);
				parameters.StoragePoolId = vds.storage_pool_id;
				parameters.VdsId = vds.vds_id;
				list.Add(parameters);
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.FenceVdsManualy, list,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		private bool updateOverrideIpTables = true;
		private bool clusterChanging;

		public void New()
		{
			if (Window != null)
			{
				return;
			}

			HostModel model = new HostModel();
			Window = model;
			model.Title = "New Host";
			model.HashName = "new_host";
			model.Port.Entity = 54321;
			model.PmType.SelectedItem = null;
			model.OverrideIpTables.Entity = true;

			//JUICOMMENT_BEGIN
			// Make sure not to set override IP tables flag back true when it was set false once.
			model.OverrideIpTables.EntityChangedEvent.addListener(new EventListener(
				(ev, o, args) =>
				{
					if (!clusterChanging)
					{
						updateOverrideIpTables = (Boolean)model.OverrideIpTables.Entity;
					}
				})
			);

			// Set override IP tables flag true for v3.0 clusters.
			model.Cluster.SelectedItemChangedEvent.addListener(new EventListener(
				(ev, o, args) =>
				{
					clusterChanging = true;

					VDSGroup cluster = (VDSGroup)model.Cluster.SelectedItem;
					if (cluster != null)
					{
						Version version3 = new Version(3, 0);

						bool isLessThan3 = cluster.compatibility_version.compareTo(version3) < 0;
						model.OverrideIpTables.IsAvailable = !isLessThan3;
						model.OverrideIpTables.Entity = !isLessThan3 && updateOverrideIpTables;
					}

					clusterChanging = false;
				})
			);
			//JUICOMMENT_END

			List<storage_pool> dataCenter = DataProvider.GetDataCenterList();
			model.DataCenter.Items = dataCenter;
			model.DataCenter.SelectedItem = Linq.FirstOrDefault(dataCenter);

			if (SystemTreeSelectedItem != null)
			{
				switch (SystemTreeSelectedItem.Type)
				{
					case SystemTreeItemType.Host:
						model.Name.IsChangable = false;
						model.Name.Info = "Cannot edit Host's Name in this tree context";
						break;
					case SystemTreeItemType.Hosts:
					case SystemTreeItemType.Cluster:
						VDSGroup cluster = (VDSGroup)SystemTreeSelectedItem.Entity;
						foreach (storage_pool dc in (List<storage_pool>)model.DataCenter.Items)
						{
							if (dc.Id.Equals(cluster.storage_pool_id))
							{
								model.DataCenter.Items = new List<storage_pool> { dc };
								model.DataCenter.SelectedItem = dc;
								break;
							}
						}
						model.DataCenter.IsChangable = false;
						model.DataCenter.Info = "Cannot choose Host's Data Center in tree context";
						model.Cluster.IsChangable = false;
						model.Cluster.Info = "Cannot choose Host's Cluster in tree context";
						break;
					case SystemTreeItemType.DataCenter:
						storage_pool selectDataCenter = (storage_pool)SystemTreeSelectedItem.Entity;
						model.DataCenter.Items = new List<storage_pool> { selectDataCenter };
						model.DataCenter.SelectedItem = selectDataCenter;
						model.DataCenter.IsChangable = false;
						model.DataCenter.Info = "Cannot choose Host's Data Center in tree context";
						break;
					default:
						break;
				}
			}

			model.Commands.Add(
				new UICommand("OnSaveFalse", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void GoToEventsTab()
		{
			ActiveDetailModel = HostEventListModel;
		}

		public void EditWithPMemphasis()
		{
			EditCommand.Execute();

			HostModel model = (HostModel)Window;
			model.IsPowerManagementSelected = true;
			model.IsPm.Entity = true;
			model.IsPm.IsChangable = false;
		}

		public void Edit()
		{
			if (Window != null)
			{
				return;
			}

			VDS host = (VDS)SelectedItem;
			HostModel model;
			PrepareModelForApproveEdit(host, out model);
			Window = model;
			model.Title = "Edit Host";
			model.HashName = "edit_host";

			model.Commands.Add(
				new UICommand("OnSaveFalse", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnSaveFalse()
		{
			OnSave(false);
		}

		public void OnSave(bool approveInitiated)
		{
			HostModel model = (HostModel)Window;

			if (!model.Validate())
			{
				return;
			}

			if (!((bool)model.IsPm.Entity))
			{
				ConfirmationModel confirmModel = new ConfirmationModel();
				ConfirmWindow = confirmModel;
				confirmModel.Title = "Power Management Configuration";
				confirmModel.HashName = "power_management_configuration";
				confirmModel.Message = "You haven't configured Power Management for this Host. Are you sure you want to continue?";

				confirmModel.Commands.Add(
					new UICommand(approveInitiated ? "OnSaveInternalFromApprove" : "OnSaveInternalNotFromApprove", this)
					{
						Title = "OK",
						IsDefault = true
					});
				confirmModel.Commands.Add(
					new UICommand("CancelConfirmFocusPM", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
			else
			{
				OnSaveInternal(approveInitiated);
			}

		}

		public void CancelConfirmFocusPM()
		{
			HostModel hostModel = (HostModel)Window;
			hostModel.IsPowerManagementSelected = true;
			hostModel.IsPm.Entity = true;

			ConfirmWindow = null;
		}

		public void OnSaveInternalNotFromApprove()
		{
			OnSaveInternal(false);
		}

		public void OnSaveInternalFromApprove()
		{
			OnSaveInternal(true);
		}

		public void OnSaveInternal(bool approveInitiated)
		{
			HostModel model = (HostModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			VDS host = model.IsNew ? new VDS() : (VDS)Cloner.Clone(SelectedItem);

			//Save changes.
			host.vds_name = (string)model.Name.Entity;
			host.host_name = (string)model.Host.Entity;
			host.ManagmentIp = (string)model.ManagementIp.Entity;
			host.port = Convert.ToInt32(model.Port.Entity.ToString());

			Guid oldClusterId = host.vds_group_id;
			Guid newClusterId = ((VDSGroup)model.Cluster.SelectedItem).ID;
			host.vds_group_id = newClusterId;
			host.pm_enabled = (bool)model.IsPm.Entity;
			host.pm_user = (string)model.PmUserName.Entity;
			host.pm_password = (string)model.PmPassword.Entity;
			host.pm_type = (string)model.PmType.SelectedItem;
			host.PmOptionsMap = new valueObjectMap(model.PmOptionsMap, false);


			CancelConfirm();
			model.StartProgress(null);

			if (model.IsNew)
			{
				AddVdsActionParameters parameters = new AddVdsActionParameters();
				parameters.VdsId = host.vds_id;
				parameters.vds = host;
				parameters.RootPassword = (string)model.RootPassword.Entity;
				parameters.OverrideFirewall = (bool)model.OverrideIpTables.Entity;

				Frontend.RunAction(VdcActionType.AddVds, parameters,
					result =>
					{
						object[] array = (object[])result.State;
						HostListModel localModel = (HostListModel)array[0];
						bool localApproveInitiated = (bool)array[1];

						localModel.PostOnSaveInternal(result.ReturnValue, localApproveInitiated);
					},
					new object[] { this, approveInitiated }
				);
			}
			else // Update VDS -> consists of changing VDS cluster first and then updating rest of VDS properties:
			{
				UpdateVdsActionParameters parameters = new UpdateVdsActionParameters();
				parameters.vds = host;
				parameters.VdsId = host.vds_id;
				parameters.RootPassword = String.Empty;
				parameters.InstallVds = false;

				if (oldClusterId != newClusterId)
				{
					Frontend.RunAction(VdcActionType.ChangeVDSCluster,
						new ChangeVDSClusterParameters(newClusterId, host.vds_id),
						result =>
						{
							object[] array = (object[])result.State;
							HostListModel localModel = (HostListModel)array[0];
							UpdateVdsActionParameters localParameters = (UpdateVdsActionParameters)array[1];
							bool localApproveInitiated = (bool)array[2];

							VdcReturnValueBase localReturnValue = result.ReturnValue;

							//If cluster was changed succesfully, proceed with save.
							if (localReturnValue != null && localReturnValue.Succeeded)
							{
								localModel.PostOnSaveInternalChangeCluster(localParameters, localApproveInitiated);
							}
							else
							{
								localModel.Window.StopProgress();
							}
						},
						new object[] { this, parameters, approveInitiated }
					);
				}
				else
				{
					PostOnSaveInternalChangeCluster(parameters, approveInitiated);
				}
			}
		}

		public void PostOnSaveInternalChangeCluster(UpdateVdsActionParameters parameters, bool approveInitiated)
		{
			Frontend.RunAction(VdcActionType.UpdateVds, parameters,
				result =>
				{
					object[] array = (object[])result.State;
					HostListModel localModel = (HostListModel)array[0];
					bool localApproveInitiated = (bool)array[1];

					localModel.PostOnSaveInternal(result.ReturnValue, localApproveInitiated);
				},
				new object[] { this, approveInitiated }
			);
		}

		public void PostOnSaveInternal(VdcReturnValueBase returnValue, bool approveInitiated)
		{
			HostModel model = (HostModel)Window;

			model.StopProgress();

			if (returnValue != null && returnValue.Succeeded)
			{
				if (approveInitiated)
				{
					OnApproveInternal();
				}
				Cancel();
			}
		}

		private void OnApproveInternal()
		{
			VDS vds = (VDS)SelectedItem;

			Frontend.RunMultipleAction(VdcActionType.ApproveVds,
				new List<VdcActionParametersBase>
                {
                    new ApproveVdsParameters(vds.vds_id)
				},
				result =>
				{
				},
				null
			);
		}

		public void Remove()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Host(s)";
			model.HashName = "remove_host";
			model.Message = "Host(s)";

			List<string> list = new List<string>();
			foreach (VDS item in Linq.Cast<VDS>(SelectedItems))
			{
				list.Add(item.vds_name);
			}
			model.Items = list;


			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnRemove()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VDS vds = (VDS)item;
				list.Add(new VdsActionParameters(vds.vds_id));
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.RemoveVds, list,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		public void Activate()
		{
			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VDS vds = (VDS)item;
				list.Add(new VdsActionParameters(vds.vds_id));
			}

			Frontend.RunMultipleAction(VdcActionType.ActivateVds, list,
				result =>
				{
				},
				null
			);
		}

		public void Maintenance()
		{
			if (ConfirmWindow != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			ConfirmWindow = model;
			model.Title = "Maintenance Host(s)";
			model.HashName = "maintenance_host";
			model.Message = @"Are you sure you want to place the following host(s) into maintenance mode?";
			//          model.Items = SelectedItems.Cast<VDS>().Select(a => a.vds_name);
			List<string> vdss = new List<string>();
			foreach (object item in SelectedItems)
			{
				VDS vds = (VDS)item;
				vdss.Add(vds.vds_name);
			}
			model.Items = vdss;


			model.Commands.Add(
				new UICommand("OnMaintenance", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("CancelConfirm", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnMaintenance()
		{
			ConfirmationModel model = (ConfirmationModel)ConfirmWindow;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			List<guid> vdss = new List<guid>();

			foreach (object item in SelectedItems)
			{
				VDS vds = (VDS)item;
				vdss.Add(vds.vds_id);
			}
			list.Add(new MaintananceNumberOfVdssParameters(vdss, false));


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.MaintananceNumberOfVdss, list,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					CancelConfirm();
				},
				model
			);
		}

		public void Approve()
		{
			VDS host = (VDS)SelectedItem;
			HostModel model;
			PrepareModelForApproveEdit(host, out model);
			Window = model;
			model.Title = "Edit and Approve Host";
			model.HashName = "edit_and_approve_host";


			model.Commands.Add(
				new UICommand("OnApprove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void PrepareModelForApproveEdit(VDS vds, out HostModel model)
		{
			model = new HostModel();
			model.HostId = vds.vds_id;
			model.RootPassword.IsAvailable = false;
			model.OverrideIpTables.IsAvailable = false;
			model.OriginalName = vds.vds_name;
			model.Name.Entity = vds.vds_name;
			model.Host.Entity = vds.host_name;
			model.Port.Entity = vds.port;
			model.IsPm.Entity = vds.pm_enabled;
			model.ManagementIp.Entity = vds.ManagmentIp;
			model.PmType.SelectedItem = vds.pm_type;
			model.PmUserName.Entity = vds.pm_user;
			model.PmPassword.Entity = vds.pm_password;
			//JUICOMMENT_BEGIN
			// * TODO: Need to find a solution for casting valueObjectMap to Dictionary<string, string> 
			// in Java, and conform the C# code to do that when a solution is found
			model.PmOptionsMap = vds.PmOptionsMap;
			//JUICOMMENT_END
			List<storage_pool> dataCenters = DataProvider.GetDataCenterList();
			model.DataCenter.Items = dataCenters;
			model.DataCenter.SelectedItem = Linq.FirstOrDefault(dataCenters, new Linq.DataCenterPredicate(vds.storage_pool_id));
			if (model.DataCenter.SelectedItem == null)
			{
				Linq.FirstOrDefault(dataCenters);
			}

			List<VDSGroup> clusters;
			if (model.Cluster.Items == null)
			{
				model.Cluster.Items = new List<VDSGroup> { new VDSGroup() { name = vds.vds_group_name, ID = vds.vds_group_id, compatibility_version = vds.vds_group_compatibility_version } };
			}
			clusters = (List<VDSGroup>)model.Cluster.Items;
			model.Cluster.SelectedItem = Linq.FirstOrDefault(clusters, new Linq.ClusterPredicate(vds.vds_group_id));
			if (model.Cluster.SelectedItem == null)
			{
				Linq.FirstOrDefault(clusters);
			}

			if (vds.status != VDSStatus.Maintenance && vds.status != VDSStatus.PendingApproval)
			{
				model.DataCenter.IsChangable = false;
				model.DataCenter.ChangeProhibitionReasons.Add("Data Center can be changed only when the Host is in Maintenance mode.");
				model.Cluster.IsChangable = false;
				model.Cluster.ChangeProhibitionReasons.Add("Cluster can be changed only when the Host is in Maintenance mode.");
			}
			else if (SystemTreeSelectedItem != null)
			{
				switch (SystemTreeSelectedItem.Type)
				{
					case SystemTreeItemType.Host:
						model.Name.IsChangable = false;
						model.Name.Info = "Cannot edit Host's Name in this tree context";
						break;
					case SystemTreeItemType.Hosts:
					case SystemTreeItemType.Cluster:
						model.Cluster.IsChangable = false;
						model.Cluster.Info = "Cannot change Host's Cluster in tree context";
						model.DataCenter.IsChangable = false;
						break;
					case SystemTreeItemType.DataCenter:
						storage_pool selectDataCenter = (storage_pool)SystemTreeSelectedItem.Entity;
						model.DataCenter.Items = new List<storage_pool> { selectDataCenter };
						model.DataCenter.SelectedItem = selectDataCenter;
						model.DataCenter.IsChangable = false;
						break;
					default:
						break;
				}
			}
		}

		public void OnApprove()
		{
			OnSave(true);
		}

		public void Restart()
		{
			ConfirmationModel model = new ConfirmationModel();
			ConfirmWindow = model;
			model.Title = "Restart Host(s)";
			model.HashName = "restart_host";
			model.Message = "Are you sure you want to Restart the following Host(s)?";
			//          model.Items = SelectedItems.Cast<VDS>().Select(a => a.vds_name);
			List<string> items = new List<string>();
			foreach (object item in SelectedItems)
			{
				VDS vds = (VDS)item;
				items.Add(vds.vds_name);
			}
			model.Items = items;


			model.Commands.Add(
				new UICommand("OnRestart", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnRestart()
		{
			ConfirmationModel model = (ConfirmationModel)ConfirmWindow;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VDS vds = (VDS)item;
				list.Add(new FenceVdsActionParameters(vds.vds_id, FenceActionType.Restart));
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.RestartVds, list,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					CancelConfirm();
				},
				model
			);
		}

		public void Start()
		{
			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VDS vds = (VDS)item;
				list.Add(new FenceVdsActionParameters(vds.vds_id, FenceActionType.Start));
			}

			Frontend.RunMultipleAction(VdcActionType.StartVds, list,
				result =>
				{
				},
				null
			);
		}

		public void Stop()
		{
			ConfirmationModel model = new ConfirmationModel();
			ConfirmWindow = model;
			model.Title = "Stop Host(s)";
			model.HashName = "stop_host";
			model.Message = "Are you sure you want to Stop the following Host(s)?";
			//          model.Items = SelectedItems.Cast<VDS>().Select(a => a.vds_name);
			List<string> items = new List<string>();
			foreach (object item in SelectedItems)
			{
				VDS vds = (VDS)item;
				items.Add(vds.vds_name);
			}
			model.Items = items;


			model.Commands.Add(
				new UICommand("OnStop", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnStop()
		{
			ConfirmationModel model = (ConfirmationModel)ConfirmWindow;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VDS vds = (VDS)item;
				list.Add(new FenceVdsActionParameters(vds.vds_id, FenceActionType.Stop));
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.StopVds, list,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					CancelConfirm();
				},
				model
			);
		}

		private void ConfigureLocalStorage()
		{
			VDS host = (VDS)SelectedItem;

			if (Window != null)
			{
				return;
			}

			ConfigureLocalStorageModel model = new ConfigureLocalStorageModel();
			Window = model;
			model.Title = "Configure Local Storage";
			model.HashName = "configure_local_storage";

			if (host.vds_type == VDSType.oVirtNode)
			{
				string prefix = DataProvider.GetLocalFSPath();
				if (!String.IsNullOrEmpty(prefix))
				{
					EntityModel pathModel = model.Storage.Path;
					pathModel.Entity = prefix;
					pathModel.IsChangable = false;
				}
			}

			bool hostSupportLocalStorage = false;
			Version version3_0 = new Version(3, 0);
			if (host.supported_cluster_levels != null)
			{
				string[] array = host.supported_cluster_levels.Split(',');
				for (int i = 0; i < array.Length; i++)
				{
					if (version3_0.CompareTo(new Version(array[i])) <= 0)
					{
						hostSupportLocalStorage = true;
						break;
					}
				}
			}

			if (hostSupportLocalStorage)
			{
				string modelMessage;
				model.SetDefaultNames(host, out modelMessage);
				model.Message = modelMessage;

				model.Commands.Add(
					new UICommand("OnConfigureLocalStorage", this)
						{
							Title = "OK",
							IsDefault = true
						});
				model.Commands.Add(
					new UICommand("Cancel", this)
						{
							Title = "Cancel",
							IsCancel = true
						});
			}
			else
			{
				model.Message = "Host doesn't support Local Storage configuration";
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsCancel = true,
						IsDefault = true
					});
			}
		}

		private void OnConfigureLocalStorage()
		{
			ConfigureLocalStorageModel model = (ConfigureLocalStorageModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			model.StartProgress("Configuring Local Storage...");

			Task.Create(this, 1).Run();
		}

		protected override void InitDetailModels()
		{
			base.InitDetailModels();

			HostGeneralModel generalModel = new HostGeneralModel();
			generalModel.RequestEditEvent.addListener(this);
			generalModel.RequestGOToEventsTabEvent.addListener(this);
			ObservableCollection<EntityModel> list = new ObservableCollection<EntityModel>();
			list.Add(generalModel);
			list.Add(new HostVmListModel());
			list.Add(new HostInterfaceListModel());
			HostEventListModel = new HostEventListModel();
			list.Add(HostEventListModel);
			list.Add(new HostHooksListModel());
			list.Add(new PermissionListModel());
			DetailModels = list;
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(HostGeneralModel.RequestEditEventDefinition))
			{
				EditWithPMemphasis();
			}
			if (ev.Equals(HostGeneralModel.RequestGOToEventsTabEventDefinition))
			{
				GoToEventsTab();
			}
		}

		public override bool IsSearchStringMatch(string searchString)
		{
			return searchString.Trim().ToLower().StartsWith("host");
		}

		protected override void SyncSearch()
		{
			base.SyncSearch(VdcQueryType.Search, new SearchParameters(SearchString, SearchType.VDS)
			{
				MaxCount = SearchPageSize
			});
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			AsyncResult = Frontend.RegisterSearch(SearchString, SearchType.VDS, SearchPageSize);
			Items = AsyncResult.Data;
		}

		public void Cancel()
		{
			CancelConfirm();
			Window = null;
		}

		public void CancelConfirm()
		{
			ConfirmWindow = null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected override void ItemsCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
		{
			base.ItemsCollectionChanged(sender, e);

			//Try to select an item corresponding to the system tree selection.
			if (SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type == SystemTreeItemType.Host)
			{
				VDS host = (VDS)SystemTreeSelectedItem.Entity;

				SelectedItem = Linq.FirstOrDefault(Linq.Cast<VDS>(Items), new Linq.HostPredicate(host.vds_id));
			}
		}

		protected override void SelectedItemPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.SelectedItemPropertyChanged(sender, e);

			if (e.PropertyName == "status" || e.PropertyName == "pm_enabled")
			{
				UpdateActionAvailability();
			}
		}

		private void UpdateActionAvailability()
		{
			List<VDS> items = SelectedItems != null ? Linq.Cast<VDS>(SelectedItems) : new List<VDS>();
			bool isAllPMEnabled = Linq.FindAllVDSByPmEnabled(items).Count == items.Count;


			EditCommand.IsExecutionAllowed = items.Count == 1
											&& VdcActionUtils.CanExecute(items, typeof(VDS), VdcActionType.UpdateVds);

			RemoveCommand.IsExecutionAllowed = items.Count > 0
											&& VdcActionUtils.CanExecute(items, typeof(VDS), VdcActionType.RemoveVds);

			ActivateCommand.IsExecutionAllowed = items.Count > 0
											  && VdcActionUtils.CanExecute(items, typeof(VDS), VdcActionType.ActivateVds);

			//or special case where its installation failed but its oVirt node
			bool approveAvailability = items.Count == 1
									   && (VdcActionUtils.CanExecute(items, typeof(VDS), VdcActionType.ApproveVds) ||
										   (items[0].status == VDSStatus.InstallFailed && items[0].vds_type == VDSType.oVirtNode));
			ApproveCommand.IsExecutionAllowed = approveAvailability;
			ApproveCommand.IsAvailable = approveAvailability;

			MaintenanceCommand.IsExecutionAllowed = items.Count > 0
				&& VdcActionUtils.CanExecute(items, typeof(VDS), VdcActionType.MaintananceVds);

			RestartCommand.IsExecutionAllowed = items.Count > 0
											  && VdcActionUtils.CanExecute(items, typeof(VDS), VdcActionType.RestartVds)
											  && isAllPMEnabled;

			StartCommand.IsExecutionAllowed = items.Count > 0
											  && VdcActionUtils.CanExecute(items, typeof(VDS), VdcActionType.StartVds)
											  && isAllPMEnabled;


			StopCommand.IsExecutionAllowed = items.Count > 0
											  && VdcActionUtils.CanExecute(items, typeof(VDS), VdcActionType.StopVds)
											  && isAllPMEnabled;

			IsPowerManagementEnabled = RestartCommand.IsExecutionAllowed ||
									   StartCommand.IsExecutionAllowed ||
									   StopCommand.IsExecutionAllowed;

			ManualFenceCommand.IsExecutionAllowed = items.Count == 1;

			AssignTagsCommand.IsExecutionAllowed = items.Count > 0;

			ConfigureLocalStorageCommand.IsExecutionAllowed = items.Count == 1 && items[0].status == VDSStatus.Maintenance;
			if (!DataProvider.HasAdminSystemPermission() && ConfigureLocalStorageCommand.IsExecutionAllowed)
			{
				ConfigureLocalStorageCommand.IsExecutionAllowed = false;
				ConfigureLocalStorageCommand.ExecuteProhibitionReasons.Add("Configuring local Storage is permitted only to Administrators with System-level permissions");
			}

			//System tree dependent actions.
			bool isAvailable = !(SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type == SystemTreeItemType.Host);

			NewCommand.IsAvailable = isAvailable;
			RemoveCommand.IsAvailable = isAvailable;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == NewCommand)
			{
				New();
			}
			else if (command == EditCommand)
			{
				Edit();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command == ActivateCommand)
			{
				Activate();
			}
			else if (command == MaintenanceCommand)
			{
				Maintenance();
			}
			else if (command == ApproveCommand)
			{
				Approve();
			}
			else if (command == RestartCommand)
			{
				Restart();
			}
			else if (command == StartCommand)
			{
				Start();
			}
			else if (command == StopCommand)
			{
				Stop();
			}
			else if (command == ManualFenceCommand)
			{
				ManualFence();
			}
			else if (command == AssignTagsCommand)
			{
				AssignTags();
			}
			else if (command == ConfigureLocalStorageCommand)
			{
				ConfigureLocalStorage();
			}
			else if (command.Name == "OnAssignTags")
			{
				OnAssignTags();
			}
			else if (command.Name == "OnManualFence")
			{
				OnManualFence();
			}
			else if (command.Name == "OnSaveFalse")
			{
				OnSaveFalse();
			}
			else if (command.Name == "OnSaveInternalFromApprove")
			{
				OnSaveInternalFromApprove();
			}
			else if (command.Name == "OnSaveInternalNotFromApprove")
			{
				OnSaveInternalNotFromApprove();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "CancelConfirm")
			{
				CancelConfirm();
			}
			else if (command.Name == "CancelConfirmFocusPM")
			{
				CancelConfirmFocusPM();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
			else if (command.Name == "OnMaintenance")
			{
				OnMaintenance();
			}
			else if (command.Name == "OnApprove")
			{
				OnApprove();
			}
			else if (command.Name == "OnRestart")
			{
				OnRestart();
			}
			else if (command.Name == "OnStop")
			{
				OnStop();
			}
			else if (command.Name == "OnConfigureLocalStorage")
			{
				OnConfigureLocalStorage();
			}
		}

		public void run(TaskContext context)
		{
			switch ((int)context.State)
			{
				case 1:
					try
					{
						//override default timeout (60 sec) with 10 minutes
						using (TransactionScope scope = new TransactionScope(TransactionScopeOption.Required, new TimeSpan(0, 10, 0)))
						{
							new AddDataCenterRM(this);
							scope.Complete();
						}
					}
					catch (TransactionAbortedException)
					{
						//Do nothing.
					}
					finally
					{
						context.InvokeUIThread(this, 2);
					}
					break;

				case 2:
					StopProgress();

					Cancel();
					break;
			}
		}


		private SystemTreeItemModel systemTreeSelectedItem;
		public SystemTreeItemModel SystemTreeSelectedItem
		{
			get { return systemTreeSelectedItem; }
			set
			{
				if (systemTreeSelectedItem != value)
				{
					systemTreeSelectedItem = value;
					OnSystemTreeSelectedItemChanged();
				}
			}
		}

		private void OnSystemTreeSelectedItemChanged()
		{
			UpdateActionAvailability();
		}
	}

	public class DataBag
	{
		public guid DataCenterId { get; set; }
		public guid ClusterId { get; set; }
		public guid OldClusterId { get; set; }
	}

	public abstract class BaseRM : IEnlistmentNotification
	{
		protected HostListModel Model { get; private set; }
		protected DataBag Data { get; private set; }

		protected BaseRM(HostListModel model, DataBag data)
		{
			Model = model;
			Data = data;
		}

		public abstract void Prepare(PreparingEnlistment preparingEnlistment);
		public abstract void Commit(Enlistment enlistment);
		public abstract void Rollback(Enlistment enlistment);
		public abstract void InDoubt(Enlistment enlistment);
	}


	public class AddDataCenterRM : BaseRM
	{
		public AddDataCenterRM(HostListModel model)
			: base(model, new DataBag())
		{
			Transaction.Current.EnlistVolatile(this, EnlistmentOptions.None);
			ConfigureLocalStorageModel configureLocalStorageModel = (ConfigureLocalStorageModel)Model.Window;
			if (configureLocalStorageModel.DontCreateDataCenter)
			{
				Data.DataCenterId = configureLocalStorageModel.DataCenter.DataCenterId.Value;
			}
			new AddClusterRM(Model, Data);
		}

		public override void Prepare(PreparingEnlistment preparingEnlistment)
		{
			ConfigureLocalStorageModel model = (ConfigureLocalStorageModel)Model.Window;
			if (!model.DontCreateDataCenter)
			{
				DataCenterModel m = model.DataCenter;
				string name = (string)m.Name.Entity;
				//Try to find existing data center with the specified name.
				storage_pool dataCenter = DataProvider.GetDataCenterByName(name);
				if (dataCenter != null)
				{
					Data.DataCenterId = dataCenter.Id;
					preparingEnlistment.Prepared();
				}

				else
				{
					dataCenter = new storage_pool();
					dataCenter.name = name;
					dataCenter.description = (string)m.Description.Entity;
					dataCenter.storage_pool_type = (StorageType)m.StorageTypeList.SelectedItem;
					dataCenter.compatibility_version = (Version)m.Version.SelectedItem;

					VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.AddEmptyStoragePool,
																		new StoragePoolManagementParameter(dataCenter));

					if (returnValue != null && returnValue.Succeeded)
					{
						Data.DataCenterId = (guid)returnValue.ActionReturnValue;
						preparingEnlistment.Prepared();
					}
					else
					{
						preparingEnlistment.ForceRollback();
					}
				}
			}
			else
			{
				preparingEnlistment.Prepared();
			}
		}


		public override void Commit(Enlistment enlistment)
		{
			enlistment.Done();
		}

		public override void Rollback(Enlistment enlistment)
		{
			if (Model == null || Model.SelectedItem == null)
			{
				return;
			}
			VDS host = (VDS)Model.SelectedItem;
			//perform rollback only when the host is in maintenance
			if (host.status != VDSStatus.Maintenance)
			{
				return;
			}

			storage_pool dataCenter = DataProvider.GetDataCenterById(Data.DataCenterId);

			//perform rollback only when the Data Center is un uninitialized
			if (dataCenter.status != StoragePoolStatus.Uninitialized)
			{
				return;
			}

			if (Data.OldClusterId != null && !host.vds_group_id.Value.Equals(Data.OldClusterId.Value))
			{
				//Switch host back to previous cluster.
				VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.ChangeVDSCluster,
					new ChangeVDSClusterParameters(Data.OldClusterId, host.vds_id));

				if (returnValue != null && returnValue.Succeeded)
				{
					//Remove cluster.
					if (Data.ClusterId != null)
					{
						Frontend.RunAction(VdcActionType.RemoveVdsGroup, new VdsGroupParametersBase(Data.ClusterId));
					}

					//Remove data center.
					if (Data.DataCenterId != null)
					{
						Frontend.RunAction(VdcActionType.RemoveStoragePool, new StoragePoolParametersBase(Data.DataCenterId));
					}
				}
			}
		}

		public override void InDoubt(Enlistment enlistment)
		{
		}
	}

	public class AddClusterRM : BaseRM
	{
		public AddClusterRM(HostListModel model, DataBag data)
			: base(model, data)
		{
			Transaction.Current.EnlistVolatile(this, EnlistmentOptions.None);
			ConfigureLocalStorageModel configureLocalStorageModel = (ConfigureLocalStorageModel)Model.Window;
			if (configureLocalStorageModel.DontCreateCluster)
			{
				Data.ClusterId = configureLocalStorageModel.Cluster.ClusterId.Value;
			}
			new ChangeHostClusterRM(Model, Data);
		}

		public override void Prepare(PreparingEnlistment preparingEnlistment)
		{
			ConfigureLocalStorageModel model = (ConfigureLocalStorageModel)Model.Window;
			if (!model.DontCreateCluster)
			{
				ClusterModel m = model.Cluster;

				string name = (string)m.Name.Entity;

				//Try to find existing cluster with the specified name.
				VDSGroup cluster = DataProvider.GetClusterByName(name);
				if (cluster != null)
				{
					Data.ClusterId = cluster.ID;
					preparingEnlistment.Prepared();
				}
				else
				{
					Version version = (Version)m.Version.SelectedItem;

					cluster = new VDSGroup();
					cluster.name = name;
					cluster.description = (string)m.Description.Entity;
					cluster.storage_pool_id = Data.DataCenterId;
					cluster.cpu_name = ((ServerCpu)m.CPU.SelectedItem).CpuName;
					cluster.max_vds_memory_over_commit = m.MemoryOverCommit;
					cluster.TransparentHugepages = version.CompareTo(new Version("3.0")) >= 0;
					cluster.compatibility_version = version;
					cluster.MigrateOnError = m.MigrateOnErrorOption;

					VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.AddVdsGroup,
																		new VdsGroupOperationParameters(cluster));

					if (returnValue != null && returnValue.Succeeded)
					{
						Data.ClusterId = (guid)returnValue.ActionReturnValue;
						preparingEnlistment.Prepared();
					}
					else
					{
						preparingEnlistment.ForceRollback();
					}
				}
			}
			else
			{
				preparingEnlistment.Prepared();
			}
		}

		public override void Commit(Enlistment enlistment)
		{
			enlistment.Done();
		}

		public override void Rollback(Enlistment enlistment)
		{
		}

		public override void InDoubt(Enlistment enlistment)
		{
		}
	}

	public class ChangeHostClusterRM : BaseRM
	{
		public ChangeHostClusterRM(HostListModel model, DataBag data)
			: base(model, data)
		{
			Transaction.Current.EnlistVolatile(this, EnlistmentOptions.None);
			ConfigureLocalStorageModel configureLocalStorageModel = (ConfigureLocalStorageModel)Model.Window;
			//Remember an old cluster, to enable rollback.
			VDS host = (VDS)Model.SelectedItem;
			Data.OldClusterId = host.vds_group_id;
			new AddStorageDomainRM(Model, Data);
		}

		public override void Prepare(PreparingEnlistment preparingEnlistment)
		{
			ConfigureLocalStorageModel model = (ConfigureLocalStorageModel)Model.Window;
			if (!model.DontChangeHostCluster)
			{
				VDS host = (VDS)Model.SelectedItem;
				VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.ChangeVDSCluster,
																	new ChangeVDSClusterParameters(Data.ClusterId, host.vds_id));

				if (returnValue != null && returnValue.Succeeded)
				{
					preparingEnlistment.Prepared();
				}
				else
				{
					preparingEnlistment.ForceRollback();
				}
			}
			else
			{
				preparingEnlistment.Prepared();
			}
		}

		public override void Commit(Enlistment enlistment)
		{
			enlistment.Done();
		}

		public override void Rollback(Enlistment enlistment)
		{
		}

		public override void InDoubt(Enlistment enlistment)
		{
		}
	}

	public class AddStorageDomainRM : BaseRM
	{
		const int WaitInterval = 5000;
		const int WaitTries = 6;

		public AddStorageDomainRM(HostListModel model, DataBag data)
			: base(model, data)
		{
			Transaction.Current.EnlistVolatile(this, EnlistmentOptions.None);
		}

		public override void Prepare(PreparingEnlistment preparingEnlistment)
		{
			VDS host = (VDS)Model.SelectedItem;
			ConfigureLocalStorageModel model = (ConfigureLocalStorageModel)Model.Window;

			//Activate host.
			VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.ActivateVds,
					new VdsActionParameters(host.vds_id));

			if (returnValue == null || !returnValue.Succeeded)
			{
				preparingEnlistment.ForceRollback();
				return;
			}

			//Wait for a host to be Up.
			for (int i = 0; i <= WaitTries; i++)
			{
				if (i == WaitTries)
				{
					preparingEnlistment.ForceRollback();
					return;
				}

				VDS tmpHost = DataProvider.GetHostById(host.vds_id);
				if (tmpHost.status != VDSStatus.Up)
				{
					//Wrap Thread.Sleep with try/catch to pass conversion to Java.
					try
					{
						Thread.Sleep(WaitInterval);
					}
					catch (InterruptedException)
					{
					}
				}
				else
				{
					break;
				}
			}

			//Add storage domain.
			storage_server_connections connection =
				new storage_server_connections
				{
					connection = (string)model.Storage.Path.Entity,
					storage_type = StorageType.LOCALFS
				};

			storage_domain_static storageDomain = new storage_domain_static();
			storageDomain.storage_type = StorageType.LOCALFS;
			storageDomain.storage_domain_type = StorageDomainType.Data;
			storageDomain.storage_name = (string)model.FormattedStorageName.Entity;

			returnValue = Frontend.RunAction(VdcActionType.AddStorageServerConnection,
				new StorageServerConnectionParametersBase(
					connection,
					host.vds_id
				)
			);

			if (returnValue == null || !returnValue.Succeeded)
			{
				//Don't rollback, just throw exception to indicate failure at this step.
				throw new TransactionAbortedException();
			}

			storageDomain.storage = (string)returnValue.ActionReturnValue;

			returnValue = Frontend.RunAction(VdcActionType.AddLocalStorageDomain,
				new StorageDomainManagementParameter(storageDomain)
				{
					VdsId = host.vds_id
				}
			);

			//Clean up connection.
			if (returnValue == null || !returnValue.Succeeded)
			{
				Frontend.RunAction(VdcActionType.RemoveStorageServerConnection,
								   new StorageServerConnectionParametersBase(connection, host.vds_id));

				throw new TransactionAbortedException();
			}

			preparingEnlistment.Prepared();
		}

		public override void Commit(Enlistment enlistment)
		{
			enlistment.Done();
		}

		public override void Rollback(Enlistment enlistment)
		{
		}

		public override void InDoubt(Enlistment enlistment)
		{
		}
	}
}