using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Reflection.Emit;
using System.Text;
using System.Web;
using System.Windows;
using System.Windows.Forms;
using System.Windows.Forms.Integration;
using System.Xml.Linq;
using org.ovirt.engine.ui.uicommon;
using org.ovirt.engine.ui.uicompat;
using Microsoft.Practices.Unity;
using org.ovirt.engine.ui.uicommon.models.vms;

namespace UI.WPFClient
{
	public class Spice : DependencyObject, ISpice
	{
		private readonly ActiveXContainer activeXContainer;

		#region ISpice Members

		public Event DisconnectedEvent { get; private set; }
		public Event MenuItemSelectedEvent { get; private set; }
		public Event UsbAutoShareChangedEvent { get; private set; }

		public Version CurrentVersion
		{
			get
			{
				string version = @"0.0.0.0";
				var spice = CreateSpiceXControl();
				if (spice != null)
				{
					version = (string)spiceType.GetProperty("Version").GetValue(spice, null);
					spiceType.GetMethod("Dispose").Invoke(spice, null);
				}

				logger.Info("Spice client version is " + version + ".");

				return new Version(version);
			}
		}

		public bool IsInstalled
		{
			get
			{
				bool installed = false;
				var spice = CreateSpiceXControl();
				if (spice != null)
				{
					installed = true;
					spiceType.GetMethod("Dispose").Invoke(spice, null);
				}
				return installed;
			}
		}

		public Version DesiredVersion { get; set; }
		public int Port { get; set; }
		public string Host { get; set; }
		public bool FullScreen { get; set; }
		public string Password { get; set; }
		public int NumberOfMonitors { get; set; }
		public int UsbListenPort { get; set; }
		public bool AdminConsole { get; set; }
		public string GuestHostName { get; set; }
		public int SecurePort { get; set; }
		public string SslChanels { get; set; }
		public string CipherSuite { get; set; }
		public string HostSubject { get; set; }
		public string TrustStore { get; set; }
		public string Title { get; set; }
		public string HotKey { get; set; }
		public string[] LocalizedStrings { get; set; }
		public string Menu { get; set; }

		public string GuestID { get; set; }
		public bool NoTaskMgrExecution { get; set; }
		public bool SendCtrlAltDelete { get; set; }
		public bool UsbAutoShare { get; set; }
		public string UsbFilter { get; set; }
		
		public void Connect()
		{
			var spiceX = spiceAssembly.CreateInstance("AxSpiceXLib.AxOSpiceX");
			activeXContainer.Add(spiceX);

			spiceType.GetProperty("HostIP").SetValue(spiceX, Host, null);
			logger.Debug("HostIP:" + Host);

			spiceType.GetProperty("Port").SetValue(spiceX, (UInt16)Port, null);
			logger.Debug("Port:" + Port);

			spiceType.GetProperty("FullScreen").SetValue(spiceX, Convert.ToUInt32(FullScreen), null);
			logger.Debug("FullScreen:" + FullScreen);

			if (!String.IsNullOrEmpty(Password))
			{
				spiceType.GetProperty("Password").SetValue(spiceX, Password, null);
				logger.Debug("Password:" + Password);
			}

			spiceType.GetProperty("NumberOfMonitors").SetValue(spiceX, (uint)NumberOfMonitors, null);
			logger.Debug("NumberOfMonitors:" + NumberOfMonitors);

			spiceType.GetProperty("UsbListenPort").SetValue(spiceX, Convert.ToUInt16(UsbListenPort), null);
			logger.Debug("UsbListenPort:" + UsbListenPort);

			spiceType.GetProperty("AdminConsole").SetValue(spiceX, AdminConsole ? 1 : 0, null);
			logger.Debug("AdminConsole:" + AdminConsole);

			spiceType.GetProperty("GuestHostName").SetValue(spiceX, GuestHostName, null);
			logger.Debug("GuestHostName:" + GuestHostName);

			spiceType.GetProperty("SecurePort").SetValue(spiceX, (ushort)SecurePort, null);
			logger.Debug("SecurePort:" + SecurePort);

			if (!String.IsNullOrEmpty(SslChanels))
			{
				spiceType.GetProperty("SSLChannels").SetValue(spiceX, SslChanels, null);
				logger.Debug("SSLChannels:" + SslChanels);
			}

			if (!String.IsNullOrEmpty(CipherSuite))
			{
				spiceType.GetProperty("CipherSuite").SetValue(spiceX, CipherSuite, null);
				logger.Debug("CipherSuite:" + CipherSuite);
			}

			if (!String.IsNullOrEmpty(HostSubject))
			{
				spiceType.GetProperty("HostSubject").SetValue(spiceX, HostSubject, null);
				logger.Debug("HostSubject:" + HostSubject);
			}

			if (!String.IsNullOrEmpty(TrustStore))
			{
				spiceType.GetProperty("TrustStore").SetValue(spiceX, TrustStore, null);
				logger.Debug("TrustStore:" + TrustStore);
			}

			spiceType.GetProperty("Title").SetValue(spiceX, Title, null);
			spiceType.GetProperty("HotKey").SetValue(spiceX, HotKey, null);
			spiceType.GetMethod("SetLanguageStrings").Invoke(spiceX, new object[] { LocalizedStrings[0], LocalizedStrings[1] });
			spiceType.GetProperty("DynamicMenu").SetValue(spiceX, Menu, null);

			delegation = delegationAssembly.CreateInstance("SpiceDelegation.EventDelegation");
			delegationType.GetProperty("Spice").SetValue(delegation, spiceX, null);
			delegationType.GetMethod("SetOnDisconnectedHandler").Invoke(delegation, new object[] { onDisconnectedHandler });
			delegationType.GetMethod("SetOnMenuItemSelectedHandler").Invoke(delegation, new object[] { onMenuItemSelectedHandler });

			spiceType.GetMethod("Connect").Invoke(spiceX, null);
		}

		public void Install()
		{
			Process.Start(App.RelativePath + HttpUtility.UrlPathEncode(string.Format("RHEV Spice Client{0}.msi", Is64BitProcess ? " 64" : String.Empty)));
		}

		void spiceX_OnMenuItemSelected(object sender, int itemId)
		{
			MenuItemSelectedEvent.raise(this, new SpiceMenuItemEventArgs(itemId));
		}

		void spiceX_OnDisconnected(object sender, int errCode)
		{
			delegationType.GetMethod("RemoveOnDisconnectedHandler").Invoke(delegation, null);
			delegationType.GetMethod("RemoveOnMenuItemSelectedHandler").Invoke(delegation, null);
			delegationType.GetProperty("Spice").SetValue(delegation, null, null);
			delegation = null;

			activeXContainer.Remove(sender);
			DisconnectedEvent.raise(this, new ErrorCodeEventArgs(errCode));
		}

		#endregion

		public ILogger logger;


		private object CreateSpiceXControl()
		{
			object spice = null;
			try
			{
				spice = spiceAssembly.CreateInstance("AxSpiceXLib.AxOSpiceX");
				spiceType.GetMethod("CreateControl").Invoke(spice, null);
			}
			catch (Exception ex)
			{
				logger.Error("CreateSpiceXControl Exception:", ex);

				if (spice != null)
				{
					spiceType.GetMethod("Dispose").Invoke(spice, null);
				}
				spice = null;
			}

			return spice;
		}

		private readonly Assembly spiceAssembly;
		private readonly Type spiceType;
		private readonly Assembly delegationAssembly;
		private readonly Type delegationType;
		private readonly Action<object, int> onDisconnectedHandler;
		private readonly Action<object, int> onMenuItemSelectedHandler;
		private object delegation;

		public Spice(IUnityContainer container)
		{
			DisconnectedEvent = new Event(SpiceConsoleModel.SpiceDisconnectedEventDefinition);
			MenuItemSelectedEvent = new Event(SpiceConsoleModel.SpiceMenuItemSelectedEventDefinition);

			logger = (ILogger)TypeResolver.Instance.Resolve(typeof(ILogger));

			activeXContainer = container.Resolve<ActiveXContainer>();

			//Load assembly wrapping Spice ActiveX depending on process bits.
			FileInfo fi = new FileInfo(Assembly.GetExecutingAssembly().Location);
			string basePath = Path.Combine(fi.Directory.FullName, Path.Combine("Spice", Is64BitProcess ? "amd64" : "x86"));

			spiceAssembly = Assembly.LoadFrom(Path.Combine(basePath, "Spice.dll"));
			spiceType = spiceAssembly.GetType("AxSpiceXLib.AxOSpiceX");

			delegationAssembly = Assembly.LoadFrom(Path.Combine(basePath, "SpiceDelegation.dll"));
			delegationType = delegationAssembly.GetType("SpiceDelegation.EventDelegation");

			onDisconnectedHandler = new Action<object, int>(spiceX_OnDisconnected);
			onMenuItemSelectedHandler = new Action<object, int>(spiceX_OnMenuItemSelected);
		}

		private bool Is64BitProcess
		{
			get
			{
				return IntPtr.Size * 8 == 64;
			}
		}
	}
}
