/*
Copyright 2019 Property of IBM.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package main

import (
	"flag"
	"net"
	"os"
	"time"

	"github.com/open-cluster-management/cis-controller/pkg/apis"
	cisv1alpha1 "github.com/open-cluster-management/cis-controller/pkg/apis/cis/v1alpha1"
	"github.com/open-cluster-management/cis-controller/pkg/controller"
	ciscontroller "github.com/open-cluster-management/cis-controller/pkg/controller/cispolicy"
	"github.com/open-cluster-management/cis-controller/pkg/webhook"
	_ "k8s.io/client-go/plugin/pkg/client/auth/gcp"
	"sigs.k8s.io/controller-runtime/pkg/client/config"
	"sigs.k8s.io/controller-runtime/pkg/manager"
	logf "sigs.k8s.io/controller-runtime/pkg/runtime/log"
	"sigs.k8s.io/controller-runtime/pkg/runtime/signals"
)

func main() {
	var clusterName, namespace, eventOnParent, cosURL string
	var frequency uint
	flag.UintVar(&frequency, "update-frequency", 60, "The status update frequency (in minutes) of a CIS policy")
	flag.StringVar(&namespace, "watch-ns", "default", "Watched Kubernetes namespace")
	flag.StringVar(&clusterName, "cluster-name", "managed-cluster", "Name of the cluster")
	flag.StringVar(&eventOnParent, "parent-event", "ifpresent", "to also send status events on parent policy. options are: yes/no/ifpresent")
	flag.StringVar(&cosURL, "cos-url", "cis-controller-minio:9000", "URL of COS where the CIS results are stored")

	flag.Parse()
	logf.SetLogger(logf.ZapLogger(false))
	log := logf.Log.WithName("cis-main-entrypoint")

	// Get a config to talk to the apiserver
	log.Info("setting up client for manager")
	cfg, err := config.GetConfig()
	if err != nil {
		log.Error(err, "unable to set up client config")
		os.Exit(1)
	}

	// Create a new Cmd to provide shared dependencies and start components
	log.Info("setting up manager")
	//SyncPeriod *time.Duration
	freq := time.Duration(int64(frequency)) * time.Minute
	opts := manager.Options{SyncPeriod: &freq}

	mgr, err := manager.New(cfg, opts)
	if err != nil {
		log.Error(err, "unable to set up overall controller manager")
		os.Exit(1)
	}

	log.Info("Registering Components.")

	// Setup Scheme for all resources
	log.Info("setting up scheme")
	if err := apis.AddToScheme(mgr.GetScheme()); err != nil {
		log.Error(err, "unable add APIs to scheme")
		os.Exit(1)
	}

	 // wait for cos to be available
        for {
		log.Info("Checking for cos")
                _, err := net.Dial("tcp", cosURL)
                if err == nil {
                        break
                } else {
                        log.Info("Waiting for cos endpoint to be available...")
                        time.Sleep(5 * time.Second)
                }
        }


	// Setup all Controllers
	log.Info("Setting up controller")
	if err := controller.AddToManager(mgr); err != nil {
		log.Error(err, "unable to register controllers to the manager")
		os.Exit(1)
	}

	log.Info("setting up webhooks")
	if err := webhook.AddToManager(mgr); err != nil {
		log.Error(err, "unable to register webhooks to the manager")
		os.Exit(1)
	}
	//cCfg store the config of the controller itself
	cCfg := &cisv1alpha1.ControllerConfig{
		ClusterName:   clusterName,
		Namespace:     namespace,
		EventOnParent: eventOnParent,
		CosURL:        cosURL,
	}
	ciscontroller.Initialize(cCfg)

	// Start the Cmd
	log.Info("Starting the Cmd.")
	if err := mgr.Start(signals.SetupSignalHandler()); err != nil {
		log.Error(err, "unable to run the manager")
		os.Exit(1)
	}
}
