// Licensed Materials - Property of IBM
// (c) Copyright IBM Corporation 2018. All Rights Reserved.
// Note to U.S. Government Users Restricted Rights:
// Use, duplication or disclosure restricted by GSA ADP Schedule
// Contract with IBM Corp.

package cispolicy

import (
	"context"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"path"
	"strconv"
	"strings"

	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"

	cisv1alpha1 "github.com/open-cluster-management/cis-controller/pkg/apis/cis/v1alpha1"
	cosminio "github.com/open-cluster-management/cis-controller/pkg/controller/cosminio"
	appsv1 "k8s.io/api/apps/v1"
	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/api/errors"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/client-go/tools/record"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller"
	"sigs.k8s.io/controller-runtime/pkg/handler"
	"sigs.k8s.io/controller-runtime/pkg/manager"
	"sigs.k8s.io/controller-runtime/pkg/reconcile"
	"sigs.k8s.io/controller-runtime/pkg/source"
)

//Finalizer used to ensure consistency when deleting a CRD
const Finalizer = "finalizer.ciscontroller.cispolicies.ibm.com"

// CtrCfg store the config of the controller itself
var CtrCfg *cisv1alpha1.ControllerConfig

// NamespaceWatched defines which namespace we can watch for the mutation policies and ignore others
var NamespaceWatched string

var cosCtl *cosminio.CosController

var cosSecretFilepath = "/etc/ciscos/" // #nosec G101

var _ reconcile.Reconciler = &ReconcileCisPolicy{}

var ClusterName string

// EventOnParent specifies if we also want to send events to the parent policy. Available options are yes/no/ifpresent
var EventOnParent string

// Initialize to initialize some controller varaibles
func Initialize(cfg *cisv1alpha1.ControllerConfig) {
	CtrCfg = cfg
	NamespaceWatched = CtrCfg.Namespace
	EventOnParent = strings.ToLower(CtrCfg.EventOnParent)
	if ClusterName == "" {
		//fmt.Printf("\n\n `%+v` \n\n", ClusterName)
		if NamespaceWatched != "default" {
			ClusterName = NamespaceWatched
		} else {

			ClusterName = CtrCfg.ClusterName
		}
	}
	log.Printf("Initializing variables: ClusterName=`%v`, NamespaceWatched=`%v`, EventOnParent=`%v`", ClusterName, NamespaceWatched, EventOnParent)

}

// Add creates a new CisPolicy Controller and adds it to the Manager with default RBAC. The Manager will set fields on the Controller
// and Start it when the Manager is Started.
// USER ACTION REQUIRED: update cmd/manager/main.go to call this mcm.Add(mgr) to install this Controller
func Add(mgr manager.Manager) error {
	return add(mgr, newReconciler(mgr))
}

// newReconciler returns a new reconcile.Reconciler
func newReconciler(mgr manager.Manager) reconcile.Reconciler {
	//Event: return &ReconcileCisPolicy{Client: mgr.GetClient(), scheme: mgr.GetScheme()}
	return &ReconcileCisPolicy{Client: mgr.GetClient(), scheme: mgr.GetScheme(), recorder: mgr.GetRecorder("cis-controller")}

}

// add adds a new Controller to mgr with r as the reconcile.Reconciler
func add(mgr manager.Manager, r reconcile.Reconciler) error {
	// Create a new controller
	c, err := controller.New("cispolicy-controller", mgr, controller.Options{Reconciler: r})
	if err != nil {
		return err
	}

	// Watch for changes to CisPolicy
	err = c.Watch(&source.Kind{Type: &cisv1alpha1.CisPolicy{}}, &handler.EnqueueRequestForObject{})
	if err != nil {
		return err
	}

	// TODO(user): Modify this to be the types you create
	// Uncomment watch a Deployment created by CisPolicy - change this for objects you create
	err = c.Watch(&source.Kind{Type: &appsv1.Deployment{}}, &handler.EnqueueRequestForOwner{
		IsController: true,
		OwnerType:    &cisv1alpha1.CisPolicy{},
	})
	if err != nil {
		return err
	}

	return nil
}

// ReconcileCisPolicy reconciles a CisPolicy object
type ReconcileCisPolicy struct {
	client.Client
	scheme   *runtime.Scheme
	recorder record.EventRecorder
}

// Reconcile reads that state of the cluster for a CisPolicy object and makes changes based on the state read
// and what is in the CisPolicy.Spec
// TODO(user): Modify this Reconcile function to implement your Controller logic.  The scaffolding writes
// a Deployment as an example
// Automatically generate RBAC rules to allow the Controller to read and write Deployments
// +kubebuilder:rbac:groups=apps,resources=deployments,verbs=get;list;watch;create;update;patch;delete
//+kubebuilder:rbac:groups="",resources=events,verbs=create;patch
// +kubebuilder:rbac:groups=mcm.ibm.com,resources=cispolicies,verbs=get;list;watch;create;update;patch;delete
func (r *ReconcileCisPolicy) Reconcile(request reconcile.Request) (reconcile.Result, error) {
	// Fetch the CisPolicy instance
	instance := &cisv1alpha1.CisPolicy{}
	err := r.Get(context.TODO(), request.NamespacedName, instance)
	if err != nil {
		if errors.IsNotFound(err) {
			// Object not found, return.  Created objects are automatically garbage collected.
			// For additional cleanup logic use finalizers.
			return reconcile.Result{}, nil
		}
		// Error reading the object - requeue the request.
		return reconcile.Result{}, err
	}

	if instance.ObjectMeta.DeletionTimestamp.IsZero() {
		if instance.Spec.Severity == "" {
			//making sure severity is there
			instance.Spec.Severity = "medium"
			if err := r.Update(context.Background(), instance); err != nil {
				return reconcile.Result{Requeue: false}, nil
			}
		}
	} else { // The object is being deleted
		return reconcile.Result{}, nil
	}

	if instance.Status.ObservedGeneration != instance.ObjectMeta.Generation {
		instance.Status.ObservedGeneration = instance.ObjectMeta.Generation
		if err = r.handleCisPolicyStatus(instance); err != nil {
			return reconcile.Result{Requeue: true}, nil
		}
		log.Printf("successfully processed the CisPolicy `%v`, with resource version = `%v`", instance.Name, instance.ResourceVersion)
	}

	return reconcile.Result{}, nil
}

//=================================================================
//initializeCisPolicyStatus making sure the maps are initialized
func (r *ReconcileCisPolicy) initializeCisPolicyStatus(instance *cisv1alpha1.CisPolicy) {
	labelClusterName := ExtractNamespaceLabel(instance)
	if labelClusterName != "" {
		ClusterName = labelClusterName
	} else {
		log.Printf("setting the cluster-namespace to %v for CisPolicy %v", ClusterName, instance.Name)
		instance.ObjectMeta.Labels["cluster-namespace"] = ClusterName
	}

	if instance.Status.CisPolicyStatus == nil {
		instance.Status.CisPolicyStatus = make(map[string]*cisv1alpha1.CisPerClusterStatus)
	}

	if _, ok := instance.Status.CisPolicyStatus[ClusterName]; !ok {
		instance.Status.CisPolicyStatus[ClusterName] = &cisv1alpha1.CisPerClusterStatus{
			Compliancy:  cisv1alpha1.UnknownCompliancy,
			ClusterName: ClusterName,
		}
	}
}

//=================================================================
// get cos credentials from secret
func getCosConfig() cosminio.CosConfig {

	var cred cosminio.Credentials
	var region string

	if _, err := os.Stat(path.Join(cosSecretFilepath, "secret_key")); err == nil {
		secretKey, _ := ioutil.ReadFile(path.Join(cosSecretFilepath, "secret_key"))
		cred.SecretKey = strings.Trim(string(secretKey), "\n")

	}
	if _, err := os.Stat(path.Join(cosSecretFilepath, "access_key")); err == nil {
		accessKey, _ := ioutil.ReadFile(path.Join(cosSecretFilepath, "access_key"))
		cred.AccessKey = strings.Trim(string(accessKey), "\n")
	}
	if _, err := os.Stat(path.Join(cosSecretFilepath, "location")); err == nil {
		location, _ := ioutil.ReadFile(path.Join(cosSecretFilepath, "location"))
		region = strings.Trim(string(location), "\n")
	}

	return cosminio.CosConfig{
		EndPoint: CtrCfg.CosURL,
		Creds:    cred,
		Region:   region,
		UseSSL:   true,
	}
}

//=================================================================
//handleCisPolicyStatus making sure the status is up to date
func (r *ReconcileCisPolicy) handleCisPolicyStatus(instance *cisv1alpha1.CisPolicy) error {
	var err error
	r.initializeCisPolicyStatus(instance)
	if cosCtl == nil {
		cosCtl = &cosminio.CosController{}
	}
	if !cosCtl.Initialized {
		//TODO don't hard code those values
		cosCtl.Config = getCosConfig()
		err = cosminio.InitializeCosController(cosCtl)
		if err != nil {
			return err
		}
	}

	var oldStatus cisv1alpha1.ComplianceState

	if instance.Status.CisPolicyStatus[ClusterName] == nil {
		oldStatus = cisv1alpha1.UnknownCompliancy
	} else {
		oldStatus = instance.Status.CisPolicyStatus[ClusterName].Compliancy
	}

	log.Printf("olds status is: %v\n", oldStatus)

	instance = cosCtl.UpdateCisCompliancy(instance) //compare CIS results and update status
	err = r.handleCisPolicyEvents(instance)
	if err != nil {
		return err
	}

	//if status changed, trigger an event on the parent.
	log.Printf("new status is: %v\n", instance.Status.CisPolicyStatus[ClusterName].Compliancy)
	if oldStatus != instance.Status.CisPolicyStatus[ClusterName].Compliancy {
		if EventOnParent != "no" && len(instance.OwnerReferences) > 0 {
			parentPlc := createParentPolicy(instance)
			reason := fmt.Sprintf("policy: %s/%s", instance.Namespace, instance.Name)
			var message string
			if instance.Status.CisPolicyStatus[ClusterName].Risk.HighestRiskScore > 0 {
				message = fmt.Sprintf("%v; highest risk score is `%v`, risk category is `%v`",
					string(instance.Status.CisPolicyStatus[ClusterName].Compliancy),
					fmt.Sprintf("%.2f", instance.Status.CisPolicyStatus[ClusterName].Risk.HighestRiskScore),
					instance.Status.CisPolicyStatus[ClusterName].Risk.RiskCategory)

			} else {
				message = fmt.Sprintf("%v", string(instance.Status.CisPolicyStatus[ClusterName].Compliancy))

			}

			//create event on parent
			if instance.Status.CisPolicyStatus[ClusterName].Compliancy == cisv1alpha1.NonCompliant {
				r.recorder.Event(&parentPlc, corev1.EventTypeWarning, reason, message)
			} else {
				r.recorder.Event(&parentPlc, corev1.EventTypeNormal, reason, message)
			}

			log.Printf("creating event on parent policy: %v/%v\n", parentPlc.Namespace, parentPlc.Name)
		} else {
			log.Printf("No event created. Either EventOnParent is no or OwnerReferences is no there.\n")
		}

	}

	err = r.Status().Update(context.Background(), instance)

	return err
}

// ExtractNamespaceLabel to find out the cluster-namespace from the label
func ExtractNamespaceLabel(instance *cisv1alpha1.CisPolicy) string {
	if instance.ObjectMeta.Labels == nil {
		return ""
	}
	if _, ok := instance.ObjectMeta.Labels["cluster-namespace"]; ok {
		return instance.ObjectMeta.Labels["cluster-namespace"]
	}
	return ""
}

// to event on the parent object
func createParentPolicy(instance *cisv1alpha1.CisPolicy) cisv1alpha1.Policy {

	ns := ExtractNamespaceLabel(instance)
	if ns == "" {
		ns = NamespaceWatched
	}
	plc := cisv1alpha1.Policy{
		ObjectMeta: metav1.ObjectMeta{
			Name:      instance.OwnerReferences[0].Name,
			Namespace: ns, // we are making an assumption here that the parent policy is in the watched-namespace passed as flag
			UID:       instance.OwnerReferences[0].UID,
		},
		TypeMeta: metav1.TypeMeta{
			Kind:       "Policy",
			APIVersion: " policy.mcm.ibm.com/v1alpha1",
		},
	}
	return plc
}

//handling same generation issue
func isNewCisPolicyGeneration(instance *cisv1alpha1.CisPolicy) bool {
	if instance.ObjectMeta.Annotations == nil {
		return true
	}
	if val, ok := instance.ObjectMeta.Annotations["generation"]; ok {
		gen, err := strconv.Atoi(val)
		log.Printf("the CR generation annotation is %v, the metadata generation is %v\n", val, instance.ObjectMeta.Generation)
		if err != nil {
			log.Printf("error parsing the CR generation %v\n", err)
			return false
		}
		if gen == int(instance.ObjectMeta.Generation) {
			log.Printf("same CR generation => ignoring the update\n")
			return false
		}
	} else {
		instance.ObjectMeta.Annotations["generation"] = strconv.Itoa(int(instance.ObjectMeta.Generation))
		log.Printf("the CR generation annotation was missing and is created, the metadata generation is %v\n", instance.ObjectMeta.Generation)
	}
	return true
}

//handling same version issue
func isNewCisPolicyVersion(instance *cisv1alpha1.CisPolicy) bool {
	if val, ok := instance.ObjectMeta.Annotations["version"]; ok {
		log.Printf("the CR generation version is %v, the metadata version is %v\n", val, instance.ResourceVersion)
		if instance.ObjectMeta.Annotations["version"] == instance.ResourceVersion {
			log.Printf("same CR version => ignoring the update\n")
			return false
		}
		instance.ObjectMeta.Annotations["version"] = instance.ResourceVersion
		return true
	}
	if instance.ObjectMeta.Annotations == nil {
		instance.ObjectMeta.Annotations = make(map[string]string)
	}
	instance.ObjectMeta.Annotations["version"] = instance.ResourceVersion
	log.Printf("the CR version annotation was missing and is created, the metadata version is %v\n", instance.ResourceVersion)

	return true
}

// handleCisPolicyEvents create events if risk is high
func (r *ReconcileCisPolicy) handleCisPolicyEvents(instance *cisv1alpha1.CisPolicy) error {
	if instance.Status.CisPolicyStatus[ClusterName].Risk.HighestRiskScore >= 7 {
		instance.Status.CisPolicyStatus[ClusterName].Risk.RiskCategory = "high"
		r.recorder.Event(instance, "Warning", "High Risk", fmt.Sprintf("High Risk detected!"))
	}
	if instance.Status.CisPolicyStatus[ClusterName].Risk.HighestRiskScore >= 4 && instance.Status.CisPolicyStatus[ClusterName].Risk.HighestRiskScore < 7 {
		instance.Status.CisPolicyStatus[ClusterName].Risk.RiskCategory = "medium"
		r.recorder.Event(instance, "Warning", "Medium Risk", fmt.Sprintf("Medium Risk detected!"))
	}
	if instance.Status.CisPolicyStatus[ClusterName].Risk.HighestRiskScore > 0 && instance.Status.CisPolicyStatus[ClusterName].Risk.HighestRiskScore < 4 {
		instance.Status.CisPolicyStatus[ClusterName].Risk.RiskCategory = "low"
		r.recorder.Event(instance, "Warning", "Low Risk", fmt.Sprintf("Low Risk detected!"))
	}
	return nil
}

//=================================================================
// we need to the list of nodes to figure out which cloud COS bucket to call and get the results
//listClusterNodes returns a list of the cluster nodes
func (r *ReconcileCisPolicy) listClusterNodes() (nodeList *corev1.NodeList, err error) {
	nodeList = &corev1.NodeList{}

	err = r.List(context.TODO(), &client.ListOptions{}, nodeList)
	if err != nil {
		return nil, err
	}
	return nodeList, err
}

//=================================================================
//deleteExternalDependency in case the CRD was related to non-k8s resource
func (r *ReconcileCisPolicy) deleteExternalDependency(instance *cisv1alpha1.CisPolicy) error {
	log.Println("deleting the CRD")
	// Ensure that delete implementation is idempotent and safe to invoke
	// multiple types for same object.
	return nil
}

//=================================================================
// Helper functions to check if a string exists in a slice of strings.
func containsString(slice []string, s string) bool {
	for _, item := range slice {
		if item == s {
			return true
		}
	}
	return false
}

//=================================================================
// Helper functions to remove a string from a slice of strings.
func removeString(slice []string, s string) (result []string) {
	for _, item := range slice {
		if item == s {
			continue
		}
		result = append(result, item)
	}
	return
}
