/*
Copyright 2019 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package disks

import (
	"context"

	"github.com/pkg/errors"
	"k8s.io/klog"
	azure "sigs.k8s.io/cluster-api-provider-azure/cloud"
)

// Spec specification for disk
type Spec struct {
	Name string
}

// Get on disk is currently no-op. OS disks should only be deleted and will create with the VM automatically.
func (s *Service) Get(ctx context.Context, spec interface{}) (interface{}, error) {
	return Spec{}, nil
}

// Reconcile on disk is currently no-op. OS disks should only be deleted and will create with the VM automatically.
func (s *Service) Reconcile(ctx context.Context, spec interface{}) error {
	return nil
}

// Delete deletes the disk associated with a VM.
func (s *Service) Delete(ctx context.Context, spec interface{}) error {
	diskSpec, ok := spec.(*Spec)
	if !ok {
		return errors.New("invalid disk specification")
	}
	klog.V(2).Infof("deleting disk %s", diskSpec.Name)
	err := s.Client.Delete(ctx, s.Scope.ResourceGroup(), diskSpec.Name)
	if err != nil && azure.ResourceNotFound(err) {
		// already deleted
		return nil
	}
	if err != nil {
		return errors.Wrapf(err, "failed to delete disk %s in resource group %s", diskSpec.Name, s.Scope.ResourceGroup())
	}

	klog.V(2).Infof("successfully deleted disk %s", diskSpec.Name)
	return nil
}
