/*
Copyright 2020 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package networkinterfaces

import (
	"context"
	"net/http"
	"testing"

	"sigs.k8s.io/cluster-api-provider-azure/cloud/services/inboundnatrules/mock_inboundnatrules"
	"sigs.k8s.io/cluster-api-provider-azure/cloud/services/internalloadbalancers/mock_internalloadbalancers"
	"sigs.k8s.io/cluster-api-provider-azure/cloud/services/networkinterfaces/mock_networkinterfaces"
	"sigs.k8s.io/cluster-api-provider-azure/cloud/services/publicips/mock_publicips"
	"sigs.k8s.io/cluster-api-provider-azure/cloud/services/publicloadbalancers/mock_publicloadbalancers"
	"sigs.k8s.io/cluster-api-provider-azure/cloud/services/subnets/mock_subnets"

	"github.com/Azure/go-autorest/autorest"
	"github.com/Azure/go-autorest/autorest/to"
	"github.com/golang/mock/gomock"

	network "github.com/Azure/azure-sdk-for-go/services/network/mgmt/2019-06-01/network"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/client-go/kubernetes/scheme"
	"k8s.io/utils/pointer"
	infrav1 "sigs.k8s.io/cluster-api-provider-azure/api/v1alpha3"
	"sigs.k8s.io/cluster-api-provider-azure/cloud/scope"
	clusterv1 "sigs.k8s.io/cluster-api/api/v1alpha3"
	"sigs.k8s.io/controller-runtime/pkg/client/fake"
)

const expectedInvalidSpec = "invalid network interface specification"

func init() {
	clusterv1.AddToScheme(scheme.Scheme)
}

func TestInvalidNetworkInterface(t *testing.T) {
	mockCtrl := gomock.NewController(t)
	netInterfaceMock := mock_networkinterfaces.NewMockClient(mockCtrl)

	cluster := &clusterv1.Cluster{
		ObjectMeta: metav1.ObjectMeta{Name: "test-cluster"},
	}

	client := fake.NewFakeClient(cluster)

	clusterScope, err := scope.NewClusterScope(scope.ClusterScopeParams{
		AzureClients: scope.AzureClients{
			SubscriptionID: "123",
			Authorizer:     autorest.NullAuthorizer{},
		},
		Client:  client,
		Cluster: cluster,
		AzureCluster: &infrav1.AzureCluster{
			Spec: infrav1.AzureClusterSpec{
				Location: "test-location",
				ResourceGroup: "my-rg",
				NetworkSpec: infrav1.NetworkSpec{
					Vnet: infrav1.VnetSpec{Name: "my-vnet", ResourceGroup: "my-rg"},
				},
			},
		},
	})
	if err != nil {
		t.Fatalf("Failed to create test context: %v", err)
	}

	s := &Service{
		Scope:  clusterScope,
		Client: netInterfaceMock,
	}

	// Wrong Spec
	wrongSpec := &network.PublicIPAddress{}

	err = s.Reconcile(context.TODO(), &wrongSpec)
	if err == nil {
		t.Fatalf("it should fail")
	}
	if err.Error() != expectedInvalidSpec {
		t.Fatalf("got an unexpected error: %v", err)
	}

	_, err = s.Get(context.TODO(), &wrongSpec)
	if err == nil {
		t.Fatalf("it should fail")
	}
	if err.Error() != expectedInvalidSpec {
		t.Fatalf("got an unexpected error: %v", err)
	}

	err = s.Delete(context.TODO(), &wrongSpec)
	if err == nil {
		t.Fatalf("it should fail")
	}
	if err.Error() != expectedInvalidSpec {
		t.Fatalf("got an unexpected error: %v", err)
	}
}

func TestGetNetworkInterface(t *testing.T) {
	testcases := []struct {
		name             string
		netInterfaceSpec Spec
		expectedError    string
		expect           func(m *mock_networkinterfaces.MockClientMockRecorder)
	}{
		{
			name: "get existing network interface",
			netInterfaceSpec: Spec{
				Name: "my-net-interface",
			},
			expectedError: "",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder) {
				m.Get(context.TODO(), "my-rg", "my-net-interface").Return(network.Interface{}, nil)
			},
		},
		{
			name: "network interface not found",
			netInterfaceSpec: Spec{
				Name: "my-net-interface",
			},
			expectedError: "network interface my-net-interface not found: #: Not found: StatusCode=404",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder) {
				m.Get(context.TODO(), "my-rg", "my-net-interface").Return(network.Interface{}, autorest.NewErrorWithResponse("", "", &http.Response{StatusCode: 404}, "Not found"))
			},
		},
		{
			name: "network interface retrieval fails",
			netInterfaceSpec: Spec{
				Name: "my-net-interface",
			},
			expectedError: "#: Internal Server Error: StatusCode=500",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder) {
				m.Get(context.TODO(), "my-rg", "my-net-interface").Return(network.Interface{}, autorest.NewErrorWithResponse("", "", &http.Response{StatusCode: 500}, "Internal Server Error"))
			},
		},
	}

	for _, tc := range testcases {
		t.Run(tc.name, func(t *testing.T) {
			mockCtrl := gomock.NewController(t)
			netInterfaceMock := mock_networkinterfaces.NewMockClient(mockCtrl)

			cluster := &clusterv1.Cluster{
				ObjectMeta: metav1.ObjectMeta{Name: "test-cluster"},
			}
			client := fake.NewFakeClient(cluster)
			clusterScope, err := scope.NewClusterScope(scope.ClusterScopeParams{
				AzureClients: scope.AzureClients{
					SubscriptionID: "123",
					Authorizer:     autorest.NullAuthorizer{},
				},
				Client:  client,
				Cluster: cluster,
				AzureCluster: &infrav1.AzureCluster{
					Spec: infrav1.AzureClusterSpec{
						Location: "test-location",
						ResourceGroup: "my-rg",
						NetworkSpec: infrav1.NetworkSpec{
							Vnet: infrav1.VnetSpec{
								Name:          "my-vnet",
								ResourceGroup: "my-rg",
							},
							Subnets: []*infrav1.SubnetSpec{{
								Name: "my-subnet",
								Role: infrav1.SubnetNode,
							}},
						},
					},
				},
			})
			if err != nil {
				t.Fatalf("Failed to create test context: %v", err)
			}

			azureMachine := &infrav1.AzureMachine{
				ObjectMeta: metav1.ObjectMeta{
					Name: "azure-test1",
					OwnerReferences: []metav1.OwnerReference{
						{
							APIVersion: clusterv1.GroupVersion.String(),
							Kind:       "Machine",
							Name:       "test1",
						},
					},
				},
			}
			machineScope, err := scope.NewMachineScope(scope.MachineScopeParams{
				Client:  client,
				Cluster: cluster,
				Machine: &clusterv1.Machine{},
				AzureClients: scope.AzureClients{
					SubscriptionID: "123",
					Authorizer:     autorest.NullAuthorizer{},
				},
				AzureMachine: azureMachine,
				AzureCluster: &infrav1.AzureCluster{},
			})
			if err != nil {
				t.Fatalf("Failed to create test context: %v", err)
			}

			tc.expect(netInterfaceMock.EXPECT())

			s := &Service{
				Scope:        clusterScope,
				MachineScope: machineScope,
				Client:       netInterfaceMock,
			}

			_, err = s.Get(context.TODO(), &tc.netInterfaceSpec)
			if err != nil {
				if tc.expectedError == "" || err.Error() != tc.expectedError {
					t.Fatalf("got an unexpected error: %v", err)
				}
			} else {
				if tc.expectedError != "" {
					t.Fatalf("expected an error: %v", tc.expectedError)
				}
			}
		})
	}
}

func TestReconcileNetworkInterface(t *testing.T) {
	testcases := []struct {
		name             string
		netInterfaceSpec Spec
		expectedError    string
		expect           func(m *mock_networkinterfaces.MockClientMockRecorder,
			mSubnet *mock_subnets.MockClientMockRecorder,
			mPublicLoadBalancer *mock_publicloadbalancers.MockClientMockRecorder,
			mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder,
			mInternalLoadBalancer *mock_internalloadbalancers.MockClientMockRecorder,
			mPublicIP *mock_publicips.MockClientMockRecorder)
	}{
		{
			name: "get subnets fails",
			netInterfaceSpec: Spec{
				Name:       "my-net-interface",
				VnetName:   "my-vnet",
				SubnetName: "my-subnet",
			},
			expectedError: "failed to get subnets: #: Internal Server Error: StatusCode=500",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder,
				mSubnet *mock_subnets.MockClientMockRecorder,
				mPublicLoadBalancer *mock_publicloadbalancers.MockClientMockRecorder,
				mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder,
				mInternalLoadBalancer *mock_internalloadbalancers.MockClientMockRecorder,
				mPublicIP *mock_publicips.MockClientMockRecorder) {
				mSubnet.Get(context.TODO(), "my-rg", "my-vnet", "my-subnet").
					Return(network.Subnet{}, autorest.NewErrorWithResponse("", "", &http.Response{StatusCode: 500}, "Internal Server Error"))
			},
		},
		{
			name: "node network interface create fails",
			netInterfaceSpec: Spec{
				Name:       "my-net-interface",
				VnetName:   "my-vnet",
				SubnetName: "my-subnet",
			},
			expectedError: "failed to create network interface my-net-interface in resource group my-rg: #: Internal Server Error: StatusCode=500",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder,
				mSubnet *mock_subnets.MockClientMockRecorder,
				mPublicLoadBalancer *mock_publicloadbalancers.MockClientMockRecorder,
				mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder,
				mInternalLoadBalancer *mock_internalloadbalancers.MockClientMockRecorder,
				mPublicIP *mock_publicips.MockClientMockRecorder) {
				gomock.InOrder(
					mSubnet.Get(context.TODO(), "my-rg", "my-vnet", "my-subnet").
						Return(network.Subnet{}, nil),
					m.CreateOrUpdate(context.TODO(), "my-rg", "my-net-interface", gomock.AssignableToTypeOf(network.Interface{})).
						Return(autorest.NewErrorWithResponse("", "", &http.Response{StatusCode: 500}, "Internal Server Error")))
			},
		},
		{
			name: "node network interface with Static private IP successfully created",
			netInterfaceSpec: Spec{
				Name:            "my-net-interface",
				VnetName:        "my-vnet",
				SubnetName:      "my-subnet",
				StaticIPAddress: "1.2.3.4",
			},
			expectedError: "",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder,
				mSubnet *mock_subnets.MockClientMockRecorder,
				mPublicLoadBalancer *mock_publicloadbalancers.MockClientMockRecorder,
				mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder,
				mInternalLoadBalancer *mock_internalloadbalancers.MockClientMockRecorder,
				mPublicIP *mock_publicips.MockClientMockRecorder) {
				gomock.InOrder(
					mSubnet.Get(context.TODO(), "my-rg", "my-vnet", "my-subnet").Return(network.Subnet{}, nil),
					m.CreateOrUpdate(context.TODO(), "my-rg", "my-net-interface", gomock.AssignableToTypeOf(network.Interface{})))
			},
		},
		{
			name: "node network interface with Dynamic private IP successfully created",
			netInterfaceSpec: Spec{
				Name:       "my-net-interface",
				VnetName:   "my-vnet",
				SubnetName: "my-subnet",
			},
			expectedError: "",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder,
				mSubnet *mock_subnets.MockClientMockRecorder,
				mPublicLoadBalancer *mock_publicloadbalancers.MockClientMockRecorder,
				mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder,
				mInternalLoadBalancer *mock_internalloadbalancers.MockClientMockRecorder,
				mPublicIP *mock_publicips.MockClientMockRecorder) {
				gomock.InOrder(
					mSubnet.Get(context.TODO(), "my-rg", "my-vnet", "my-subnet").Return(network.Subnet{}, nil),
					m.CreateOrUpdate(context.TODO(), "my-rg", "my-net-interface", gomock.AssignableToTypeOf(network.Interface{})))
			},
		},
		{
			name: "control plane network interface successfully created",
			netInterfaceSpec: Spec{
				Name:                     "my-net-interface",
				VnetName:                 "my-vnet",
				SubnetName:               "my-subnet",
				PublicLoadBalancerName:   "my-publiclb",
				InternalLoadBalancerName: "my-internal-lb",
			},
			expectedError: "",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder,
				mSubnet *mock_subnets.MockClientMockRecorder,
				mPublicLoadBalancer *mock_publicloadbalancers.MockClientMockRecorder,
				mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder,
				mInternalLoadBalancer *mock_internalloadbalancers.MockClientMockRecorder,
				mPublicIP *mock_publicips.MockClientMockRecorder) {
				gomock.InOrder(
					mSubnet.Get(context.TODO(), "my-rg", "my-vnet", "my-subnet").
						Return(network.Subnet{ID: to.StringPtr("my-subnet-id")}, nil),
					mPublicLoadBalancer.Get(context.TODO(), "my-rg", "my-publiclb").Return(network.LoadBalancer{
						Name: to.StringPtr("my-publiclb"),
						ID:   pointer.StringPtr("my-publiclb-id"),
						LoadBalancerPropertiesFormat: &network.LoadBalancerPropertiesFormat{
							FrontendIPConfigurations: &[]network.FrontendIPConfiguration{
								{
									ID: to.StringPtr("frontend-ip-config-id"),
								},
							},
							BackendAddressPools: &[]network.BackendAddressPool{
								{
									ID: pointer.StringPtr("my-backend-pool-id"),
								},
							},
							InboundNatRules: &[]network.InboundNatRule{},
						}}, nil),
					mInboundNATRules.CreateOrUpdate(context.TODO(), "my-rg", "my-publiclb", "azure-test1", network.InboundNatRule{
						Name: pointer.StringPtr("azure-test1"),
						InboundNatRulePropertiesFormat: &network.InboundNatRulePropertiesFormat{
							FrontendPort:         to.Int32Ptr(22),
							BackendPort:          to.Int32Ptr(22),
							EnableFloatingIP:     to.BoolPtr(false),
							IdleTimeoutInMinutes: to.Int32Ptr(4),
							FrontendIPConfiguration: &network.SubResource{
								ID: to.StringPtr("frontend-ip-config-id"),
							},
							Protocol: network.TransportProtocolTCP,
						},
					}),
					mInternalLoadBalancer.Get(context.TODO(), "my-rg", "my-internal-lb").
						Return(network.LoadBalancer{
							ID: pointer.StringPtr("my-internal-lb-id"),
							LoadBalancerPropertiesFormat: &network.LoadBalancerPropertiesFormat{
								BackendAddressPools: &[]network.BackendAddressPool{
									{
										ID: pointer.StringPtr("my-internal-backend-pool-id"),
									},
								},
							}}, nil),
					m.CreateOrUpdate(context.TODO(), "my-rg", "my-net-interface", network.Interface{
						Location: to.StringPtr("test-location"),
						InterfacePropertiesFormat: &network.InterfacePropertiesFormat{
							IPConfigurations: &[]network.InterfaceIPConfiguration{
								{
									Name: to.StringPtr("pipConfig"),
									InterfaceIPConfigurationPropertiesFormat: &network.InterfaceIPConfigurationPropertiesFormat{
										Subnet:                          &network.Subnet{ID: to.StringPtr("my-subnet-id")},
										PrivateIPAllocationMethod:       network.Dynamic,
										LoadBalancerInboundNatRules:     &[]network.InboundNatRule{{ID: to.StringPtr("my-publiclb-id/inboundNatRules/azure-test1")}},
										LoadBalancerBackendAddressPools: &[]network.BackendAddressPool{{ID: to.StringPtr("my-backend-pool-id")}, {ID: to.StringPtr("my-internal-backend-pool-id")}},
									},
								},
							},
						},
					}))
			},
		},
		{
			name: "control plane network interface fail to get public LB",
			netInterfaceSpec: Spec{
				Name:                     "my-net-interface",
				VnetName:                 "my-vnet",
				SubnetName:               "my-subnet",
				PublicLoadBalancerName:   "my-publiclb",
				InternalLoadBalancerName: "my-internal-lb",
			},
			expectedError: "failed to get publicLB: #: Internal Server Error: StatusCode=500",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder,
				mSubnet *mock_subnets.MockClientMockRecorder,
				mPublicLoadBalancer *mock_publicloadbalancers.MockClientMockRecorder,
				mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder,
				mInternalLoadBalancer *mock_internalloadbalancers.MockClientMockRecorder,
				mPublicIP *mock_publicips.MockClientMockRecorder) {
				gomock.InOrder(
					mSubnet.Get(context.TODO(), "my-rg", "my-vnet", "my-subnet").Return(network.Subnet{}, nil),
					mPublicLoadBalancer.Get(context.TODO(), "my-rg", "my-publiclb").
						Return(network.LoadBalancer{}, autorest.NewErrorWithResponse("", "", &http.Response{StatusCode: 500}, "Internal Server Error")))
			},
		},
		{
			name: "control plane network interface fail to create NAT rule",
			netInterfaceSpec: Spec{
				Name:                     "my-net-interface",
				VnetName:                 "my-vnet",
				SubnetName:               "my-subnet",
				PublicLoadBalancerName:   "my-publiclb",
				InternalLoadBalancerName: "my-internal-lb",
			},
			expectedError: "failed to create NAT rule: #: Internal Server Error: StatusCode=500",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder,
				mSubnet *mock_subnets.MockClientMockRecorder,
				mPublicLoadBalancer *mock_publicloadbalancers.MockClientMockRecorder,
				mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder,
				mInternalLoadBalancer *mock_internalloadbalancers.MockClientMockRecorder,
				mPublicIP *mock_publicips.MockClientMockRecorder) {
				gomock.InOrder(
					mSubnet.Get(context.TODO(), "my-rg", "my-vnet", "my-subnet").Return(network.Subnet{}, nil),
					mPublicLoadBalancer.Get(context.TODO(), "my-rg", "my-publiclb").Return(network.LoadBalancer{
						Name: to.StringPtr("my-publiclb"),
						ID:   pointer.StringPtr("my-publiclb-id"),
						LoadBalancerPropertiesFormat: &network.LoadBalancerPropertiesFormat{
							FrontendIPConfigurations: &[]network.FrontendIPConfiguration{
								{
									ID: to.StringPtr("frontend-ip-config-id"),
								},
							},
							BackendAddressPools: &[]network.BackendAddressPool{
								{
									ID: pointer.StringPtr("my-backend-pool-id"),
								},
							},
							InboundNatRules: &[]network.InboundNatRule{
								{
									Name: pointer.StringPtr("other-machine-nat-rule"),
									ID:   pointer.StringPtr("some-natrules-id"),
									InboundNatRulePropertiesFormat: &network.InboundNatRulePropertiesFormat{
										FrontendPort: to.Int32Ptr(22),
									},
								},
								{
									Name: pointer.StringPtr("other-machine-nat-rule-2"),
									ID:   pointer.StringPtr("some-natrules-id-2"),
									InboundNatRulePropertiesFormat: &network.InboundNatRulePropertiesFormat{
										FrontendPort: to.Int32Ptr(2201),
									},
								},
							},
						}}, nil),
					mInboundNATRules.CreateOrUpdate(context.TODO(), "my-rg", "my-publiclb", "azure-test1", network.InboundNatRule{
						Name: pointer.StringPtr("azure-test1"),
						InboundNatRulePropertiesFormat: &network.InboundNatRulePropertiesFormat{
							FrontendPort:         to.Int32Ptr(2202),
							BackendPort:          to.Int32Ptr(22),
							EnableFloatingIP:     to.BoolPtr(false),
							IdleTimeoutInMinutes: to.Int32Ptr(4),
							FrontendIPConfiguration: &network.SubResource{
								ID: to.StringPtr("frontend-ip-config-id"),
							},
							Protocol: network.TransportProtocolTCP,
						},
					}).
						Return(autorest.NewErrorWithResponse("", "", &http.Response{StatusCode: 500}, "Internal Server Error")))
			},
		},
		{
			name: "control plane network interface fail to get internal LB",
			netInterfaceSpec: Spec{
				Name:                     "my-net-interface",
				VnetName:                 "my-vnet",
				SubnetName:               "my-subnet",
				InternalLoadBalancerName: "my-internal-lb",
			},
			expectedError: "failed to get internalLB: #: Internal Server Error: StatusCode=500",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder,
				mSubnet *mock_subnets.MockClientMockRecorder,
				mPublicLoadBalancer *mock_publicloadbalancers.MockClientMockRecorder,
				mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder,
				mInternalLoadBalancer *mock_internalloadbalancers.MockClientMockRecorder,
				mPublicIP *mock_publicips.MockClientMockRecorder) {
				gomock.InOrder(
					mSubnet.Get(context.TODO(), "my-rg", "my-vnet", "my-subnet").Return(network.Subnet{}, nil),
					mInternalLoadBalancer.Get(context.TODO(), "my-rg", "my-internal-lb").
						Return(network.LoadBalancer{}, autorest.NewErrorWithResponse("", "", &http.Response{StatusCode: 500}, "Internal Server Error")))
			},
		},
		{
			name: "network interface with Public IP successfully created",
			netInterfaceSpec: Spec{
				Name:         "my-net-interface",
				VnetName:     "my-vnet",
				SubnetName:   "my-subnet",
				PublicIPName: "my-public-ip",
			},
			expectedError: "",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder,
				mSubnet *mock_subnets.MockClientMockRecorder,
				mPublicLoadBalancer *mock_publicloadbalancers.MockClientMockRecorder,
				mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder,
				mInternalLoadBalancer *mock_internalloadbalancers.MockClientMockRecorder,
				mPublicIP *mock_publicips.MockClientMockRecorder) {
				gomock.InOrder(
					mSubnet.Get(context.TODO(), "my-rg", "my-vnet", "my-subnet").Return(network.Subnet{}, nil),
					mPublicIP.Get(context.TODO(), "my-rg", "my-public-ip").Return(network.PublicIPAddress{}, nil),
					m.CreateOrUpdate(context.TODO(), "my-rg", "my-net-interface", gomock.AssignableToTypeOf(network.Interface{})))
			},
		},
		{
			name: "network interface with Public IP fail to get Public IP",
			netInterfaceSpec: Spec{
				Name:         "my-net-interface",
				VnetName:     "my-vnet",
				SubnetName:   "my-subnet",
				PublicIPName: "my-public-ip",
			},
			expectedError: "failed to get publicIP: #: Internal Server Error: StatusCode=500",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder,
				mSubnet *mock_subnets.MockClientMockRecorder,
				mPublicLoadBalancer *mock_publicloadbalancers.MockClientMockRecorder,
				mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder,
				mInternalLoadBalancer *mock_internalloadbalancers.MockClientMockRecorder,
				mPublicIP *mock_publicips.MockClientMockRecorder) {
				gomock.InOrder(
					mSubnet.Get(context.TODO(), "my-rg", "my-vnet", "my-subnet").Return(network.Subnet{}, nil),
					mPublicIP.Get(context.TODO(), "my-rg", "my-public-ip").Return(network.PublicIPAddress{}, autorest.NewErrorWithResponse("", "", &http.Response{StatusCode: 500}, "Internal Server Error")),
					m.CreateOrUpdate(context.TODO(), "my-rg", "my-net-interface", gomock.AssignableToTypeOf(network.Interface{})))
			},
		},
	}

	for _, tc := range testcases {
		t.Run(tc.name, func(t *testing.T) {
			mockCtrl := gomock.NewController(t)
			netInterfaceMock := mock_networkinterfaces.NewMockClient(mockCtrl)
			subnetMock := mock_subnets.NewMockClient(mockCtrl)
			publicLoadBalancerMock := mock_publicloadbalancers.NewMockClient(mockCtrl)
			inboundNatRulesMock := mock_inboundnatrules.NewMockClient(mockCtrl)
			internalLoadBalancerMock := mock_internalloadbalancers.NewMockClient(mockCtrl)
			publicIPsMock := mock_publicips.NewMockClient(mockCtrl)

			cluster := &clusterv1.Cluster{
				ObjectMeta: metav1.ObjectMeta{Name: "test-cluster"},
			}
			client := fake.NewFakeClient(cluster)
			clusterScope, err := scope.NewClusterScope(scope.ClusterScopeParams{
				AzureClients: scope.AzureClients{
					SubscriptionID: "123",
					Authorizer:     autorest.NullAuthorizer{},
				},
				Client:  client,
				Cluster: cluster,
				AzureCluster: &infrav1.AzureCluster{
					Spec: infrav1.AzureClusterSpec{
						Location: "test-location",
						ResourceGroup: "my-rg",
						NetworkSpec: infrav1.NetworkSpec{
							Vnet: infrav1.VnetSpec{
								Name:          "my-vnet",
								ResourceGroup: "my-rg",
							},
							Subnets: []*infrav1.SubnetSpec{{
								Name: "my-subnet",
								Role: infrav1.SubnetNode,
							}},
						},
					},
				},
			})
			if err != nil {
				t.Fatalf("Failed to create test context: %v", err)
			}

			azureMachine := &infrav1.AzureMachine{
				ObjectMeta: metav1.ObjectMeta{
					Name: "azure-test1",
					OwnerReferences: []metav1.OwnerReference{
						{
							APIVersion: clusterv1.GroupVersion.String(),
							Kind:       "Machine",
							Name:       "test1",
						},
					},
				},
			}
			machineScope, err := scope.NewMachineScope(scope.MachineScopeParams{
				Client:  client,
				Cluster: cluster,
				Machine: &clusterv1.Machine{},
				AzureClients: scope.AzureClients{
					SubscriptionID: "123",
					Authorizer:     autorest.NullAuthorizer{},
				},
				AzureMachine: azureMachine,
				AzureCluster: &infrav1.AzureCluster{},
			})
			if err != nil {
				t.Fatalf("Failed to create test context: %v", err)
			}

			tc.expect(netInterfaceMock.EXPECT(), subnetMock.EXPECT(),
				publicLoadBalancerMock.EXPECT(), inboundNatRulesMock.EXPECT(),
				internalLoadBalancerMock.EXPECT(), publicIPsMock.EXPECT())

			s := &Service{
				Scope:                       clusterScope,
				MachineScope:                machineScope,
				Client:                      netInterfaceMock,
				SubnetsClient:               subnetMock,
				PublicLoadBalancersClient:   publicLoadBalancerMock,
				InboundNATRulesClient:       inboundNatRulesMock,
				InternalLoadBalancersClient: internalLoadBalancerMock,
				PublicIPsClient:             publicIPsMock,
			}

			if err := s.Reconcile(context.TODO(), &tc.netInterfaceSpec); err != nil {
				if tc.expectedError == "" || err.Error() != tc.expectedError {
					t.Fatalf("got an unexpected error: %v", err)
				}
			} else {
				if tc.expectedError != "" {
					t.Fatalf("expected an error: %v", tc.expectedError)

				}
			}
		})
	}
}

func TestDeleteNetworkInterface(t *testing.T) {
	testcases := []struct {
		name             string
		netInterfaceSpec Spec
		expectedError    string
		expect           func(m *mock_networkinterfaces.MockClientMockRecorder, mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder)
	}{
		{
			name: "successfully delete an existing network interface",
			netInterfaceSpec: Spec{
				Name:                   "my-net-interface",
				PublicLoadBalancerName: "my-public-lb",
			},
			expectedError: "",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder, mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder) {
				m.Delete(context.TODO(), "my-rg", "my-net-interface")
				mInboundNATRules.Delete(context.TODO(), "my-rg", "my-public-lb", "azure-test1")
			},
		},
		{
			name: "network interface already deleted",
			netInterfaceSpec: Spec{
				Name:                   "my-net-interface",
				PublicLoadBalancerName: "my-public-lb",
			},
			expectedError: "",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder, mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder) {
				m.Delete(context.TODO(), "my-rg", "my-net-interface").
					Return(autorest.NewErrorWithResponse("", "", &http.Response{StatusCode: 404}, "Not found"))
				mInboundNATRules.Delete(context.TODO(), "my-rg", "my-public-lb", "azure-test1")
			},
		},
		{
			name: "network interface deletion fails",
			netInterfaceSpec: Spec{
				Name:                   "my-net-interface",
				PublicLoadBalancerName: "my-public-lb",
			},
			expectedError: "failed to delete network interface my-net-interface in resource group my-rg: #: Internal Server Error: StatusCode=500",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder, mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder) {
				m.Delete(context.TODO(), "my-rg", "my-net-interface").
					Return(autorest.NewErrorWithResponse("", "", &http.Response{StatusCode: 500}, "Internal Server Error"))
			},
		},
		{
			name: "NAT rule already deleted",
			netInterfaceSpec: Spec{
				Name:                   "my-net-interface",
				PublicLoadBalancerName: "my-public-lb",
			},
			expectedError: "",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder, mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder) {
				m.Delete(context.TODO(), "my-rg", "my-net-interface").
					Return(autorest.NewErrorWithResponse("", "", &http.Response{StatusCode: 404}, "Not found"))
				mInboundNATRules.Delete(context.TODO(), "my-rg", "my-public-lb", "azure-test1")
			},
		},
		{
			name: "NAT rule deletion fails",
			netInterfaceSpec: Spec{
				Name:                   "my-net-interface",
				PublicLoadBalancerName: "my-public-lb",
			},
			expectedError: "failed to delete inbound NAT rule azure-test1 in load balancer my-public-lb: #: Internal Server Error: StatusCode=500",
			expect: func(m *mock_networkinterfaces.MockClientMockRecorder, mInboundNATRules *mock_inboundnatrules.MockClientMockRecorder) {
				m.Delete(context.TODO(), "my-rg", "my-net-interface")
				mInboundNATRules.Delete(context.TODO(), "my-rg", "my-public-lb", "azure-test1").
					Return(autorest.NewErrorWithResponse("", "", &http.Response{StatusCode: 500}, "Internal Server Error"))
			},
		},
	}

	for _, tc := range testcases {
		t.Run(tc.name, func(t *testing.T) {
			mockCtrl := gomock.NewController(t)
			netInterfaceMock := mock_networkinterfaces.NewMockClient(mockCtrl)
			inboundNatRulesMock := mock_inboundnatrules.NewMockClient(mockCtrl)

			cluster := &clusterv1.Cluster{
				ObjectMeta: metav1.ObjectMeta{Name: "test-cluster"},
			}
			client := fake.NewFakeClient(cluster)
			clusterScope, err := scope.NewClusterScope(scope.ClusterScopeParams{
				AzureClients: scope.AzureClients{
					SubscriptionID: "123",
					Authorizer:     autorest.NullAuthorizer{},
				},
				Client:  client,
				Cluster: cluster,
				AzureCluster: &infrav1.AzureCluster{
					Spec: infrav1.AzureClusterSpec{
						Location: "test-location",
						ResourceGroup: "my-rg",
						NetworkSpec: infrav1.NetworkSpec{
							Vnet: infrav1.VnetSpec{Name: "my-vnet", ResourceGroup: "my-rg"},
							Subnets: []*infrav1.SubnetSpec{{
								Name: "my-subnet",
								Role: infrav1.SubnetNode,
							}},
						},
					},
				},
			})
			if err != nil {
				t.Fatalf("Failed to create test context: %v", err)
			}

			azureMachine := &infrav1.AzureMachine{
				ObjectMeta: metav1.ObjectMeta{
					Name: "azure-test1",
					OwnerReferences: []metav1.OwnerReference{
						{
							APIVersion: clusterv1.GroupVersion.String(),
							Kind:       "Machine",
							Name:       "test1",
						},
					},
				},
			}
			machineScope, err := scope.NewMachineScope(scope.MachineScopeParams{
				Client:  client,
				Cluster: cluster,
				Machine: &clusterv1.Machine{},
				AzureClients: scope.AzureClients{
					SubscriptionID: "123",
					Authorizer:     autorest.NullAuthorizer{},
				},
				AzureMachine: azureMachine,
				AzureCluster: &infrav1.AzureCluster{},
			})
			if err != nil {
				t.Fatalf("Failed to create test context: %v", err)
			}

			tc.expect(netInterfaceMock.EXPECT(), inboundNatRulesMock.EXPECT())

			s := &Service{
				Scope:                 clusterScope,
				MachineScope:          machineScope,
				Client:                netInterfaceMock,
				InboundNATRulesClient: inboundNatRulesMock,
			}

			if err := s.Delete(context.TODO(), &tc.netInterfaceSpec); err != nil {
				if tc.expectedError == "" || err.Error() != tc.expectedError {
					t.Fatalf("got an unexpected error: %v", err)
				}
			} else {
				if tc.expectedError != "" {
					t.Fatalf("expected an error: %v", tc.expectedError)
				}
			}
		})
	}
}
