/*
Copyright 2019 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package publicips

import (
	"context"

	"github.com/Azure/azure-sdk-for-go/services/network/mgmt/2019-06-01/network"
	"github.com/Azure/go-autorest/autorest"
	azure "sigs.k8s.io/cluster-api-provider-azure/cloud"
)

// Client wraps go-sdk
type Client interface {
	Get(context.Context, string, string) (network.PublicIPAddress, error)
	CreateOrUpdate(context.Context, string, string, network.PublicIPAddress) error
	Delete(context.Context, string, string) error
}

// AzureClient contains the Azure go-sdk Client
type AzureClient struct {
	publicips network.PublicIPAddressesClient
}

var _ Client = &AzureClient{}

// NewClient creates a new public IP client from subscription ID.
func NewClient(subscriptionID string, authorizer autorest.Authorizer) *AzureClient {
	c := newPublicIPAddressesClient(subscriptionID, authorizer)
	return &AzureClient{c}
}

// newPublicIPAddressesClient creates a new public IP client from subscription ID.
func newPublicIPAddressesClient(subscriptionID string, authorizer autorest.Authorizer) network.PublicIPAddressesClient {
	publicIPsClient := network.NewPublicIPAddressesClient(subscriptionID)
	publicIPsClient.Authorizer = authorizer
	publicIPsClient.AddToUserAgent(azure.UserAgent)
	return publicIPsClient
}

// Get gets the specified public IP address in a specified resource group.
func (ac *AzureClient) Get(ctx context.Context, resourceGroupName, ipName string) (network.PublicIPAddress, error) {
	return ac.publicips.Get(ctx, resourceGroupName, ipName, "")
}

// CreateOrUpdate creates or updates a static or dynamic public IP address.
func (ac *AzureClient) CreateOrUpdate(ctx context.Context, resourceGroupName string, ipName string, ip network.PublicIPAddress) error {
	future, err := ac.publicips.CreateOrUpdate(ctx, resourceGroupName, ipName, ip)
	if err != nil {
		return err
	}
	err = future.WaitForCompletionRef(ctx, ac.publicips.Client)
	if err != nil {
		return err
	}
	_, err = future.Result(ac.publicips)
	return err
}

// Delete deletes the specified public IP address.
func (ac *AzureClient) Delete(ctx context.Context, resourceGroupName, ipName string) error {
	future, err := ac.publicips.Delete(ctx, resourceGroupName, ipName)
	if err != nil {
		return err
	}
	err = future.WaitForCompletionRef(ctx, ac.publicips.Client)
	if err != nil {
		return err
	}
	_, err = future.Result(ac.publicips)
	return err
}
