/*
Copyright 2020 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package client

import (
	"context"
	"sort"
	"testing"

	. "github.com/onsi/gomega"

	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	clusterv1 "sigs.k8s.io/cluster-api/api/v1alpha3"
	clusterctlv1 "sigs.k8s.io/cluster-api/cmd/clusterctl/api/v1alpha3"
	"sigs.k8s.io/cluster-api/cmd/clusterctl/client/cluster"
	"sigs.k8s.io/cluster-api/cmd/clusterctl/client/config"
)

func Test_clusterctlClient_ApplyUpgrade(t *testing.T) {
	g := NewWithT(t)

	type fields struct {
		client *fakeClient
	}
	type args struct {
		options ApplyUpgradeOptions
	}
	tests := []struct {
		name          string
		fields        fields
		args          args
		wantProviders *clusterctlv1.ProviderList
		wantErr       bool
	}{
		{
			name: "apply a plan",
			fields: fields{
				client: fakeClientFoUpgrade(), // core v1.0.0 (v1.0.1 available), infra v2.0.0 (v2.0.1 available)
			},
			args: args{
				options: ApplyUpgradeOptions{
					Kubeconfig:              "kubeconfig",
					ManagementGroup:         "cluster-api-system/cluster-api",
					Contract:                "v1alpha3",
					CoreProvider:            "",
					BootstrapProviders:      nil,
					ControlPlaneProviders:   nil,
					InfrastructureProviders: nil,
				},
			},
			wantProviders: &clusterctlv1.ProviderList{
				TypeMeta: metav1.TypeMeta{
					APIVersion: clusterctlv1.GroupVersion.String(),
					Kind:       "ProviderList",
				},
				ListMeta: metav1.ListMeta{},
				Items: []clusterctlv1.Provider{ // both providers should be upgraded
					fakeProvider("cluster-api", clusterctlv1.CoreProviderType, "v1.0.1", "cluster-api-system"),
					fakeProvider("infra", clusterctlv1.InfrastructureProviderType, "v2.0.1", "infra-system"),
				},
			},
			wantErr: false,
		},
		{
			name: "apply a custom plan - core provider only",
			fields: fields{
				client: fakeClientFoUpgrade(), // core v1.0.0 (v1.0.1 available), infra v2.0.0 (v2.0.1 available)
			},
			args: args{
				options: ApplyUpgradeOptions{
					Kubeconfig:              "kubeconfig",
					ManagementGroup:         "cluster-api-system/cluster-api",
					Contract:                "",
					CoreProvider:            "cluster-api-system/cluster-api:v1.0.1",
					BootstrapProviders:      nil,
					ControlPlaneProviders:   nil,
					InfrastructureProviders: nil,
				},
			},
			wantProviders: &clusterctlv1.ProviderList{
				TypeMeta: metav1.TypeMeta{
					APIVersion: clusterctlv1.GroupVersion.String(),
					Kind:       "ProviderList",
				},
				ListMeta: metav1.ListMeta{},
				Items: []clusterctlv1.Provider{ // only one provider should be upgraded
					fakeProvider("cluster-api", clusterctlv1.CoreProviderType, "v1.0.1", "cluster-api-system"),
					fakeProvider("infra", clusterctlv1.InfrastructureProviderType, "v2.0.0", "infra-system"),
				},
			},
			wantErr: false,
		},
		{
			name: "apply a custom plan - infra provider only",
			fields: fields{
				client: fakeClientFoUpgrade(), // core v1.0.0 (v1.0.1 available), infra v2.0.0 (v2.0.1 available)
			},
			args: args{
				options: ApplyUpgradeOptions{
					Kubeconfig:              "kubeconfig",
					ManagementGroup:         "cluster-api-system/cluster-api",
					Contract:                "",
					CoreProvider:            "",
					BootstrapProviders:      nil,
					ControlPlaneProviders:   nil,
					InfrastructureProviders: []string{"infra-system/infra:v2.0.1"},
				},
			},
			wantProviders: &clusterctlv1.ProviderList{
				TypeMeta: metav1.TypeMeta{
					APIVersion: clusterctlv1.GroupVersion.String(),
					Kind:       "ProviderList",
				},
				ListMeta: metav1.ListMeta{},
				Items: []clusterctlv1.Provider{ // only one provider should be upgraded
					fakeProvider("cluster-api", clusterctlv1.CoreProviderType, "v1.0.0", "cluster-api-system"),
					fakeProvider("infra", clusterctlv1.InfrastructureProviderType, "v2.0.1", "infra-system"),
				},
			},
			wantErr: false,
		},
		{
			name: "apply a custom plan - both providers",
			fields: fields{
				client: fakeClientFoUpgrade(), // core v1.0.0 (v1.0.1 available), infra v2.0.0 (v2.0.1 available)
			},
			args: args{
				options: ApplyUpgradeOptions{
					Kubeconfig:              "kubeconfig",
					ManagementGroup:         "cluster-api-system/cluster-api",
					Contract:                "",
					CoreProvider:            "cluster-api-system/cluster-api:v1.0.1",
					BootstrapProviders:      nil,
					ControlPlaneProviders:   nil,
					InfrastructureProviders: []string{"infra-system/infra:v2.0.1"},
				},
			},
			wantProviders: &clusterctlv1.ProviderList{
				TypeMeta: metav1.TypeMeta{
					APIVersion: clusterctlv1.GroupVersion.String(),
					Kind:       "ProviderList",
				},
				ListMeta: metav1.ListMeta{},
				Items: []clusterctlv1.Provider{ // only one provider should be upgraded
					fakeProvider("cluster-api", clusterctlv1.CoreProviderType, "v1.0.1", "cluster-api-system"),
					fakeProvider("infra", clusterctlv1.InfrastructureProviderType, "v2.0.1", "infra-system"),
				},
			},
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			err := tt.fields.client.ApplyUpgrade(tt.args.options)
			if tt.wantErr {
				g.Expect(err).To(HaveOccurred())
				return
			}
			g.Expect(err).NotTo(HaveOccurred())

			proxy := tt.fields.client.clusters["kubeconfig"].Proxy()
			gotProviders := &clusterctlv1.ProviderList{}

			c, err := proxy.NewClient()
			g.Expect(err).NotTo(HaveOccurred())

			g.Expect(c.List(context.Background(), gotProviders)).To(Succeed())

			sort.Slice(gotProviders.Items, func(i, j int) bool {
				return gotProviders.Items[i].Name < gotProviders.Items[j].Name
			})
			sort.Slice(tt.wantProviders.Items, func(i, j int) bool {
				return tt.wantProviders.Items[i].Name < tt.wantProviders.Items[j].Name
			})
			for i := range gotProviders.Items {
				tt.wantProviders.Items[i].ResourceVersion = gotProviders.Items[i].ResourceVersion
			}
			g.Expect(gotProviders).To(Equal(tt.wantProviders))
		})
	}
}

func fakeClientFoUpgrade() *fakeClient {
	core := config.NewProvider("cluster-api", "https://somewhere.com", clusterctlv1.CoreProviderType)
	infra := config.NewProvider("infra", "https://somewhere.com", clusterctlv1.InfrastructureProviderType)

	config1 := newFakeConfig().
		WithProvider(core).
		WithProvider(infra)

	repository1 := newFakeRepository(core, config1).
		WithPaths("root", "components.yaml").
		WithDefaultVersion("v1.0.1").
		WithFile("v1.0.1", "components.yaml", componentsYAML("ns2")).
		WithVersions("v1.0.0", "v1.0.1").
		WithMetadata("v1.0.1", &clusterctlv1.Metadata{
			ReleaseSeries: []clusterctlv1.ReleaseSeries{
				{Major: 1, Minor: 0, Contract: "v1alpha3"},
			},
		})
	repository2 := newFakeRepository(infra, config1).
		WithPaths("root", "components.yaml").
		WithDefaultVersion("v2.0.0").
		WithFile("v2.0.1", "components.yaml", componentsYAML("ns2")).
		WithVersions("v2.0.0", "v2.0.1").
		WithMetadata("v2.0.1", &clusterctlv1.Metadata{
			ReleaseSeries: []clusterctlv1.ReleaseSeries{
				{Major: 2, Minor: 0, Contract: "v1alpha3"},
			},
		})

	cluster1 := newFakeCluster("kubeconfig", config1).
		WithRepository(repository1).
		WithRepository(repository2).
		WithProviderInventory(core.Name(), core.Type(), "v1.0.0", "cluster-api-system", "").
		WithProviderInventory(infra.Name(), infra.Type(), "v2.0.0", "infra-system", "")

	client := newFakeClient(config1).
		WithRepository(repository1).
		WithRepository(repository2).
		WithCluster(cluster1)

	return client
}

func fakeProvider(name string, providerType clusterctlv1.ProviderType, version, targetNamespace string) clusterctlv1.Provider {
	return clusterctlv1.Provider{
		TypeMeta: metav1.TypeMeta{
			APIVersion: clusterctlv1.GroupVersion.String(),
			Kind:       "Provider",
		},
		ObjectMeta: metav1.ObjectMeta{
			Namespace: targetNamespace,
			Name:      clusterctlv1.ManifestLabel(name, providerType),
			Labels: map[string]string{
				clusterctlv1.ClusterctlLabelName:     "",
				clusterv1.ProviderLabelName:          clusterctlv1.ManifestLabel(name, providerType),
				clusterctlv1.ClusterctlCoreLabelName: "inventory",
			},
		},
		ProviderName:     name,
		Type:             string(providerType),
		Version:          version,
		WatchedNamespace: "",
	}
}

func Test_parseUpgradeItem(t *testing.T) {
	g := NewWithT(t)

	type args struct {
		provider string
	}
	tests := []struct {
		name    string
		args    args
		want    *cluster.UpgradeItem
		wantErr bool
	}{
		{
			name: "namespace/provider",
			args: args{
				provider: "namespace/provider",
			},
			want: &cluster.UpgradeItem{
				Provider: clusterctlv1.Provider{
					ObjectMeta: metav1.ObjectMeta{
						Namespace: "namespace",
						Name:      clusterctlv1.ManifestLabel("provider", clusterctlv1.CoreProviderType),
					},
					ProviderName: "provider",
					Type:         string(clusterctlv1.CoreProviderType),
				},
				NextVersion: "",
			},
			wantErr: false,
		},
		{
			name: "namespace/provider:version",
			args: args{
				provider: "namespace/provider:version",
			},
			want: &cluster.UpgradeItem{
				Provider: clusterctlv1.Provider{
					ObjectMeta: metav1.ObjectMeta{
						Namespace: "namespace",
						Name:      clusterctlv1.ManifestLabel("provider", clusterctlv1.CoreProviderType),
					},
					ProviderName: "provider",
					Type:         string(clusterctlv1.CoreProviderType),
				},
				NextVersion: "version",
			},
			wantErr: false,
		},
		{
			name: "namespace missing",
			args: args{
				provider: "provider:version",
			},
			want:    nil,
			wantErr: true,
		},
		{
			name: "namespace empty",
			args: args{
				provider: "/provider:version",
			},
			want:    nil,
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := parseUpgradeItem(tt.args.provider, clusterctlv1.CoreProviderType)
			if tt.wantErr {
				g.Expect(err).To(HaveOccurred())
				return
			}
			g.Expect(err).NotTo(HaveOccurred())

			g.Expect(got).To(Equal(tt.want))
		})
	}
}
