// Copyright 2017 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Package gophers is a list of names, emails, and Github usernames of people
// from the Go git repos and issue trackers.
package gophers

import (
	"strings"

	"golang.org/x/build/gerrit"
)

type Person struct {
	Name    string   // "Foo Bar"
	Github  string   // "FooBar" (orig case, no '@')
	Gerrit  string   // "foo@bar.com" (lowercase)
	Emails  []string // all lower
	Googler bool
	Bot     bool
}

func strSliceContains(ss []string, s string) bool {
	for _, v := range ss {
		if v == s {
			return true
		}
	}
	return false
}

func (p *Person) mergeIDs(ids ...string) {
	for _, id := range ids {
		switch {
		case strings.HasPrefix(id, "@"):
			p.Github = id[1:]
			idToPerson[strings.ToLower(id)] = p
		case strings.Contains(id, "@"):
			email := strings.ToLower(id)
			if !strSliceContains(p.Emails, email) {
				p.Emails = append(p.Emails, email)
			}
			idToPerson[email] = p
			if strings.HasSuffix(email, "@golang.org") || strings.HasSuffix(email, "@google.com") {
				p.Googler = true
			}
			// The first email seen is considered the person's Gerrit email.
			if len(p.Emails) == 1 {
				p.Gerrit = email
			}
		case id == "*goog":
			p.Googler = true
		case id == "*bot":
			p.Bot = true
		default:
			p.Name = id
			idToPerson[strings.ToLower(id)] = p
		}
	}
}

// keys are "@lowercasegithub", "lowercase name", "lowercase@email.com".
var idToPerson = map[string]*Person{}

func GetPerson(id string) *Person {
	return idToPerson[strings.ToLower(id)]
}

func GetGerritPerson(ai gerrit.AccountInfo) *Person {
	if p := GetPerson(ai.Name); p != nil {
		return p
	}
	if p := GetPerson(ai.Email); p != nil {
		return p
	}
	return nil
}

func addPerson(ids ...string) *Person {
	var p *Person
	for _, id := range ids {
		p = GetPerson(id)
		if p != nil {
			break
		}
	}
	if p == nil {
		p = &Person{}
	}
	p.mergeIDs(ids...)
	return p
}

func init() {
	// Not people, but hereby granted personhood:
	addPerson("Gopherbot", "gobot@golang.org", "@gopherbot", "5976@62eb7196-b449-3ce5-99f1-c037f21e1705", "*bot")
	addPerson("Gerrit Bot", "letsusegerrit@gmail.com", "12446@62eb7196-b449-3ce5-99f1-c037f21e1705", "*bot")

	addPerson("212472270", "ggp493@gmail.com", "@ggriffiths")
	addPerson("9.nashi", "9.nashi@gmail.com", "@80nashi")
	addPerson("AJ Yoo", "ajarusan@arista.com")
	addPerson("Aamir Khan", "syst3m.w0rm@gmail.com", "7715@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Aamir Khan", "syst3m.w0rm@gmail.com", "@syst3mw0rm")
	addPerson("Aaron Cannon", "cannona@fireantproductions.com", "@cannona")
	addPerson("Aaron Clawson", "Aaron.Clawson@gmail.com")
	addPerson("Aaron France", "aaron.l.france@gmail.com", "@AeroNotix")
	addPerson("Aaron Jacobs", "jacobsa@google.com", "6475@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Aaron Jacobs", "jacobsa@google.com", "@jacobsa")
	addPerson("Aaron Kemp", "kemp.aaron@gmail.com", "@writeonlymemory")
	addPerson("Aaron Kemp", "kemp@google.com")
	addPerson("Aaron Torres", "tcboox@gmail.com", "6165@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Aaron Torres", "tcboox@gmail.com", "@agtorre")
	addPerson("Aaron Zinman", "aaron@azinman.com", "@azinman")
	addPerson("Aarti Parikh", "aarti.parikh@gmail.com", "@aarti")
	addPerson("Aashish Karki", "0133asis@gmail.com")
	addPerson("Abe Haskins", "abeisgreat@abeisgreat.com")
	addPerson("Abhijit Pai", "abhijitpai05@gmail.com")
	addPerson("Abhinav Gupta", "abhinav.g90@gmail.com", "@abhinav")
	addPerson("Adam Azarchs", "adam.azarchs@10xgenomics.com", "@adam-azarchs")
	addPerson("Adam Bender", "abender@google.com", "@bitlux")
	addPerson("Adam Eijdenberg", "adam@continusec.com")
	addPerson("Adam Harvey", "aharvey@php.net")
	addPerson("Adam Jones", "adam@modsrus.com")
	addPerson("Adam Kisala", "adam.kisala@gmail.com", "@adamkisala")
	addPerson("Adam Langley", "agl@golang.org", "5425@62eb7196-b449-3ce5-99f1-c037f21e1705", "@agl", "agl@google.com", "7285@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Adam Medzinski", "adam.medzinski@gmail.com", "@medzin")
	addPerson("Adam Ostor", "adam.ostor@gmail.com")
	addPerson("Adam Ryman", "adamryman@gmail.com")
	addPerson("Adam Shannon", "adamkshannon@gmail.com", "26193@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Adam Shannon", "adamkshannon@gmail.com", "@adamdecaf")
	addPerson("Adam Sindelar", "adamsh@google.com", "27224@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Adam Thomason", "athomason@gmail.com")
	addPerson("Adam Wolfe Gordon", "awg@xvx.ca")
	addPerson("Adam Woodbeck", "adam@woodbeck.net", "@awoodbeck")
	addPerson("Adam Yi", "i@adamyi.com")
	addPerson("Adin Scannell", "ascannell@google.com")
	addPerson("Aditya Mukerjee", "dev@chimeracoder.net", "@ChimeraCoder")
	addPerson("Aditya Rastogi", "adirastogi@google.com")
	addPerson("Adrian Hesketh", "adrianhesketh@hushmail.com", "24533@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Adrian Hesketh", "adrianhesketh@hushmail.com", "@a-h")
	addPerson("Adrian O'Grady", "elpollouk@gmail.com", "@elpollouk")
	addPerson("Aeneas Rekkas (arekkas)", "aeneas@ory.am")
	addPerson("Afanasev Stanislav", "phpprogger@gmail.com")
	addPerson("Agis Anastasopoulos", "agis.anast@gmail.com", "@agis")
	addPerson("Agniva De Sarker", "agniva.quicksilver@gmail.com", "24096@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Agniva De Sarker", "agniva.quicksilver@gmail.com", "@agnivade")
	addPerson("Agniva De Sarker", "agnivade@yahoo.co.in", "@agnivade")
	addPerson("Ahmed W.", "oneofone@gmail.com", "5255@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ahmet Alp Balkan", "ahmetb@google.com", "@ahmetb")
	addPerson("Ahmet Soormally", "ahmet@mangomm.co.uk", "@asoorm")
	addPerson("Ahmy Yulrizka", "yulrizka@gmail.com", "@yulrizka")
	addPerson("Aiden Scandella", "ai@uber.com", "@sectioneight")
	addPerson("Aiden Scandella", "sc@ndella.com")
	addPerson("Ainar Garipov", "gugl.zadolbal@gmail.com", "@ainar-g")
	addPerson("Aishraj", "aishraj@users.noreply.github.com", "@aishraj")
	addPerson("Akhil Indurti", "aindurti@gmail.com", "17921@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Akhil Indurti", "aindurti@gmail.com", "@smasher164")
	addPerson("Akihiko Odaki", "akihiko.odaki.4i@stu.hosei.ac.jp")
	addPerson("Akihiro Suda", "suda.kyoto@gmail.com", "13030@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Akihiro Suda", "suda.kyoto@gmail.com", "@AkihiroSuda")
	addPerson("Alan Bradley", "alan@gangleri.net")
	addPerson("Alan Braithwaite", "alan@ipaddr.org", "@abraithwaite")
	addPerson("Alan Donovan", "adonovan@google.com", "5195@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alan Donovan", "adonovan@google.com", "@alandonovan") // work profile
	addPerson("Alan Donovan", "alan@alandonovan.net", "@adonovan")   // personal profile
	addPerson("Alan Gardner", "alanctgardner@gmail.com")
	addPerson("Alan Shreve", "alan@inconshreveable.com", "@inconshreveable")
	addPerson("Albert Nigmatzianov", "albertnigma@gmail.com", "15270@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Albert Smith", "albert@horde.today")
	addPerson("Albert Strasheim", "fullung@gmail.com", "@alberts")
	addPerson("Albert Yu", "yukinying@gmail.com", "@yukinying")
	addPerson("Alberto Bertogli", "albertito@blitiri.com.ar", "10985@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alberto Bertogli", "albertito@blitiri.com.ar", "@albertito")
	addPerson("Alberto Donizetti", "alb.donizetti@gmail.com", "5385@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alberto Donizetti", "alb.donizetti@gmail.com", "@ALTree")
	addPerson("Alberto García Hierro", "alberto@garciahierro.com", "@fiam")
	addPerson("Aleksandar Dezelin", "dezelin@gmail.com", "@dezelin")
	addPerson("Aleksandr Demakin", "alexander.demakin@gmail.com", "8245@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Aleksandr Demakin", "alexander.demakin@gmail.com", "@avdva")
	addPerson("Aleksandr Razumov", "ar@cydev.ru")
	addPerson("Aleksandr Razumov", "ar@cydev.ru", "@ernado")
	addPerson("Alekseev Artem", "a.artem060@gmail.com", "@fexolm")
	addPerson("Alessandro Arzilli", "alessandro.arzilli@gmail.com", "5821@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alessandro Arzilli", "alessandro.arzilli@gmail.com", "@aarzilli")
	addPerson("Alessandro Baffa", "alessandro.baffa@gmail.com", "@alebaffa")
	addPerson("Alex A Skinner", "alex@lx.lc")
	addPerson("Alex Brainman", "alex.brainman@gmail.com", "5070@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alex Brainman", "alex.brainman@gmail.com", "@alexbrainman")
	addPerson("Alex Bramley", "a.bramley@gmail.com", "@fluffle")
	addPerson("Alex Browne", "stephenalexbrowne@gmail.com", "@albrow")
	addPerson("Alex Carol", "alex.carol.c@gmail.com", "@alexcarol")
	addPerson("Alex Crawford", "alex@acrawford.com")
	addPerson("Alex Flint", "alex.flint@gmail.com")
	addPerson("Alex Jin", "toalexjin@gmail.com", "@toalexjin")
	addPerson("Alex Kohler", "alexjohnkohler@gmail.com", "@alexkohler")
	addPerson("Alex Myasoedov", "msoedov@gmail.com", "@msoedov")
	addPerson("Alex Plugaru", "alex@plugaru.org", "@xarg")
	addPerson("Alex Schroeder", "alex@gnu.org", "@kensanata")
	addPerson("Alex Sergeyev", "abc@alexsergeyev.com", "@asergeyev")
	addPerson("Alex Seubert", "alexseubert@gmail.com")
	addPerson("Alex Skinner", "alex@lx.lc", "6090@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alex Stoddard", "alex.stoddard@comcast.net")
	addPerson("Alex Tokarev", "aleksator@gmail.com", "@aleksator")
	addPerson("Alex Vaghin", "crhyme@google.com")
	addPerson("Alex Vaghin", "ddos@google.com")
	addPerson("Alex Vaghin", "ddos@google.com", "6347@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alex Vaghin", "ddos@google.com", "@x1ddos")
	addPerson("Alex Yu", "yu.alex96@gmail.com")
	addPerson("AlexRudd", "rudd.alex1@gmail.com")
	addPerson("Alexander A. Klimov", "alexander.klimov@netways.de")
	addPerson("Alexander Ben Nasrallah", "me@abn.sh")
	addPerson("Alexander Döring", "email@alexd.ch", "15115@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alexander Döring", "email@alexd.ch", "@alexd765")
	addPerson("Alexander F Rødseth", "alexander.rodseth@appeartv.com", "@xyproto")
	addPerson("Alexander F Rødseth", "rodseth@gmail.com")
	addPerson("Alexander Guz", "kalimatas@gmail.com", "@kalimatas")
	addPerson("Alexander Kauer", "alexander@affine.space", "@kaueraal")
	addPerson("Alexander Kucherenko", "alxkchr@gmail.com")
	addPerson("Alexander Kuleshov", "kuleshovmail@gmail.com")
	addPerson("Alexander Larsson", "alexander.larsson@gmail.com", "@alexlarsson")
	addPerson("Alexander Menzhinsky", "amenzhinsky@gmail.com", "16045@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alexander Menzhinsky", "amenzhinsky@gmail.com", "@amenzhinsky")
	addPerson("Alexander Milyutin", "alexander.milyutin@lazada.com")
	addPerson("Alexander Morozov", "lk4d4math@gmail.com", "8340@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alexander Morozov", "lk4d4math@gmail.com", "@LK4D4")
	addPerson("Alexander Neumann", "alexander@bumpern.de", "@fd0")
	addPerson("Alexander Orlov", "alexander.orlov@loxal.net", "@loxal")
	addPerson("Alexander Polcyn", "apolcyn@google.com", "16623@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alexander Polcyn", "apolcyn@google.com", "@apolcyn")
	addPerson("Alexander Reece", "awreece@gmail.com", "@awreece")
	addPerson("Alexander Shopov", "ash@kambanaria.org", "@alshopov")
	addPerson("Alexander Zhavnerchik", "alex.vizor@gmail.com", "@alxzh")
	addPerson("Alexander Zolotov", "goldifit@gmail.com", "@zolotov")
	addPerson("Alexandre Cesaro", "alexandre.cesaro@gmail.com", "5647@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alexandre Cesaro", "alexandre.cesaro@gmail.com", "@alexcesaro")
	addPerson("Alexandre Fiori", "fiorix@gmail.com", "@fiorix")
	addPerson("Alexandre Maari", "draeron@gmail.com", "@draeron")
	addPerson("Alexandre Normand", "alexandre.normand@gmail.com", "@alexandre-normand")
	addPerson("Alexandre Parenteau", "aubonbeurre@gmail.com")
	addPerson("Alexandre Viau", "viau.alexandre@gmail.com", "27580@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alexandru Moșoi", "alexandru@mosoi.ro")
	addPerson("Alexandru Moșoi", "alexandru@mosoi.ro", "6173@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alexandru Moșoi", "brtzsnr@gmail.com", "5930@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alexandru Moșoi", "brtzsnr@gmail.com", "@brtzsnr")
	addPerson("Alexandru Moșoi", "mosoi@google.com")
	addPerson("Alexei Sholik", "alcosholik@gmail.com", "@alco")
	addPerson("Alexey Alexandrov", "aalexand@google.com", "@aalexand")
	addPerson("Alexey Borzenkov", "snaury@gmail.com", "@snaury")
	addPerson("Alexey Naidonov", "alexey.naidyonov@gmail.com")
	addPerson("Alexey Naidonov", "alexey.naidyonov@gmail.com", "@growler")
	addPerson("Alexey Neganov", "neganovalexey@gmail.com", "@neganovalexey")
	addPerson("Alexey Nezhdanov", "snakeru@gmail.com")
	addPerson("Alexey Nezhdanov", "snakeru@gmail.com", "9000@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alexey Palazhchenko", "alexey.palazhchenko@gmail.com", "13090@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Alexey Palazhchenko", "alexey.palazhchenko@gmail.com", "@AlekSi")
	addPerson("Alexey Vilenskiy", "vilenskialeksei@gmail.com")
	addPerson("Alexis Hildebrandt", "surryhill@gmail.com")
	addPerson("Alexis Horgix Chotard", "alexis.horgix.chotard@gmail.com")
	addPerson("Alexis Hunt", "lexer@google.com")
	addPerson("Alexis Imperial-Legrand", "ail@google.com", "@ailg")
	addPerson("Ali Rizvi-Santiago", "arizvisa@gmail.com", "@arizvisa")
	addPerson("Aliaksandr Valialkin", "valyala@gmail.com", "9525@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Aliaksandr Valialkin", "valyala@gmail.com", "@valyala")
	addPerson("Alif Rachmawadi", "subosito@gmail.com", "@subosito")
	addPerson("Alistair Barrell", "alistair.t.barrell@gmail.com")
	addPerson("Allan Simon", "allan.simon@supinfo.com", "@allan-simon")
	addPerson("Alok Menghrajani", "alok.menghrajani@gmail.com", "@alokmenghrajani")
	addPerson("Aman Gupta", "aman@tmm1.net", "20002@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Aman Gupta", "aman@tmm1.net", "@tmm1")
	addPerson("Amanuel Bogale", "abogale2@gmail.com")
	addPerson("Amir Mohammad Saied", "amir@gluegadget.com", "@amir")
	addPerson("Amit Ghadge", "amitg.b14@gmail.com")
	addPerson("Ammar Bandukwala", "ammar@ammar.io")
	addPerson("Amr A.Mohammed", "merodiro@gmail.com")
	addPerson("Amrut Joshi", "amrut.joshi@gmail.com", "@rumple")
	addPerson("Amy Schlesener", "amyschlesener@gmail.com")
	addPerson("Anand K. Mistry", "anand@mistry.ninja")
	addPerson("Anders Pearson", "anders@columbia.edu", "@thraxil")
	addPerson("Andrea Nodari", "andrea.nodari91@gmail.com")
	addPerson("Andrea Nodari", "andrea.nodari91@gmail.com", "@nodo")
	addPerson("Andrea Spadaccini", "spadaccio@google.com", "@lupino3")
	addPerson("Andreas Auernhamer", "andreas_golang@mail.de")
	addPerson("Andreas Auernhammer", "aead@mail.de", "14805@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Andreas Auernhammer", "aead@mail.de", "@aead")
	addPerson("Andreas Auernhammer", "enceve@mail.de")
	addPerson("Andreas Jellinghaus", "andreas@ionisiert.de", "@tolonuga")
	addPerson("Andreas Litt", "andreas.litt@gmail.com")
	addPerson("Andrei Gherzan", "andrei@resin.io")
	addPerson("Andrei Korzhevskii", "a.korzhevskiy@gmail.com", "@nordligulv")
	addPerson("Andrei Tudor Călin", "mail@acln.ro", "27279@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Andrei Tudor Călin", "mail@acln.ro", "@acln0")
	addPerson("Andres Erbsen", "andres.erbsen@gmail.com")
	addPerson("Andrew Austin", "andrewaclt@gmail.com", "@andrewaustin")
	addPerson("Andrew Benton", "andrewmbenton@gmail.com", "@andrewmbenton")
	addPerson("Andrew Bonventre", "365204+andybons@users.noreply.github.com")
	addPerson("Andrew Bonventre", "andybons@golang.org", "22285@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Andrew Bonventre", "andybons@golang.org", "andybons@gmail.com", "@andybons")
	addPerson("Andrew Bonventre", "andybons@google.com", "10660@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Andrew Bonventre", "hello@obvy.co")
	addPerson("Andrew Brampton", "bramp@google.com")
	addPerson("Andrew Braunstein", "awbraunstein@gmail.com", "@awbraunstein")
	addPerson("Andrew Ekstedt", "andrew.ekstedt@gmail.com", "6255@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Andrew Ekstedt", "andrew.ekstedt@gmail.com", "@magical")
	addPerson("Andrew Etter", "andrew.etter@gmail.com", "@andrewetter")
	addPerson("Andrew Gerrand", "adg@golang.org", "5010@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Andrew Gerrand", "adg@golang.org", "@adg")
	addPerson("Andrew Gerrand", "nf@wh3rd.net", "@nf")
	addPerson("Andrew Harding", "andrew@spacemonkey.com", "@azdagron")
	addPerson("Andrew M Bursavich", "abursavich@gmail.com", "@abursavich")
	addPerson("Andrew Patzer", "andrew.patzer@gmail.com")
	addPerson("Andrew Pilloud", "andrewpilloud@igneoussystems.com", "@apilloud")
	addPerson("Andrew Pogrebnoy", "absourd.noise@gmail.com", "@dAdAbird")
	addPerson("Andrew Poydence", "apoydence@pivotal.io", "@poy")
	addPerson("Andrew Pritchard", "awpritchard@gmail.com", "@awpr")
	addPerson("Andrew Radev", "andrey.radev@gmail.com", "@AndrewRadev")
	addPerson("Andrew Skiba", "skibaa@gmail.com", "@skibaa")
	addPerson("Andrew Szeto", "andrew@jabagawee.com", "@jabagawee")
	addPerson("Andrew Wilkins", "axwalk@gmail.com", "8640@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Andrew Wilkins", "axwalk@gmail.com", "@axw")
	addPerson("Andrew Williams", "williams.andrew@gmail.com", "@williamsandrew")
	addPerson("Andrey Petrov", "andrey.petrov@shazow.net", "@shazow")
	addPerson("Andrii Soldatenko", "andrii.soldatenko@gmail.com", "@andriisoldatenko")
	addPerson("Andrii Soluk", "isoluchok@gmail.com", "24501@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Andrii Zakharov", "andrii@messagebird.com")
	addPerson("Andris Valums", "avalums.spam@linelane.com")
	addPerson("Andriy Lytvynov", "lytvynov.a.v@gmail.com", "@awly")
	addPerson("Andrzej Żeżel", "andrii.zhezhel@gmail.com", "@zhezhel")
	addPerson("André Carvalho", "asantostc@gmail.com", "@andrestc")
	addPerson("Andy Balholm", "andy@balholm.com", "6535@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Andy Balholm", "andy@balholm.com", "@andybalholm")
	addPerson("Andy Balholm", "andybalholm@gmail.com")
	addPerson("Andy Bursavich", "bursavich@google.com")
	addPerson("Andy Davis", "andy@bigandian.com", "@bigandian")
	addPerson("Andy Finkenstadt", "afinkenstadt@zynga.com", "@afinkenstadt")
	addPerson("Andy Lindeman", "andy@lindeman.io")
	addPerson("Andy Maloney", "asmaloney@gmail.com", "@asmaloney")
	addPerson("Andy Walker", "walkeraj@gmail.com")
	addPerson("Anfernee Yongkun Gui", "anfernee.gui@gmail.com", "@anfernee")
	addPerson("Angelo Bulfone", "mbulfone@gmail.com", "@boomshroom")
	addPerson("Angelo Compagnucci", "angelo.compagnucci@gmail.com")
	addPerson("Anh Hai Trinh", "anh.hai.trinh@gmail.com", "@aht")
	addPerson("Anit Gandhi", "anitgandhi@gmail.com", "@anitgandhi")
	addPerson("Ankit Goyal", "ankit3goyal@gmail.com", "@goyalankit")
	addPerson("Anmol Sethi", "anmol@aubble.com", "@nhooyr")
	addPerson("Anmol Sethi", "hi@nhooyr.io")
	addPerson("Anmol Sethi", "me+git@anmol.io", "@nhooyr")
	addPerson("Anmol Sethi", "me@anmol.io", "9620@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Anschel Schaffer-Cohen", "anschelsc@gmail.com", "@anschelsc")
	addPerson("Anthony Alves", "cvballa3g0@gmail.com")
	addPerson("Anthony Canino", "anthony.canino1@gmail.com", "@anthonycanino1")
	addPerson("Anthony Eufemio", "anthony.eufemio@gmail.com", "@tymat")
	addPerson("Anthony Martin", "ality@pbrane.org", "5635@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Anthony Martin", "ality@pbrane.org", "@ality")
	addPerson("Anthony Pesch", "inolen@gmail.com")
	addPerson("Anthony Romano", "anthony.romano@coreos.com")
	addPerson("Anthony Sottile", "asottile@umich.edu", "@asottile")
	addPerson("Anthony Starks", "ajstarks@gmail.com", "@ajstarks")
	addPerson("Antoine Martin", "antoine97.martin@gmail.com", "@alarsyo")
	addPerson("Anton Gyllenberg", "anton@iki.fi", "@antong")
	addPerson("Antonin Amand", "antonin.amand@gmail.com", "@gwik")
	addPerson("Antonio Antelo", "aantelov87@gmail.com")
	addPerson("Antonio Bibiano", "antbbn@gmail.com", "@antbbn")
	addPerson("Antonio Murdaca", "runcom@redhat.com", "@runcom")
	addPerson("Aram Hăvărneanu", "aram@mgk.ro", "5036@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Aram Hăvărneanu", "aram@mgk.ro", "@4ad")
	addPerson("Arash Bina", "arash@arash.io")
	addPerson("Arash Bina", "arash@arash.io", "@arashbina")
	addPerson("Areski Belaid", "areski@gmail.com", "5825@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ariel Mashraki", "ariel@mashraki.co.il", "@a8m")
	addPerson("Arlo Breault", "arlolra@gmail.com", "@arlolra")
	addPerson("Arnaud Ysmal", "stacktic@netbsd.org", "@stacktic")
	addPerson("Arne Hormann", "arnehormann@gmail.com", "@arnehormann")
	addPerson("Arnout Engelen", "arnout@bzzt.net")
	addPerson("Aron Nopanen", "aron.nopanen@gmail.com", "@aroneous")
	addPerson("Artem V. Navrotskiy", "bozaro@gmail.com")
	addPerson("Artemiy Ryabinkov", "getmylag@gmail.com")
	addPerson("Arthur Khashaev", "arthur@khashaev.ru", "@Invizory")
	addPerson("Arthur Mello", "arthur.mello85@gmail.com")
	addPerson("Artyom Pervukhin", "artyom.pervukhin@gmail.com", "9870@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Artyom Pervukhin", "artyom.pervukhin@gmail.com", "@artyom")
	addPerson("Arvindh Rajesh Tamilmani", "art@a-30.net", "@arvindht")
	addPerson("Asad Mehmood", "asad78611@googlemail.com")
	addPerson("Ashish Gandhi", "ag@ashishgandhi.org", "@ashishgandhi")
	addPerson("Asim Shankar", "asimshankar@gmail.com", "@asimshankar")
	addPerson("Atin M", "amalaviy@akamai.com", "@amalaviy")
	addPerson("Ato Araki", "ato.araki@gmail.com", "@atotto")
	addPerson("Attila Tajti", "attila.tajti@gmail.com")
	addPerson("Audrey Lim", "audreylh@gmail.com", "13190@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Audrey Lim", "audreylh@gmail.com", "@audreylim")
	addPerson("Audrius Butkevicius", "audrius.butkevicius@gmail.com", "25277@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Audrius Butkevicius", "audrius.butkevicius@gmail.com", "@AudriusButkevicius")
	addPerson("Augusto Roman", "aroman@gmail.com", "@augustoroman")
	addPerson("Aulus Egnatius Varialus", "varialus@gmail.com", "@varialus")
	addPerson("Aurélien Rainone", "aurelien.rainone@gmail.com")
	addPerson("Aurélien Rainone", "aurelien.rainone@gmail.com", "@arl")
	addPerson("Austin Clements", "austin@google.com", "5167@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Austin Clements", "austin@google.com", "@aclements")
	addPerson("Austin J. Alexander", "austinjalexander@gmail.com")
	addPerson("Author Name", "aaronstein12@gmail.com", "@aastein")
	addPerson("Author Name", "brett.j.merrill94@gmail.com", "@bmerrill42")
	addPerson("Author Name", "mikemitchellwebdev@gmail.com")
	addPerson("Author: grantseltzer", "grantseltzer@gmail.com")
	addPerson("Avelino", "t@avelino.xxx", "8805@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Avelino", "t@avelino.xxx", "@avelino")
	addPerson("Awn Umar", "awn@cryptolosophy.org", "21940@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Awn", "awn@cryptolosophy.io")
	addPerson("Axel Wagner", "axel.wagner.hh@googlemail.com", "@Merovius")
	addPerson("Ayan George", "ayan@ayan.net")
	addPerson("Ayke van Laethem", "aykevanlaethem@gmail.com")
	addPerson("Aymerick", "aymerick@jehanne.org", "@aymerick")
	addPerson("B.G.Adrian", "aditza8@gmail.com")
	addPerson("Baiju Muthukadan", "baiju.m.mail@gmail.com", "@baijum")
	addPerson("Bakin Aleksey", "kultihell@gmail.com")
	addPerson("Balaram Makam", "bmakam.qdt@qualcommdatacenter.com", "25702@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Balaram Makam", "bmakam.qdt@qualcommdatacenter.com", "@bmakam-qdt")
	addPerson("Balazs Lecz", "leczb@google.com", "@leczb")
	addPerson("Baokun Lee", "nototon@gmail.com", "9646@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Baokun Lee", "nototon@gmail.com", "@oiooj")
	addPerson("Bartek Plotka", "bwplotka@gmail.com")
	addPerson("Bartosz Modelski", "modelski.bartosz@gmail.com")
	addPerson("Bastian Ike", "bastian.ike@gmail.com")
	addPerson("Baylee Feore", "baylee.feore@gmail.com")
	addPerson("Ben Burkert", "ben@benburkert.com", "5673@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ben Burkert", "ben@benburkert.com", "@benburkert")
	addPerson("Ben Fried", "ben.fried@gmail.com", "@benfried")
	addPerson("Ben Haines", "bhainesva@gmail.com")
	addPerson("Ben Hoyt", "benhoyt@gmail.com", "@benhoyt")
	addPerson("Ben Laurie", "ben@links.org", "21925@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ben Lubar", "ben.lubar@gmail.com", "@BenLubar")
	addPerson("Ben Lynn", "benlynn@gmail.com", "@blynn")
	addPerson("Ben Olive", "sionide21@gmail.com", "@sionide21")
	addPerson("Ben Schwartz", "bemasc@google.com", "20251@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ben Schwartz", "bemasc@google.com", "@bemasc")
	addPerson("Ben Shi", "powerman1st@163.com", "16935@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ben Shi", "powerman1st@163.com", "@benshi001")
	addPerson("Ben Toews", "mastahyeti@gmail.com")
	addPerson("Benjamin Black", "b@b3k.us", "@b")
	addPerson("Benjamin Cable", "cable.benjamin@gmail.com", "@ladydascalie")
	addPerson("Benjamin Hsieh", "tanookiben@users.noreply.github.com")
	addPerson("Benjamin Prosnitz", "bprosnitz@google.com", "6965@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Benjamin Prosnitz", "bprosnitz@google.com", "@bprosnitz")
	addPerson("Benjamin Wester", "bwester@squareup.com", "@bwester")
	addPerson("Benny Siegert", "bsiegert@gmail.com", "5184@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Benny Siegert", "bsiegert@gmail.com", "@bsiegert")
	addPerson("Benny Siegert", "bsiegert@google.com", "@bsiegert")
	addPerson("Benoit Sigoure", "tsunanet@gmail.com", "9643@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Benoit Sigoure", "tsunanet@gmail.com", "@tsuna")
	addPerson("Bernat Moix", "bmoix@bmoix.io")
	addPerson("Bernd Fix", "brf@hoi-polloi.org")
	addPerson("BigMikes", "giulio.micheloni@gmail.com", "@BigMikes")
	addPerson("Bill Neubauer", "wcn@google.com")
	addPerson("Bill O'Farrell", "billo@ca.ibm.com", "@wgo")
	addPerson("Bill O'Farrell", "billotosyr@gmail.com", "11191@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Bill Prin", "waprin@google.com")
	addPerson("Bill Thiede", "couchmoney@gmail.com", "6175@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Bill Thiede", "couchmoney@gmail.com", "@wathiede", "*goog")
	addPerson("Bill Zissimopoulos", "billziss@navimatics.com", "@billziss-gh")
	addPerson("Billie H. Cleek", "bhcleek@gmail.com", "@bhcleek")
	addPerson("Billy Lynch", "wlynch@google.com", "@wlynch")
	addPerson("Blain Smith", "blain.smith@gmail.com", "22696@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Blain Smith", "rebelgeek@blainsmith.com", "@blainsmith")
	addPerson("Blake Gentry", "blakesgentry@gmail.com", "5683@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Blake Gentry", "blakesgentry@gmail.com", "@bgentry")
	addPerson("Blake Mesdag", "blakemesdag@gmail.com")
	addPerson("Blake Mizerany", "blake.mizerany@gmail.com", "10551@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Blake Mizerany", "blake.mizerany@gmail.com", "@bmizerany")
	addPerson("Blixt", "me@blixt.nyc", "@blixt")
	addPerson("Bob B.", "rbriski@gmail.com", "26997@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Bob Potter", "bobby.potter@gmail.com")
	addPerson("Bobby DeSimone", "bobbydesimone@gmail.com", "@desimone")
	addPerson("Bobby Powers", "bobbypowers@gmail.com", "@bpowers")
	addPerson("Bodo Junglas", "bodo.junglas@leanovate.de")
	addPerson("Boris Nagaev", "nagaev@google.com")
	addPerson("Boris Schrijver", "bschrijver@schubergphilis.com")
	addPerson("Borja Clemente", "borja.clemente@gmail.com", "@clebs")
	addPerson("Brad Burch", "brad.burch@gmail.com", "@brad-burch")
	addPerson("Brad Fitzpatrick", "bradfitz@golang.org", "brad@danga.com", "@bradfitz", "5065@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Brad Jones", "rbjones@google.com")
	addPerson("Brad Morgan", "brad@morgabra.com")
	addPerson("Brad Whitaker", "brad.whitaker@gmail.com")
	addPerson("Braden Bassingthwaite", "bbassingthwaite@vendasta.com")
	addPerson("Bradley Kemp", "bradleyjkemp96@gmail.com")
	addPerson("Bradley Schoch", "bschoch@gmail.com")
	addPerson("Brady Catherman", "brady@gmail.com")
	addPerson("Brady Sullivan", "brady@bsull.com", "@d1str0")
	addPerson("Brandon Bennett", "bbennett@fb.com", "@brbe")
	addPerson("Brandon Bennett", "bbennett@fb.com", "@nemith")
	addPerson("Brandon Dyck", "brandon@dyck.us")
	addPerson("Brandon Gilmore", "varz@google.com", "@bgilmore")
	addPerson("Brandon Gonzalez", "bg@lightstep.com")
	addPerson("Brandon Lum", "lumjjb@gmail.com")
	addPerson("Brendan Ashworth", "brendan.ashworth@me.com")
	addPerson("Brendan Daniel Tracey", "tracey.brendan@gmail.com", "@btracey")
	addPerson("Brendan Tracey", "tracey.brendan@gmail.com", "7155@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Brett Cannon", "bcannon@gmail.com", "@brettcannon")
	addPerson("Brett Jones", "bjones027@gmail.com")
	addPerson("Brian Dellisanti", "briandellisanti@gmail.com", "@briandellisanti")
	addPerson("Brian Downs", "brian.downs@gmail.com", "@briandowns")
	addPerson("Brian Flanigan", "brian_flanigan@cable.comcast.com")
	addPerson("Brian G. Merrell", "bgmerrell@gmail.com", "@bgmerrell")
	addPerson("Brian Gitonga Marete", "bgm@google.com", "@marete")
	addPerson("Brian Gitonga Marete", "marete@toshnix.com", "@marete")
	addPerson("Brian Kennedy", "btkennedy@gmail.com", "@briantkennedy")
	addPerson("Brian Kessler", "brian.m.kessler@gmail.com", "20650@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Brian Kessler", "brian.m.kessler@gmail.com", "@bmkessler")
	addPerson("Brian Ketelsen", "bketelsen@gmail.com", "@bketelsen")
	addPerson("Brian Slesinskya", "skybrian@google.com", "@skybrian")
	addPerson("Brian Smith", "ohohvi@gmail.com", "@sirwart")
	addPerson("Brian Starke", "brian.starke@gmail.com", "@brianstarke")
	addPerson("Brian Starkey", "stark3y@gmail.com")
	addPerson("Bruno Clermont", "bruno.clermont@gmail.com")
	addPerson("Bryan Alexander", "kozical@msn.com", "@Kozical")
	addPerson("Bryan C. Mills", "bcmills@google.com", "6365@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Bryan C. Mills", "bcmills@google.com", "@bcmills")
	addPerson("Bryan Chan", "bryan.chan@ca.ibm.com", "@bryanpkc")
	addPerson("Bryan Chan", "bryanpkc@gmail.com", "6576@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Bryan Ford", "brynosaurus@gmail.com", "5500@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Bryan Ford", "brynosaurus@gmail.com", "@bford")
	addPerson("Bryan Heden", "b.heden@gmail.com", "@hedenface")
	addPerson("Bryan Mills", "bcmills@google.com", "@bcmills")
	addPerson("Bryan Turley", "bryanturley@gmail.com")
	addPerson("Bulat Gaifullin", "gaifullinbf@gmail.com", "@bgaifullin")
	addPerson("Burak Guven", "bguven@gmail.com", "@burakguven")
	addPerson("Caine Tighe", "arctanofyourface@gmail.com", "@nilnilnil")
	addPerson("Caio Marcelo de Oliveira Filho", "caio.oliveira@intel.com", "@cmarcelo")
	addPerson("Caio Oliveira", "caio.oliveira@intel.com", "12640@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Caleb Doxsey", "caleb@doxsey.net")
	addPerson("Caleb Martinez", "accounts@calebmartinez.com", "@conspicuousClockwork")
	addPerson("Caleb Spare", "cespare@gmail.com", "5615@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Caleb Spare", "cespare@gmail.com", "@cespare")
	addPerson("Calvin Behling", "calvin.behling@gmail.com")
	addPerson("Calvin Leung Huang", "cleung2010@gmail.com")
	addPerson("Cameron Howey", "chowey@ualberta.net")
	addPerson("Carl Chatfield", "carlchatfield@gmail.com", "@0xfaded")
	addPerson("Carl Henrik Lunde", "chlunde@ifi.uio.no")
	addPerson("Carl Henrik Lunde", "chlunde@ifi.uio.no", "@chlunde")
	addPerson("Carl Jackson", "carl@stripe.com", "@carl-stripe")
	addPerson("Carl Johnson", "me@carlmjohnson.net", "12425@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Carl Johnson", "me@carlmjohnson.net", "@carlmjohnson")
	addPerson("Carl Mastrangelo", "carl.mastrangelo@gmail.com")
	addPerson("Carl Mastrangelo", "carlmastrangelo@gmail.com")
	addPerson("Carl Mastrangelo", "notcarl@google.com", "12225@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Carl Mastrangelo", "notcarl@google.com", "@carl-mastrangelo")
	addPerson("Carlisia Campos", "carlisia@grokkingtech.io", "@carlisia")
	addPerson("Carlo Alberto Ferraris", "cafxx@strayorange.com", "11500@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Carlo Alberto Ferraris", "cafxx@strayorange.com", "@CAFxX")
	addPerson("Carlos C", "uldericofilho@gmail.com", "@ucirello")
	addPerson("Carlos Castillo", "cookieo9@gmail.com", "5141@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Carlos Castillo", "cookieo9@gmail.com", "@cookieo9")
	addPerson("Carlos Eduardo Seo", "cseo@linux.vnet.ibm.com", "13015@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Carlos Eduardo Seo", "cseo@linux.vnet.ibm.com", "@ceseo")
	addPerson("Carolyn Van Slyck", "me@carolynvanslyck.com", "@carolynvs")
	addPerson("Carrie Bynon", "cbynon@gmail.com", "@cbynon")
	addPerson("Casey Callendrello", "squeed@gmail.com")
	addPerson("Casey Marshall", "casey.marshall@gmail.com", "@cmars")
	addPerson("Casey Smith", "smithc@homesandland.com")
	addPerson("Cassandra Salisbury", "cls@golang.org")
	addPerson("Cassandra Salisbury", "salisburycl@gmail.com")
	addPerson("Catalin Nicutar", "cnicutar@google.com")
	addPerson("Catalin Nicutar", "cnicutar@google.com", "12526@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Catalin Patulea", "catalinp@google.com", "@cpatulea")
	addPerson("Cedric Staub", "cs@squareup.com", "@csstaub")
	addPerson("Cezar Espinola", "cezarsa@gmail.com", "9010@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Cezar Sa Espinola", "cezarsa@gmail.com", "@cezarsa")
	addPerson("Chad Kunde", "Kunde21@gmail.com")
	addPerson("Chad Rosier", "mrosier.qdt@qualcommdatacenter.com", "25690@62eb7196-b449-3ce5-99f1-c037f21e1705", "@mrosier-qdt")
	addPerson("ChaiShushan", "chaishushan@gmail.com", "@chai2010")
	addPerson("Chance Zibolski", "chance.zibolski@coreos.com")
	addPerson("Changsoo Kim", "broodkcs@gmail.com")
	addPerson("Charle Demers", "charle.demers@gmail.com")
	addPerson("Charles Fenwick Elliott", "Charles@FenwickElliott.io")
	addPerson("Charles Kenney", "charlesc.kenney@gmail.com", "@Charliekenney23")
	addPerson("Charles Weill", "weill@google.com", "@cweill")
	addPerson("Charlie Dorian", "cldorian@gmail.com", "5435@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Charlie Dorian", "cldorian@gmail.com", "@cldorian")
	addPerson("Cheng-Lung Sung", "clsung@gmail.com", "@clsung")
	addPerson("Cherry Zhang", "cherryyz@google.com", "13315@62eb7196-b449-3ce5-99f1-c037f21e1705", "@cherrymui", "lunaria21@gmail.com", "9670@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Chew Choon Keat", "choonkeat@gmail.com", "@choonkeat")
	addPerson("Chintan Sheth", "shethchintan7@gmail.com")
	addPerson("Cholerae Hu", "choleraehyq@gmail.com", "15760@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Cholerae Hu", "choleraehyq@gmail.com", "@choleraehyq")
	addPerson("Chotepud Teo", "alexrousg@users.noreply.github.com", "@AlexRouSg")
	addPerson("Chris Ball", "chris@printf.net", "@cjb")
	addPerson("Chris Biscardi", "chris@christopherbiscardi.com", "@ChristopherBiscardi")
	addPerson("Chris Broadfoot", "cbro@golang.org", "7935@62eb7196-b449-3ce5-99f1-c037f21e1705", "cbro@google.com", "@broady", "7440@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Chris Dollin", "ehog.hedge@gmail.com", "@ehedgehog")
	addPerson("Chris Donnelly", "cmd@ceedon.io")
	addPerson("Chris Duarte", "csduarte@gmail.com")
	addPerson("Chris Farmiloe", "chrisfarms@gmail.com", "@chrisfarms")
	addPerson("Chris H (KruftMaster)", "chrusty@gmail.com")
	addPerson("Chris Hines", "chris.cs.guy@gmail.com", "7850@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Chris Hines", "chris.cs.guy@gmail.com", "@ChrisHines")
	addPerson("Chris J Arges", "christopherarges@gmail.com")
	addPerson("Chris Jones", "chris@cjones.org", "@cjyar")
	addPerson("Chris K", "c@chrisko.ch")
	addPerson("Chris Kastorff", "encryptio@gmail.com", "@encryptio")
	addPerson("Chris Lennert", "calennert@gmail.com", "@calennert")
	addPerson("Chris Lewis", "cflewis@golang.org")
	addPerson("Chris Lewis", "cflewis@google.com")
	addPerson("Chris Liles", "caveryliles@gmail.com", "26297@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Chris Manghane", "cmang@golang.org", "5130@62eb7196-b449-3ce5-99f1-c037f21e1705", "@paranoiacblack")
	addPerson("Chris Marchesi", "chrism@vancluevertech.com", "@vancluever")
	addPerson("Chris McGee", "newton688@gmail.com", "15452@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Chris McGee", "sirnewton_01@yahoo.ca", "@sirnewton01")
	addPerson("Chris Raynor", "raynor@google.com")
	addPerson("Chris Roche", "rodaine@gmail.com", "@rodaine")
	addPerson("Chris Stockton", "chrisstocktonaz@gmail.com")
	addPerson("Chris Zou", "chriszou@ca.ibm.com", "@ChrisXZou")
	addPerson("ChrisALiles", "caveryliles@gmail.com", "@ChrisALiles")
	addPerson("Christian Alexander", "christian@linux.com", "@ChristianAlexander")
	addPerson("Christian Couder", "chriscool@tuxfamily.org", "@chriscool")
	addPerson("Christian Couder", "christian.couder@gmail.com", "11200@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Christian Haas", "christian.haas@sevensuns.at")
	addPerson("Christian Himpel", "chressie@googlemail.com", "@chressie")
	addPerson("Christian Mauduit", "ufoot@ufoot.org")
	addPerson("Christian Pellegrin", "chri@evolware.org")
	addPerson("Christian Simon", "simon@swine.de")
	addPerson("Christoph Hack", "christoph@tux21b.org", "@tux21b")
	addPerson("Christophe Kamphaus", "christophe.kamphaus@gmail.com")
	addPerson("Christophe Taton", "taton@google.com")
	addPerson("Christopher Boumenot", "chrboum@microsoft.com")
	addPerson("Christopher Cahoon", "chris.cahoon@gmail.com", "@ccahoon")
	addPerson("Christopher Guiney", "chris@guiney.net", "@chrisguiney")
	addPerson("Christopher Koch", "chrisko@google.com", "@hugelgupf")
	addPerson("Christopher Nelson", "nadiasvertex@gmail.com", "11675@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Christopher Nelson", "nadiasvertex@gmail.com", "@nadiasvertex")
	addPerson("Christopher Nielsen", "m4dh4tt3r@gmail.com", "@m4dh4tt3r")
	addPerson("Christopher Redden", "christopher.redden@gmail.com", "@topherredden")
	addPerson("Christopher Wedgwood", "cw@f00f.org", "@cwedgwood")
	addPerson("Christos Zoulas", "christos@zoulas.com", "@zoulasc")
	addPerson("Christos Zoulas", "zoulasc@gmail.com")
	addPerson("Christy Perez", "christy@linux.vnet.ibm.com", "@clnperez")
	addPerson("Cindy Pallares", "cindy@gitlab.com")
	addPerson("Cixtor", "cixtords@gmail.com", "@cixtor")
	addPerson("Claire Wang", "cw773@cornell.edu")
	addPerson("Clement Courbet", "courbet@google.com")
	addPerson("Clement Skau", "clementskau@gmail.com", "@cskau")
	addPerson("Clément Chigot", "clement.chigot@atos.net", "@Helflym")
	addPerson("Clément Denis", "clement@altirnao.com")
	addPerson("Coda Hale", "coda.hale@gmail.com")
	addPerson("Colby Ranger", "cranger@google.com", "@crangeratgoogle")
	addPerson("Colin Cross", "ccross@android.com", "@colincross")
	addPerson("Colin Edwards", "colin@recursivepenguin.com", "@DDRBoxman")
	addPerson("Colin Kennedy", "moshen.colin@gmail.com", "@moshen")
	addPerson("Colin", "clr@google.com")
	addPerson("Connor McGuinness", "connor.mcguinness@izettle.com")
	addPerson("Conrad Irwin", "conrad.irwin@gmail.com", "@ConradIrwin")
	addPerson("Conrad Meyer", "cemeyer@cs.washington.edu", "@cemeyer")
	addPerson("Conrad Taylor", "conradwt@gmail.com")
	addPerson("Conrado Gouvea", "conradoplg@gmail.com", "@conradoplg")
	addPerson("Constantijn Schepens", "constantijnschepens@gmail.com")
	addPerson("Constantin Konstantinidis", "constantinkonstantinidis@gmail.com", "26957@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Constantin Konstantinidis", "constantinkonstantinidis@gmail.com", "@iWdGo")
	addPerson("Corey Thomasson", "cthom.lists@gmail.com", "@cthom06")
	addPerson("Cory LaNou", "cory@lanou.com")
	addPerson("Cosmos Nicolaou", "cnicolaou@grailbio.com")
	addPerson("Costin Chirvasuta", "ctin@google.com", "@ct1n")
	addPerson("Craig Citro", "craigcitro@google.com", "@craigcitro")
	addPerson("Craig Peterson", "cpeterson@stackoverflow.com")
	addPerson("Cristian Staretu", "unclejacksons@gmail.com", "@unclejack")
	addPerson("Cuihtlauac ALVARADO", "cuihtlauac.alvarado@orange.com", "@cuihtlauac")
	addPerson("Cuong Manh Le", "cuong.manhle.vn@gmail.com", "14665@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Cyrill Schumacher", "cyrill@schumacher.fm", "@SchumacherFM")
	addPerson("Daker Fernandes Pinheiro", "daker.fernandes.pinheiro@intel.com", "@dakerfp")
	addPerson("Dalton Scott", "dscott.jobs@gmail.com")
	addPerson("Damian Gryski", "damian@gryski.com", "@dgryski")
	addPerson("Damian Gryski", "dgryski@gmail.com", "7050@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Damian Gryski", "dgryski@gmail.com", "@dgryski")
	addPerson("Damien Lespiau", "damien.lespiau@gmail.com", "13855@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Damien Lespiau", "damien.lespiau@intel.com", "damien.lespiau@gmail.com", "@dlespiau")
	addPerson("Damien Mathieu", "42@dmathieu.com", "@dmathieu")
	addPerson("Damien Neil", "dneil@google.com", "5305@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Damien Neil", "dneil@google.com", "@neild")
	addPerson("Damien Tournoud", "damien@platform.sh")
	addPerson("Dan Adkins", "dadkins@gmail.com")
	addPerson("Dan Ballard", "dan@mindstab.net")
	addPerson("Dan Barry", "dan@bakineggs.com")
	addPerson("Dan Bentley", "dtbentley@gmail.com")
	addPerson("Dan Caddigan", "goldcaddy77@gmail.com", "@goldcaddy77")
	addPerson("Dan Callahan", "dan.callahan@gmail.com", "@callahad")
	addPerson("Dan Ertman", "dtertman@gmail.com")
	addPerson("Dan Goldsmith", "dan@d2g.org.uk")
	addPerson("Dan Harrington", "harringtond@google.com")
	addPerson("Dan Jacques", "dnj@google.com")
	addPerson("Dan Johnson", "computerdruid@google.com", "@ComputerDruid")
	addPerson("Dan Kortschak", "dan.kortschak@adelaide.edu.au", "6480@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Dan Kortschak", "dan@kortschak.io", "@kortschak")
	addPerson("Dan Luedtke", "mail@danrl.com")
	addPerson("Dan Moore", "mooreds@gmail.com")
	addPerson("Dan Peterson", "dpiddy@gmail.com", "5665@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Dan Peterson", "dpiddy@gmail.com", "@danp")
	addPerson("Dan Richards", "dan.m.richards@gmail.com")
	addPerson("Dan Richelson", "drichelson@gmail.com")
	addPerson("Dan Sinclair", "dan.sinclair@gmail.com", "@dj2")
	addPerson("Dana Hoffman", "danahoffman@google.com")
	addPerson("Daniel Cormier", "daniel.cormier@gmail.com")
	addPerson("Daniel Fleischman", "danielfleischman@gmail.com", "@danielf")
	addPerson("Daniel Heckrath", "d.heckrath@maple-apps.com")
	addPerson("Daniel Hultqvist", "daniel@typedef.se")
	addPerson("Daniel Ingram", "ingramds@appstate.edu", "@daniel-s-ingram")
	addPerson("Daniel Johansson", "dajo2002@gmail.com", "9663@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Daniel Johansson", "dajo2002@gmail.com", "@dajoo75")
	addPerson("Daniel Kerwin", "d.kerwin@gini.net", "@dkerwin")
	addPerson("Daniel Krech", "eikeon@eikeon.com", "@eikeon")
	addPerson("Daniel Mahu", "dmahu@google.com")
	addPerson("Daniel Martí", "mvdan@mvdan.cc", "13550@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Daniel Martí", "mvdan@mvdan.cc", "@mvdan")
	addPerson("Daniel Morsing", "daniel.morsing@gmail.com", "5310@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Daniel Morsing", "daniel.morsing@gmail.com", "@DanielMorsing")
	addPerson("Daniel Nephin", "dnephin@gmail.com", "@dnephin")
	addPerson("Daniel Ortiz Pereira da Silva", "daniel.particular@gmail.com", "@dopsilva")
	addPerson("Daniel Skinner", "daniel@dasa.cc", "10675@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Daniel Skinner", "daniel@dasa.cc", "@dskinner")
	addPerson("Daniel Speichert", "daniel@speichert.pl", "@DSpeichert")
	addPerson("Daniel Theophanes", "kardianos@gmail.com", "5080@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Daniel Theophanes", "kardianos@gmail.com", "@kardianos")
	addPerson("Daniel Toebe", "dtoebe@gmail.com")
	addPerson("Daniel Upton", "daniel@floppy.co", "@boxofrad")
	addPerson("Daniel Wagner-Hall", "dawagner@gmail.com")
	addPerson("Daniel", "danielfs.ti@gmail.com")
	addPerson("Daniel, Dao Quang Minh", "dqminh89@gmail.com")
	addPerson("Daniela Petruzalek", "daniela.petruzalek@gmail.com", "@danicat")
	addPerson("Daniël de Kok", "me@danieldk.eu", "@danieldk")
	addPerson("Danny Hadley", "dadleyy@gmail.com")
	addPerson("Danny Rosseau", "daniel.rosseau@gmail.com")
	addPerson("Danny Wyllie", "wylliedanny@gmail.com")
	addPerson("Danny Yoo", "dannyyoo@google.com")
	addPerson("Dante Shareiff", "prophesional@gmail.com")
	addPerson("Darien Raymond", "admin@v2ray.com")
	addPerson("Darien Raymond", "admin@v2ray.com", "@DarienRaymond")
	addPerson("Darren Elwood", "darren@textnode.com", "@textnode")
	addPerson("Darron Froese", "dfroese@salesforce.com")
	addPerson("Darshan Parajuli", "parajulidarshan@gmail.com", "@darshanparajuli")
	addPerson("Datong Sun", "dndx@idndx.com", "@dndx")
	addPerson("Dave Borowitz", "dborowitz@google.com", "@dborowitz")
	addPerson("Dave Cheney", "dave@cheney.net", "5150@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Dave Cheney", "dave@cheney.net", "@davecheney")
	addPerson("Dave Day", "djd@golang.org", "5170@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Dave Day", "djd@golang.org", "@okdave")
	addPerson("Dave MacFarlane", "driusan@gmail.com")
	addPerson("Dave Russell", "forfuncsake@gmail.com", "@forfuncsake")
	addPerson("Dave Setzke", "daveset73@gmail.com")
	addPerson("Dave Wyatt", "dlwyatt115@gmail.com")
	addPerson("David Anderson", "danderson@google.com", "13070@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("David Anderson", "danderson@google.com", "@danderson")
	addPerson("David Barnett", "dbarnett@google.com", "@dbarnett")
	addPerson("David Bartley", "bartle@stripe.com")
	addPerson("David Benjamin", "davidben@google.com", "7805@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("David Benjamin", "davidben@google.com", "@davidben")
	addPerson("David Benque", "dbenque@gmail.com")
	addPerson("David Brophy", "dave@brophy.uk", "@dave")
	addPerson("David Bürgin", "676c7473@gmail.com", "@glts")
	addPerson("David Calavera", "david.calavera@gmail.com", "@calavera")
	addPerson("David Carlier", "devnexen@gmail.com", "@devnexen")
	addPerson("David Chase", "drchase@google.com", "7061@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("David Chase", "drchase@google.com", "@dr2chase")
	addPerson("David Crawshaw", "crawshaw@golang.org", "5030@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("David Crawshaw", "crawshaw@golang.org", "@crawshaw")
	addPerson("David Deng", "daviddengcn@gmail.com")
	addPerson("David Finkel", "david.finkel@gmail.com")
	addPerson("David Forsythe", "dforsythe@gmail.com", "@dforsyth")
	addPerson("David G. Andersen", "dave.andersen@gmail.com", "@dave-andersen")
	addPerson("David Glasser", "glasser@meteor.com", "9556@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("David Glasser", "glasser@meteor.com", "@glasser")
	addPerson("David Good", "dgood@programminggoody.com")
	addPerson("David Heuschmann", "heuschmann.d@gmail.com", "@dddent")
	addPerson("David Howden", "dhowden@gmail.com")
	addPerson("David Hubbard", "dsp@google.com")
	addPerson("David Jakob Fritz", "david.jakob.fritz@gmail.com", "@djfritz")
	addPerson("David Kitchen", "david@buro9.com")
	addPerson("David Lazar", "lazard@golang.org", "16260@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("David Lazar", "lazard@golang.org", "@davidlazar")
	addPerson("David Leon Gil", "coruus@gmail.com", "5830@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("David Leon Gil", "coruus@gmail.com", "@coruus")
	addPerson("David Ndungu", "dnjuguna@gmail.com")
	addPerson("David NewHamlet", "david@newhamlet.com", "@wheelcomplex")
	addPerson("David Newhamlet", "david@newhamlet.com", "13738@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("David Presotto", "presotto@gmail.com", "@presotto")
	addPerson("David R. Jenni", "david.r.jenni@gmail.com", "@davidrjenni")
	addPerson("David R. Jenni", "davidrjenni@protonmail.com", "6180@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("David Sansome", "me@davidsansome.com")
	addPerson("David Stainton", "dstainton415@gmail.com", "@david415")
	addPerson("David Symonds", "dsymonds@golang.org", "5045@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("David Symonds", "dsymonds@golang.org", "@dsymonds")
	addPerson("David Thomas", "davidthomas426@gmail.com", "@davidthomas426")
	addPerson("David Timm", "dtimm@pivotal.io", "@dtimm")
	addPerson("David Titarenco", "david.titarenco@gmail.com", "@dvx")
	addPerson("David Tolpin", "david.tolpin@gmail.com", "@dtolpin")
	addPerson("David Url", "david@urld.io", "26506@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("David Url", "david@urld.io", "@urld")
	addPerson("David Volquartz Lebech", "david@lebech.info")
	addPerson("David Wimmer", "davidlwimmer@gmail.com", "@dwimmer")
	addPerson("David du Colombier", "0intro@gmail.com")
	addPerson("David du Colombier", "0intro@gmail.com", "5060@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("David du Colombier", "0intro@gmail.com", "@0intro")
	addPerson("Davies Liu", "davies.liu@gmail.com", "@davies")
	addPerson("Davor Kapsa", "davor.kapsa@gmail.com", "@dvrkps")
	addPerson("Ddo", "joeddo89@gmail.com")
	addPerson("Dean Prichard", "dean.prichard@gmail.com", "@zard49")
	addPerson("Deepak Jois", "deepak.jois@gmail.com", "@deepakjois")
	addPerson("Deepali Raina", "deepali.raina@gmail.com")
	addPerson("Denis Bernard", "db047h@gmail.com", "@db47h")
	addPerson("Denis Nagorny", "denis.nagorny@intel.com", "10734@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Denis Nagorny", "denis.nagorny@intel.com", "@dvnagorny")
	addPerson("Dennis Kuhnert", "mail.kuhnert@gmail.com", "26874@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Dennis Kuhnert", "mail.kuhnert@gmail.com", "@kyroy")
	addPerson("Denys Honsiorovskyi", "honsiorovskyi@gmail.com", "@honsiorovskyi")
	addPerson("Denys Smirnov", "denis.smirnov.91@gmail.com", "@dennwc")
	addPerson("Derek Bruening", "bruening@google.com")
	addPerson("Derek Buitenhuis", "derek.buitenhuis@gmail.com", "@dwbuiten")
	addPerson("Derek Che", "drc@yahoo-inc.com")
	addPerson("Derek Che", "drc@yahoo-inc.com", "5750@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Derek McGowan", "derek@mcgstyle.net")
	addPerson("Derek Parker", "parkerderek86@gmail.com", "@derekparker")
	addPerson("Derek Perkins", "derek@derekperkins.com")
	addPerson("Derek Phan", "derekphan94@gmail.com", "@dphan72")
	addPerson("Derek Shockey", "derek.shockey@gmail.com", "@derelk")
	addPerson("Dev Ojha", "dojha12@gmail.com", "27059@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Deval Shah", "devalshah88@gmail.com")
	addPerson("Devon H. O'Dell", "devon.odell@gmail.com", "25956@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Devon H. O'Dell", "devon.odell@gmail.com", "@dhobsd")
	addPerson("Dhaivat Pandit", "dhaivatpandit@gmail.com", "15030@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Dhaivat Pandit", "dhaivatpandit@gmail.com", "@ceocoder")
	addPerson("Dhananjay Nakrani", "dhananjayn@google.com")
	addPerson("Dhananjay Nakrani", "dhananjayn@google.com", "15558@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Dhananjay Nakrani", "dhananjaynakrani@gmail.com", "@dhananjay92")
	addPerson("Dhiru Kholia", "dhiru.kholia@gmail.com", "@kholia")
	addPerson("Dhruvdutt Jadhav", "dhruvdutt.jadhav@gmail.com", "@dhruvdutt")
	addPerson("Di Xiao", "dixiao@google.com")
	addPerson("Di Xiao", "xiaodi.larry@gmail.com")
	addPerson("Didier Spezia", "didier.06@gmail.com", "7795@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Didier Spezia", "didier.06@gmail.com", "@dspezia")
	addPerson("Diego Saint Esteben", "diego@saintesteben.me")
	addPerson("Diego Siqueira", "diego9889@gmail.com", "@DiSiqueira")
	addPerson("Dieter Plaetinck", "dieter@raintank.io")
	addPerson("Dieter Plaetinck", "dieter@raintank.io", "@Dieterbe")
	addPerson("Dimitri Tcaciuc", "dtcaciuc@gmail.com", "@dtcaciuc")
	addPerson("Dimitrios Arethas", "darethas@gmail.com")
	addPerson("Dina Garmash", "dgrmsh@gmail.com", "@dgrmsh")
	addPerson("Dinesh Kumar", "dinesh.kumar@go-jek.com")
	addPerson("Diogo Pinela", "diogoid7400@gmail.com")
	addPerson("Diogo Pinela", "diogoid7400@gmail.com", "16943@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Diogo Pinela", "diogoid7400@gmail.com", "@dpinela")
	addPerson("Dirk Gadsden", "dirk@esherido.com", "@dirk")
	addPerson("Diwaker Gupta", "diwakergupta@gmail.com", "@diwakergupta")
	addPerson("Dmitri Popov", "operator@cv.dp-net.com", "@pin")
	addPerson("Dmitri Shuralyov", "dmitshur@golang.org", "6005@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Dmitri Shuralyov", "dmitshur@golang.org", "dmitri@shuralyov.com", "shurcool@gmail.com", "@dmitshur")
	addPerson("Dmitriy Dudkin", "dudkin.dmitriy@gmail.com", "@tmwh")
	addPerson("Dmitriy", "dchenk@users.noreply.github.com")
	addPerson("Dmitry Chestnykh", "dchest@gmail.com", "@dchest")
	addPerson("Dmitry Doroginin", "doroginin@gmail.com", "@doroginin")
	addPerson("Dmitry Mottl", "dmitry.mottl@gmail.com", "@Mottl")
	addPerson("Dmitry Neverov", "dmitry.neverov@gmail.com", "@nd")
	addPerson("Dmitry Pokidov", "dooman87@gmail.com")
	addPerson("Dmitry Savintsev", "dsavints@gmail.com", "6190@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Dmitry Savintsev", "dsavints@gmail.com", "@dmitris")
	addPerson("Dmitry Vyukov", "dvyukov@google.com", "5400@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Dmitry Vyukov", "dvyukov@google.com", "@dvyukov")
	addPerson("DocMerlin", "landivar@gmail.com")
	addPerson("Dominic Barnes", "dominic@dbarnes.info")
	addPerson("Dominic Green", "dominicgreen1@gmail.com")
	addPerson("Dominik Honnef", "dominik@honnef.co", "5020@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Dominik Honnef", "dominik@honnef.co", "@dominikh")
	addPerson("Dominik Vogt", "vogt@linux.vnet.ibm.com", "6065@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Dominik Vogt", "vogt@linux.vnet.ibm.com", "@vogtd")
	addPerson("Don Byington", "don@dbyington.com", "@dbyington")
	addPerson("Donald Huang", "don.hcd@gmail.com")
	addPerson("Dong-hee Na", "donghee.na92@gmail.com", "17352@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Dong-hee Na", "donghee.na92@gmail.com", "@corona10")
	addPerson("Donovan Hide", "donovanhide@gmail.com", "@donovanhide")
	addPerson("Doug Evans", "dje@google.com")
	addPerson("Doug Fawley", "dfawley@google.com")
	addPerson("Dragoslav Mitrinovic", "fdm224@motorola.com")
	addPerson("Drew Flower", "drewvanstone@gmail.com", "@drewvanstone")
	addPerson("Drew Hintz", "adhintz@google.com", "@adhintz")
	addPerson("Duco van Amstel", "duco@improbable.io")
	addPerson("Duncan Holm", "mail@frou.org", "@frou")
	addPerson("Dusan Kasan", "me@dusankasan.com")
	addPerson("Dustin Carlino", "dcarlino@google.com")
	addPerson("Dustin Shields-Cloues", "dcloues@gmail.com", "@dcloues")
	addPerson("Dylan Carney", "dcarney@gmail.com")
	addPerson("Dylan Waits", "dylan@waits.io", "@waits")
	addPerson("EKR", "ekr@rtfm.com", "@ekr")
	addPerson("Edan B", "3d4nb3@gmail.com", "@edanbe")
	addPerson("Eddie Ringle", "eddie@ringle.io")
	addPerson("Eden Li", "eden.li@gmail.com", "@eden")
	addPerson("Edson Medina", "edsonmedina@gmail.com")
	addPerson("EduRam", "eduardo.ramalho@gmail.com", "@EduRam")
	addPerson("Eduard Urbach", "e.urbach@gmail.com")
	addPerson("Eduard Urbach", "e.urbach@gmail.com", "@blitzprog")
	addPerson("Edward Muller", "edwardam@interlix.com", "9641@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Edward Muller", "edwardam@interlix.com", "@freeformz")
	addPerson("Egon Elbre", "egonelbre@gmail.com", "6785@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Egon Elbre", "egonelbre@gmail.com", "@egonelbre")
	addPerson("Ehden Sinai", "ehdens@gmail.com")
	addPerson("Ehren Kret", "ehren.kret@gmail.com", "@eakret")
	addPerson("Eitan Adler", "lists@eitanadler.com", "@grimreaper")
	addPerson("Eivind Uggedal", "eivind@uggedal.com", "@uggedal")
	addPerson("Elbert Fliek", "efliek@gmail.com", "@Nr90")
	addPerson("Eldar Rakhimberdin", "ibeono@gmail.com")
	addPerson("Elen Eisendle", "elen@eisendle.ee")
	addPerson("Elena Grahovac", "elena@grahovac.me")
	addPerson("Elias Naur", "mail@eliasnaur.com", "@eliasnaur")
	addPerson("Elias Naur", "mail@eliasnaur.com", "elias.naur@gmail.com", "@eliasnaur", "7435@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Elliot Morrison-Reed", "elliotmr@gmail.com", "@elliotmr")
	addPerson("Emanuele Iannone", "emanuele@fondani.it")
	addPerson("Emerson Lin", "linyintor@gmail.com", "21970@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Emil Hessman", "emil@hessman.se", "5555@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Emil Hessman", "emil@hessman.se", "c.emil.hessman@gmail.com", "@ceh")
	addPerson("Emmanuel Odeke", "emm.odeke@gmail.com", "5137@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Emmanuel Odeke", "emm.odeke@gmail.com", "@odeke-em")
	addPerson("Emmanuel Odeke", "emmanuel@orijtech.com", "27585@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Emmanuel Odeke", "odeke@ualberta.ca")
	addPerson("Emmanuel Odeke", "odeke@ualberta.ca", "5735@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Eno Compton", "enocom@google.com")
	addPerson("Enrico Candino", "enrico.candino@gmail.com")
	addPerson("Eoghan Sherry", "ejsherry@gmail.com", "@ejsherry")
	addPerson("Eric Adams", "ercadams@gmail.com")
	addPerson("Eric Brown", "browne@vmware.com")
	addPerson("Eric Chiang", "eric.chiang.m@gmail.com", "@ericchiang")
	addPerson("Eric Clark", "zerohp@gmail.com", "@eclark")
	addPerson("Eric Daniels", "eric@erdaniels.com", "25196@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Eric Daniels", "eric@erdaniels.com", "@edaniels")
	addPerson("Eric Dube", "eric.alex.dube@gmail.com")
	addPerson("Eric Engestrom", "eric@engestrom.ch", "@1ace")
	addPerson("Eric Garrido", "ekg@google.com", "@minusnine")
	addPerson("Eric Hopper", "hopper@omnifarious.org")
	addPerson("Eric Koleda", "ekoleda+devrel@google.com")
	addPerson("Eric Lagergren", "eric@ericlagergren.com", "@ericlagergren")
	addPerson("Eric Lagergren", "ericscottlagergren@gmail.com", "7276@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Eric Lagergren", "ericscottlagergren@gmail.com", "@ericlagergren")
	addPerson("Eric Milliken", "emilliken@gmail.com", "@emilliken")
	addPerson("Eric Pauley", "eric@pauley.me", "@ericpauley")
	addPerson("Eric Ponce", "tricokun@gmail.com", "@trico")
	addPerson("Eric Roshan-Eisner", "eric.d.eisner@gmail.com", "@eisner")
	addPerson("Eric Rykwalder", "e.rykwalder@gmail.com", "@erykwalder")
	addPerson("Eric Schow", "eric.schow@gmail.com")
	addPerson("Erik Aigner", "aigner.erik@gmail.com", "@eaigner")
	addPerson("Erik Dubbelboer", "erik@dubbelboer.com", "8976@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Erik Dubbelboer", "erik@dubbelboer.com", "@erikdubbelboer")
	addPerson("Erik St. Martin", "alakriti@gmail.com", "@erikstmartin")
	addPerson("Erik Staab", "estaab@google.com", "@erikus")
	addPerson("Erik Westrup", "erik.westrup@gmail.com", "@erikw")
	addPerson("Erin Call", "hello@erincall.com")
	addPerson("Erin Masatsugu", "erin.masatsugu@gmail.com", "@emasatsugu")
	addPerson("Ernest Chiang", "ernest_chiang@htc.com")
	addPerson("Erwin Oegema", "blablaechthema@hotmail.com", "@diamondo25")
	addPerson("Esko Luontola", "esko.luontola@gmail.com", "@orfjackal")
	addPerson("Etai Lev Ran", "etail@il.ibm.com")
	addPerson("Ethan Burns", "eaburns@google.com")
	addPerson("Ethan Miller", "eamiller@us.ibm.com", "@millere")
	addPerson("Euan Kemp", "euank@euank.com", "@euank")
	addPerson("Eugene Kalinin", "e.v.kalinin@gmail.com", "12380@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Eugene Kalinin", "e.v.kalinin@gmail.com", "@ekalinin")
	addPerson("Evan Broder", "evan@stripe.com", "@evan-stripe")
	addPerson("Evan Brown", "evanbrown@google.com")
	addPerson("Evan Brown", "evanbrown@google.com", "9260@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Evan Brown", "evanbrown@google.com", "@evandbrown")
	addPerson("Evan Farrar", "evanfarrar@gmail.com")
	addPerson("Evan Hicks", "evan.hicks2@gmail.com", "@FearlessDestiny")
	addPerson("Evan Jones", "ej@evanjones.ca", "@evanj")
	addPerson("Evan Klitzke", "evan@eklitzke.org")
	addPerson("Evan Klitzke", "evan@eklitzke.org", "@eklitzke")
	addPerson("Evan Kroske", "evankroske@google.com", "@evankroske")
	addPerson("Evan Martin", "evan.martin@gmail.com", "@evmar")
	addPerson("Evan Phoenix", "evan@phx.io", "6330@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Evan Phoenix", "evan@phx.io", "@evanphx")
	addPerson("Evan Shaw", "chickencha@gmail.com")
	addPerson("Evan Shaw", "edsrzf@gmail.com", "@edsrzf")
	addPerson("Evgeniy Polyakov", "zbr@ioremap.net")
	addPerson("Evgeniy Polyakov", "zbr@ioremap.net", "17055@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Evgeniy Polyakov", "zbr@ioremap.net", "@bioothod")
	addPerson("Ewan Chou", "coocood@gmail.com", "@coocood")
	addPerson("Eyal Posener", "posener@gmail.com")
	addPerson("Fab>rizio (Misto) Milo", "mistobaan@gmail.com", "@Mistobaan")
	addPerson("Fabian Wickborn", "fabian@wickborn.net", "@fawick")
	addPerson("Fabian", "fabian@youremail.eu")
	addPerson("Fabien Silberstein", "silberfab@gmail.com")
	addPerson("Fabio Alessandro Locati", "me@fale.io")
	addPerson("Faiyaz Ahmed", "ahmedf@vmware.com", "@fdawg4l")
	addPerson("Fan Hongjian", "fan.howard@gmail.com", "@fango")
	addPerson("Fan Jiang", "fan.torchz@gmail.com")
	addPerson("Fangming Fang", "Fangming.Fang@arm.com", "19276@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Fangming.Fang", "fangming.fang@arm.com", "@zorrorffm")
	addPerson("Fatih Arslan", "fatih@arslan.io", "@fatih")
	addPerson("Fatih Arslan", "ftharsln@gmail.com", "@fatih")
	addPerson("Fazal Majid", "majid@apsalar.com")
	addPerson("Fazlul Shahriar", "fshahriar@gmail.com", "@fhs")
	addPerson("Federico Simoncelli", "fsimonce@redhat.com", "@simon3z")
	addPerson("Fedor Indutny", "fedor@indutny.com", "@indutny")
	addPerson("Felix Kollmann", "felix.kollmann@twinpoint.de")
	addPerson("Felix Kollmann", "fk@konsorten.de")
	addPerson("Felix Kollmann", "mail@fkollmann.de", "26861@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Feng Liyuan", "darktemplar.f@gmail.com")
	addPerson("Filip Gruszczyński", "gruszczy@gmail.com", "17532@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Filip Gruszczyński", "gruszczy@gmail.com", "@gruszczy")
	addPerson("Filip Haglund", "drathier@users.noreply.github.com")
	addPerson("Filip Ochnik", "filip.ochnik@gmail.com")
	addPerson("Filip Stanis", "fstanis@google.com")
	addPerson("Filippo Valsorda", "", "6195@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Filippo Valsorda", "filippo@cloudflare.com")
	addPerson("Filippo Valsorda", "filippo@golang.org", "11715@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Filippo Valsorda", "filippo@golang.org", "hi@filippo.io", "@FiloSottile")
	addPerson("Firmansyah Adiputra", "frm.adiputra@gmail.com", "@frm-adiputra")
	addPerson("Florian Forster", "octo@google.com", "@octo")
	addPerson("Florian Uekermann", "florian@uekermann.me", "13410@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Florian Uekermann", "florian@uekermann.me", "@FlorianUekermann")
	addPerson("Florian Uekermann", "florian@uekermann.me", "@MaVo159")
	addPerson("Florian Weimer", "fw@deneb.enyo.de", "@fweimer")
	addPerson("Florian", "sinnlosername@users.noreply.github.com")
	addPerson("Florian", "sinnlosername@users.noreply.github.com", "@sinnlosername")
	addPerson("Florin Patan", "florinpatan@gmail.com", "6473@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Florin Patan", "florinpatan@gmail.com", "@dlsniper")
	addPerson("Ford Hurley", "ford.hurley@gmail.com", "@fordhurley")
	addPerson("FourSeventy", "msiggy@gmail.com")
	addPerson("Francesc Campoy Flores", "campoy@golang.org", "5955@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Francesc Campoy Flores", "campoy@google.com", "7455@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Francesc Campoy", "campoy@golang.org", "campoy@google.com", "@campoy")
	addPerson("Francesc Campoy", "francesc@campoy.cat")
	addPerson("Francisco Claude", "fclaude@recoded.cl", "@fclaude")
	addPerson("Francisco Rojas", "francisco.rojas.gallegos@gmail.com", "@frojasg")
	addPerson("Francisco Souza", "franciscossouza@gmail.com", "@fsouza")
	addPerson("Francisco Souza", "fsouza@users.noreply.github.com")
	addPerson("Frank Rehwinkel", "frankrehwinkel@gmail.com")
	addPerson("Frank Schroeder", "frank.schroeder@gmail.com")
	addPerson("Frank Schroeder", "frank.schroeder@gmail.com", "@magiconair")
	addPerson("Frank Schröder", "frank.schroeder@gmail.com", "11300@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Frank Somers", "fsomers@arista.com", "@somersf")
	addPerson("Franz Bettag", "franz@bett.ag")
	addPerson("Fred Carle", "fred.carle@thorium90.io")
	addPerson("Frederick Kelly Mayle III", "frederickmayle@gmail.com", "@fkm3")
	addPerson("Frederik Ring", "frederik.ring@gmail.com")
	addPerson("Fredrik Enestad", "fredrik.enestad@soundtrackyourbrand.com", "@fredr")
	addPerson("Fredrik Forsmo", "fredrik.forsmo@gmail.com", "@frozzare")
	addPerson("Fredrik Wallgren", "fredrik.wallgren@gmail.com")
	addPerson("Frits van Bommel", "fvbommel@gmail.com", "13460@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Frits van Bommel", "fvbommel@gmail.com", "@fvbommel")
	addPerson("Frédéric Guillot", "frederic.guillot@gmail.com", "@fguillot")
	addPerson("Fumitoshi Ukai", "ukai@google.com", "@ukai")
	addPerson("G. Hussain Chinoy", "ghchinoy@gmail.com", "@ghchinoy")
	addPerson("Gaal Yahas", "gaal@google.com")
	addPerson("Gabe Dalay", "gabedalay@gmail.com")
	addPerson("Gabriel Aszalos", "gabriel.aszalos@gmail.com", "5465@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Gabriel Aszalos", "gabriel.aszalos@gmail.com", "@gbbr")
	addPerson("Gabriel Nicolas Avellaneda", "avellaneda.gabriel@gmail.com", "@GabrielNicolasAvellaneda")
	addPerson("Gabriel Rosenhouse", "rosenhouse@gmail.com")
	addPerson("Gabriel Russell", "gabriel.russell@gmail.com", "@wiccatech")
	addPerson("Gabríel Arthúr Pétursson", "gabriel@system.is", "@polarina")
	addPerson("Gareth Paul Jones", "gpj@foursquare.com", "@garethpaul")
	addPerson("Garret Kelly", "gdk@google.com")
	addPerson("Gary Burd", "gary@beagledreams.com", "@garyburd")
	addPerson("Gary Elliott", "garyelliott@google.com")
	addPerson("Gaurish Sharma", "contact@gaurishsharma.com", "@gaurish")
	addPerson("Gautam Dey", "gautam.dey77@gmail.com")
	addPerson("Gautham Thambidorai", "gautham.dorai@gmail.com", "@gauthamt")
	addPerson("Gauthier Jolly", "gauthier.jolly@gmail.com")
	addPerson("Geert-Johan Riemer", "gjr19912@gmail.com")
	addPerson("Genevieve Luyt", "genevieve.luyt@gmail.com", "@genevieveluyt")
	addPerson("Geoff Berry", "gberry.qdt@qualcommdatacenter.com", "25768@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Geoff Berry", "gberry.qdt@qualcommdatacenter.com", "@gberry-qdt")
	addPerson("Georg Reinke", "guelfey@gmail.com", "@guelfey")
	addPerson("George Gkirtsou", "ggirtsou@gmail.com", "@ggirtsou")
	addPerson("George Shammas", "george@shamm.as", "@georgyo")
	addPerson("George Tankersley", "george.tankersley@gmail.com")
	addPerson("Gepser Hoil", "geharold@gmail.com")
	addPerson("Gerasimos (Makis) Maropoulos", "kataras2006@hotmail.com", "@kataras")
	addPerson("Gerasimos Dimitriadis", "gedimitr@gmail.com", "@gedimitr")
	addPerson("Gergely Brautigam", "skarlso777@gmail.com", "@Skarlso")
	addPerson("Getulio Sánchez", "valentin2507@gmail.com")
	addPerson("Ggicci", "ggicci.t@gmail.com", "@ggicci")
	addPerson("Gianguido Sora`", "g.sora4@gmail.com")
	addPerson("Giannis Kontogianni", "giannis2792@gmail.com")
	addPerson("GiantsLoveDeathMetal", "sebastien@cytora.com", "@foxyblue")
	addPerson("Gil Raphaelli", "g@raphaelli.com")
	addPerson("Giovanni Bajo", "rasky@develer.com", "5340@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Giovanni Bajo", "rasky@develer.com", "@rasky")
	addPerson("Giulio Iotti", "dullgiulio@gmail.com", "@dullgiulio")
	addPerson("Giuseppe Valente", "gvalente@arista.com")
	addPerson("Gleb Smirnoff", "glebius@netflix.com")
	addPerson("Gleb Stepanov", "glebstepanov1992@gmail.com", "14596@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Gleb Stepanov", "glebstepanov1992@gmail.com", "@stgleb")
	addPerson("Glenn Brown", "glennb@google.com")
	addPerson("Glenn Griffin", "glenng@google.com")
	addPerson("Glenn Lewis", "gmlewis@google.com", "@gmlewis")
	addPerson("Glib Smaga", "code@gsmaga.com")
	addPerson("Go Team", "no-reply@golang.org")
	addPerson("Goo", "liuwanle2010@gmail.com", "@l-we")
	addPerson("Gordon Klaus", "gordon.klaus@gmail.com", "5780@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Gordon Klaus", "gordon.klaus@gmail.com", "@gordonklaus")
	addPerson("Graham Miller", "graham.miller@gmail.com", "@laslowh")
	addPerson("Greg Poirier", "greg.istehbest@gmail.com", "@grepory")
	addPerson("Greg Ward", "greg@gerg.ca", "@gward")
	addPerson("Gregory Colella", "gcolella@google.com")
	addPerson("Gregory Haskins", "gregory.haskins@gmail.com")
	addPerson("Gregory Man", "man.gregory@gmail.com")
	addPerson("Gregory Man", "man.gregory@gmail.com", "@gregory-m")
	addPerson("Greyh4t", "greyh4t1337@gmail.com")
	addPerson("Grim", "megaskyhawk@gmail.com")
	addPerson("Grégoire Delattre", "gregoire.delattre@gmail.com", "@gregdel")
	addPerson("Guilherme Garnier", "guilherme.garnier@gmail.com", "@ggarnier")
	addPerson("Guilherme Goncalves", "guilhermeaugustosg@gmail.com", "@guilhermeasg")
	addPerson("Guilherme Rezende", "guilhermebr@gmail.com", "22856@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Guilherme Rezende", "guilhermebr@gmail.com", "@guilhermebr")
	addPerson("Guilherme Santos", "guilherme.santos@foodora.com")
	addPerson("GuilhermeCaruso", "gui.martinscaruso@gmail.com", "@GuilhermeCaruso")
	addPerson("Guillaume J. Charmes", "gcharmes@magicleap.com")
	addPerson("Guillaume J. Charmes", "guillaume@charmes.net", "@creack")
	addPerson("Guillaume Koenig", "guillaume.edward.koenig@gmail.com")
	addPerson("Guillaume Leroi", "leroi.g@gmail.com")
	addPerson("Guillermo López-Anglada", "guillermo.lopez@outlook.com", "@guillermooo")
	addPerson("Guobiao Mei", "meiguobiao@gmail.com", "@guobiao")
	addPerson("Guoliang Wang", "iamwgliang@gmail.com", "@wgliang")
	addPerson("Gurpartap Singh", "hi@gurpartap.com")
	addPerson("Gustav Paul", "gustav.paul@gmail.com", "@gpaul")
	addPerson("Gustav Westling", "gustav@westling.xyz", "@zegl")
	addPerson("Gustav Westling", "zegl@westling.xyz", "@zegl")
	addPerson("Gustavo Niemeyer", "gustavo@niemeyer.net", "n13m3y3r@gmail.com", "@niemeyer")
	addPerson("Gustavo Picón", "tabo@tabo.pe")
	addPerson("Gyu-Ho Lee", "gyuhox@gmail.com", "@gyuho")
	addPerson("H. İbrahim Güngör", "igungor@gmail.com", "@igungor")
	addPerson("HAMANO Tsukasa", "hamano@osstech.co.jp", "@hamano")
	addPerson("HENRY-PC\\Henry", "henry.adisumarto@gmail.com")
	addPerson("Hajime Hoshi", "hajimehoshi@gmail.com", "7938@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Hajime Hoshi", "hajimehoshi@gmail.com", "@hajimehoshi")
	addPerson("Hamit Burak Emre", "hamitburakemre@gmail.com")
	addPerson("Han-Wen Nienhuys", "hanwen@google.com", "5893@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Han-Wen Nienhuys", "hanwen@google.com", "@hanwen")
	addPerson("Han-Wen Nienhuys", "hanwenn@gmail.com", "6115@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Hana Kim", "hyangah@gmail.com", "@hyangah")
	addPerson("Hang Qian", "hangqian90@gmail.com")
	addPerson("Hanjun Kim", "hallazzang@gmail.com", "@hallazzang")
	addPerson("Hannes Landeholm", "hnsl@google.com")
	addPerson("Haosdent Huang", "haosdent@gmail.com")
	addPerson("Harald Nordgren", "haraldnordgren@gmail.com", "26145@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("HaraldNordgren", "haraldnordgren@gmail.com", "@HaraldNordgren")
	addPerson("Hari haran", "hariharan.uno@gmail.com", "@hariharan-uno")
	addPerson("Hariharan Srinath", "srinathh@gmail.com", "@srinathh")
	addPerson("Harry Moreno", "morenoh149@gmail.com", "@morenoh149")
	addPerson("Harshavardhana", "harsha@minio.io")
	addPerson("Harshavardhana", "hrshvardhana@gmail.com", "11900@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Harshavardhana", "hrshvardhana@gmail.com", "@harshavardhana")
	addPerson("Hauke Löffler", "hloeffler@users.noreply.github.com", "@hloeffler")
	addPerson("He Liu", "liulonnie@gmail.com")
	addPerson("Hector Chu", "hectorchu@gmail.com", "@hectorchu")
	addPerson("Hector Jusforgues", "hector.jusforgues@gmail.com")
	addPerson("Hector Martin Cantero", "hector@marcansoft.com", "@marcan")
	addPerson("Hector Rivas Gandara", "keymon@gmail.com")
	addPerson("Henning Schmiedehausen", "henning@schmiedehausen.org", "@hgschmie")
	addPerson("Henrik Hodne", "henrik@hodne.io", "@henrikhodne")
	addPerson("Henrique Vicente", "henriquevicente@gmail.com")
	addPerson("Henry Chang", "mr.changyuheng@gmail.com")
	addPerson("Henry Clifford", "h.a.clifford@gmail.com", "@hcliff")
	addPerson("Henry D. Case", "kris@amongbytes.com")
	addPerson("Henry", "google@mindeco.de")
	addPerson("Herbert Georg Fischer", "herbert.fischer@gmail.com", "@hgfischer")
	addPerson("Herbie Ong", "herbie@google.com", "17100@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Herbie Ong", "herbie@google.com", "@cybrcodr")
	addPerson("Heschi Kreinick", "heschi@google.com", "17090@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Heschi Kreinick", "heschi@google.com", "@heschik")
	addPerson("Hidetatsu Yaginuma", "ygnmhdtt@gmail.com", "@yagi5")
	addPerson("Hilko Bengen", "bengen@hilluzination.de")
	addPerson("Hiroaki Nakamura", "hnakamur@gmail.com")
	addPerson("Hiroaki Nakamura", "hnakamur@gmail.com", "17745@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Hironao OTSUBO", "motemen@gmail.com", "@motemen")
	addPerson("Hiroshi Ioka", "hirochachacha@gmail.com", "11631@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Hiroshi Ioka", "hirochachacha@gmail.com", "@hirochachacha")
	addPerson("Hitoshi Mitake", "mitake.hitoshi@gmail.com", "@mitake")
	addPerson("Homan Chou", "homanchou@gmail.com")
	addPerson("Hong Ruiqi", "hongruiqi@gmail.com", "@hongruiqi")
	addPerson("Hsin Tsao", "tsao@google.com")
	addPerson("Hsin Tsao", "tsao@google.com", "@lazyhackeratwork")
	addPerson("HuKeping", "hukeping@huawei.com", "@HuKeping")
	addPerson("Huadcu Sulivan", "huadcu@gmail.com")
	addPerson("Hugo Rut", "hugorut@gmail.com")
	addPerson("Hugues Bruant", "hugues.bruant@gmail.com", "17586@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Hugues Bruant", "hugues.bruant@gmail.com", "@huguesb")
	addPerson("Hyang-Ah Hana Kim", "hyangah@gmail.com", "5190@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Hyang-Ah Hana Kim", "hyangah@gmail.com", "hakim@google.com", "@hyangah")
	addPerson("Håvard Haugen", "havard.haugen@gmail.com", "5505@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Håvard Haugen", "havard.haugen@gmail.com", "@osocurioso")
	addPerson("INADA Naoki", "songofacandy@gmail.com", "@methane")
	addPerson("Ian Cottrell", "iancottrell@google.com", "9711@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ian Cottrell", "iancottrell@google.com", "@ianthehat")
	addPerson("Ian Davis", "nospam@iandavis.com")
	addPerson("Ian Davis", "nospam@iandavis.com", "@iand")
	addPerson("Ian Ennis", "michaelian.ennis@gmail.com")
	addPerson("Ian Gudger", "igudger@google.com", "12625@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ian Gudger", "igudger@google.com", "ian@loosescre.ws", "@iangudger")
	addPerson("Ian Haken", "ihaken@netflix.com")
	addPerson("Ian Johnson", "person.uwsome@gmail.com")
	addPerson("Ian Kent", "iankent85@gmail.com", "@ian-kent")
	addPerson("Ian Lance Taylor", "iant@golang.org", "@ianlancetaylor", "5206@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ibrahim AshShohail", "ibra.sho@gmail.com", "@ibrasho")
	addPerson("Ibrahim AshShohail", "me@ibrasho.com")
	addPerson("Iccha Sethi", "icchasethi@gmail.com", "@isethi")
	addPerson("Idora Shinatose", "idora.shinatose@gmail.com", "@idora")
	addPerson("Igor Bernstein", "igorbernstein@google.com")
	addPerson("Igor Dolzhikov", "bluesriverz@gmail.com", "@takama")
	addPerson("Igor Vashyst", "ivashyst@gmail.com", "@ivashyst")
	addPerson("Igor Zhilianin", "igor.zhilianin@gmail.com", "@igorzhilianin")
	addPerson("Ilan Pillemer", "ilan.pillemer@gmail.com")
	addPerson("Ilia Filippov", "ilia.filippov@intel.com")
	addPerson("Ilya Tocar", "ilya.tocar@intel.com", "26817@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ilya Tocar", "ilya.tocar@intel.com", "8585@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ilya Tocar", "ilya.tocar@intel.com", "@TocarIP")
	addPerson("Inanc Gumus", "m@inanc.io", "25354@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Inanc Gumus", "m@inanc.io", "@inancgumus")
	addPerson("Ingo Gottwald", "in.gottwald@gmail.com")
	addPerson("Ingo Krabbe", "ikrabbe.ask@gmail.com", "@ikrabbe")
	addPerson("Ingo Oeser", "nightlyone@googlemail.com", "5021@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ingo Oeser", "nightlyone@googlemail.com", "@nightlyone")
	addPerson("Ioannis Georgoulas", "geototti21@hotmail.com", "@geototti21")
	addPerson("Ishani Garg", "ishani.garg@gmail.com")
	addPerson("Iskander Sharipov", "iskander.sharipov@intel.com", "24037@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Iskander Sharipov", "iskander.sharipov@intel.com", "@Quasilyte")
	addPerson("Iskander Sharipov", "quasilyte@gmail.com", "25422@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Issac Trotts", "issac.trotts@gmail.com")
	addPerson("Issac Trotts", "issactrotts@google.com", "@ijt")
	addPerson("Ivan Babrou", "ivan@cloudflare.com", "@bobrik")
	addPerson("Ivan Bertona", "ivan.bertona@gmail.com", "@ibrt")
	addPerson("Ivan Jovanovic", "ivan@loopthrough.ch")
	addPerson("Ivan Krasin", "krasin@golang.org", "@krasin2")
	addPerson("Ivan Kruglov", "ivan.kruglov@yahoo.com")
	addPerson("Ivan Kutuzov", "arbrix@gmail.com")
	addPerson("Ivan Kutuzov", "arbrix@gmail.com", "@arbrix")
	addPerson("Ivan Markin", "sw@nogoegst.net")
	addPerson("Ivan Markin", "twim@riseup.net")
	addPerson("Ivan Moscoso", "moscoso@gmail.com", "@ivan3bx")
	addPerson("Ivan Sharavuev", "shpiwan@gmail.com", "@Shiwin")
	addPerson("Ivan Ukhov", "ivan.ukhov@gmail.com", "@IvanUkhov")
	addPerson("Ivy Evans", "ivy@ivyevans.net", "@ivy")
	addPerson("J. Mroz", "nalik.nal@gmail.com")
	addPerson("JBD (DO NOT USE)", "jbd@golang.org", "10107@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("JBD", "jbd@google.com", "5040@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("JP Sugarbroad", "jpsugar@google.com", "@taralx")
	addPerson("JT Olds", "hello@jtolds.com", "@jtolds")
	addPerson("Jaana Burcu Dogan", "jbd@google.com", "jbd@golang.org", "@rakyll")
	addPerson("Jack Christensen", "jack@jackchristensen.com")
	addPerson("Jack Lindamood", "jlindamo@justin.tv", "@cep21")
	addPerson("Jack Parkinson", "jdparkinson93@gmail.com")
	addPerson("Jack", "jackxbritton@gmail.com", "@jackxbritton")
	addPerson("Jackson Owens", "jackson_owens@alumni.brown.edu")
	addPerson("Jacob H. Haven", "jacob@cloudflare.com")
	addPerson("Jacob H. Haven", "jacob@jhaven.me", "@jacobhaven")
	addPerson("Jacob Haven", "jacob@cloudflare.com", "5346@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jacob Hoffman-Andrews", "github@hoffman-andrews.com", "10927@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jacob Hoffman-Andrews", "github@hoffman-andrews.com", "@jsha")
	addPerson("Jacob Kobernik", "jkobernik@gmail.com")
	addPerson("Jacob Marble", "jacobmarble@google.com")
	addPerson("Jacob Walker", "jacobwalker0814@gmail.com")
	addPerson("Jade Auer", "jda@tapodi.net")
	addPerson("Jae Kwon", "jae@tendermint.com", "@jaekwon")
	addPerson("Jaime Geiger", "jaime@grimm-co.com")
	addPerson("Jake B", "doogie1012@gmail.com")
	addPerson("Jake B", "doogie1012@gmail.com", "@silbinarywolf")
	addPerson("Jake Burkhead", "jake.b@socialcodeinc.com")
	addPerson("Jakob Borg", "jakob@nym.se", "@calmh")
	addPerson("Jakob Weisblat", "jakobw@mit.edu", "@jakob223")
	addPerson("Jakub Katarzynski", "kkatarzynski@gmail.com")
	addPerson("Jakub Ryszard Czarnowicz", "j.czarnowicz@gmail.com", "@Naranim")
	addPerson("Jakub Čajka", "jcajka@redhat.com", "11002@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jakub Čajka", "jcajka@redhat.com", "@jcajka")
	addPerson("James Abley", "james.abley@gmail.com")
	addPerson("James Bardin", "j.bardin@gmail.com", "@jbardin")
	addPerson("James Chacon", "jchacon@google.com")
	addPerson("James Clarke", "jrtc27@jrtc27.com", "15676@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("James Clarke", "jrtc27@jrtc27.com", "@jrtc27")
	addPerson("James Cowgill", "James.Cowgill@imgtec.com")
	addPerson("James Cowgill", "james.cowgill@mips.com", "17679@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("James Cowgill", "james.cowgill@mips.com", "@jcowgill")
	addPerson("James Craig Burley", "james-github@burleyarch.com", "@jcburley")
	addPerson("James F. Carter", "jfc.org.uk@gmail.com")
	addPerson("James Fysh", "james.fysh@gmail.com", "@JamesFysh")
	addPerson("James Gray", "james@james4k.com", "@james4k")
	addPerson("James Greenhill", "fuziontech@gmail.com")
	addPerson("James Hall", "james.hall@shopify.com")
	addPerson("James Hartig", "fastest963@gmail.com", "17920@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("James Hartig", "fastest963@gmail.com", "@fastest963")
	addPerson("James Lawrence", "jljatone@gmail.com", "@james-lawrence")
	addPerson("James Munnelly", "james@munnelly.eu")
	addPerson("James Myers", "jfmyers9@gmail.com", "@jfmyers9")
	addPerson("James Neve", "jamesoneve@gmail.com", "@jamesneve")
	addPerson("James Robinson", "jamesr@google.com", "@jamesr")
	addPerson("James Schofield", "james@shoeboxapp.com", "@jamesshoebox")
	addPerson("James Smith", "jrs1995@icloud.com", "@jimmysmith95")
	addPerson("James Sweet", "james.sweet88@googlemail.com", "@Omegaice")
	addPerson("James Toy", "nil@opensesame.st", "@jamestoy")
	addPerson("James Treanor", "jtreanor3@gmail.com")
	addPerson("James Tucker", "raggi@google.com", "@raggi")
	addPerson("James Whitehead", "jnwhiteh@gmail.com", "@jnwhiteh")
	addPerson("Jamie Barnett", "jamiebarnett1992@gmail.com")
	addPerson("Jamie Beverly", "jamie.r.beverly@gmail.com", "@jbeverly")
	addPerson("Jamie Hall", "jamiehall@google.com")
	addPerson("Jamie Kerr", "jkerr113@googlemail.com")
	addPerson("Jamie Liu", "jamieliu@google.com", "@nixprime")
	addPerson("Jamie Stackhouse", "contin673@gmail.com", "@itsjamie")
	addPerson("Jamie Wilkinson", "jaq@spacepants.org", "@jaqx0r")
	addPerson("Jamil Djadala", "djadala@gmail.com", "@djadala")
	addPerson("Jan Berktold", "jan@berktold.co")
	addPerson("Jan Berktold", "jan@berktold.co", "@JanBerktold")
	addPerson("Jan H. Hosang", "jan.hosang@gmail.com", "@hosang")
	addPerson("Jan Kratochvil", "jan.kratochvil@redhat.com", "@jankratochvil")
	addPerson("Jan Lehnardt", "jan@apache.org", "@janl")
	addPerson("Jan Mercl", "0xjnml@gmail.com", "5295@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jan Mercl", "0xjnml@gmail.com", "@cznic")
	addPerson("Jan Mercl", "befelemepeseveze@gmail.com", "@bflm")
	addPerson("Jan Pilzer", "jan.pilzer@gmx.de")
	addPerson("Jan Ziak", "0xe2.0x9a.0x9b@gmail.com", "@atomsymbol")
	addPerson("Janne Snabb", "snabb@epipe.com", "@snabb")
	addPerson("Jason A. Donenfeld", "jason@zx2c4.com", "@zx2c4")
	addPerson("Jason Barnett", "jason.w.barnett@gmail.com", "@jasonwbarnett")
	addPerson("Jason Buberel", "jbuberel@google.com", "8445@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jason Buberel", "jbuberel@google.com", "jason@buberel.org", "@jbuberel")
	addPerson("Jason Chu", "jasonchujc@gmail.com", "@1lann")
	addPerson("Jason Cwik", "jason@cwik.org")
	addPerson("Jason Del Ponte", "delpontej@gmail.com", "@jasdel")
	addPerson("Jason Donenfeld", "jason.donenfeld@gmail.com", "20556@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jason E. Aten", "j.e.aten@gmail.com")
	addPerson("Jason Hall", "imjasonh@gmail.com")
	addPerson("Jason Hall", "jasonhall@google.com")
	addPerson("Jason Hewes", "jasonhewes5@gmail.com")
	addPerson("Jason Keene", "jasonkeene@gmail.com")
	addPerson("Jason Keene", "jasonkeene@gmail.com", "@jasonkeene")
	addPerson("Jason LeBrun", "jblebrun@gmail.com", "@jblebrun")
	addPerson("Jason McVetta", "jason.mcvetta@gmail.com", "@jmcvetta")
	addPerson("Jason Murray", "jason@chaosaffe.io")
	addPerson("Jason Smale", "jsmale@zendesk.com")
	addPerson("Jason Travis", "infomaniac7@gmail.com", "@corburn")
	addPerson("Jason Wangsadinata", "jwangsadinata@gmail.com", "@jwangsadinata")
	addPerson("Jason Wilder", "mail@jasonwilder.com")
	addPerson("Javier Kohen", "jkohen@google.com", "@jkohen")
	addPerson("Javier Segura", "javism@gmail.com", "@jsegura")
	addPerson("Jay Conrod", "jayconrod@google.com", "17092@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jay Conrod", "jayconrod@google.com", "@jayconrod")
	addPerson("Jay Satiro", "raysatiro@yahoo.com")
	addPerson("Jay Stramel", "js@ionactual.com")
	addPerson("Jay Weisskopf", "jay@jayschwa.net", "@jayschwa")
	addPerson("Jayabaskar Rajagopal", "jayabaskar.rajagopal@gmail.com")
	addPerson("Jean de Klerk", "deklerk@google.com", "26615@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jean de Klerk", "deklerk@google.com", "@jadekler")
	addPerson("Jean de Klerk", "jadekler@gmail.com")
	addPerson("Jean-André Santoni", "jean.andre.santoni@gmail.com")
	addPerson("Jean-Francois Cantin", "jfcantin@gmail.com", "@jfcantin")
	addPerson("Jean-Marc Eurin", "jmeurin@google.com", "@jmeurin")
	addPerson("Jean-Nicolas Moal", "jn.moal@gmail.com", "@jnmoal")
	addPerson("Jed Denlea", "jed@fastly.com", "5550@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jed Denlea", "jed@fastly.com", "@jeddenlea")
	addPerson("Jeet Parekh", "jeetparekh96@gmail.com", "24716@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jeet Parekh", "jeetparekh96@gmail.com", "@jeet-parekh")
	addPerson("Jeevanandam M", "jeeva@myjeeva.com")
	addPerson("Jeff (Zhefu) Jiang", "jeffjiang@google.com")
	addPerson("Jeff Buchbinder", "jeff@ourexchange.net")
	addPerson("Jeff Craig", "jeffcraig@google.com", "@foxxtrot")
	addPerson("Jeff Dupont", "jeff.dupont@gmail.com", "@jeffdupont")
	addPerson("Jeff Grafton", "jgrafton@google.com")
	addPerson("Jeff Hodges", "jeff@somethingsimilar.com", "@jmhodges")
	addPerson("Jeff Johnson", "jrjohnson@google.com", "16958@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jeff Johnson", "jrjohnson@google.com", "@johnsonj")
	addPerson("Jeff R. Allen", "jra@nella.org", "5646@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jeff R. Allen", "jra@nella.org", "@jeffallen")
	addPerson("Jeff Sickel", "jas@corpus-callosum.com", "@vat9")
	addPerson("Jeff Wendling", "jeff@spacemonkey.com", "@zeebo")
	addPerson("Jeff Williams", "jefesaurus@google.com")
	addPerson("Jeff", "jeffreyh192@gmail.com", "@jeffizhungry")
	addPerson("Jeffrey Yong", "jeffreyyong10@gmail.com")
	addPerson("Jelte Fennema", "github-tech@jeltef.nl", "@JelteF")
	addPerson("Jens Frederich", "jfrederich@gmail.com", "@frederich")
	addPerson("Jeremiah Harmsen", "jeremiah@google.com", "@jharmsen")
	addPerson("Jeremy Baumont", "jeremy.baumont@gmail.com")
	addPerson("Jeremy Jackins", "jeremyjackins@gmail.com", "5300@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jeremy Jackins", "jeremyjackins@gmail.com", "@jnjackins")
	addPerson("Jeremy Jay", "jeremy@pbnjay.com")
	addPerson("Jeremy Jay", "jeremy@pbnjay.com", "@pbnjay")
	addPerson("Jeremy Loy", "jeremy.b.loy@icloud.com")
	addPerson("Jeremy Schlatter", "jeremy.schlatter@gmail.com", "@jeremyschlatter")
	addPerson("Jeremy", "jcanady@gmail.com")
	addPerson("Jeroen Bobbeldijk", "jerbob92@gmail.com", "@jerbob92")
	addPerson("Jerrin Shaji George", "jerrinsg@gmail.com", "@jerrinsg")
	addPerson("Jess Frazelle", "acidburn@google.com", "@jessfraz")
	addPerson("Jess Frazelle", "me@jessfraz.com", "@jessfraz")
	addPerson("Jesse Szwedko", "jesse.szwedko@gmail.com", "@jszwedko")
	addPerson("Jessie Frazelle", "me@jessfraz.com", "6071@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jesús Espino", "jespinog@gmail.com")
	addPerson("Jian Zhen", "zhenjl@gmail.com")
	addPerson("Jianing Yu", "jnyu@google.com")
	addPerson("Jianqiao Li", "jianqiaoli@google.com")
	addPerson("Jianqiao Li", "jianqiaoli@jianqiaoli.svl.corp.google.com")
	addPerson("Jihyun Yu", "yjh0502@gmail.com", "@yjh0502")
	addPerson("Jille Timmermans", "quis@google.com")
	addPerson("Jim Cote", "jfcote87@gmail.com", "5320@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jim Cote", "jfcote87@gmail.com", "@jfcote87")
	addPerson("Jim Kingdon", "jim@bolt.me", "@jkingdon")
	addPerson("Jim McGrath", "jimmc2@gmail.com", "@mcgoo")
	addPerson("Jim Minter", "jminter@redhat.com")
	addPerson("Jim Myers", "jfmyers9@gmail.com", "16855@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jimmy Zelinskie", "jimmyzelinskie@gmail.com", "@jzelinskie")
	addPerson("Jin-wook Jeong", "jeweljar@hanmail.net", "@jeweljar")
	addPerson("Jingcheng Zhang", "diogin@gmail.com", "@diogin")
	addPerson("Jingguo Yao", "yaojingguo@gmail.com", "@yaojingguo")
	addPerson("Jiong Du", "londevil@gmail.com", "@lodevil")
	addPerson("Jirka Daněk", "dnk@mail.muni.cz", "@jirkadanek")
	addPerson("Jiulong Wang", "jiulongw@gmail.com")
	addPerson("Jizhong Jiang", "jiangjizhong@gmail.com")
	addPerson("Joakim Sernbrant", "serbaut@gmail.com", "@serbaut")
	addPerson("Joe Cortopassi", "joe@joecortopassi.com", "@JoeCortopassi")
	addPerson("Joe Farrell", "joe2farrell@gmail.com", "@joe2far")
	addPerson("Joe Harrison", "joehazzers@gmail.com", "@sigwinch28")
	addPerson("Joe Henke", "joed.henke@gmail.com", "@jdhenke")
	addPerson("Joe Kyo", "xunianzu@gmail.com", "21935@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Joe Kyo", "xunianzu@gmail.com", "@joekyo")
	addPerson("Joe Poirier", "jdpoirier@gmail.com", "@jpoirier")
	addPerson("Joe Richey", "joerichey@google.com", "17411@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Joe Shaw", "joe@joeshaw.org")
	addPerson("Joe Shaw", "joe@joeshaw.org", "5185@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Joe Shaw", "joe@joeshaw.org", "@joeshaw")
	addPerson("Joe Sylve", "joe.sylve@gmail.com", "11851@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Joe Sylve", "joe.sylve@gmail.com", "@jtsylve")
	addPerson("Joe Tsai", "joetsai@digital-static.net", "8495@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Joe Tsai", "joetsai@google.com", "12850@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Joe Tsai", "joetsai@google.com", "joetsai@digital-static.net", "thebrokentoaster@gmail.com", "@dsnet")
	addPerson("Joe Tsai", "thebrokentoaster@gmail.com", "9735@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Joel Sing", "joel@sing.id.au", "13640@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Joel Sing", "joel@sing.id.au", "jsing@google.com", "@4a6f656c")
	addPerson("Joel Sing", "jsing@google.com", "5770@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Johan Brandhorst", "johan.brandhorst@gmail.com", "16585@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Johan Brandhorst", "johan.brandhorst@gmail.com", "@johanbrandhorst")
	addPerson("Johan Brandhorst", "johan@cognitivelogic.com")
	addPerson("Johan Brandhorst", "johan@infosum.com")
	addPerson("Johan Euphrosine", "proppy@google.com", "5480@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Johan Euphrosine", "proppy@google.com", "@proppy")
	addPerson("Johan Sageryd", "j@1616.se", "@jsageryd")
	addPerson("Johan Schuijt-Li", "johan@300.nl")
	addPerson("Johanna Mantilla Duque", "johanna1431@gmail.com")
	addPerson("Johannes Ebke", "johannes@ebke.org")
	addPerson("John Asmuth", "jasmuth@gmail.com", "jasmuth@google.com", "@skelterjohn")
	addPerson("John Beisley", "huin@google.com", "@huin-google")
	addPerson("John Dethridge", "jcd@golang.org", "5515@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("John Dethridge", "jcd@golang.org", "@jcd2")
	addPerson("John Eikenberry", "jae@zhar.net")
	addPerson("John Gibb", "johngibb@gmail.com", "@johngibb")
	addPerson("John Howard Palevich", "jack.palevich@gmail.com", "@jackpal")
	addPerson("John Jeffery", "jjeffery@sp.com.au", "@jjeffery")
	addPerson("John Leidegren", "john.leidegren@gmail.com")
	addPerson("John Paul Adrian Glaubitz", "glaubitz@physik.fu-berlin.de")
	addPerson("John Potocny", "johnp@vividcortex.com", "@potocnyj")
	addPerson("John R. Lenton", "jlenton@gmail.com", "@chipaca")
	addPerson("John Schnake", "schnake.john@gmail.com", "@johnSchnake")
	addPerson("John ShaggyTwoDope Jenkins", "twodopeshaggy@gmail.com", "@shaggytwodope")
	addPerson("John Shahid", "jvshahid@gmail.com", "@jvshahid")
	addPerson("John Starks", "jostarks@microsoft.com")
	addPerson("John Tuley", "john@tuley.org", "@jmtuley")
	addPerson("JohnCGriffin", "griffinish@gmail.com")
	addPerson("Johnny Boursiquot", "jboursiquot@gmail.com")
	addPerson("Johnny Luo", "johnnyluo1980@gmail.com", "19155@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Johnny Luo", "johnnyluo1980@gmail.com", "@johnnyluo")
	addPerson("Jon Chen", "jchen@justin.tv", "@bsdlp")
	addPerson("Jon Jenkins", "invultussolis@gmail.com")
	addPerson("Jon Jenkins", "jon@mj12.su")
	addPerson("Jonathan Amsterdam", "jba@google.com", "14570@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jonathan Amsterdam", "jba@google.com", "@jba")
	addPerson("Jonathan Anderson", "jonathan.anderson@mun.ca")
	addPerson("Jonathan Boulle", "jonathanboulle@gmail.com", "@jonboulle")
	addPerson("Jonathan Chen", "dijonkitchen@users.noreply.github.com", "@dijonkitchen")
	addPerson("Jonathan Doklovic", "doklovic@atlassian.com")
	addPerson("Jonathan ES Lin", "ernsheong@gmail.com")
	addPerson("Jonathan Feinberg", "feinberg@google.com", "@google-feinberg")
	addPerson("Jonathan Hseu", "jhseu@google.com", "@jhseu")
	addPerson("Jonathan Lloyd", "j.lloyd.email@gmail.com")
	addPerson("Jonathan Mark", "jhmark@xenops.com", "@jhmark")
	addPerson("Jonathan Mayer", "jonmayer@google.com")
	addPerson("Jonathan Nieder", "jrn@google.com", "@jrn")
	addPerson("Jonathan Pentecost", "pentecostjonathan@gmail.com")
	addPerson("Jonathan Pittman", "jmpittman@google.com", "@jonathanpittman")
	addPerson("Jonathan Rudenberg", "jonathan@titanous.com", "5431@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jonathan Rudenberg", "jonathan@titanous.com", "@titanous")
	addPerson("Jonathan Turner", "jt@jtnet.co.uk")
	addPerson("Jonathan Wills", "runningwild@gmail.com", "@runningwild")
	addPerson("Jongmin Kim", "atomaths@gmail.com", "@atomaths")
	addPerson("Jongmin Kim", "jmkim@pukyong.ac.kr", "@jmkim")
	addPerson("Jono Gould", "jono.gould@gmail.com")
	addPerson("Joonas Kuorilehto", "joneskoo@derbian.fi", "14770@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Joonas Kuorilehto", "joneskoo@derbian.fi", "@joneskoo")
	addPerson("Joop Kiefte", "joop@kiefte.net", "@LaPingvino")
	addPerson("Jordan Lewis", "jordanthelewis@gmail.com", "@jordanlewis")
	addPerson("Jordan Liggitt", "jliggitt@redhat.com")
	addPerson("Jordan Rhee", "jordanrh@microsoft.com", "28473@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Jordan Rhee", "jordanrh@microsoft.com", "@jordanrh1")
	addPerson("Jos Visser", "josv@google.com", "@gosbisser")
	addPerson("Jose Luis Vázquez González", "josvazg@gmail.com", "@josvazg")
	addPerson("Joseph Herlant", "herlantj@gmail.com")
	addPerson("Joseph Holsten", "joseph@josephholsten.com", "@josephholsten")
	addPerson("Joseph Poirier", "jdpoirier@gmail.com")
	addPerson("Joseph Richey", "joerichey@google.com", "@josephlr")
	addPerson("Joseph Spurrier", "code@josephspurrier.com")
	addPerson("Josh Bleecher Snyder", "josharian@gmail.com", "5143@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Josh Bleecher Snyder", "josharian@gmail.com", "@josharian")
	addPerson("Josh Chorlton", "jchorlton@gmail.com", "@jchorl")
	addPerson("Josh Deprez", "josh.deprez@gmail.com", "@DrJosh9000")
	addPerson("Josh Goebel", "dreamer3@gmail.com", "@yyyc514")
	addPerson("Josh Hoak", "jhoak@google.com", "@Kashomon")
	addPerson("Josh Lubawy", "jlubawy@gmail.com")
	addPerson("Josh Roppo", "joshroppo@gmail.com", "@Ropes")
	addPerson("Josh Varga", "josh.varga@gmail.com")
	addPerson("Joshua Blakeley", "jtblakeley@gmail.com", "27898@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Joshua Boelter", "joshua.boelter@intel.com")
	addPerson("Joshua Boelter", "joshua.boelter@intel.com", "@duckized")
	addPerson("Joshua Humphries", "jhumphries131@gmail.com")
	addPerson("Joshua Rubin", "joshua@rubixconsulting.com", "@joshuarubin")
	addPerson("Joshua T Corbin", "joshua@uber.com")
	addPerson("Josselin Costanzi", "josselin@costanzi.fr")
	addPerson("Josselin Costanzi", "josselin@costanzi.fr", "16720@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Josselin Costanzi", "josselin@costanzi.fr", "@josselin-c")
	addPerson("Jostein Stuhaug", "js@solidsystem.no")
	addPerson("José Carlos Nieto", "jose.carlos@menteslibres.net")
	addPerson("João Lucas Melo Brasio", "jaumlucas@gmail.com")
	addPerson("Joël Stemmer", "jstemmer@google.com", "@jstemmer")
	addPerson("Joël Stemmer", "stemmertech@gmail.com", "@jstemmer")
	addPerson("Juan Carlos", "juanjcsr@gmail.com", "@juanjcsr")
	addPerson("JuciÊ Dias Andrade", "ojucie@gmail.com")
	addPerson("Jude Pereira", "judebpereira@gmail.com")
	addPerson("Jukka-Pekka Kekkonen", "karatepekka@gmail.com", "@madari")
	addPerson("Julia Hansbrough", "flowerhack@google.com", "@flowerhack")
	addPerson("Julian Kornberger", "jk+github@digineo.de", "@corny")
	addPerson("Julian Pastarmov", "pastarmovj@google.com")
	addPerson("Julian Phillips", "julian@quantumfyre.co.uk", "@qur")
	addPerson("Julie Qiu", "julieyeqiu@gmail.com", "@julieqiu", "julieqiu@google.com", "julie@golang.org")
	addPerson("Julien Kauffmann", "julien.kauffmann@freelan.org")
	addPerson("Julien Salleyron", "julien.salleyron@gmail.com", "@Juliens")
	addPerson("Julien Schmidt", "google@julienschmidt.com", "@julienschmidt")
	addPerson("Julio Montes", "julio.montes@intel.com", "@devimc")
	addPerson("Junda Liu", "junda@celer.network")
	addPerson("Jungho Ahn", "jhahn@google.com", "@jhahn21")
	addPerson("Junya Hayashi", "ledmonster@gmail.com")
	addPerson("Juraj Sukop", "sukop@users.noreply.github.com", "@sukop")
	addPerson("Jure Ham", "jure.ham@zemanta.com", "@hamaxx")
	addPerson("Jurgen De Commer", "jurgen.decommer@gmail.com")
	addPerson("Justin Gracenin", "jgracenin@gmail.com")
	addPerson("Justin Li", "git@justinli.net", "@pushrax")
	addPerson("Justin Nuß", "nuss.justin@gmail.com", "5475@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Justin Nuß", "nuss.justin@gmail.com", "@nussjustin")
	addPerson("Justyn Temme", "justyntemme@gmail.com", "@justyntemme")
	addPerson("KB Sriram", "kbsriram@google.com", "@kbsriram")
	addPerson("Kaarthik Rao Bekal Radhakrishna", "karthik.0703@gmail.com")
	addPerson("Kale Blankenship", "kale@lemnisys.com", "15430@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Kale Blankenship", "kale@lemnisys.com", "@vcabbage")
	addPerson("Kaleb Elwert", "kelwert@atlassian.com", "@belak")
	addPerson("Kamal Aboul-Hosn", "aboulhosn@google.com")
	addPerson("Kamal Aboul-Hosn", "kamal.aboulhosn@gmail.com")
	addPerson("Kamil Kisiel", "kamil@kamilkisiel.net", "@kisielk")
	addPerson("Kamil Rytarowski", "krytarowski@users.noreply.github.com", "@krytarowski")
	addPerson("Kang Hu", "hukangustc@gmail.com", "@mkhu")
	addPerson("Kanitkorn S", "k.sujautra@gmail.com")
	addPerson("Karan Misra", "kidoman@gmail.com")
	addPerson("Karel Pazdera", "pazderak@gmail.com", "@pazderak")
	addPerson("Karsten Köhler", "karsten.koehler95@gmail.com", "@SchiffFlieger")
	addPerson("Karthik Karanth", "karanth.karthik@gmail.com")
	addPerson("Kashav Madan", "kshvmdn@gmail.com", "@kshvmdn")
	addPerson("Kasper Nilsson", "kaspern@google.com")
	addPerson("Kate Manson", "kate.manson@izettle.com", "@kamanson")
	addPerson("Katie Hockman", "katie@golang.org", "28759@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Katie Hockman", "katie@golang.org", "@katiehockman")
	addPerson("Kato Kazuyoshi", "kato.kazuyoshi@gmail.com", "@kzys")
	addPerson("Katrina Owen", "katrina.owen@gmail.com", "10395@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Katrina Owen", "katrina.owen@gmail.com", "@kytrinyx")
	addPerson("Katsuya Miyachi", "kattu0426@gmail.com")
	addPerson("Kaviraj", "kavirajkanagaraj@gmail.com", "@kavirajk")
	addPerson("Kazuhiro Kubota", "k2.wanko@gmail.com")
	addPerson("Kazuhiro Sera", "seratch@gmail.com", "@seratch")
	addPerson("Keegan Carruthers-Smith", "keegan.csmith@gmail.com", "@keegancsmith")
	addPerson("Kei Son", "hey.calmdown@gmail.com", "@heycalmdown")
	addPerson("Keiji Yoshida", "keijiyoshida.mail@gmail.com", "@keijiyoshida")
	addPerson("Keith Ball", "inflatablewoman@gmail.com")
	addPerson("Keith Randall", "khr@golang.org", "5200@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Keith Randall", "khr@golang.org", "khr@google.com", "keithr@alum.mit.edu", "@randall77")
	addPerson("Keith Rarick", "kr@xph.us", "@kr")
	addPerson("Kelsey Hightower", "kelsey.hightower@gmail.com", "5491@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Kelsey Hightower", "kelsey.hightower@gmail.com", "@kelseyhightower")
	addPerson("Ken Friedenbach", "kenliz@cruzio.com", "@Ken1JF")
	addPerson("Ken Rockot", "ken@oz.gs", "@krockot")
	addPerson("Ken Sedgwick", "ken@bonsai.com", "@ksedgwic")
	addPerson("Ken Thompson", "ken@golang.org", "@ken")
	addPerson("Kenji Kaneda", "kenji.kaneda@gmail.com", "@kkaneda")
	addPerson("Kenji Yano", "kenji.yano@gmail.com", "@yanolab")
	addPerson("Kenneth Shaw", "kenshaw@gmail.com", "@kenshaw")
	addPerson("Kenny Grant", "kennygrant@gmail.com", "10235@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Kenny Grant", "kennygrant@gmail.com", "@kennygrant")
	addPerson("Ketan Parmar", "ketanbparmar@gmail.com")
	addPerson("Ketan Parmar", "ketanbparmar@gmail.com", "@kpbird")
	addPerson("Kevin Ballard", "kevin@sb.org", "@kballard")
	addPerson("Kevin Burke", "kev@inburke.com", "13437@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Kevin Burke", "kev@inburke.com", "@kevinburke")
	addPerson("Kevin Kirsche", "kev.kirsche@gmail.com", "@kkirsche")
	addPerson("Kevin Klues", "klueska@gmail.com", "@klueska")
	addPerson("Kevin Lozandier", "lozandier@gmail.com")
	addPerson("Kevin Malachowski", "chowski@google.com")
	addPerson("Kevin Ruffin", "kruffin@gmail.com")
	addPerson("Kevin Vu", "kevin.m.vu@gmail.com", "@kvu787")
	addPerson("Kevin Wang", "kevin@kevinwang.com")
	addPerson("Kevin Zita", "bleedgreenandgold@gmail.com", "@kzisme")
	addPerson("Khramov Anton", "anton@endocode.com")
	addPerson("Kieran Colford", "kieran@kcolford.com")
	addPerson("Kim Shrier", "kshrier@racktopsystems.com", "@kim-racktop")
	addPerson("Kim YongBin", "kybinz@gmail.com", "5154@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Kir Kolyshkin", "kolyshkin@gmail.com", "@kolyshkin")
	addPerson("Kirill Korotaev", "kirillx@gmail.com")
	addPerson("Kirill Smelkov", "kirr@nexedi.com", "16286@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Kirill Smelkov", "kirr@nexedi.com", "@navytux")
	addPerson("Kirklin McDonald", "kirklin.mcdonald@gmail.com", "@KirkMcDonald")
	addPerson("Klaus Post", "klauspost@gmail.com", "6545@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Klaus Post", "klauspost@gmail.com", "@klauspost")
	addPerson("Koala Yeung", "koalay@gmail.com")
	addPerson("Kodie", "kodiegoodwin@gmail.com")
	addPerson("Koen Rouwhorst", "info@koenrouwhorst.nl")
	addPerson("Koichi Shiraishi", "zchee.io@gmail.com", "10420@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Koichi Shiraishi", "zchee.io@gmail.com", "@zchee")
	addPerson("Koki Ide", "niconegoto@yahoo.co.jp", "@niconegoto")
	addPerson("Konstantin Shaposhnikov", "k.shaposhnikov@gmail.com", "8065@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Konstantin Shaposhnikov", "k.shaposhnikov@gmail.com", "@kostya-sh")
	addPerson("Kris Kwiatkowski", "kris@cloudflare.com", "27471@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Kris Nova", "kris@nivenly.com", "@kris-nova")
	addPerson("Kris", "krousey@google.com")
	addPerson("Krish Munot", "krishmunot@gmail.com")
	addPerson("Kristopher Watts", "traetox@gmail.com", "@traetox")
	addPerson("Kropekk", "kamilkropiewnicki@gmail.com")
	addPerson("Kun", "likunarmstrong@gmail.com", "@cnbuff410")
	addPerson("Kunpei Sakai", "namusyaka@gmail.com", "23250@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Kunpei Sakai", "namusyaka@gmail.com", "@namusyaka")
	addPerson("Kyle Consalus", "consalus@gmail.com", "@kcons")
	addPerson("Kyle Isom", "kyle@gokyle.net", "@kisom")
	addPerson("Kyle Jones", "kyle@kyledj.com")
	addPerson("Kyle Lemons", "kyle@kylelemons.net", "@kylelemons")
	addPerson("Kyle Shannon", "kyle@pobox.com", "@ksshannon")
	addPerson("Kyle Spiers", "eiais@google.com")
	addPerson("Kyle Wood", "kyle@kylewood.cc", "@DemonWav")
	addPerson("Kyohei Kadota", "lufia@lufia.org")
	addPerson("Kyrylo Silin", "silin@kyrylo.org", "@kyrylo")
	addPerson("L Campbell", "unpantsu@gmail.com", "@lye")
	addPerson("L. Alberto Giménez", "lagimenez@gmail.com")
	addPerson("LE Manh Cuong", "cuong.manhle.vn@gmail.com", "@Gnouc")
	addPerson("Lai Jiangshan", "eag0628@gmail.com", "@laijs")
	addPerson("Lakshay Garg", "lakshay.garg.1996@gmail.com", "21860@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Lakshay Garg", "lakshay.garg.1996@gmail.com", "@lakshayg")
	addPerson("Landon Jones", "lbj@landonjones.com")
	addPerson("Lann Martin", "lannm@google.com")
	addPerson("Lanre Adelowo", "yo@lanre.wtf")
	addPerson("Lantao Liu", "lantaol@google.com")
	addPerson("Larry Clapp", "larry@theclapp.org", "@theclapp")
	addPerson("Larry Hosken", "lahosken@gmail.com", "@lahosken")
	addPerson("Lars Jeppesen", "jeppesen.lars@gmail.com")
	addPerson("Lars Lehtonen", "lars.lehtonen@gmail.com")
	addPerson("Lars Wiegman", "lars@namsral.com", "@namsral")
	addPerson("Larz Conwell", "larzconwell@gmail.com", "@larzconwell")
	addPerson("Laurent Voisin", "lpvoisin@gmail.com")
	addPerson("Laurie Clark-Michalek", "laurie@qubit.com", "@lclarkmichalek")
	addPerson("Laurynas", "LaurynasUsas@gmail.com")
	addPerson("Lee Hinman", "hinman@gmail.com", "@hinman")
	addPerson("Lee Packham", "lpackham@gmail.com", "@leepa")
	addPerson("Lehner Florian", "dev@der-flo.net")
	addPerson("Lehner Florian", "dev@der-flo.net", "@florianl")
	addPerson("Leigh McCulloch", "leigh@mcchouse.com")
	addPerson("Leigh McCulloch", "leighmcc@gmail.com", "21426@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Leigh McCulloch", "leighmcc@gmail.com", "@leighmcculloch")
	addPerson("Leo Antunes", "leo@costela.net")
	addPerson("Leo Rudberg", "ljr@google.com")
	addPerson("Leon Klingele", "git@leonklingele.de", "16005@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Leon Klingele", "git@leonklingele.de", "@leonklingele")
	addPerson("Leonel Quinteros", "leonel.quinteros@gmail.com")
	addPerson("Letian Yi", "letian0805@gmail.com")
	addPerson("Lev Shamardin", "shamardin@gmail.com", "@abbot")
	addPerson("Lewin Bormann", "lbo@spheniscida.de")
	addPerson("Lewin Bormann", "lewin.bormann@gmail.com", "@dermesser")
	addPerson("Liam Missin", "liam.missin@gmail.com")
	addPerson("Lifu Huang", "lifu.hlf@gmail.com")
	addPerson("Lin Haowen", "linhaowen99@gmail.com")
	addPerson("Lineu Felipe", "lineufelipe@gmail.com")
	addPerson("Lion Yang", "lion@aosc.xyz", "@LionNatsu")
	addPerson("Liz Rice", "liz@lizrice.com")
	addPerson("Lloyd Dewolf", "foolswisdom@gmail.com", "@lloydde")
	addPerson("Logan", "businesspapers@gmail.com")
	addPerson("Lorenz Bauer", "lmb@cloudflare.com", "14200@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Lorenz Bauer", "lmb@cloudflare.com", "@lmb")
	addPerson("Lorenzo Masini", "rugginoso@develer.com", "17340@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Lorenzo Masini", "rugginoso@develer.com", "@rugginoso")
	addPerson("Lorenzo Stoakes", "lstoakes@gmail.com", "@lorenzo-stoakes")
	addPerson("LotusFenn", "fenn.lotus@gmail.com")
	addPerson("LotusFenn", "fenn.lotus@gmail.com", "@LotusFenn")
	addPerson("Luan Santos", "cfcluan@gmail.com", "@luan")
	addPerson("Lubomir I. Ivanov (VMware)", "neolit123@gmail.com", "@neolit123")
	addPerson("Lubomir I. Ivanov", "neolit123@gmail.com", "26534@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Luca Bruno", "luca.bruno@coreos.com", "@lucab")
	addPerson("Luca Greco", "luca.greco@alcacoop.it", "@rpl")
	addPerson("Lucas Bremgartner", "lucas.bremgartner@gmail.com", "16630@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Lucas Bremgartner", "lucas.bremgartner@gmail.com", "@breml")
	addPerson("Lucas Clemente", "lclemente@google.com", "@lucas-clemente")
	addPerson("Lucas Garron", "lgarron@chromium.org", "*goog")
	addPerson("Lucas Halim", "luketheflyingcoder@gmail.com")
	addPerson("Lucien Stuker", "lucien.stuker@gmail.com", "@LStuker")
	addPerson("Lucio De Re", "lucio.dere@gmail.com", "@lootch")
	addPerson("Ludi Rehak", "ludi317@gmail.com", "@ludi317")
	addPerson("Luigi Riefolo", "luigi.riefolo@gmail.com", "@luigi-riefolo")
	addPerson("Luit van Drongelen", "luit@luit.it", "@Luit")
	addPerson("Luit van Drongelen", "luitvd@gmail.com", "@Luit")
	addPerson("Luka Zakrajšek", "tr00.g33k@gmail.com")
	addPerson("Luka", "luka@blow.sh")
	addPerson("Lukasz Dobrzanski", "lukasz.m.dobrzanski@gmail.com")
	addPerson("Lukasz Milewski", "lmmilewski@gmail.com", "lmilewski@google.com", "@LMMilewski")
	addPerson("Luke Curley", "qpingu@gmail.com", "@kixelated")
	addPerson("Luke Granger-Brown", "git@lukegb.com")
	addPerson("Luna Duclos", "luna.duclos@palmstonegames.com")
	addPerson("Lyle Franklin", "lylejfranklin@gmail.com", "@ljfranklin")
	addPerson("Lynn Boger", "laboger@linux.vnet.ibm.com", "6320@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Lynn Boger", "laboger@linux.vnet.ibm.com", "@laboger")
	addPerson("Lyoness", "carmen.andoh@gmail.com")
	addPerson("Maarten Bezemer", "maarten.bezemer@gmail.com")
	addPerson("Maciej Dębski", "maciejd@google.com", "26521@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Maciej Galkowski", "maciejgalkowski@gmail.com")
	addPerson("Maggie Nolan", "nolanmar@google.com")
	addPerson("Magnus Hiie", "magnus.hiie@gmail.com")
	addPerson("Mahmoud Bassiouny", "mbassiouny@google.com")
	addPerson("Mak Kolybabi", "mak@kolybabi.com", "@mogigoma")
	addPerson("Maksym Trykur", "maksym.trykur@gmail.com", "@mak73kur")
	addPerson("Mal Curtis", "mal@mal.co.nz", "@snikch")
	addPerson("Malcolm Rebughini", "malcolm.rebughini@gmail.com")
	addPerson("Malhar Vora", "mlvora.2010@gmail.com")
	addPerson("Manfred Touron", "m@42.am", "@moul")
	addPerson("Maniacal", "mike.glenney@gmail.com")
	addPerson("Manigandan Dharmalingam", "manigandan.jeff@gmail.com")
	addPerson("Manish Goregaokar", "manishsmail@gmail.com", "@Manishearth")
	addPerson("Mansour Rahimi", "rahimi.mnr@gmail.com", "25524@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Mansour Rahimi", "rahimi.mnr@gmail.com", "@m4ns0ur")
	addPerson("Manu Garg", "manugarg@gmail.com")
	addPerson("Manu S Ajith", "neo@codingarena.in", "@manusajith")
	addPerson("Manuel FernandezaaZ", "sourvivor@gmail.com")
	addPerson("Marc Coury", "gloriphobia@gmail.com")
	addPerson("Marc Lopez", "marc5.12@outlook.com")
	addPerson("Marc Vandenbosch", "marc.vandenbosch@gmail.com")
	addPerson("Marc-Antoine Ruel", "maruel@chromium.org", "7845@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Marc-Antoine Ruel", "maruel@chromium.org", "@maruel", "*goog")
	addPerson("Marcel Edmund Franke", "marcel.edmund.franke@gmail.com", "@donutloop")
	addPerson("Marcel Lanz", "marcel.lanz@n-1.ch")
	addPerson("Marcel van Lohuizen", "mpvl@golang.org", "5182@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Marcel van Lohuizen", "mpvl@golang.org", "@mpvl")
	addPerson("Marcelo Cantos", "marcelo.cantos@gmail.com")
	addPerson("Marcio Feroni", "consultoria.feroni@gmail.com")
	addPerson("Marco Hennings", "marco.hennings@freiheit.com", "@mhennings")
	addPerson("Marco Peereboom", "marco@peereboom.us")
	addPerson("Marcos Minond", "minond.marcos@gmail.com")
	addPerson("Marcus Comstedt", "marcus@mc.pp.se")
	addPerson("Marcus Willock", "crazcalm@gmail.com", "@crazcalm")
	addPerson("Marek Polacek", "polacek@redhat.com", "@mpolacek")
	addPerson("Marga Manterola", "marga@google.com", "@margamanterola")
	addPerson("Marin", "marin.basic02@gmail.com", "@MarinX")
	addPerson("Mario Arranz", "marioarranzr@gmail.com", "@marioarranzr")
	addPerson("Marius Kittler", "mariuskittler@gmx.de")
	addPerson("Marius Nuennerich", "mnu@google.com")
	addPerson("Mark Adams", "mark@markadams.me", "@mark-adams")
	addPerson("Mark Bucciarelli", "mkbucc@gmail.com", "@mbucc")
	addPerson("Mark Harrison", "marhar@google.com")
	addPerson("Mark Lee", "code0x9@gmail.com")
	addPerson("Mark Pulford", "mark@kyne.com.au", "15920@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Mark Pulford", "mark@kyne.com.au", "@mpx")
	addPerson("Mark Rushakoff", "mark.rushakoff@gmail.com", "@mark-rushakoff")
	addPerson("Mark Rushakoff", "mark@influxdata.com")
	addPerson("Mark Ryan", "mark.d.ryan@intel.com", "@markdryan")
	addPerson("Mark Severson", "miquella@gmail.com", "11540@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Mark Severson", "miquella@gmail.com", "@miquella")
	addPerson("Mark Theunissen", "mark.theunissen@gmail.com", "@marktheunissen")
	addPerson("Mark Wolfe", "mark@wolfe.id.au", "@wolfeidau")
	addPerson("Mark Zavislak", "zavislak@google.com", "@zavislak")
	addPerson("Marko Juhani Silokunnas", "marko.silokunnas@gmail.com", "@marant")
	addPerson("Marko Kevac", "marko@kevac.org")
	addPerson("Marko Kevac", "marko@kevac.org", "@mkevac")
	addPerson("Marko Mudrinic", "mudrinic.mare@gmail.com", "@xmudrii")
	addPerson("Marko Mudrinić", "mudrinic.mare@gmail.com", "17318@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Marko Tiikkaja", "marko@joh.to", "5446@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Marko Tiikkaja", "marko@joh.to", "@johto")
	addPerson("Markus Sonderegger", "marraison@gmail.com", "@mars9")
	addPerson("Markus Zimmermann", "markus.zimmermann@nethead.at")
	addPerson("Markus Zimmermann", "zimmski@gmail.com", "@zimmski")
	addPerson("Markus", "m.walther97@gmail.com", "@markus-wa")
	addPerson("Marten Seemann", "martenseemann@gmail.com")
	addPerson("Marten Seemann", "martenseemann@gmail.com", "@marten-seemann")
	addPerson("Martijn Janssen", "martijn9612+github@gmail.com")
	addPerson("Martin Bertschler", "mbertschler@gmail.com", "@mbertschler")
	addPerson("Martin Drlik", "martadrlik@gmail.com")
	addPerson("Martin Garton", "garton@gmail.com", "13346@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Martin Garton", "garton@gmail.com", "@MartinGarton")
	addPerson("Martin Garton", "garton@gmail.com", "@mjgarton")
	addPerson("Martin Habbecke", "marhab@google.com")
	addPerson("Martin Hamrle", "martin.hamrle@gmail.com", "@mhamrle")
	addPerson("Martin Hoefling", "martin.hoefling@gmx.de")
	addPerson("Martin Kreichgauer", "martinkr@google.com", "16331@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Martin Kreichgauer", "martinkr@google.com", "@kreichgauer")
	addPerson("Martin Kunc", "martinkunc@users.noreply.github.com")
	addPerson("Martin Kunc", "mk@Martins-MacBook-Pro.local")
	addPerson("Martin Lee", "martin@martinlee.org")
	addPerson("Martin Lindhe", "martin.j.lindhe@gmail.com", "@martinlindhe")
	addPerson("Martin Möhrmann", "martisch@uos.de", "5846@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Martin Möhrmann", "moehrmann@google.com", "16006@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Martin Möhrmann", "moehrmann@google.com", "martisch@uos.de", "@martisch")
	addPerson("Martin Olsen", "github.com@martinolsen.net", "@martinolsen")
	addPerson("Martin Olsson", "martin@minimum.se", "@mo")
	addPerson("Martin Probst", "martin@probst.io")
	addPerson("Martin Redmond", "mrtodo@gmail.com")
	addPerson("Martin Sucha", "anty.sk+git@gmail.com", "@martin-sucha")
	addPerson("Martin Tournoij", "martin@arp242.net")
	addPerson("Martins Sipenko", "martins.sipenko@gmail.com", "@martinssipenko")
	addPerson("Martynas Budriūnas", "mabu@google.com", "@mabu")
	addPerson("Marvin Stenger", "marvin.stenger94@gmail.com")
	addPerson("Marvin Stenger", "marvin.stenger94@gmail.com", "9850@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Masa Sekimura", "sekimura@gmail.com")
	addPerson("Masahiro Furudate", "masahiro.furudate@gmail.com")
	addPerson("Massimiliano Ghilardi", "massimiliano.ghilardi@gmail.com")
	addPerson("Mat Byczkowski", "mbyczkowski@gmail.com", "@mbyczkowski")
	addPerson("Mat Evans", "matzhouse@gmail.com")
	addPerson("Mat Ryer", "thatmatryer@gmail.com")
	addPerson("Matej Baćo", "matejbaco@gmail.com", "@matejb")
	addPerson("Mateus Amin", "mateus.amin@gmail.com")
	addPerson("Mateusz Czapliński", "czapkofan@gmail.com", "10525@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Mateusz Czapliński", "czapkofan@gmail.com", "@akavel")
	addPerson("Mathias Beke", "git@denbeke.be", "7490@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Mathias Hall-Andersen", "mathias@hall-andersen.dk")
	addPerson("Mathias Leppich", "mleppich@muhqu.de", "@muhqu")
	addPerson("MathiasB", "git@denbeke.be", "@DenBeke")
	addPerson("Mathieu Lonjaret", "mathieu.lonjaret@gmail.com", "8466@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Mats Lidell", "mats.lidell@cag.se", "@matsl")
	addPerson("Matt Aimonetti", "mattaimonetti@gmail.com", "13882@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Matt Aimonetti", "mattaimonetti@gmail.com", "@mattetti")
	addPerson("Matt Blair", "me@matthewblair.net", "@mblair")
	addPerson("Matt Bostock", "matt@mattbostock.com", "@mattbostock")
	addPerson("Matt Dee", "mdee@hioscar.com")
	addPerson("Matt Drollette", "matt@drollette.com", "@MDrollette")
	addPerson("Matt DuVall", "matt@stripe.com")
	addPerson("Matt Harden", "matt.harden@gmail.com")
	addPerson("Matt Harden", "matt.harden@gmail.com", "8785@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Matt Harden", "matt.harden@gmail.com", "@nerdatmath")
	addPerson("Matt Jibson", "matt.jibson@gmail.com", "@mjibson")
	addPerson("Matt Joiner", "anacrolix@gmail.com", "@anacrolix")
	addPerson("Matt Jones", "matt@mhjones.org")
	addPerson("Matt Juran", "thepciet@gmail.com", "@pciet")
	addPerson("Matt Keenan", "github@mattkeenan.net", "@mattkeenan")
	addPerson("Matt Layher", "mdlayher@gmail.com", "7860@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Matt Layher", "mdlayher@gmail.com", "@mdlayher")
	addPerson("Matt Proud", "matt.proud@gmail.com", "6400@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Matt Reiferson", "mreiferson@gmail.com", "@mreiferson")
	addPerson("Matt Robenolt", "matt@ydekproductions.com", "@mattrobenolt")
	addPerson("Matt T. Proud", "matt.proud@gmail.com", "@matttproud")
	addPerson("Matteo Croce", "matteo.croce@canonical.com")
	addPerson("Matthew Allen Moltzau", "Matthew_Moltzau@comcast.com")
	addPerson("Matthew Brennan", "matty.brennan@gmail.com", "@mattyb")
	addPerson("Matthew Broberg", "gogetmb@gmail.com", "@mbbroberg")
	addPerson("Matthew Broberg", "matthewbbroberg@gmail.com")
	addPerson("Matthew Byrne", "mjw.byrne@gmail.com")
	addPerson("Matthew Cottingham", "mattcottingham@gmail.com", "@mattrco")
	addPerson("Matthew Dempsky", "matthew@dempsky.org", "8715@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Matthew Dempsky", "mdempsky@google.com", "5440@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Matthew Dempsky", "mdempsky@google.com", "@mdempsky")
	addPerson("Matthew Denton", "mdenton@skyportsystems.com", "@mdentonSkyport")
	addPerson("Matthew Endsley", "mendsley@gmail.com")
	addPerson("Matthew Herrmann", "mherr@google.com")
	addPerson("Matthew Holt", "matthew.holt+git@gmail.com", "@mholt")
	addPerson("Matthew Holt", "matthew.holt@gmail.com", "7611@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Matthew Hooker", "mwhooker@gmail.com")
	addPerson("Matthew Horsnell", "matthew.horsnell@gmail.com", "@matt2909")
	addPerson("Matthew LJ Smith", "matthew.lj.smith@gmail.com")
	addPerson("Matthew Rudy Jacobs", "matthewrudyjacobs@gmail.com")
	addPerson("Matthew Waters", "mwwaters@gmail.com", "@mwwaters")
	addPerson("Matthew Whisenhunt", "matt.whisenhunt@gmail.com")
	addPerson("Matthieu Hauglustaine", "matt.hauglustaine@gmail.com", "@MattHauglustaine")
	addPerson("Matthieu Sarter", "matthieu.sarter.external@atos.net", "16325@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Matthieu Sarter", "matthieu.sarter.external@atos.net", "@MatthieuSarter")
	addPerson("Matthijs Kooijman", "matthijs@stdin.nl", "@matthijskooijman")
	addPerson("Max Moroz", "maxmoroz@gmail.com")
	addPerson("Max Renaud", "maxrd@google.com")
	addPerson("Max Riveiro", "kavu13@gmail.com", "@kavu")
	addPerson("Max Schmitt", "max@schmitt.mx")
	addPerson("Max Ushakov", "ushmax@gmail.com", "@ushakov")
	addPerson("Maxime de Roucy", "maxime.deroucy@gmail.com", "@tchernomax")
	addPerson("Maxwell Krohn", "themax@gmail.com", "@maxtaco")
	addPerson("Mayank Kumar", "krmayankk@gmail.com", "@krmayankk")
	addPerson("Mayank Sharma", "maksharma231@gmail.com")
	addPerson("Mayank Sharma", "mayank.sharma@tokopedia.com")
	addPerson("Mayank Sharma", "mayank@qlogic.io")
	addPerson("Mayra Cabrera", "mcabrera1087@gmail.com")
	addPerson("Mehul Choube", "mchoube@gmail.com")
	addPerson("Meir Fischer", "meirfischer@gmail.com", "@meirf")
	addPerson("Meng Zhuo", "mengzhuo1203@gmail.com", "@mengzhuo")
	addPerson("Menghan Li", "menghanl@google.com")
	addPerson("Mhd Sulhan", "m.shulhan@gmail.com", "@shuLhan")
	addPerson("MiLk", "hello@emilienkenler.com", "@MiLk")
	addPerson("Michael Andersen", "michael@steelcode.com")
	addPerson("Michael Anthony Knyszek", "mknyszek@google.com", "@mknyszek")
	addPerson("Michael Brandenburg", "mbrandenburg@bolste.com")
	addPerson("Michael Chaten", "mchaten@gmail.com", "@chaten")
	addPerson("Michael D Henderson", "mdhender@mdhender.com")
	addPerson("Michael Darakananda", "pongad@gmail.com")
	addPerson("Michael Darakananda", "pongad@google.com")
	addPerson("Michael Darakananda", "pongad@google.com", "@pongad")
	addPerson("Michael Darakananda", "pongad@pongad-linuxworkstation1.sea.corp.google.com")
	addPerson("Michael Dorner", "mail@michaeldorner.de")
	addPerson("Michael Edwards", "medwards@walledcity.ca")
	addPerson("Michael Elkins", "michael.elkins@gmail.com", "@sigpipe")
	addPerson("Michael Ellis", "micellis@justin.tv", "@mellis")
	addPerson("Michael Folkson", "michael@riskbazaar.org")
	addPerson("Michael Fraenkel", "michael.fraenkel@gmail.com", "5889@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Michael Fraenkel", "michael.fraenkel@gmail.com", "@fraenkel")
	addPerson("Michael Gehring", "mg@ebfe.org", "6715@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Michael Gehring", "mg@ebfe.org", "@ebfe")
	addPerson("Michael Henderson", "mdhender@users.noreply.github.com", "@mdhender")
	addPerson("Michael Hendricks", "michael@ndrix.org", "@mndrix")
	addPerson("Michael Hoisie", "hoisie@gmail.com", "@hoisie")
	addPerson("Michael Hudson-Doyle", "michael.hudson@canonical.com", "5153@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Michael Hudson-Doyle", "michael.hudson@canonical.com", "@mwhudson")
	addPerson("Michael Kasch", "michael.kasch@gmail.com", "@MyChaOS87")
	addPerson("Michael Kuryshev", "me@mk9.name")
	addPerson("Michael Käufl", "golang@c.michael-kaeufl.de", "@michael-k")
	addPerson("Michael Lewis", "mikelikespie@gmail.com", "@mikelikespie")
	addPerson("Michael MacInnis", "michael.p.macinnis@gmail.com", "6355@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Michael MacInnis", "michael.p.macinnis@gmail.com", "@michaelmacinnis")
	addPerson("Michael Marineau", "michael.marineau@coreos.com", "mike@marineau.org", "@marineam")
	addPerson("Michael Matloob", "matloob@golang.org", "10033@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Michael Matloob", "matloob@golang.org", "matloob@google.com", "@matloob")
	addPerson("Michael Matloob", "michaelmatloob@gmail.com", "5270@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Michael McConville", "momcconville@gmail.com")
	addPerson("Michael McGreevy", "mcgreevy@golang.org", "@mcgreevy")
	addPerson("Michael McLoughlin", "mmcloughlin@gmail.com", "@mmcloughlin")
	addPerson("Michael Munday", "mike.munday@ibm.com", "11990@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Michael Munday", "mike.munday@ibm.com", "munday@ca.ibm.com", "@mundaym")
	addPerson("Michael Pearson", "mipearson@gmail.com", "@mipearson")
	addPerson("Michael Pratt", "mpratt@google.com", "12120@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Michael Pratt", "mpratt@google.com", "@prattmic")
	addPerson("Michael Schaller", "michael@5challer.de", "@michael-schaller")
	addPerson("Michael Schurter", "michael.schurter@gmail.com", "@schmichael")
	addPerson("Michael Shields", "mshields@google.com", "@shields")
	addPerson("Michael Smith", "mikejsmitty@gmail.com")
	addPerson("Michael Spiegel", "michael.m.spiegel@gmail.com")
	addPerson("Michael Stapelberg", "stapelberg@google.com", "8470@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Michael Stapelberg", "stapelberg@google.com", "@stapelberg")
	addPerson("Michael Steinert", "mike.steinert@gmail.com", "@msteinert")
	addPerson("Michael Sterle-Contala", "mike.sterle@gmail.com")
	addPerson("Michael Teichgräber", "mteichgraeber@gmx.de", "@knieriem")
	addPerson("Michael Vetter", "g.bluehut@gmail.com", "@jubalh")
	addPerson("Michal Bohuslávek", "mbohuslavek@gmail.com", "9715@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Michal Bohuslávek", "mbohuslavek@gmail.com", "@mibk")
	addPerson("Michal Cierniak", "cierniak@google.com", "@cierniak")
	addPerson("Michal Franc", "lam.michal.franc@gmail.com")
	addPerson("Michal Pristas", "michal.pristas@gmail.com")
	addPerson("Michalis Kargakis", "michaliskargakis@gmail.com")
	addPerson("Michel Lespinasse", "walken@google.com", "12938@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Michel Lespinasse", "walken@google.com", "@walken-google")
	addPerson("Miek Gieben", "miek@miek.nl", "@miekg", "*goog")
	addPerson("Miguel Molina", "hi@mvader.me", "@erizocosmico")
	addPerson("Miguel Perez", "miguel250@gmail.com")
	addPerson("Mihai Borobocea", "MihaiBorobocea@gmail.com")
	addPerson("Mihai Todor", "todormihai@gmail.com", "@mihaitodor")
	addPerson("Mike Appleby", "mike@app.leby.org", "14930@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Mike Appleby", "mike@app.leby.org", "@appleby")
	addPerson("Mike Danese", "mikedanese@google.com", "@mikedanese")
	addPerson("Mike Dour", "mdour@google.com")
	addPerson("Mike Graf", "mikegraf000@gmail.com")
	addPerson("Mike Houston", "mike@kothar.net", "@kothar")
	addPerson("Mike Kabischev", "kabischev@gmail.com")
	addPerson("Mike Lloyd", "kevin.michael.lloyd@gmail.com")
	addPerson("Mike Lloyd", "kevin.michael.lloyd@gmail.com", "10091@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Mike Rosset", "mike.rosset@gmail.com", "@mrosset")
	addPerson("Mike Samuel", "mikesamuel@gmail.com", "17511@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Mike Samuel", "mikesamuel@gmail.com", "@mikesamuel")
	addPerson("Mike Solomon", "msolo@gmail.com", "@msolo")
	addPerson("Mike Strosaker", "strosake@us.ibm.com")
	addPerson("Mike Tsao", "mike@sowbug.com")
	addPerson("Mike Wiacek", "mjwiacek@google.com", "@mikewiacek")
	addPerson("Mike Wiacek", "mjwiacek@mjwiacek-macbookpro.roam.corp.google.com")
	addPerson("Mikhail Gusarov", "dottedmag@dottedmag.net", "@dottedmag")
	addPerson("Miki Habryn", "dichro@rcpt.to")
	addPerson("Miki Tebeka", "miki.tebeka@gmail.com", "@tebeka")
	addPerson("Mikio Hara", "mikioh.public.networking@gmail.com", "mikioh.mikioh@gmail.com", "@mikioh", "29736@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Mikkel Krautz", "mikkel@krautz.dk", "@mkrautz")
	addPerson("Mikołaj Baranowski", "m.baranowski@travelaudience.com")
	addPerson("Milan Knezevic", "milan.knezevic@mips.com", "@milanknezevic")
	addPerson("Minaev Mike", "minaev.mike@gmail.com", "23800@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Minaev Mike", "minaev.mike@gmail.com", "@minaevmike")
	addPerson("Minux Ma", "minux@golang.org", "5055@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Miquel Sabaté Solà", "mikisabate@gmail.com", "@mssola")
	addPerson("Miroslav Genov", "mgenov@gmail.com", "@mgenov")
	addPerson("Misty De Meo", "mistydemeo@gmail.com", "@mistydemeo")
	addPerson("Mitchell-Riley", "tug72074@temple.edu")
	addPerson("Mithun Sasidharan", "mithunsasidharan89@gmail.com")
	addPerson("Miyakawa Taku", "miyakawa.taku@gmail.com")
	addPerson("Mofizur Rahman", "moficodes@gmail.com")
	addPerson("Mohan Pawar", "mohanpawary1990@gmail.com")
	addPerson("Mohit Agarwal", "mohit@sdf.org", "10715@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Mohit Agarwal", "mohit@sdf.org", "@0xmohit")
	addPerson("Mohit kumar Bajoria", "mohitbajo36@gmail.com", "@mbj36")
	addPerson("Momchil Velikov", "momchil.velikov@gmail.com", "@momchil-velikov")
	addPerson("Monis Khan", "mkhan@redhat.com", "@enj")
	addPerson("Monty Taylor", "mordred@inaugust.com", "@emonty")
	addPerson("Moriyoshi Koizumi", "mozo@mozo.jp", "@moriyoshi")
	addPerson("Morten Siebuhr", "sbhr@sbhr.dk", "10928@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Morten Siebuhr", "sbhr@sbhr.dk", "@msiebuhr")
	addPerson("Mostyn Bramley-Moore", "mostyn@antipode.se")
	addPerson("Mostyn Bramley-Moore", "mostyn@antipode.se", "18980@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Mostyn Bramley-Moore", "mostyn@antipode.se", "@mostynb")
	addPerson("Mrunal Patel", "mrunalp@gmail.com", "@mrunalp")
	addPerson("Muhammad Falak R Wani", "falakreyaz@gmail.com", "23560@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Muhammad Falak R Wani", "falakreyaz@gmail.com", "@mfrw")
	addPerson("Muhammed Uluyol", "uluyol0@gmail.com", "@uluyol")
	addPerson("Mura Li", "mura_li@castech.com.tw", "10925@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Mura Li", "mura_li@castech.com.tw", "@typeless")
	addPerson("Máximo Cuadros Ortiz", "mcuadros@gmail.com", "@mcuadros")
	addPerson("NODA, Kai", "nodakai@gmail.com")
	addPerson("Nan Deng", "monnand@gmail.com", "@monnand")
	addPerson("Naoki INADA", "songofacandy@gmail.com", "5895@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Naoki Kanatani", "k12naoki@gmail.com", "@kanata2")
	addPerson("Nate Wilkinson", "nathanwilk7@gmail.com", "@nathanwilk7")
	addPerson("Nathan Cantelmo", "n.cantelmo@gmail.com", "@ncantelmo")
	addPerson("Nathan Caza", "mastercactapus@gmail.com", "@mastercactapus")
	addPerson("Nathan Davies", "nathanjamesdavies@gmail.com")
	addPerson("Nathan John Youngman", "nj@nathany.com", "@nathany")
	addPerson("Nathan Otterness", "otternes@cs.unc.edu")
	addPerson("Nathan P Finch", "nate.finch@gmail.com", "@natefinch")
	addPerson("Nathan VanBenschoten", "nvanbenschoten@gmail.com", "@nvanbenschoten")
	addPerson("Nathan Youngman", "git@nathany.com", "@nathany")
	addPerson("Nathan Youngman", "hello@nathany.com", "5235@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Nathan(yinian) Hu", "nathanhu@google.com", "@nathandfox")
	addPerson("Nathaniel Caza", "mastercactapus@gmail.com", "17183@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Naveen Kumar Sangi", "naveenkumarsangi@protonmail.com")
	addPerson("Neelesh Chandola", "neelesh.c98@gmail.com")
	addPerson("Neil Basu", "nbasu02@gmail.com")
	addPerson("Neil Lyons", "nwjlyons@googlemail.com", "@nwjlyons")
	addPerson("Neil Owen", "neil.anthony.owen@gmail.com")
	addPerson("Nelz", "nelz9999@gmail.com")
	addPerson("Nemin Sun", "sunnemin@gmail.com")
	addPerson("Neven Sajko", "nsajko@gmail.com", "15048@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Neven Sajko", "nsajko@gmail.com", "@nsajko")
	addPerson("Nevins Bartolomeo", "nevins.bartolomeo@gmail.com", "@nevins-b")
	addPerson("Niall Sheridan", "nsheridan@gmail.com", "13755@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Niall Sheridan", "nsheridan@gmail.com", "@nsheridan")
	addPerson("Nic Day", "nic.day@me.com", "@nicday")
	addPerson("Nicholas Anderson", "nick@miletwo.net")
	addPerson("Nicholas Katsaros", "nick@nickkatsaros.com", "@nkatsaros")
	addPerson("Nicholas Maniscalco", "nicholas@maniscalco.com")
	addPerson("Nicholas Maniscalco", "nicholas@maniscalco.com", "@nicholasmaniscalco")
	addPerson("Nicholas Ng", "nickng@nickng.io")
	addPerson("Nicholas Presta", "nick@nickpresta.ca", "@nickpresta")
	addPerson("Nicholas Rawlings", "nicholasorenrawlings@gmail.com")
	addPerson("Nicholas Waples", "nwaples@gmail.com", "@nwaples")
	addPerson("Nick Cooper", "nmvc@google.com")
	addPerson("Nick Cooper", "nmvc@google.com", "5776@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Nick Craig-Wood", "nick@craig-wood.com", "@ncw")
	addPerson("Nick Craig-Wood", "nickcw@gmail.com", "5175@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Nick Harper", "nharper@google.com")
	addPerson("Nick Kubala", "nkubala@google.com", "@nkubala")
	addPerson("Nick McCrory", "nickmhc14@gmail.com")
	addPerson("Nick Miyake", "nmiyake@gmail.com")
	addPerson("Nick Patavalis", "nick.patavalis@gmail.com", "9880@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Nick Patavalis", "nick.patavalis@gmail.com", "@npat-efault")
	addPerson("Nick Petroni", "npetroni@cs.umd.edu")
	addPerson("Nick Sullivan", "nicholas.sullivan@gmail.com", "@grittygrease")
	addPerson("Nickolay Turpitko", "nikolay@turpitko.com", "7015@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Nicolas BRULEZ", "n.brulez@gmail.com", "@N-Bz")
	addPerson("Nicolas S. Dade", "nic.dade@gmail.com", "@nsd20463")
	addPerson("Niek Sanders", "niek.sanders@gmail.com", "19925@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Niek Sanders", "niek.sanders@gmail.com", "@nieksand")
	addPerson("Nigel Kerr", "nigel.kerr@gmail.com", "@nigelkerr")
	addPerson("Nigel Tao", "nigeltao@golang.org", "5899@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Nigel Tao", "nigeltao@golang.org", "@nigeltao")
	addPerson("Nik Nyby", "nnyby@columbia.edu", "@nikolas")
	addPerson("Nikhil Benesch", "nikhil.benesch@gmail.com", "25418@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Nikhil Benesch", "nikhil.benesch@gmail.com", "@benesch")
	addPerson("Nikhita Raghunath", "nikitaraghunath@gmail.com")
	addPerson("Niklas Lindblad", "niklas@lindblad.info")
	addPerson("Niklas Schnelle", "niklas.schnelle@gmail.com", "@niklas88")
	addPerson("Niko Dziemba", "niko@dziemba.com", "@dziemba")
	addPerson("Nikolay Ponomarev", "itsnikolay@gmail.com")
	addPerson("Nikolay Turpitko", "nikolay@turpitko.com", "@nikolay-turpitko")
	addPerson("Nilesh Jagnik", "nileshj@google.com")
	addPerson("Nils Larsgård", "nilsmagnus@gmail.com", "@nilsmagnus")
	addPerson("Nir Soffer", "nirsof@gmail.com")
	addPerson("Nishanth Shanmugham", "nishanth.gerrard@gmail.com")
	addPerson("Nitin Patil", "patil16nit@gmail.com")
	addPerson("Noah Campbell", "noahcampbell@gmail.com")
	addPerson("Noble Johnson", "noblepoly@gmail.com")
	addPerson("Nodir Turakulov", "nodir@google.com", "7877@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Nodir Turakulov", "nodir@google.com", "@nodirt")
	addPerson("Noel Georgi", "git@frezbo.com")
	addPerson("Norberto Lopes", "nlopes.ml@gmail.com", "@nlopes")
	addPerson("Nyah Check", "check.nyah@gmail.com", "22747@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("ObesePenguin", "srburnham@gmail.com")
	addPerson("Odin Ugedal", "odin@ugedal.com", "@odinuge")
	addPerson("Oleg Bulatov", "dmage@yandex-team.ru", "@dmage")
	addPerson("Oleg Bulatov", "oleg@bulatov.me", "@dmage")
	addPerson("Oleg Vakheta", "helginet@gmail.com", "@helginet")
	addPerson("Oleku Konko", "oleku.konko@gmail.com", "@olekukonko")
	addPerson("OlgaVlPetrova", "ovppetrova@gmail.com", "@OlgaVlPetrova")
	addPerson("Oling Cat", "olingcat@gmail.com", "5136@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Oling Cat", "olingcat@gmail.com", "@OlingCat")
	addPerson("Oliver Hookins", "ohookins@gmail.com", "@ohookins")
	addPerson("Oliver Skånberg-Tippen", "oliverskanbergtippen@gmail.com")
	addPerson("Oliver Stenbom", "ostenbom@pivotal.io", "@ostenbom")
	addPerson("Oliver Tonnhofer", "olt@bogosoft.com", "@olt")
	addPerson("Olivier Duperray", "duperray.olivier@gmail.com", "@dupoxy")
	addPerson("Olivier Mengué", "olivier.mengue@gmail.com")
	addPerson("Olivier Poitrey", "rs@netflix.com", "rs@dailymotion.com", "rs@rhapsodyk.net", "10610@62eb7196-b449-3ce5-99f1-c037f21e1705", "@rs")
	addPerson("Olivier Saingre", "osaingre@gmail.com", "@osaingre")
	addPerson("Olivier", "desylva@gmail.com")
	addPerson("Omar Jarjur", "ojarjur@google.com")
	addPerson("OneOfOne", "oneofone@gmail.com", "@OneOfOne")
	addPerson("Oryan Moshe", "iamoryanmoshe@gmail.com", "28422@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Oryan Moshe", "iamoryanmoshe@gmail.com", "@oryanmoshe")
	addPerson("Osamu TONOMORI", "osamingo@gmail.com")
	addPerson("Osamu TONOMORI", "osamingo@gmail.com", "@osamingo")
	addPerson("Oscar Forner Martinez", "oscar.forner.martinez@gmail.com")
	addPerson("Otto Giron", "ottog2486@gmail.com")
	addPerson("Owen Marshall", "om@om.ht")
	addPerson("Pablo Lalloni", "plalloni@gmail.com", "@lalloni")
	addPerson("Pablo Santiago Blum de Aguiar", "scorphus@gmail.com", "@scorphus")
	addPerson("Paddy Foran", "paddy@secondbit.org", "@paddyforan")
	addPerson("Paddy Steed", "jarktasaa@gmail.com")
	addPerson("Padraig Kitterick", "padraigkitterick@gmail.com", "@padraigkitterick")
	addPerson("Panagiotis Xynos", "panagiotis.xinos@gmail.com")
	addPerson("Paolo Giarrusso", "p.giarrusso@gmail.com", "@Blaisorblade")
	addPerson("Paolo Martini", "mrtnpaolo@gmail.com", "@ear")
	addPerson("Parker Moore", "parkrmoore@gmail.com", "6501@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Parker Moore", "parkrmoore@gmail.com", "@parkr")
	addPerson("Pascal Corpet", "lascap@google.com")
	addPerson("Pascal Muetschard", "pmuetschard@google.com")
	addPerson("Pascal S. de Kloe", "pascal@quies.net", "@pascaldekloe")
	addPerson("Pascal de Kloe", "pascal@quies.net", "8310@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Pat Moroney", "pat@pat.email", "@pmoroney")
	addPerson("Patrick Bennett", "patrick@thebennetts.com")
	addPerson("Patrick Edgett", "pedgett@gmail.com")
	addPerson("Patrick Gavlin", "pgavlin@gmail.com", "@pgavlin")
	addPerson("Patrick Higgins", "patrick.allen.higgins@gmail.com", "@patrick-higgins")
	addPerson("Patrick Lee", "pattyshack101@gmail.com", "@pattyshack")
	addPerson("Patrick Mezard", "patrick@mezard.eu", "@pmezard")
	addPerson("Patrick Mylund Nielsen", "patrick@patrickmn.com", "@patrickmn")
	addPerson("Patrick Mézard", "patrick@mezard.eu", "7915@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Patrick Pelletier", "pp.pelletier@gmail.com", "@skinp")
	addPerson("Patrick Riley", "pfr@google.com", "@pfrstg")
	addPerson("Patrick Smith", "pat42smith@gmail.com", "@pat42smith")
	addPerson("Patrick Uiterwijk", "patrick@puiterwijk.org")
	addPerson("Patrik Nyblom", "pnyb@google.com")
	addPerson("Paul A Querna", "paul.querna@gmail.com", "@pquerna")
	addPerson("Paul Borman", "borman@google.com", "@pborman")
	addPerson("Paul Boyd", "boyd.paul2@gmail.com", "@pboyd")
	addPerson("Paul Chang", "paulchang@google.com", "@pchx")
	addPerson("Paul Gier", "pgier@redhat.com")
	addPerson("Paul Hankin", "paulhankin@google.com")
	addPerson("Paul Jolly", "paul@myitcv.io", "@myitcv")
	addPerson("Paul Jolly", "paul@myitcv.org.uk", "16375@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Paul Jolly", "paul@myitcv.org.uk", "@myitcv")
	addPerson("Paul Lalonde", "paul.a.lalonde@gmail.com", "@paul-lalonde")
	addPerson("Paul M Furley", "paul@paulfurley.com")
	addPerson("Paul Marks", "pmarks@google.com", "6050@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Paul Marks", "pmarks@google.com", "@pmarks-net")
	addPerson("Paul Meyer", "paul.meyer@microsoft.com", "@paulmey")
	addPerson("Paul Nasrat", "pnasrat@google.com")
	addPerson("Paul PISCUC", "paul.piscuc@gmail.com", "@ppiscuc")
	addPerson("Paul Querna", "pquerna@apache.org", "@pquerna", "paul@querna.org", "14273@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Paul Rosania", "paul.rosania@gmail.com", "@paulrosania")
	addPerson("Paul Sbarra", "sbarra.paul@gmail.com", "@tones111")
	addPerson("Paul Smith", "paulsmith@pobox.com", "@paulsmith")
	addPerson("Paul Tyng", "paul@paultyng.net")
	addPerson("Paul Tyng", "ptyng@underarmour.com")
	addPerson("Paul Wankadia", "junyer@google.com", "@junyer")
	addPerson("Paul van Brouwershaven", "paul@vanbrouwershaven.com", "5920@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Paul van Brouwershaven", "paul@vanbrouwershaven.com", "@vanbroup")
	addPerson("Paulo Casaretto", "pcasaretto@gmail.com", "@pcasaretto")
	addPerson("Paulo Flabiano Smorigo", "pfsmorigo@linux.vnet.ibm.com")
	addPerson("Paulo Flabiano Smorigo", "pfsmorigo@linux.vnet.ibm.com", "@pfsmorigo")
	addPerson("Pavlo Sumkin", "psumkin@mirantis.com")
	addPerson("Pavlo Sumkin", "ymkins@gmail.com")
	addPerson("Pawel Knap", "pawelknap88@gmail.com", "@ppknap")
	addPerson("Pawel Pisarzewski", "morris@morris.io")
	addPerson("Peng Gao", "peng.gao.dut@gmail.com")
	addPerson("Percy Wegmann", "ox.to.a.cart@gmail.com", "@oxtoacart")
	addPerson("Perry Abbott", "perry.j.abbott@gmail.com", "@pabbott0")
	addPerson("Petar Maymounkov", "petarm@gmail.com", "@petar")
	addPerson("Peter Armitage", "peter.armitage@gmail.com", "@pja")
	addPerson("Peter Armitage", "pja@google.com")
	addPerson("Peter Bourgon", "peter@bourgon.org", "@peterbourgon")
	addPerson("Peter Collingbourne", "pcc@google.com", "5535@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Peter Collingbourne", "pcc@google.com", "@pcc")
	addPerson("Peter Ebden", "peter.ebden@gmail.com")
	addPerson("Peter Froehlich", "peter.hans.froehlich@gmail.com", "@phf")
	addPerson("Peter Gonda", "pgonda@google.com")
	addPerson("Peter Gonda", "pgonda@google.com", "@pgonda")
	addPerson("Peter Gonda", "ptrgonda@gmail.com")
	addPerson("Peter Götz", "peter.gtz@gmail.com")
	addPerson("Peter Hoyes", "pahoyes@gmail.com")
	addPerson("Peter Kieltyka", "peter.kieltyka@pressly.com")
	addPerson("Peter Kleiweg", "pkleiweg@xs4all.nl", "@pebbe")
	addPerson("Peter Mattis", "petermattis@gmail.com")
	addPerson("Peter McKenzie", "petermck@google.com", "@peter-mckenzie")
	addPerson("Peter Moody", "peter.moody@gmail.com")
	addPerson("Peter Moody", "pmoody@uber.com", "8905@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Peter Moody", "pmoody@uber.com", "@pmoody-")
	addPerson("Peter Morjan", "pmorjan@gmail.com", "@pmorjan")
	addPerson("Peter Mundy", "go.peter.90@gmail.com", "@peterGo")
	addPerson("Peter Nguyen", "peter@mictis.com", "@pengux")
	addPerson("Peter S", "speter.go1@gmail.com", "5351@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Peter Sanford", "psanford@sanford.io")
	addPerson("Peter Sutherland", "peter@pedrosland.co.uk")
	addPerson("Peter Teichman", "peter@teichman.org")
	addPerson("Peter Teichman", "pteichman@fastly.com", "@pteichman")
	addPerson("Peter Tseng", "ptseng@squareup.com", "@petertseng")
	addPerson("Peter Waldschmidt", "peter@waldschmidt.com", "6340@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Peter Waldschmidt", "peter@waldschmidt.com", "@peterwald")
	addPerson("Peter Waller", "p@pwaller.net", "5822@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Peter Waller", "p@pwaller.net", "@pwaller")
	addPerson("Peter Wathall", "peter.wathall@gmail.com")
	addPerson("Peter Weinberger", "pjw@golang.org", "pjw@google.com", "@pjweinb", "@pjweinbgo")
	addPerson("Peter Weinberger", "pjw@google.com", "5260@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Peter Williams", "pwil3058@gmail.com", "@pwil3058")
	addPerson("Peter Wu", "peter@lekensteyn.nl", "24681@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Peter Wu", "pwu@cloudflare.com", "@Lekensteyn")
	addPerson("Peterson, David", "davidtpeterson@gmail.com")
	addPerson("Phil Pearl", "philip.j.r.pearl@gmail.com", "@philpearl")
	addPerson("Philip Brown", "phil@bolthole.com")
	addPerson("Philip Børgesen", "philip.borgesen@gmail.com", "@PhilipBorgesen")
	addPerson("Philip Børgesen", "philipborgesen@users.noreply.github.com", "@PhilipBorgesen")
	addPerson("Philip Hofer", "phofer@umich.edu")
	addPerson("Philip Hofer", "phofer@umich.edu", "9055@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Philip K. Warren", "pkwarren@gmail.com", "@pkwarren")
	addPerson("Philip Pearl", "philip.j.r.pearl@gmail.com", "27852@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Philipp Kern", "pkern@google.com")
	addPerson("Philippe Lafoucrière", "philippe.lafoucriere@gmail.com")
	addPerson("Pierre Durand", "pierredurand@gmail.com", "@pierrre")
	addPerson("Pierre Prinetti", "pierreprinetti@gmail.com", "@pierreprinetti")
	addPerson("Pierre Roullon", "pierre.roullon@gmail.com", "@proullon")
	addPerson("Pieter Droogendijk", "pieter@binky.org.uk", "@PieterD")
	addPerson("Pieterjan Lambein", "pieterjan@otainsight.com")
	addPerson("Pietro Gagliardi", "pietro10@mac.com", "9190@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Pietro Gagliardi", "pietro10@mac.com", "@andlabs")
	addPerson("Piotr Kowalczuk", "p.kowalczuk.priv@gmail.com")
	addPerson("Piyush Mishra", "piyush@codeitout.com", "@ofpiyush")
	addPerson("Plekhanov Maxim", "kishtatix@gmail.com")
	addPerson("Plekhanov Maxim", "kishtatix@gmail.com", "@kishtatik")
	addPerson("Pontus Leitzler", "leitzler@gmail.com")
	addPerson("Pontus Leitzler", "leitzler@users.noreply.github.com", "@leitzler")
	addPerson("Pradeep Singh", "rautelap@gmail.com")
	addPerson("Prasanna Swaminathan", "prasanna@mediamath.com", "@pswaminathan")
	addPerson("Prasanna V. Loganathar", "pvl@prasannavl.com")
	addPerson("Prashant Varanasi", "prashant@prashantv.com", "@prashantv")
	addPerson("Prashanth Pai", "ppai@redhat.com")
	addPerson("Praveen Bathala", "praveen.bathala@gmail.com")
	addPerson("Pravendra Singh", "hackpravj@gmail.com", "@pravj")
	addPerson("Preetam Jinka", "pj@preet.am", "@Preetam")
	addPerson("ProhtMeyhet", "sebastian@prohtmeyhet.de")
	addPerson("Péter Surányi", "speter.go1@gmail.com", "@speter")
	addPerson("Péter Szilágyi", "peterke@gmail.com", "5786@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Péter Szilágyi", "peterke@gmail.com", "@karalabe")
	addPerson("Qais Patankar", "qaisjp@gmail.com", "@qaisjp")
	addPerson("Qi Zhao", "zhaoq@google.com")
	addPerson("Qi Zhao", "zhaoq@google.com", "9480@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Qiu", "ilsh1022@gmail.com", "21240@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Qiuxuan Zhu", "ilsh1022@gmail.com", "@Kinghack")
	addPerson("QtRoS", "mrqtros@gmail.com", "@QtRoS")
	addPerson("Quan Yong Zhai", "qyzhai@gmail.com", "@qyzhai")
	addPerson("Quentin Perez", "qperez@ocs.online.net", "@QuentinPerez")
	addPerson("Quentin Perez", "quentin@zen.ly")
	addPerson("Quentin Renard", "contact@asticode.com", "12775@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Quentin Renard", "contact@asticode.com", "@asticode")
	addPerson("Quentin Smith", "quentin@golang.org")
	addPerson("Quentin Smith", "quentin@golang.org", "13020@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Quentin Smith", "quentin@golang.org", "@quentinmit")
	addPerson("Quinn Slack", "sqs@sourcegraph.com", "@sqs")
	addPerson("Quoc-Viet Nguyen", "afelion@gmail.com", "@nqv")
	addPerson("R Primus", "rprimus@gmail.com")
	addPerson("Radek Sohlich", "sohlich@gmail.com", "@sohlich")
	addPerson("Radu Berinde", "radu@cockroachlabs.com", "12530@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Radu Berinde", "radu@cockroachlabs.com", "@RaduBerinde")
	addPerson("Rafal Jeczalik", "rjeczalik@gmail.com", "@rjeczalik")
	addPerson("Raghavendra Nagaraj", "jamdagni86@gmail.com")
	addPerson("Raghavendra Nagaraj", "jamdagni86@gmail.com", "@jamdagni86")
	addPerson("Rahul Chaudhry", "rahulchaudhry@chromium.org", "@rahulchaudhry", "*goog")
	addPerson("Rahul Chaudhry", "rahulchaudhry@google.com", "5211@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Raj Mahey", "raj.axisos@gmail.com")
	addPerson("Rajat Goel", "rajat.goel2010@gmail.com", "@rajatgoel")
	addPerson("Rajath Agasthya", "rajathagasthya@gmail.com", "24258@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Rajath Agasthya", "rajathagasthya@gmail.com", "@rajathagasthya")
	addPerson("Ralph Corderoy", "ralph.corderoy@gmail.com", "7020@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ralph Corderoy", "ralph@inputplus.co.uk")
	addPerson("Ralph Corderoy", "ralph@inputplus.co.uk", "10961@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ralph Corderoy", "ralph@inputplus.co.uk", "@RalphCorderoy")
	addPerson("Ralph Ligtenberg", "ralph.ligtenberg@gmail.com")
	addPerson("Ramazan AYYILDIZ", "rayyildiz@gmail.com", "@rayyildiz")
	addPerson("Ramesh Dharan", "dharan@google.com")
	addPerson("Ramon Nogueira", "rmn@google.com")
	addPerson("Randy Reddig", "ydnar@shaderlab.com")
	addPerson("Raph Levien", "raph@google.com", "@raphlinus")
	addPerson("Raphael Geronimi", "raphael.geronimi@gmail.com", "@rgeronimi")
	addPerson("Raul Silvera", "rauls5382@gmail.com")
	addPerson("Raul Silvera", "rsilvera@google.com", "10031@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Raul Silvera", "rsilvera@google.com", "@rauls5382")
	addPerson("RaviTeja", "ravi.tezu@gmail.com")
	addPerson("Ray Tung", "rtung@thoughtworks.com", "@raytung")
	addPerson("Raymond Kazlauskas", "raima220@gmail.com", "@Rhymond")
	addPerson("Rebecca Stambler", "rstambler@golang.org", "16140@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Rebecca Stambler", "rstambler@golang.org", "@stamblerre")
	addPerson("Reese Wilson", "reese@shinymayhem.com")
	addPerson("Reilly Watson", "reillywatson@gmail.com", "@reillywatson")
	addPerson("Reinaldo de Souza Jr", "juniorz@gmail.com", "@juniorz")
	addPerson("Remi Gillig", "remigillig@gmail.com", "@speps")
	addPerson("Rens Rikkerink", "ikkerens@users.noreply.github.com", "@ikkerens")
	addPerson("Rhett Garber", "rhettg@gmail.com")
	addPerson("Rhys Hiltner", "rhys@justin.tv", "9210@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Rhys Hiltner", "rhys@justin.tv", "@rhysh")
	addPerson("Ricardo Padilha", "ricardospadilha@gmail.com", "@ricardopadilha")
	addPerson("Ricardo Rey", "rrey@google.com")
	addPerson("Ricardo Smania", "ricsmania@gmail.com")
	addPerson("Ricardo Vegas", "ricardovegas@gmail.com")
	addPerson("Riccardo Paccagnella", "ricpacca@gmail.com")
	addPerson("Richard Barnes", "rlb@ipv.sx")
	addPerson("Richard Crowley", "r@rcrowley.org", "@rcrowley")
	addPerson("Richard Dingwall", "rdingwall@gmail.com", "@rdingwall")
	addPerson("Richard Eric Gavaletz", "gavaletz@gmail.com", "@gavaletz")
	addPerson("Richard Gibson", "richard.gibson@gmail.com", "13081@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Richard Gibson", "richard.gibson@gmail.com", "@gibson042")
	addPerson("Richard Miller", "millerresearch@gmail.com", "12217@62eb7196-b449-3ce5-99f1-c037f21e1705", "miller.research@gmail.com")
	addPerson("Richard Musiol", "mail@richard-musiol.de", "@neelance", "neelance@gmail.com", "13620@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Rick Arnold", "rickarnoldjr@gmail.com", "@rickar")
	addPerson("Rick Hudson", "rlh@golang.org", "5186@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Rick Hudson", "rlh@golang.org", "@RLH")
	addPerson("Rick Sayre", "whorfin@gmail.com")
	addPerson("Rijnard van Tonder", "rvantonder@gmail.com", "@rvantonder")
	addPerson("Rik van der Heijden", "rikvdh@users.noreply.github.com")
	addPerson("Riku Voipio", "riku.voipio@linaro.org", "12765@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Riku Voipio", "riku.voipio@linaro.org", "@suihkulokki")
	addPerson("Risto Jaakko Saarelma", "rsaarelm@gmail.com", "@rsaarelm")
	addPerson("Rob Bradford", "robert.bradford@intel.com")
	addPerson("Rob Earhart", "earhart@google.com")
	addPerson("Rob Phoenix", "rob@robphoenix.com", "17946@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Rob Phoenix", "rob@robphoenix.com", "@robphoenix")
	addPerson("Rob Pike", "r@golang.org", "5015@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Rob Pike", "r@golang.org", "@robpike")
	addPerson("Rob Strong", "robert.a.strong@gmail.com")
	addPerson("Robert Anthony Bellamy", "rabellamy@gmail.com")
	addPerson("Robert Bittle", "guywithnose@gmail.com")
	addPerson("Robert Figueiredo", "robfig@gmail.com", "@robfig")
	addPerson("Robert Gogolok", "gogolok@gmail.com")
	addPerson("Robert Griesemer", "gri@golang.org", "5210@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Robert Griesemer", "gri@golang.org", "gri@gri-macbookair.roam.corp.google.com", "@griesemer")
	addPerson("Robert Hencke", "robert.hencke@gmail.com", "@rhencke")
	addPerson("Robert Iannucci", "iannucci@google.com")
	addPerson("Robert Kuska", "rkuska@gmail.com")
	addPerson("Robert Obryk", "robryk@gmail.com", "@robryk")
	addPerson("Robert Sesek", "rsesek@google.com", "@rsesek")
	addPerson("Robert Stepanek", "robert.stepanek@gmail.com", "6062@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Robert Stepanek", "robert.stepanek@gmail.com", "@rsto")
	addPerson("Robert Weber", "robertweber95@gmail.com")
	addPerson("Robert Xu", "robxu9@gmail.com")
	addPerson("Robert-André Mauchin", "zebob.m@gmail.com")
	addPerson("Roberto Clapis", "robclap8@gmail.com", "@empijei")
	addPerson("Roberto Lublinerman Reitzes", "rluble@google.com", "@rluble")
	addPerson("Roberto Selbach", "roberto@selbach.ca")
	addPerson("Roberto Selbach", "roberto@selbach.ca", "@robteix")
	addPerson("Roberto", "empijei@users.noreply.github.com", "@empijei")
	addPerson("Robin Eklind", "r.eklind.87@gmail.com")
	addPerson("Rodolfo Carvalho", "rhcarvalho@gmail.com", "@rhcarvalho")
	addPerson("Rodolfo Rodriguez", "rodolfobgibson@gmail.com", "@techmexdev")
	addPerson("Rodrigo Moraes de Oliveira", "rodrigo.moraes@gmail.com", "@moraes")
	addPerson("Rodrigo Rafael Monti Kochenburger", "divoxx@gmail.com", "@divoxx")
	addPerson("Roger Guldbrandsen", "roger@kinbiko.com")
	addPerson("Roger Pau Monné", "royger@gmail.com", "@royger")
	addPerson("Roger Peppe", "rogpeppe@gmail.com", "@rogpeppe")
	addPerson("Roger Simms", "roger.simms@gmail.com")
	addPerson("Rohit Agarwal", "agarwalrohit@google.com")
	addPerson("Roland Illig", "roland.illig@gmx.de")
	addPerson("Roland Illig", "roland.illig@gmx.de", "@rillig")
	addPerson("Roland Shoemaker", "rolandshoemaker@gmail.com", "12545@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Roland Shoemaker", "rolandshoemaker@gmail.com", "@rolandshoemaker")
	addPerson("Rollie Ma", "rollie.ma@gmail.com")
	addPerson("Roman Budnikov", "romanyx90@yandex.ru", "23999@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ron Hashimoto", "mail@h2so5.net", "@h2so5")
	addPerson("Ronald G. Minnich", "rminnich@gmail.com", "@rminnich")
	addPerson("Ronan Guilloux", "ronan.guilloux@gmail.com")
	addPerson("Ross Chater", "rdchater@gmail.com", "@rdcx")
	addPerson("Ross Light", "light@google.com", "8285@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ross Light", "light@google.com", "@zombiezen")
	addPerson("Rowan Marshall", "rowanajmarshall@gmail.com", "@RowanAJMarshall")
	addPerson("Rowan Worth", "sqweek@gmail.com", "@sqweek")
	addPerson("Ruben Vermeersch", "ruben@rocketeer.be")
	addPerson("Rudi Kramer", "rudi.kramer@gmail.com")
	addPerson("Rui Ueyama", "ruiu@google.com", "@rui314")
	addPerson("Ruslan Nigmatullin", "elessar@dropbox.com")
	addPerson("Russ Cox", "rsc@golang.org", "5056@62eb7196-b449-3ce5-99f1-c037f21e1705", "@rsc")
	addPerson("Russell Haering", "russellhaering@gmail.com", "@russellhaering")
	addPerson("Ryan Boehning", "ryan.boehning@apcera.com", "@y0ssar1an")
	addPerson("Ryan Brown", "ribrdb@google.com", "6136@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ryan Brown", "ribrdb@google.com", "@ribrdb")
	addPerson("Ryan Canty", "jrcanty@gmail.com")
	addPerson("Ryan Dahl", "ry@tinyclouds.org", "@ry")
	addPerson("Ryan Hitchman", "hitchmanr@gmail.com", "@rmmh")
	addPerson("Ryan Lower", "rpjlower@gmail.com", "@ryanlower")
	addPerson("Ryan Seys", "ryan@ryanseys.com", "@ryanseys")
	addPerson("Ryan Slade", "ryanslade@gmail.com", "@ryanslade")
	addPerson("Ryan Thomas", "rnt@google.com")
	addPerson("Ryan Zhang", "ryan.zhang@docker.com")
	addPerson("Ryoichi KATO", "ryo1kato@gmail.com", "@ryo1kato")
	addPerson("Ryuji Iwata", "qt.luigi@gmail.com", "@qt-luigi")
	addPerson("Ryuma Yoshida", "ryuma.y1117@gmail.com", "@ryysud")
	addPerson("Ryuzo Yamamoto", "ryuzo.yamamoto@gmail.com", "@dragon3")
	addPerson("Rémy Oudompheng", "oudomphe@phare.normalesup.org", "@remyoudompheng")
	addPerson("S.Çağlar Onur", "caglar@10ur.org", "@caglar10ur")
	addPerson("Sabin Mihai Rapan", "sabin.rapan@gmail.com", "@sabin-rapan")
	addPerson("Sagiv Ofek", "sagiv4@gmail.com")
	addPerson("Sai Cheemalapati", "saicheems@google.com")
	addPerson("Salman Aljammaz", "s@0x65.net", "5220@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Salman Aljammaz", "s@0x65.net", "@saljam")
	addPerson("Sam Broughton", "sambroughton@hotmail.co.uk")
	addPerson("Sam Hug", "samuel.b.hug@gmail.com", "@samuelhug")
	addPerson("Sam Mulube", "sam@thingful.net")
	addPerson("Sam Sendelbach", "sbsends@gmail.com")
	addPerson("Sam Thorogood", "thorogood@google.com", "@samthor")
	addPerson("Sam Whited", "sam@samwhited.com", "11106@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Sam Whited", "sam@samwhited.com", "@SamWhited")
	addPerson("Saman Barghi", "saman.b@gmail.com")
	addPerson("Sameer Ajmani", "sameer@golang.org", "5265@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Sameer Ajmani", "sameer@golang.org", "@Sajmani")
	addPerson("Sami Commerot", "samic@google.com")
	addPerson("Sami Pönkänen", "sami.ponkanen@gmail.com")
	addPerson("Samuel Cochran", "sj26@sj26.com")
	addPerson("Samuel Kelemen", "sckelemen@users.noreply.github.com", "@SCKelemen")
	addPerson("Samuel Tan", "samueltan@google.com", "16020@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Samuel Tan", "samueltan@google.com", "@stjj89")
	addPerson("Samuele Pedroni", "pedronis@lucediurna.net", "@pedronis")
	addPerson("Sandip Bhattacharya", "sandipb@sandipb.net")
	addPerson("Sandy McPherson", "sandyzwin6@gmail.com")
	addPerson("Sandy", "openset.wang@gmail.com", "@openset")
	addPerson("Sanjay Menakuru", "balasanjay@gmail.com", "@balasanjay")
	addPerson("Santhosh Kumar Tekuri", "santhosh.tekuri@gmail.com", "@santhosh-tekuri")
	addPerson("Santosh Ananthakrishnan", "santosh@dropbox.com")
	addPerson("Sarah Adams", "shadams@google.com", "16850@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Sarah Adams", "shadams@google.com", "shadams@shadams0.mtv.corp.google.com", "@adams-sarah")
	addPerson("Sarah Chacko", "SJC1982.1992@gmail.com")
	addPerson("Sascha Brawer", "sascha@brawer.ch", "@brawer")
	addPerson("Sasha Lionheart", "lionhearts@google.com")
	addPerson("Satyajit Ranjeev", "s@ranjeev.in")
	addPerson("Sawood Alam", "ibnesayeed@gmail.com")
	addPerson("Scott Bell", "scott@sctsm.com", "13380@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Scott Bell", "scott@sctsm.com", "@sctb")
	addPerson("Scott Crunkleton", "crunk1@gmail.com")
	addPerson("Scott Ferguson", "scottwferg@gmail.com", "@scottferg")
	addPerson("Scott Lawrence", "bytbox@gmail.com", "@bytbox")
	addPerson("Scott Mansfield", "smansfield@netflix.com", "@ScottMansfield")
	addPerson("Scott Pakin", "scott+gpg@pakin.org")
	addPerson("Sean Chittenden", "seanc@joyent.com", "@sean-")
	addPerson("Sean Christopherson", "sean.j.christopherson@intel.com", "@sean-jc")
	addPerson("Sean Dolphin", "Sean.Dolphin@kpcompass.com")
	addPerson("Sean Harger", "sharger@google.com")
	addPerson("Sean Rees", "sean@erifax.org", "@seanrees")
	addPerson("SeanBurford", "sburford@google.com", "@sburford")
	addPerson("Sebastiaan van Stijn", "github@gone.nl", "@thaJeztah")
	addPerson("Sebastian Schmidt", "mrschmidt@google.com")
	addPerson("Sebastian Schuberth", "sschuberth@gmail.com")
	addPerson("Sebastian Willing", "sewi.de@gmail.com")
	addPerson("Sebastien Binet", "binet@cern.ch")
	addPerson("Sebastien Binet", "seb.binet@gmail.com", "5810@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Sebastien Binet", "seb.binet@gmail.com", "@sbinet")
	addPerson("Seebs", "seebs@sourcegraph.com", "@seebs")
	addPerson("Seiji Takahashi", "timaki.st@gmail.com", "15570@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Seiji Takahashi", "timaki.st@gmail.com", "@timakin")
	addPerson("Sergey 'SnakE' Gromov", "snake.scaly@gmail.com", "@snake-scaly")
	addPerson("Sergey Frolov", "sfrolov@google.com")
	addPerson("Sergey Lemeshkin", "sergeilem@gmail.com")
	addPerson("Sergey Lukjanov", "me@slukjanov.name")
	addPerson("Sergey Mishin", "sergeymishine@gmail.com", "@dartkron")
	addPerson("Sergey Mudrik", "sergey.mudrik@gmail.com")
	addPerson("Sergey Rogulenko", "rogulenko@laserlike.com")
	addPerson("Sergey Semin", "gray12511@gmail.com", "@Gray5")
	addPerson("Sergiusz Bazanski", "bazanski@gmail.com")
	addPerson("Serhat Şevki Dinçer", "jfcgauss@gmail.com")
	addPerson("Serhii Aheienko", "serhii.aheienko@gmail.com")
	addPerson("Serhii Bratus", "sergiibratus@gmail.com")
	addPerson("Seshachalam Malisetti", "abbiya@gmail.com")
	addPerson("Seth Greenstein", "sgreens@google.com")
	addPerson("Seth Hoenig", "seth.a.hoenig@gmail.com", "@shoenig")
	addPerson("Seth Hollyman", "shollyman@google.com")
	addPerson("Seth Shelnutt", "Shelnutt2@gmail.com")
	addPerson("Seth Vargo", "sethvargo@gmail.com", "@sethvargo")
	addPerson("Shahar Kohanim", "skohanim@gmail.com")
	addPerson("Shahar Kohanim", "skohanim@gmail.com", "12700@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Shamil Garatuev", "garatuev@gmail.com", "@FluorescentTouch")
	addPerson("Shane Hansen", "shanemhansen@gmail.com", "@shanemhansen")
	addPerson("Shannon Wynter", "freman@users.noreply.github.com")
	addPerson("Shaun Dunning", "shaun.dunning@uservoice.com")
	addPerson("Shawn Pearce", "sop@google.com")
	addPerson("Shawn Smith", "shawn.p.smith@gmail.com", "7245@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Shawn Walker-Salas", "shawn.walker@oracle.com", "7291@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Shawn Walker-Salas", "shawn.walker@oracle.com", "@binarycrusader")
	addPerson("Shenghou Ma", "minux@golang.org", "minux.ma@gmail.com", "@minux")
	addPerson("Shengyu Zhang", "shengyu.zhang@chaitin.com", "@SilverRainZ")
	addPerson("Shi Han Ng", "shihanng@gmail.com")
	addPerson("Shinichi Nishimura", "nshmura.s@gmail.com")
	addPerson("Shinji Tanaka", "shinji.tanaka@gmail.com", "@stanaka")
	addPerson("Shintaro Kaneko", "kaneshin0120@gmail.com", "@kaneshin")
	addPerson("Shivakumar GN", "shivakumar.gn@gmail.com", "@shivakumargn")
	addPerson("Shivansh Rai", "shivansh@freebsd.org", "@shivansh")
	addPerson("Shubheksha Jalan", "jshubheksha@gmail.com")
	addPerson("Shun Fan", "sfan@google.com")
	addPerson("Shushan Chai", "chaishushan@gmail.com", "5095@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Shuvo Debnath", "shuvo.debnath@gmail.com")
	addPerson("Silvan Jegen", "s.jegen@gmail.com", "@Shugyousha")
	addPerson("Simon Inman", "simoninman@google.com")
	addPerson("Simon Jefford", "simon.jefford@gmail.com", "@simonjefford")
	addPerson("Simon Johansson", "simon@simonjohansson.com")
	addPerson("Simon Ordish", "simon.ordish@masagi.co.uk")
	addPerson("Simon Rawet", "simon@rawet.se", "@KilledKenny")
	addPerson("Simon Thulbourn", "simon+github@thulbourn.com", "@sthulb")
	addPerson("Simon Whitehead", "chemnova@gmail.com", "@simon-whitehead")
	addPerson("Simone Carletti", "weppos@gmail.com")
	addPerson("Sina Siadat", "siadat@gmail.com", "14140@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Sina Siadat", "siadat@gmail.com", "@siadat")
	addPerson("Sokolov Yura", "funny.falcon@gmail.com", "@funny-falcon")
	addPerson("Song Gao", "song@gao.io", "@songgao")
	addPerson("Spencer Nelson", "s@spenczar.com", "10000@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Spencer Nelson", "s@spenczar.com", "@spenczar")
	addPerson("Spencer Tung", "spencertung@google.com", "20245@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Spring Mc", "heresy.mc@gmail.com", "@mcspring")
	addPerson("Srdjan Petrovic", "spetrovic@google.com", "6605@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Srdjan Petrovic", "spetrovic@google.com", "@spetrovic77")
	addPerson("Sridhar Venkatakrishnan", "sridhar@laddoo.net", "9665@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Sridhar Venkatakrishnan", "sridhar@laddoo.net", "@sridharv")
	addPerson("StalkR", "stalkr@stalkr.net")
	addPerson("Stan Chan", "stanchan@gmail.com")
	addPerson("Stan Schwertly", "stan@schwertly.com", "@Stantheman")
	addPerson("Stanislav Afanasev", "php.progger@gmail.com", "@superstas")
	addPerson("Stanislav Paskalev", "kshorg@gmail.com")
	addPerson("Stanislav Petrov", "s.e.petrov@gmail.com")
	addPerson("Steeve Morin", "steeve.morin@gmail.com", "@steeve")
	addPerson("Stefan Schmidt", "stschmidt@google.com")
	addPerson("Stepan Shabalin", "neverliberty@gmail.com", "@Neverik")
	addPerson("Stephan Renatus", "srenatus@chef.io", "@srenatus")
	addPerson("Stephen Gutekanst", "stephen.gutekanst@gmail.com")
	addPerson("Stephen L", "36011612+steuhs@users.noreply.github.com", "@steuhs")
	addPerson("Stephen Lewis", "stephen@sock.org.uk")
	addPerson("Stephen McQuay (smcquay)", "stephen@mcquay.me", "@smcquay")
	addPerson("Stephen McQuay", "stephen@mcquay.me", "13960@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Stephen Searles", "stephens2424@gmail.com", "stephen.searles@gmail.com", "@stephens2424")
	addPerson("Stephen Solka", "stephen0q@gmail.com")
	addPerson("Stephen Sugden", "glurgle@gmail.com")
	addPerson("Stephen Weinberg", "stephen@q5comm.com", "stephenmw@google.com", "@stephenmw")
	addPerson("Stephen Weinberg", "stephenmw@google.com", "13156@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Steve Francia", "spf@golang.org", "14840@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Steve Francia", "spf@golang.org", "@spf13")
	addPerson("Steve Gilbert", "stevegilbert23@gmail.com")
	addPerson("Steve McCoy", "mccoyst@gmail.com", "@mccoyst")
	addPerson("Steve Phillips", "steve@tryingtobeawesome.com", "@elimisteve")
	addPerson("Steve Reed", "sreed@zulily.com")
	addPerson("Steve Streeting", "steve@stevestreeting.com", "@sinbad")
	addPerson("Steve Wills", "steve@mouf.net")
	addPerson("Steven Berlanga", "zabawaba99@gmail.com")
	addPerson("Steven Buss", "sbuss@google.com")
	addPerson("Steven Elliot Harris", "seharris@gmail.com", "@seh")
	addPerson("Steven Erenst", "stevenerenst@gmail.com")
	addPerson("Steven Hartland", "steven.hartland@multiplay.co.uk", "10210@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Steven Hartland", "steven.hartland@multiplay.co.uk", "@stevenh")
	addPerson("Steven Kabbes", "stevenkabbes@gmail.com")
	addPerson("Steven Ruckdashel", "steve.ruckdashel@gmail.com")
	addPerson("Steven Selph", "sselph@google.com")
	addPerson("Steven Wilkin", "stevenwilkin@gmail.com", "@stevenwilkin")
	addPerson("Stéphane Travostino", "stephane.travostino@gmail.com", "@1player")
	addPerson("Sue Spence", "virtuallysue@gmail.com")
	addPerson("Sugu Sougoumarane", "ssougou@gmail.com", "@sougou")
	addPerson("Suharsh Sivakumar", "suharshs@google.com", "@suharshs")
	addPerson("Suriyaa Sundararuban", "isc.suriyaa@gmail.com")
	addPerson("Suriyaa Sundararuban", "suriyaasundararuban@gmail.com", "27899@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Suriyaa Sundararuban", "suriyaasundararuban@gmail.com", "@SuriyaaKudoIsc")
	addPerson("Surma Surma", "surma@google.com")
	addPerson("Sutton Yamanashi", "syamanashi@gmail.com")
	addPerson("Suyash", "dextrous93@gmail.com", "15015@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Suyash", "dextrous93@gmail.com", "@suyash")
	addPerson("Suzy Mueller", "suzmue@golang.org", "21300@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Suzy Mueller", "suzmue@golang.org", "@suzmue")
	addPerson("Sven Almgren", "sven@tras.se", "@blindmatrix")
	addPerson("Sven Blumenstein", "svbl@google.com")
	addPerson("Sven Dowideit", "svendowideit@home.org.au")
	addPerson("Sylvain Zimmer", "sylvain@sylvainzimmer.com", "@sylvinus")
	addPerson("Syohei YOSHIDA", "syohex@gmail.com", "@syohex")
	addPerson("Sébastien Paolacci", "sebastien.paolacci@gmail.com", "@spaolacci")
	addPerson("Sébastien Portebois", "sportebois@gmail.com")
	addPerson("TSUYUSATO Kitsune", "make.just.on@gmail.com")
	addPerson("Tad Fisher", "tadfisher@gmail.com")
	addPerson("Tad Glines", "tad.glines@gmail.com", "@tadglines")
	addPerson("Taesu Pyo", "pyotaesu@gmail.com", "@bigflood")
	addPerson("Tair Sabirgaliev", "tair.sabirgaliev@gmail.com")
	addPerson("Taj Khattra", "taj.khattra@gmail.com", "@tkhattra")
	addPerson("Takashi Matsuo", "tmatsuo@google.com")
	addPerson("Takayoshi Nishida", "takayoshi.nishida@gmail.com", "@takp")
	addPerson("Takuto Ikuta", "tikuta@google.com", "@atetubou")
	addPerson("Takuya Sato", "takuya0219@gmail.com")
	addPerson("Takuya Ueda", "uedatakuya@gmail.com", "@tenntenn")
	addPerson("Tal Shprecher", "tshprecher@gmail.com", "11915@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Tal Shprecher", "tshprecher@gmail.com", "@tshprecher")
	addPerson("Tamir Duberstein", "tamird@gmail.com", "7955@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Tamir Duberstein", "tamird@gmail.com", "@tamird")
	addPerson("Tamás Gulácsi", "tgulacsi78@gmail.com")
	addPerson("Tao Wang", "twang2218@gmail.com")
	addPerson("Tardis Xu", "xiaoxubeii@gmail.com")
	addPerson("Tarmigan Casebolt", "tarmigan@gmail.com")
	addPerson("Tarmigan Casebolt", "tarmigan@gmail.com", "9697@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Taro Aoki", "aizu.s1230022@gmail.com", "@ktr0731")
	addPerson("Tarrant", "tarrant@keyneston.com", "@tarrant")
	addPerson("Taru Karttunen", "taruti@taruti.net", "@taruti")
	addPerson("Tatsuhiro Tsujikawa", "tatsuhiro.t@gmail.com", "@tatsuhiro-t")
	addPerson("Taufiq Rahman", "taufiqrx8@gmail.com", "@Inconnu08")
	addPerson("Ted Hahn", "teh@uber.com")
	addPerson("Ted Kornish", "golang@tedkornish.com", "@tedkornish")
	addPerson("Tejasvi Nareddy", "tejunareddy@gmail.com")
	addPerson("Terin Stock", "terinjokes@gmail.com", "25203@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Terin Stock", "terinjokes@gmail.com", "@terinjokes")
	addPerson("Terrel Shumway", "gopher@shumway.us")
	addPerson("Terry Wong", "terry.wong2@yahoo.com")
	addPerson("Tess Rinearson", "tess.rinearson@gmail.com")
	addPerson("Tetsuo Kiso", "tetsuokiso9@gmail.com", "@tetsuok")
	addPerson("Than McIntosh", "thanm@google.com", "14020@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Than McIntosh", "thanm@google.com", "@thanm")
	addPerson("Thanabodee Charoenpiriyakij", "wingyminus@gmail.com", "19095@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Thanabodee Charoenpiriyakij", "wingyminus@gmail.com", "@wingyplus")
	addPerson("Theo Schlossnagle", "jesus@lethargy.org")
	addPerson("Thiago Farina", "tfarina@chromium.org")
	addPerson("Thiago Fransosi Farina", "thiago.farina@gmail.com", "@thiagofarina")
	addPerson("Thibault Falque", "thibault_falque@ens.univ-artois.fr")
	addPerson("Thibaut Colar", "tcolar@colar.net")
	addPerson("Thomas Alan Copeland", "talan.copeland@gmail.com", "@talanc")
	addPerson("Thomas Bonfort", "thomas.bonfort@gmail.com", "@tbonfort")
	addPerson("Thomas Bouldin", "inlined@google.com")
	addPerson("Thomas Bruyelle", "thomas.bruyelle@gmail.com", "@tbruyelle")
	addPerson("Thomas Bushnell, BSG", "tbushnell@google.com")
	addPerson("Thomas Desrosiers", "thomasdesr@gmail.com", "@thomaso-mirodin")
	addPerson("Thomas Habets", "habets@google.com", "@ThomasHabets")
	addPerson("Thomas Johnson", "NTmatter@gmail.com")
	addPerson("Thomas Kappler", "tkappler@gmail.com", "@thomas11")
	addPerson("Thomas Meson", "zllak@hycik.org")
	addPerson("Thomas Sauvaget", "sauvaget.thomas@gmail.com")
	addPerson("Thomas Wanielista", "tomwans@gmail.com", "@tomwans")
	addPerson("Thomas de Zeeuw", "thomasdezeeuw@gmail.com", "@Thomasdezeeuw")
	addPerson("Thorben Krueger", "thorben.krueger@gmail.com", "@benthor")
	addPerson("Thordur Bjornsson", "thorduri@secnorth.net", "@thorduri")
	addPerson("Tiago Queiroz", "contato@tiago.eti.br")
	addPerson("Tilman Dilo", "tilman.dilo@gmail.com", "@tdilo")
	addPerson("Tim 'mithro' Ansell", "tansell@google.com")
	addPerson("Tim Burks", "timburks@google.com")
	addPerson("Tim Cooijmans", "timcooijmans@gmail.com", "@timcooijmans")
	addPerson("Tim Cooper", "tim.cooper@layeh.com", "24935@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Tim Cooper", "tim.cooper@layeh.com", "@bontibon")
	addPerson("Tim Ebringer", "tim.ebringer@gmail.com")
	addPerson("Tim Heckman", "t@heckman.io", "@theckman")
	addPerson("Tim Henderson", "tim.tadh@gmail.com", "@timtadh")
	addPerson("Tim Hockin", "thockin@google.com", "@thockin")
	addPerson("Tim Shen", "timshen@google.com", "@timshen91")
	addPerson("Tim St. Clair", "stclair@google.com")
	addPerson("Tim Swast", "swast@google.com", "@tswast")
	addPerson("Tim Wright", "tenortim@gmail.com", "25424@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Tim Wright", "tenortim@gmail.com", "@tenortim")
	addPerson("Tim Xu", "xiaoxubeii@gmail.com", "@xiaoxubeii")
	addPerson("Tim", "tdhutt@gmail.com", "@Timmmm")
	addPerson("Timo Savola", "timo.savola@gmail.com", "@tsavola")
	addPerson("Timothy Raymond", "xtjraymondx@gmail.com")
	addPerson("Timothy Studd", "tim@timstudd.com", "@timstudd")
	addPerson("Tipp Moseley", "tipp@google.com", "@tippjammer")
	addPerson("Tobias Assarsson", "tobias.assarsson@gmail.com")
	addPerson("Tobias Columbus", "tobias.columbus@gmail.com", "@tc-0")
	addPerson("Tobias Klauser", "tklauser@distanz.ch", "@tklauser")
	addPerson("Tobias Klauser", "tobias.klauser@gmail.com", "19560@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Tobias Klauser", "tobias.klauser@gmail.com", "@tklauser")
	addPerson("Tobias Schottdorf", "tobias.schottdorf@gmail.com")
	addPerson("Toby Burress", "kurin@google.com")
	addPerson("Todd Neal", "todd@tneal.org", "12836@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Todd Neal", "todd@tneal.org", "@tzneal")
	addPerson("Todd Neal", "tolchz@gmail.com", "8481@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Todd Rafferty", "webRat@gmail.com")
	addPerson("Todd Wang", "toddwang@gmail.com", "@tatatodd")
	addPerson("Tom Bergan", "tombergan@google.com", "10820@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Tom Bergan", "tombergan@google.com", "@tombergan")
	addPerson("Tom Elliott", "tom.w.elliott@gmail.com")
	addPerson("Tom Heng", "zhm20070928@gmail.com", "7380@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Tom Heng", "zhm20070928@gmail.com", "@tomheng")
	addPerson("Tom Holmes", "tom@wandb.com")
	addPerson("Tom Lanyon", "tomlanyon@google.com", "@tomlanyon")
	addPerson("Tom Levy", "tomlevy93@gmail.com", "@tom93")
	addPerson("Tom Limoncelli", "tal@whatexit.org", "@TomOnTime")
	addPerson("Tom Linford", "tomlinford@gmail.com", "@tomlinford")
	addPerson("Tom Thorogood", "me+google@tomthorogood.co.uk")
	addPerson("Tom Thorogood", "me+google@tomthorogood.co.uk", "@tmthrgd")
	addPerson("Tom Wilkie", "tom.wilkie@gmail.com", "tom@weave.works", "@tomwilkie")
	addPerson("Tomas Basham", "tomasbasham@gmail.com")
	addPerson("Tommy Schaefer", "tommy.schaefer@teecom.com", "@tommyschaefer")
	addPerson("Tonis Tiigi", "tonistiigi@gmail.com", "@tonistiigi")
	addPerson("Tony Reix", "Tony.Reix@bull.net", "16326@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Tony Walker", "walkert.uk@gmail.com", "@walkert")
	addPerson("Tooru Takahashi", "tooru.takahashi134@gmail.com", "@tooru")
	addPerson("Tor Andersson", "tor.andersson@gmail.com", "@ccxvii")
	addPerson("Tormod Erevik Lea", "tormodlea@gmail.com", "@tormoder")
	addPerson("Toshiki Shima", "haya14busa@gmail.com", "16861@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Totoro W", "tw19881113@gmail.com", "5975@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Travis Beatty", "travisby@gmail.com")
	addPerson("Travis Bischel", "travis.bischel@gmail.com", "26898@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Travis Bischel", "travis.bischel@gmail.com", "@twmb")
	addPerson("Travis Cline", "travis.cline@gmail.com", "@tmc")
	addPerson("Trevor Prater", "trevor.prater@gmail.com")
	addPerson("Trey Lawrence", "lawrence.trey@gmail.com", "@TreyLawrence")
	addPerson("Tristan Colgate", "tcolgate@gmail.com", "@tcolgate")
	addPerson("Tristan Ooohry", "ooohry@gmail.com", "@golantrevize")
	addPerson("Tristan Rice", "rice@fn.lc")
	addPerson("Troels Thomsen", "troels@thomsen.io", "@tt")
	addPerson("Trung Nguyen", "trung.n.k@gmail.com")
	addPerson("Tugdual Saunier", "tucksaun@gmail.com", "23797@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Tugdual Saunier", "tugdual.saunier@gmail.com")
	addPerson("Tuo Shan", "shantuo@google.com", "12855@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Tuo Shan", "shantuo@google.com", "@shantuo")
	addPerson("Tuo Shan", "sturbo89@gmail.com")
	addPerson("Tuo Shan", "sturbo89@gmail.com", "12857@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Tw", "tw19881113@gmail.com", "@tw4452852")
	addPerson("Tyler Bui-Palsulich", "tbp@google.com")
	addPerson("Tyler Bui-Palsulich", "tpalsulich@google.com", "@tbpg")
	addPerson("Tyler Bunnell", "tylerbunnell@gmail.com", "@tylerb")
	addPerson("Tyler Compton", "xaviosx@gmail.com")
	addPerson("Tyler Treat", "ttreat31@gmail.com")
	addPerson("Tyler Treat", "tyler.treat@apcera.com")
	addPerson("Tyler Yahn", "tyler.yahn@urbanairship.com")
	addPerson("Tzu-Jung Lee", "roylee17@currant.com", "@roylee17")
	addPerson("Ugorji Nwoke", "ugorji@gmail.com", "@ugorji")
	addPerson("Ulderico Cirello", "uldericofilho@gmail.com", "7250@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ulrich Kunitz", "uli.kunitz@gmail.com", "@ulikunitz")
	addPerson("Umang Parmar", "umangjparmar@gmail.com", "@darkLord19")
	addPerson("Uriel Mangado", "uriel@berlinblue.org", "@uriel")
	addPerson("Urvil Patel", "patelurvil38@gmail.com", "@urvil38")
	addPerson("Uttam C Pawar", "uttam.c.pawar@intel.com", "@uttampawar")
	addPerson("Vadim Grek", "vadimprog@gmail.com", "@brainiac84")
	addPerson("Val Polouchkine", "vpolouch@justin.tv")
	addPerson("Valentin Vidic", "vvidic@valentin-vidic.from.hr", "@vvidic")
	addPerson("Vanesa", "mail@vanesaortiz.com")
	addPerson("Vega Garcia Luis Alfonso", "vegacom@gmail.com", "@vegacom")
	addPerson("Venil Noronha", "veniln@vmware.com", "@venilnoronha")
	addPerson("Veselkov Konstantin", "kostozyb@gmail.com", "@KosToZyB")
	addPerson("Viacheslav Poturaev", "vearutop@gmail.com", "@vearutop")
	addPerson("Vicki Niu", "vicki.niu@gmail.com")
	addPerson("Victor Chudnovsky", "vchudnov@google.com")
	addPerson("Victor Vrantchan", "vrancean+github@gmail.com", "@groob")
	addPerson("Vignesh Ramachandra", "vickyramachandra@gmail.com")
	addPerson("Vikas Kedia", "vikask@google.com")
	addPerson("Vikram Jadhav", "vikramcse.10@gmail.com")
	addPerson("Vince0000", "522341976@qq.com")
	addPerson("Vincent Batts", "vbatts@hashbangbash.com", "@vbatts")
	addPerson("Vincent Bernat", "vincent@bernat.ch")
	addPerson("Vincent Demeester", "vinc.demeester@gmail.com")
	addPerson("Vincent Vanackere", "vincent.vanackere@gmail.com", "@vanackere")
	addPerson("Vincenzo Pupillo", "v.pupillo@gmail.com", "24134@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Vinu Rajashekhar", "vinutheraj@gmail.com", "@vinuraja")
	addPerson("Vishvananda Ishaya", "vishvananda@gmail.com", "@vishvananda")
	addPerson("Vitor De Mario", "vitor.demario@mendelics.com.br")
	addPerson("Vitor De Mario", "vitordemario@gmail.com", "@vdemario")
	addPerson("Vivek Ayer", "vivek@restlessbandit.com")
	addPerson("Vivek Sekhar", "vivek@viveksekhar.ca")
	addPerson("Vlad Krasnov", "vlad@cloudflare.com", "7601@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Vlad Krasnov", "vlad@cloudflare.com", "@vkrasnov")
	addPerson("Vladimir Kovpak", "cn007b@gmail.com", "@cn007b")
	addPerson("Vladimir Kuzmin", "vkuzmin@uber.com", "26409@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Vladimir Kuzmin", "vkuzmin@uber.com", "@vkuzmin-uber")
	addPerson("Vladimir Mezentsev", "vladimir.mezentsev@oracle.com")
	addPerson("Vladimir Mihailenco", "vladimir.webdev@gmail.com", "@vmihailenco")
	addPerson("Vladimir Nikishenko", "vova616@gmail.com", "@vova616")
	addPerson("Vladimir Stefanovic", "vladimir.stefanovic@imgtec.com", "@vstefanovic")
	addPerson("Vladimir Stefanovic", "vladimir.stefanovic@mips.com", "15150@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Vladimir Varankin", "nek.narqo@gmail.com")
	addPerson("Vladimir Varankin", "vladimir@varank.in", "@narqo")
	addPerson("Volker Dobler", "dr.volker.dobler@gmail.com", "5050@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Volker Dobler", "dr.volker.dobler@gmail.com", "@vdobler")
	addPerson("Volodymyr Paprotski", "vpaprots@ca.ibm.com", "@vpaprots")
	addPerson("W. Mark Kubacki", "wmark@hurrikane.de")
	addPerson("W. Trevor King", "wking@tremily.us")
	addPerson("Wade Simmons", "wade@wades.im", "@wadey")
	addPerson("Waldemar Quevedo", "waldemar.quevedo@gmail.com")
	addPerson("Walter Poupore", "wpoupore@google.com")
	addPerson("Wander Lairson Costa", "wcosta@mozilla.com", "@walac")
	addPerson("Warren Fernandes", "warren.f.fernandes@gmail.com")
	addPerson("Warren Fernandes", "warren.f.fernandes@gmail.com", "@wfernandes")
	addPerson("Warren Harper", "warrenjharper@gmail.com")
	addPerson("Wayne Ashley Berry", "wayneashleyberry@gmail.com", "@wayneashleyberry")
	addPerson("Wedson Almeida Filho", "wedsonaf@google.com", "12200@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Wedson Almeida Filho", "wedsonaf@google.com", "@wedsonaf")
	addPerson("Weerasak Chongnguluam", "singpor@gmail.com")
	addPerson("Weerasak Chongnguluam", "singpor@gmail.com", "@iporsut")
	addPerson("Wei Fu", "fhfuwei@163.com")
	addPerson("Wei Guangjing", "vcc.163@gmail.com", "@wgj-zz")
	addPerson("Wei Xiao", "Wei.Xiao@arm.com", "16227@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Wei Xiao", "wei.xiao@arm.com", "@williamweixiao")
	addPerson("Weichao Tang", "tevic.tt@gmail.com")
	addPerson("Weichao Tang", "tevic.tt@gmail.com", "@Tevic")
	addPerson("Wembley G. Leach, Jr", "wembley.gl@gmail.com", "@wemgl")
	addPerson("Wes Widner", "kai5263499@gmail.com")
	addPerson("Wesley Hill", "hakobyte@gmail.com")
	addPerson("WhisperRain", "2516435583@qq.com", "@WhisperRain")
	addPerson("Wil Selwood", "wselwood@gmail.com")
	addPerson("Wil Selwood", "wselwood@gmail.com", "@wselwood")
	addPerson("Wilfried Teiken", "wteiken@google.com")
	addPerson("Will Beason", "willbeason@gmail.com", "@willbeason")
	addPerson("Will Bond", "will@wbond.net", "9815@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Will Chan", "willchan@google.com")
	addPerson("Will Faught", "will.faught@gmail.com", "@willfaught")
	addPerson("Will Madison", "wmadisonDev@GMail.com")
	addPerson("Will Morrow", "wmorrow.qdt@qualcommdatacenter.com")
	addPerson("Will Norris", "willnorris@google.com", "@willnorris")
	addPerson("Will Storey", "will@summercat.com", "@horgh")
	addPerson("Will", "willow.pine.2011@gmail.com")
	addPerson("Willem van der Schyff", "willemvds@gmail.com", "@willemvds")
	addPerson("William Chan", "willchan@chromium.org", "@willchan", "*goog")
	addPerson("William Chang", "mr.williamchang@gmail.com", "27627@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("William Orr", "will@worrbase.com", "@worr")
	addPerson("Wisdom Omuya", "deafgoat@gmail.com", "@deafgoat")
	addPerson("Wèi Cōngruì", "crvv.mail@gmail.com", "22895@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Wèi Cōngruì", "crvv.mail@gmail.com", "@crvv")
	addPerson("XAX", "xaxiclouddev@gmail.com")
	addPerson("Xargin", "cao1988228@163.com")
	addPerson("Xi Ruoyao", "xry23333@gmail.com")
	addPerson("Xia Bin", "snyh@snyh.org", "12161@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Xia Bin", "snyh@snyh.org", "@snyh")
	addPerson("Xing Xing", "mikespook@gmail.com", "@mikespook")
	addPerson("Xudong Zhang", "felixmelon@gmail.com")
	addPerson("Xudong Zheng", "7pkvm5aw@slicealias.com", "@xudongzheng")
	addPerson("Xuyang Kang", "xuyang@google.com")
	addPerson("Xuyang Kang", "xuyangkang@gmail.com", "@xuyangkang")
	addPerson("Yaacov Akiba Slama", "yaslama@gmail.com")
	addPerson("Yamagishi Kazutoshi", "ykzts@desire.sh")
	addPerson("Yann Hodique", "yhodique@google.com", "@sigma")
	addPerson("Yann Kerhervé", "yann.kerherve@gmail.com", "@yannk")
	addPerson("Yaron de Leeuw", "jarondl@google.com")
	addPerson("Yasha Bubnov", "girokompass@gmail.com")
	addPerson("Yasha Bubnov", "girokompass@gmail.com", "@ybubnov")
	addPerson("Yasser Abdolmaleki", "yasser@yasser.ca", "@spring1843")
	addPerson("Yasuharu Goto", "matope.ono@gmail.com", "8070@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Yasuharu Goto", "matope.ono@gmail.com", "@matope")
	addPerson("Yasuhiro MATSUMOTO", "mattn.jp@gmail.com", "5025@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Yasuhiro Matsumoto", "mattn.jp@gmail.com", "@mattn")
	addPerson("Yazen2017", "yazen.shunnar@gmail.com", "@yazsh")
	addPerson("Yestin", "ylh@pdx.edu", "@ylih")
	addPerson("Yesudeep Mangalapilly", "yesudeep@google.com", "@gorakhargosh")
	addPerson("Ying Zou", "xpzouying@gmail.com")
	addPerson("Yissakhar Z. Beck", "yissakhar.beck@gmail.com", "@DeedleFake")
	addPerson("Yogesh Desai", "er.yogeshdesai@gmail.com")
	addPerson("Yongjian Xu", "i3dmaster@gmail.com", "@i3d")
	addPerson("Yoon", "learder@gmail.com")
	addPerson("Yoshi Yamaguchi", "ymotongpoo@gmail.com")
	addPerson("Yoshiya Hinosawa", "stibium121@gmail.com")
	addPerson("Yoshiyuki Kanno", "nekotaroh@gmail.com", "@mocchira")
	addPerson("Yuki Yugui Sonoda", "yugui@google.com")
	addPerson("Yury Smolsky", "yury@smolsky.by", "26536@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Yury Smolsky", "yury@smolsky.by", "@ysmolsky")
	addPerson("Yusuke Kagiwada", "block.rxckin.beats@gmail.com", "@Jxck")
	addPerson("Yuusei Kuwana", "kuwana@kumama.org", "@kumama")
	addPerson("Yuval Pavel Zholkover", "paulzhol@gmail.com", "5781@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Yuval Pavel Zholkover", "paulzhol@gmail.com", "@paulzhol")
	addPerson("Yuwei Ba", "xiaobayuwei@gmail.com")
	addPerson("Yuya Kusakabe", "yuya.kusakabe@gmail.com")
	addPerson("Yves Junqueira", "yves.junqueira@gmail.com", "@nictuku")
	addPerson("ZZMarquis", "zhonglingjian3821@163.com", "@ZZMarquis")
	addPerson("Zac Bergquist", "zbergquist99@gmail.com", "9250@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Zac Bergquist", "zbergquist99@gmail.com", "@zmb3")
	addPerson("Zach Auclair", "zach101@gmail.com")
	addPerson("Zach Bintliff", "zbintliff@gmail.com", "@zbintliff")
	addPerson("Zach Gershman", "zachgersh@gmail.com", "6360@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Zachary  Madigan", "zachary.madigan@apollovideo.com")
	addPerson("Zachary Amsden", "zach@thundertoken.com")
	addPerson("Zachary Amsden", "zach@thundertoken.com", "@zamsden")
	addPerson("Zachary Gershman", "zgershman@pivotal.io")
	addPerson("Zachary Madigan", "zacharywmadigan@gmail.com", "25899@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Zachary Romero", "zacromero3@gmail.com")
	addPerson("Zacharya", "zacharya19@gmail.com")
	addPerson("Zaq? Wiedmann", "zaquestion@gmail.com")
	addPerson("Zero King", "l2d4y3@gmail.com")
	addPerson("Zev Goldstein", "zev.goldstein@gmail.com", "@zevdg")
	addPerson("Zezhou Yu", "ray.zezhou@gmail.com")
	addPerson("Zhang Qiang", "dotslash.lu@gmail.com")
	addPerson("Zhang Wei", "zhangwei198900@gmail.com")
	addPerson("Zheng Dayu", "davidzheng23@gmail.com", "@ceshihao")
	addPerson("Zheng Xu", "zheng.xu@arm.com")
	addPerson("Zheng Xu", "zheng.xu@arm.com", "@Zheng-Xu")
	addPerson("Zheng Yang", "zhengyang4k@gmail.com")
	addPerson("Zhengyu He", "hzy@google.com")
	addPerson("ZhiFeng Hu", "hufeng1987@gmail.com")
	addPerson("Zhongpeng Lin", "zplin@uber.com", "@linzhp")
	addPerson("Zhongwei Yao", "zhongwei.yao@arm.com", "@zhongweiy")
	addPerson("Zhou Peng", "p@ctriple.cn", "26955@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Zhou Peng", "p@ctriple.cn", "@ctriple")
	addPerson("Zhuo Meng", "mengzhuo1203@gmail.com", "7530@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("Ziad Hatahet", "hatahet@gmail.com", "@hatahet")
	addPerson("Zorion Arrizabalaga", "zorionk@gmail.com", "@zorion")
	addPerson("a.lukinykh", "a.lukinykh@xsolla.com")
	addPerson("abdul.mannan", "abdul.mannan@thirdbridge.com")
	addPerson("acoshift", "acoshift@gmail.com")
	addPerson("adrienpetel", "peteladrien@gmail.com", "@feliixx")
	addPerson("aecdanjun", "aeciodantasjunior@gmail.com", "@aecdanjun")
	addPerson("ajackura", "ajackura@localhost")
	addPerson("ajnirp", "ajnirp@users.noreply.github.com", "@ajnirp")
	addPerson("akushman", "zeusakm@gmail.com")
	addPerson("alexpantyukhin", "apantykhin@gmail.com", "@alexpantyukhin")
	addPerson("alkesh26", "alkesh26@gmail.com", "@alkesh26")
	addPerson("alokc", "alokkr1090@gmail.com")
	addPerson("alpha.wong", "alpha.wong@lalamove.com")
	addPerson("amandhora", "aman.usa07@gmail.com")
	addPerson("amirrezaask", "raskarpour@gmail.com")
	addPerson("anatoly techtonik", "techtonik@gmail.com")
	addPerson("andrew werner", "andrew@upthere.com")
	addPerson("andrey mirtchovski", "mirtchovski@gmail.com", "@mirtchovski")
	addPerson("andrius4669", "andrius4669@gmail.com", "@andrius4669")
	addPerson("andy", "andyjgarfield@gmail.com")
	addPerson("apoorvam", "app.apoorva@gmail.com")
	addPerson("areski", "areski@gmail.com", "@areski")
	addPerson("as", "as.utf8@gmail.com", "@as")
	addPerson("asgaines", "andrew.s.gaines@gmail.com")
	addPerson("avi", "hi@avi.im")
	addPerson("aviau", "alexandre@alexandreviau.net")
	addPerson("avsharapov", "analytics.kzn@gmail.com", "@avsharapov")
	addPerson("awaw fumin", "awawfumin@gmail.com", "@fumin")
	addPerson("ayanamist", "ayanamist@gmail.com", "@ayanamist")
	addPerson("azat", "kaumov.a.r@gmail.com", "@akaumov")
	addPerson("azretkenzhaliev", "azret.kenzhaliev@gmail.com")
	addPerson("bbrodriges", "bender.rodriges@gmail.com")
	addPerson("benjamin-rood", "bisr@icloud.com")
	addPerson("berkant ipek", "41230766+0xbkt@users.noreply.github.com", "@0xbkt")
	addPerson("bogem", "albertnigma@gmail.com", "@bogem")
	addPerson("bontequero", "bontequero@gmail.com", "@bontequero")
	addPerson("boreq", "boreq@sourcedrops.com")
	addPerson("buddhamagnet", "buddhamagnet@gmail.com")
	addPerson("c9s", "yoanlin93@gmail.com", "@c9s")
	addPerson("calerogers", "cale.rogers.m@gmail.com")
	addPerson("caosz", "cszznbb@gmail.com")
	addPerson("catatsuy", "m.ddotx.f@gmail.com", "@catatsuy")
	addPerson("cch123", "buaa.cch@gmail.com", "@cch123")
	addPerson("chanxuehong", "chanxuehong@gmail.com", "@chanxuehong")
	addPerson("christopher-henderson", "chris@chenderson.org", "@christopher-henderson")
	addPerson("cia-rana", "kiwamura0314@gmail.com", "@cia-rana")
	addPerson("closs", "the.cody.oss@gmail.com", "@codyoss")
	addPerson("conorbroderick", "cjayjayb@gmail.com")
	addPerson("cyacco", "cyacco@gmail.com")
	addPerson("dalyk", "dalyk@google.com")
	addPerson("danoscarmike", "danom@google.com")
	addPerson("datianshi", "dsz0111@gmail.com", "@datianshi")
	addPerson("dchenk", "dcherchenko@gmail.com", "@dchenk")
	addPerson("dechen-sherpa", "Dechen.Sherpa@dal.ca")
	addPerson("delioda", "delioda@consenteye.com")
	addPerson("diana ortega", "dicaormu@gmail.com")
	addPerson("diplozoon", "huyuumi.dev@gmail.com", "@JohnTitor")
	addPerson("djherbis", "djherbis@gmail.com", "@djherbis")
	addPerson("dsivalingam", "dayansivalingam@gmail.com")
	addPerson("dupoxy", "dupoxy@users.noreply.github.com", "@dupoxy")
	addPerson("elmar", "ktye78@gmail.com")
	addPerson("elpinal", "6elpinal@gmail.com", "@elpinal")
	addPerson("emersion", "contact@emersion.fr")
	addPerson("epkann", "epkann@gmail.com")
	addPerson("erdi", "erdi@google.com")
	addPerson("eric fang", "eric.fang@arm.com", "24534@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("erifan01", "eric.fang@arm.com", "@erifan")
	addPerson("esell", "eujon.sellers@gmail.com")
	addPerson("esell", "eujon.sellers@gmail.com", "@esell")
	addPerson("fannie zhang", "Fannie.Zhang@arm.com", "21345@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("fanzha02", "fannie.zhang@arm.com", "@zhangfannie")
	addPerson("feilengcui008", "feilengcui008@gmail.com", "@feilengcui008")
	addPerson("feng pengfei", "mountainfpf@gmail.com")
	addPerson("fenwickelliott", "charles@fenwickelliott.io")
	addPerson("ferhat elmas", "elmas.ferhat@gmail.com")
	addPerson("filewalkwithme", "maiconscosta@gmail.com", "@filewalkwithme")
	addPerson("gangachris", "ganga.chris@gmail.com")
	addPerson("garrickevans", "garrick@google.com")
	addPerson("gbbr", "ga@stripetree.co.uk", "@gbbr")
	addPerson("glorieux", "lorieux.g@gmail.com", "@glorieux")
	addPerson("gmarik", "gmarik@gmail.com", "@gmarik")
	addPerson("go101", "tapir.liu@gmail.com", "@TapirLiu")
	addPerson("guitarbum722", "johnkenneth.moore@gmail.com")
	addPerson("gulyasm", "mgulyas86@gmail.com", "@gulyasm")
	addPerson("guyfedwards", "guyfedwards@gmail.com")
	addPerson("hagen1778", "hagen1778@gmail.com", "@hagen1778")
	addPerson("halfcrazy", "hackzhuyan@gmail.com")
	addPerson("halgrimur", "douga@google.com")
	addPerson("hanyang.tay", "htay@wesleyan.edu")
	addPerson("haormj", "haormj@gmail.com", "@haormj")
	addPerson("harshit777", "harshit.g.0702@gmail.com")
	addPerson("haya14busa", "haya14busa@gmail.com", "@haya14busa")
	addPerson("haya14busa", "hayabusa1419@gmail.com", "@haya14busa")
	addPerson("hearot", "gabriel@hearot.it", "@hearot")
	addPerson("helloPiers", "google@hellopiers.pro")
	addPerson("hellozee", "hellozee@disroot.org", "@hellozee")
	addPerson("hengwu0", "41297446+hengwu0@users.noreply.github.com", "@hengwu0")
	addPerson("hertzbach", "rhertzbach@gmail.com")
	addPerson("hezhenwei", "3711971@qq.com")
	addPerson("hsinhoyeh", "yhh92u@gmail.com")
	addPerson("huangyonglin", "1249107551@qq.com")
	addPerson("ia", "isaac.ardis@gmail.com")
	addPerson("iamqizhao", "toqizhao@gmail.com")
	addPerson("ianzapolsky", "ianzapolsky@gmail.com", "@ianzapolsky")
	addPerson("irfan sharif", "irfanmahmoudsharif@gmail.com")
	addPerson("ivan parra", "ivantrips1@gmail.com")
	addPerson("jaredculp", "jculp14@gmail.com", "@jaredculp")
	addPerson("jerome-laforge", "jerome.laforge@gmail.Com")
	addPerson("jimmy frasche", "soapboxcicero@gmail.com", "13220@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("jimmyfrasche", "soapboxcicero@gmail.com", "@jimmyfrasche")
	addPerson("jirawat001", "paoji@icloud.com")
	addPerson("joshua stein", "jcs@jcs.org")
	addPerson("kanapuliAthavan", "athavankanapuli@gmail.com")
	addPerson("kargakis", "mkargaki@redhat.com", "@kargakis")
	addPerson("khr", "khr@khr-glaptop.roam.corp.google.com")
	addPerson("kim yongbin", "kybinz@gmail.com", "@kybin")
	addPerson("kirinrastogi", "kirin.rastogi@shopify.com")
	addPerson("kirk", "kirk91.han@gmail.com", "@kirk91")
	addPerson("knqyf263", "knqyf263@gmail.com")
	addPerson("komuW", "komuw05@gmail.com", "@komuw")
	addPerson("komuw", "komuw05@gmail.com")
	addPerson("konstantin8105", "konstantin8105@gmail.com", "@Konstantin8105")
	addPerson("kortschak", "dan.kortschak@adelaide.edu.au", "@kortschak")
	addPerson("kujenga", "ataylor0123@gmail.com")
	addPerson("lcd1232", "8745863+lcd1232@users.noreply.github.com")
	addPerson("leigh schrandt", "leigh@null.net")
	addPerson("linatiantamade", "linqiyo@gmail.com")
	addPerson("lotus.wu", "lotus.wu@outlook.com")
	addPerson("lsytj0413", "511121939@qq.com")
	addPerson("ltnwgl", "ltnwgl@gmail.com")
	addPerson("ltnwgl", "ltnwgl@gmail.com", "@gengliangwang")
	addPerson("lucor", "lu.corbo@gmail.com")
	addPerson("ludweeg", "mursalimovemeel@gmail.com", "@ludweeg")
	addPerson("lukechampine", "luke.champine@gmail.com")
	addPerson("lukechampine", "luke.champine@gmail.com", "@lukechampine")
	addPerson("maiyang", "yangwen.yw@gmail.com")
	addPerson("majiang", "ma.jiang@zte.com.cn", "@zte-majiang")
	addPerson("mapeiqi", "mapeiqi2017@gmail.com")
	addPerson("marwan-at-work", "marwan.sameer@gmail.com", "@marwan-at-work")
	addPerson("matematik7", "domen@ipavec.net")
	addPerson("mattyw", "gh@mattyw.net", "@mattyw")
	addPerson("mdp", "m@mdp.im")
	addPerson("meir fischer", "meirfischer@gmail.com", "8955@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("mewmew", "rnd0x00@gmail.com", "@mewmew")
	addPerson("mihasya", "m@mihasya.com", "@mihasya")
	addPerson("mike andrews", "mra@xoba.com", "@xoba")
	addPerson("milad arabi", "milad.arabi@gmail.com")
	addPerson("mingrammer", "mingrammer@gmail.com", "@mingrammer")
	addPerson("mischief", "mischief@offblast.org", "@mischief")
	addPerson("mmaldo329", "michael_maldonado@comcast.com")
	addPerson("molivier", "olivier.matthieu@gmail.com", "@molivier")
	addPerson("monkeybutter", "pablo.larraondo@anu.edu.au")
	addPerson("moznion", "moznion@gmail.com")
	addPerson("mpl", "mathieu.lonjaret@gmail.com", "@mpl")
	addPerson("mstrong", "mstrong1341@gmail.com", "@xmattstrongx")
	addPerson("musgravejw", "musgravejw@gmail.com")
	addPerson("nicerobot", "golang@nicerobot.org")
	addPerson("nick.grange", "nicolas.grange@retrievercommunications.com")
	addPerson("nkhumphreys", "nkhumphreys@gmail.com")
	addPerson("nobonobo", "irieda@gmail.com", "@nobonobo")
	addPerson("nogoegst", "nogoegst@users.noreply.github.com", "@nogoegst")
	addPerson("nwidger", "niels.widger@gmail.com", "@nwidger")
	addPerson("oiooj", "nototon@gmail.com")
	addPerson("omarvides", "omarvides@gmail.com")
	addPerson("pallat", "yod.pallat@gmail.com")
	addPerson("pamelin", "amelin.paul@gmail.com")
	addPerson("pankona", "yosuke.akatsuka@gmail.com")
	addPerson("pavel-paulau", "pavel.paulau@gmail.com", "@pavel-paulau")
	addPerson("pbberlin", "peter.buchmann@web.de")
	addPerson("peter zhang", "i@ddatsh.com")
	addPerson("phayes", "patrick.d.hayes@gmail.com")
	addPerson("philhofer", "phofer@umich.edu", "@philhofer")
	addPerson("pityonline", "pityonline@gmail.com", "@pityonline")
	addPerson("prateekgogia", "prateekgogia42@gmail.com")
	addPerson("pvoicu", "pvoicu@paypal.com", "@pvoicu")
	addPerson("pytimer", "lixin20101023@gmail.com")
	addPerson("qeed", "qeed.quan@gmail.com", "@qeedquan")
	addPerson("ragavendra", "ragavendra.bn@gmail.com")
	addPerson("rajender", "rajenderreddykompally@gmail.com", "@rajender")
	addPerson("rajnikant", "rajnikant12345@gmail.com")
	addPerson("rhysd", "lin90162@yahoo.co.jp")
	addPerson("robnorman", "rob.norman@infinitycloud.com", "@robnorman")
	addPerson("roger peppe", "rogpeppe@gmail.com", "6010@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("romanyx", "romanyx90@yandex.ru", "@romanyx")
	addPerson("ron minnich", "rminnich@gmail.com", "12935@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("rubyist", "scott.barron@github.com", "@rubyist")
	addPerson("rust", "pruest@gmail.com", "@gazed")
	addPerson("rwaweber", "rwaweber@gmail.com")
	addPerson("saberuster", "saberuster@gmail.com")
	addPerson("sagarkrkv", "sagarkrkv@gmail.com")
	addPerson("sam boyer", "tech@samboyer.org", "@sdboyer")
	addPerson("sandyskies", "chenmingjie0828@163.com")
	addPerson("sasha-s", "sasha@scaledinference.com")
	addPerson("sayden", "mariocaster@gmail.com")
	addPerson("sbramin", "s@sbramin.com")
	addPerson("sdheisenberg", "nicholasleli@gmail.com", "@smugcloud")
	addPerson("sergey", "sngasuan@gmail.com", "@Asuan")
	addPerson("sergey.arseev", "sergey.arseev@intel.com", "@sergeyarseev")
	addPerson("sergey.dobrodey", "sergey.dobrodey@synesis.ru")
	addPerson("sevki", "s@sevki.org", "@sevki")
	addPerson("shaharko", "skohanim@gmail.com", "@skohanim")
	addPerson("shawnps", "shawnpsmith@gmail.com", "@shawnps")
	addPerson("shinofara", "shinofara@gmail.com")
	addPerson("shogo-ma", "choroma194@gmail.com", "@shogo-ma")
	addPerson("shwsun", "jethro.sun7@gmail.com")
	addPerson("slene", "vslene@gmail.com")
	addPerson("smasher164", "contact@akhilindurti.com")
	addPerson("softctrl", "carlostimoshenkorodrigueslopes@gmail.com")
	addPerson("soluchok", "isoluchok@gmail.com", "@soluchok")
	addPerson("spring1843", "yasser@yasser.ca")
	addPerson("stephane benoit", "stefb965@gmail.com")
	addPerson("stxmendez", "stxmendez@gmail.com")
	addPerson("sukrithanda", "sukrit.handa@utoronto.ca")
	addPerson("tal@whatexit.org", "tal@whatexit.org")
	addPerson("taylorza", "taylorza@gmail.com")
	addPerson("tbunyk", "tbunyk@gmail.com", "@bunyk")
	addPerson("teague", "tnc1443@gmail.com", "@teaguecole")
	addPerson("telecoda", "robbaines@gmail.com")
	addPerson("templexxx", "lucas1x1x@gmail.com", "@templexxx")
	addPerson("tengufromsky", "nick27surgut@gmail.com", "@tengufromsky")
	addPerson("theairkit", "theairkit@gmail.com")
	addPerson("themester", "Garriga975@gmail.com")
	addPerson("themester", "dgrripoll@gmail.com")
	addPerson("themihai", "mihai@epek.com")
	addPerson("thoeni", "thoeni@gmail.com")
	addPerson("thoeni", "thoeni@gmail.com", "@thoeni")
	addPerson("thor wolpert", "thor@wolpert.ca")
	addPerson("tkivisik", "taavi.kivisik@gmail.com", "@tkivisik")
	addPerson("tliu", "terry.liu.y@gmail.com")
	addPerson("tnt", "alkaloid.btx@gmail.com", "@trtstm")
	addPerson("tom", "tommiemeyer290@gmail.com")
	addPerson("tro3", "trey.roessig@gmail.com", "@tro3")
	addPerson("ttacon", "ttacon@gmail.com", "@ttacon")
	addPerson("ttyh061", "ttyh061@gmail.com")
	addPerson("tuxpy", "q8886888@qq.com")
	addPerson("unknown", "daria.kolistratova@intel.com", "@DarKol13")
	addPerson("unknown", "geon0250@gmail.com", "@KimMachineGun")
	addPerson("unknown", "nonamezeil@gmail.com", "@zeil")
	addPerson("uropek", "uropek@gmail.com", "@uropek")
	addPerson("vabr-g", "vabr@google.com")
	addPerson("viswesr", "r.visweswara@gmail.com")
	addPerson("voutasaurus", "voutasaurus@gmail.com", "@voutasaurus")
	addPerson("vvakame", "vvakame+dev@gmail.com")
	addPerson("wbond", "will@wbond.net")
	addPerson("weeellz", "weeellz12@gmail.com", "@weeellz")
	addPerson("wheelcomplex yin", "wheelcomplex@gmail.com")
	addPerson("woodsaj", "awoods@raintank.io", "@woodsaj")
	addPerson("wozz", "wozz@users.noreply.github.com")
	addPerson("wrfly", "mr.wrfly@gmail.com")
	addPerson("wu-heng", "41297446+wu-heng@users.noreply.github.com")
	addPerson("wuyunzhou", "yunzhouwu@gmail.com", "@wuyunzhou")
	addPerson("wzshiming", "wzshiming@foxmail.com")
	addPerson("xiezhenye", "xiezhenye@gmail.com")
	addPerson("xufei_Alex", "badgangkiller@gmail.com", "18915@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("xufei_Alex", "badgangkiller@gmail.com", "@knightXun")
	addPerson("yansal", "yannsalaun1@gmail.com", "@yansal")
	addPerson("yanyiwu", "wuyanyi09@gmail.com")
	addPerson("yazver", "ya.zver@gmail.com")
	addPerson("yo-tak", "yo.tak0812@gmail.com", "@yo-tak")
	addPerson("yuuji.yaginuma", "yuuji.yaginuma@gmail.com", "@y-yagi")
	addPerson("zachgersh", "zachgersh@gmail.com")
	addPerson("zaq1tomo", "zaq1tomo@gmail.com", "@zaq1tomo")
	addPerson("zhongtao.chen", "chenzhongtao@126.com", "@chenzhongtao")
	addPerson("zhoujun", "dev.zhoujun@gmail.com")
	addPerson("Özgür Kesim", "oec-go@kesim.org")
	addPerson("Максим Федосеев", "max.faceless.frei@gmail.com", "@codesenberg")
	addPerson("Фахриддин Балтаев", "faxriddinjon@gmail.com", "@faxriddin")
	addPerson("Юрий Соколов", "funny.falcon@gmail.com", "7215@62eb7196-b449-3ce5-99f1-c037f21e1705")
	addPerson("一痕 刘", "liuyihen@gmail.com")
	addPerson("张嵩", "zs349596@gmail.com", "@zs1379")
	addPerson("沈涛", "shentaoskyking@gmail.com", "@smileusd")
	addPerson("祥曦 徐", "lucas1x1x@gmail.com", "28434@62eb7196-b449-3ce5-99f1-c037f21e1705")
}

// GithubOfGomoteUser returns the GitHub username for the provided gomote user.
func GithubOfGomoteUser(gomoteUser string) (githubUser string) {
	switch gomoteUser {
	case "austin":
		return "aclements"
	case "cbro":
		return "broady"
	case "cherryyz":
		return "cherrymui"
	case "cmang":
		return "paranoiacblack"
	case "drchase":
		return "dr2chase"
	case "gri":
		return "griesemer"
	case "hakim":
		return "hyangah"
	case "herbie":
		return "cybrcodr"
	case "iant":
		return "ianlancetaylor"
	case "jbd":
		return "rakyll"
	case "joetsai":
		return "dsnet"
	case "jrjohnson":
		return "johnsonj"
	case "khr":
		return "randall77"
	case "lazard":
		return "davidlazar"
	case "pjw":
		return "pjweinbgo"
	case "r":
		return "robpike"
	case "rstambler":
		return "stamblerre"
	case "sameer":
		return "Sajmani"
	case "shadams":
		return "adams-sarah"
	case "spf":
		return "spf13"
	case "valsorda":
		return "FiloSottile"
	}
	return gomoteUser
}
