/*
Copyright 2018 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package manager

import (
	"path/filepath"

	"sigs.k8s.io/controller-tools/pkg/scaffold/input"
)

var _ input.File = &Config{}

// Config scaffolds yaml config for the manager.
type Config struct {
	input.Input
	// Image is controller manager image name
	Image string
}

// GetInput implements input.File
func (c *Config) GetInput() (input.Input, error) {
	if c.Path == "" {
		c.Path = filepath.Join("config", "manager", "manager.yaml")
	}
	c.TemplateBody = configTemplate
	return c.Input, nil
}

var configTemplate = `apiVersion: v1
kind: Namespace
metadata:
  labels:
    controller-tools.k8s.io: "1.0"
  name: system
---
apiVersion: v1
kind: Service
metadata:
  name: controller-manager-service
  namespace: system
  labels:
    control-plane: controller-manager
    controller-tools.k8s.io: "1.0"
spec:
  selector:
    control-plane: controller-manager
    controller-tools.k8s.io: "1.0"
  ports:
  - port: 443
---
apiVersion: apps/v1
kind: StatefulSet
metadata:
  name: controller-manager
  namespace: system
  labels:
    control-plane: controller-manager
    controller-tools.k8s.io: "1.0"
spec:
  selector:
    matchLabels:
      control-plane: controller-manager
      controller-tools.k8s.io: "1.0"
  serviceName: controller-manager-service
  template:
    metadata:
      labels:
        control-plane: controller-manager
        controller-tools.k8s.io: "1.0"
    spec:
      containers:
      - command:
        - /root/manager
        image: {{ .Image }}
        imagePullPolicy: Always
        name: manager
        env:
          - name: POD_NAMESPACE
            valueFrom:
              fieldRef:
                fieldPath: metadata.namespace
          - name: SECRET_NAME
            value: $(WEBHOOK_SECRET_NAME)
        resources:
          limits:
            cpu: 100m
            memory: 30Mi
          requests:
            cpu: 100m
            memory: 20Mi
        ports:
        - containerPort: 9876
          name: webhook-server
          protocol: TCP
        volumeMounts:
        - mountPath: /tmp/cert
          name: cert
          readOnly: true
      terminationGracePeriodSeconds: 10
      volumes:
      - name: cert
        secret:
          defaultMode: 420
          secretName: webhook-server-secret
---
apiVersion: v1
kind: Secret
metadata:
  name: webhook-server-secret
  namespace: system
`
