// Copyright ©2015 The Gonum Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package testlapack

import (
	"testing"

	"gonum.org/v1/gonum/floats"
)

type Dlasq5er interface {
	Dlasq5(i0, n0 int, z []float64, pp int, tau, sigma float64) (i0Out, n0Out, ppOut int, tauOut, sigmaOut, dmin, dmin1, dmin2, dn, dnm1, dnm2 float64)
}

func Dlasq5Test(t *testing.T, impl Dlasq5er) {
	dTol := 1e-14
	for _, test := range []struct {
		z          []float64
		i0, n0, pp int
		tau, sigma float64

		i0Out, n0Out, ppOut         int
		tauOut, sigmaOut            float64
		dminOut, dmin1Out, dmin2Out float64
		dnOut, dnm1Out, dnm2Out     float64
	}{
		{
			z:        []float64{0.1914365246180821E+01, 0.1564384297703890E+01, 0.2493389162143899E+00, 0.3499809484769305E+00, 0.1315996513131545E+01, 0.1363862112490627E+01, 0.9898466611970759E-01, 0.2014733168553078E+00, 0.6023973979587287E+00, 0.6465544792741794E+00, 0.2210033410638781E-02, 0.5482758480425683E-01, 0.9861857233678967E-01, 0.2428190810745492E-01, 0.4756321484454819E+00, 0.7654669763997353E-01, 0.2588748143677115E+00, 0.6127784069508770E+00, 0.1078611376690004E+00, 0.1217285558623164E+00, 0.6442896492255246E+00, 0.2293835804898155E+00, 0.6203230486639705E+00, 0.5227672064047094E+00, 0.3695660678607585E+00, 0.7645233184745865E+00, 0.5378838054252265E+00, 0.2253657980501426E+00, 0.3562533181264623E+00, 0.8820486722335483E+00, 0.2222132496436145E-01, 0.1208845131814035E-01, 0.1275094303021685E+01, 0.6548746852163357E+00, 0.1647324354821218E+00, 0.6424409427697111E+00, 0.1007530576543866E+01, 0.3269551736546701E+00, 0.3453881601783118E+00, 0.8453078383713172E+00, 0.2679391719153404E+00, 0.4116714838778281E+00, 0.7328677736683723E+00, 0.2016558482158241E+00, 0.8360828138307410E+00, 0.9737579452195326E+00, 0.4813660709592822E+00, 0.5951926422795808E+00, 0.6495370513676459E+00, 0.6761876248148171E+00, 0.2325475880222648E+00, 0.4547154975121112E+00, 0.1993624802893807E+00, 0.3321819367342255E+00, 0.3782318916911257E+00, 0.9972813157741996E-01, 0.9830449403503746E+00, 0.7561080996844842E+00, 0.4429733864040367E+00, 0.6051687323570161E+00, 0.1173279550602403E+01, 0.7195724480316686E+00, 0.5035524069144587E+00, 0.8966804889747714E+00, 0.3058980395058521E+00, 0.6588832353928662E+00, 0.3014634433415453E+00, 0.1505672110274446E+00, 0.1289422237567578E+01, 0.6124645310993601E+00, 0.7583364305799440E+00, 0.9784211498097629E+00, 0.4977814779461571E+00, 0.9993813577491869E+00, 0.2841468847862598E+00, 0.2567365507769143E+00, 0.9257539794205765E+00, 0.5509268385614666E+00, 0.5231355605450990E-04, 0.6589740256453697E+00, 0.2117869221381033E-04, 0.7349224826832024E-04, 0.0000000000000000E+00, 0.0000000000000000E+00},
			i0:       1,
			n0:       21,
			pp:       0,
			tau:      0.0000000000000000,
			sigma:    0.0000000000000000,
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.0000000000000000,
			dminOut:  2.1175426017541180E-005,
			dmin1Out: 4.4311601260836921E-002,
			dmin2Out: 4.4311601260836921E-002,
			dnOut:    2.1175426017541180E-005,
			dnm1Out:  0.33915960483100382,
			dnm2Out:  0.16428924199195991,
		},
		{
			z:        []float64{0.1914365246180821E+01, 0.2163704162395211E+01, 0.2493389162143899E+00, 0.1516515751224039E+00, 0.1315996513131545E+01, 0.1263329604128848E+01, 0.9898466611970759E-01, 0.4719916727467415E-01, 0.6023973979587287E+00, 0.5574082640946934E+00, 0.2210033410638781E-02, 0.3910066531356214E-03, 0.9861857233678967E-01, 0.5738597141291359E+00, 0.4756321484454819E+00, 0.2145632131068746E+00, 0.2588748143677115E+00, 0.1521727389298373E+00, 0.1078611376690004E+00, 0.4566771620366771E+00, 0.6442896492255246E+00, 0.8079355358528180E+00, 0.6203230486639705E+00, 0.2837483186776231E+00, 0.3695660678607585E+00, 0.6237015546083620E+00, 0.5378838054252265E+00, 0.3072349091217998E+00, 0.3562533181264623E+00, 0.7123973396902394E-01, 0.2222132496436145E-01, 0.3977314805803597E+00, 0.1275094303021685E+01, 0.1042095257923447E+01, 0.1647324354821218E+00, 0.1592685164190333E+00, 0.1007530576543866E+01, 0.1193650220303144E+01, 0.3453881601783118E+00, 0.7752942700755104E-01, 0.2679391719153404E+00, 0.9232775185761617E+00, 0.7328677736683723E+00, 0.6636554427529671E+00, 0.8360828138307410E+00, 0.6537934420370561E+00, 0.4813660709592822E+00, 0.4782322339990674E+00, 0.6495370513676459E+00, 0.4038524053908432E+00, 0.2325475880222648E+00, 0.1147975431483785E+00, 0.1993624802893807E+00, 0.4627968288321279E+00, 0.3782318916911257E+00, 0.8034172324482011E+00, 0.9830449403503746E+00, 0.6226010943062101E+00, 0.4429733864040367E+00, 0.8347746582554776E+00, 0.1173279550602403E+01, 0.8420572992613844E+00, 0.5035524069144587E+00, 0.1829278057427913E+00, 0.3058980395058521E+00, 0.4244336771046062E+00, 0.3014634433415453E+00, 0.9158407747236312E+00, 0.1289422237567578E+01, 0.1131917893423890E+01, 0.7583364305799440E+00, 0.3334922359541972E+00, 0.4977814779461571E+00, 0.4484361267782198E+00, 0.2841468847862598E+00, 0.5865943745895725E+00, 0.9257539794205765E+00, 0.3392119183870583E+00, 0.5231355605450990E-04, 0.3266196269153995E-08, 0.2117869221381033E-04, 0.2117542601754118E-04, 0.0000000000000000E+00, 0.3910066531356214E-03, 0.1037537856266618 - 321},
			i0:       1,
			n0:       21,
			pp:       1,
			tau:      0.0000000000000000,
			sigma:    0.0000000000000000,
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.0000000000000000,
			dminOut:  2.1175425069176302E-005,
			dmin1Out: 2.9944624525135358E-002,
			dmin2Out: 2.9944624525135358E-002,
			dnOut:    2.1175425069176302E-005,
			dnm1Out:  7.2928780948497918E-002,
			dnm2Out:  0.16065460645225654,
		},
		{
			z:        []float64{0.2315355737517615E+01, 0.2163704162395211E+01, 0.8274578340618610E-01, 0.1516515751224039E+00, 0.1227782987997336E+01, 0.1263329604128848E+01, 0.2142822156235013E-01, 0.4719916727467415E-01, 0.5363710491854788E+00, 0.5574082640946934E+00, 0.4183353417969536E-03, 0.3910066531356214E-03, 0.7880045918942136E+00, 0.5738597141291359E+00, 0.4143462125464707E-01, 0.2145632131068746E+00, 0.5674152797118673E+00, 0.1521727389298373E+00, 0.6502569120260687E+00, 0.4566771620366771E+00, 0.4414269425043723E+00, 0.8079355358528180E+00, 0.4009140594652070E+00, 0.2837483186776231E+00, 0.5300224042649548E+00, 0.6237015546083620E+00, 0.4129510944388858E-01, 0.3072349091217998E+00, 0.4276761051054951E+00, 0.7123973396902394E-01, 0.9691308092544145E+00, 0.3977314805803597E+00, 0.2322329650880660E+00, 0.1042095257923447E+01, 0.8186215063776209E+00, 0.1592685164190333E+00, 0.4525581409330741E+00, 0.1193650220303144E+01, 0.1581701233715052E+00, 0.7752942700755104E-01, 0.1428762837957623E+01, 0.9232775185761617E+00, 0.3036848136842134E+00, 0.6636554427529671E+00, 0.8283408623519102E+00, 0.6537934420370561E+00, 0.2331591338951825E+00, 0.4782322339990674E+00, 0.2854908146440392E+00, 0.4038524053908432E+00, 0.1860933389154074E+00, 0.1147975431483785E+00, 0.1080120722364922E+01, 0.4627968288321279E+00, 0.4631042046962229E+00, 0.8034172324482011E+00, 0.9942715478654648E+00, 0.6226010943062101E+00, 0.7069779837626068E+00, 0.8347746582554776E+00, 0.3180071212415688E+00, 0.8420572992613844E+00, 0.2441477440283845E+00, 0.1829278057427913E+00, 0.1096126707799853E+01, 0.4244336771046062E+00, 0.9457451890006905E+00, 0.9158407747236312E+00, 0.5196649403773971E+00, 0.1131917893423890E+01, 0.2877815203259632E+00, 0.3334922359541972E+00, 0.7472489810418290E+00, 0.4484361267782198E+00, 0.2662831374385604E+00, 0.5865943745895725E+00, 0.7292878421469419E-01, 0.3392119183870583E+00, 0.9483648767903632E-12, 0.3266196269153995E-08, 0.2117542506917630E-04, 0.2117542601754118E-04, 0.4183353417969536E-03, 0.3910066531356214E-03},
			i0:       1,
			n0:       21,
			pp:       0,
			tau:      2.1175313795360271E-005,
			sigma:    0.0000000000000000,
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			tauOut:   2.1175313795360271E-005,
			sigmaOut: 0.0000000000000000,
			dminOut:  1.1127325659669794E-010,
			dmin1Out: 3.1433071595911154E-002,
			dmin2Out: 3.1433071595911154E-002,
			dnOut:    1.1127325659669794E-010,
			dnm1Out:  3.5896964560873705E-002,
			dnm2Out:  0.25842281720128102,
		},
		{
			z:        []float64{0.2315355737517615E+01, 0.2398080345610006E+01, 0.8274578340618610E-01, 0.4236466279397526E-01, 0.1227782987997336E+01, 0.1206825371451915E+01, 0.2142822156235013E-01, 0.9523728911788614E-02, 0.5363710491854788E+00, 0.5272444803016919E+00, 0.4183353417969536E-03, 0.6252320936560726E-03, 0.7880045918942136E+00, 0.8287928057414093E+00, 0.4143462125464707E-01, 0.2836732781232222E-01, 0.5674152797118673E+00, 0.1189283688611819E+01, 0.6502569120260687E+00, 0.2413561400585997E+00, 0.4414269425043723E+00, 0.6009636865971842E+00, 0.4009140594652070E+00, 0.3535878097802652E+00, 0.5300224042649548E+00, 0.2177085286147829E+00, 0.4129510944388858E-01, 0.8112190955144877E-01, 0.4276761051054951E+00, 0.1315663829494665E+01, 0.9691308092544145E+00, 0.1710650671895379E+00, 0.2322329650880660E+00, 0.8797682289623537E+00, 0.8186215063776209E+00, 0.4211038940233675E+00, 0.4525581409330741E+00, 0.1896031949674164E+00, 0.1581701233715052E+00, 0.1191897606932286E+01, 0.1428762837957623E+01, 0.5405288693957555E+00, 0.3036848136842134E+00, 0.4653859482687157E+00, 0.8283408623519102E+00, 0.5960928726645816E+00, 0.2331591338951825E+00, 0.1116684901463164E+00, 0.2854908146440392E+00, 0.3598944880993349E+00, 0.1860933389154074E+00, 0.5585061130503639E+00, 0.1080120722364922E+01, 0.9846976386969850E+00, 0.4631042046962229E+00, 0.4676068229793028E+00, 0.9942715478654648E+00, 0.1233621533334973E+01, 0.7069779837626068E+00, 0.1822471700779458E+00, 0.3180071212415688E+00, 0.3798865198782122E+00, 0.2441477440283845E+00, 0.7044652781161848E+00, 0.1096126707799853E+01, 0.1337385443370563E+01, 0.9457451890006905E+00, 0.3674861422265960E+00, 0.5196649403773971E+00, 0.4399391431629689E+00, 0.2877815203259632E+00, 0.4888049885267526E+00, 0.7472489810418290E+00, 0.5247059546398414E+00, 0.2662831374385604E+00, 0.3701064434002514E-01, 0.7292878421469419E-01, 0.3589696456182207E-01, 0.9483648767903632E-12, 0.5594353069081231E-15, 0.2117542506917630E-04, 0.1112732565966979E-09, 0.4183353417969536E-03, 0.6252320936560726E-03, 0.1037537856266618 - 321},
			i0:       1,
			n0:       21,
			pp:       1,
			tau:      2.1175313795360271E-005,
			sigma:    2.1175313795360271E-005,
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			tauOut:   2.1175313795360271E-005,
			sigmaOut: 2.1175313795360271E-005,
			dminOut:  -2.1175202522103674E-005,
			dmin1Out: 2.9116497146097618E-002,
			dmin2Out: 4.9396687496051764E-002,
			dnOut:    -2.1175202522103674E-005,
			dnm1Out:  2.9116497146097618E-002,
			dnm2Out:  0.15954393093937583,
		},
		{
			z:        []float64{0.2440423833090186E+01, 0.2398080345610006E+01, 0.2094994698033050E-01, 0.4236466279397526E-01, 0.1195377978069578E+01, 0.1206825371451915E+01, 0.4200624064314086E-02, 0.9523728911788614E-02, 0.5236479130172386E+00, 0.5272444803016919E+00, 0.9895730475750664E-03, 0.6252320936560726E-03, 0.8561493851923613E+00, 0.8287928057414093E+00, 0.3940527300515336E-01, 0.2836732781232222E-01, 0.1391213380351470E+01, 0.1189283688611819E+01, 0.1042588274099539E+00, 0.2413561400585997E+00, 0.8502714936537001E+00, 0.6009636865971842E+00, 0.9053470847599347E-01, 0.3535878097802652E+00, 0.2082745543764428E+00, 0.2177085286147829E+00, 0.5124445590385125E+00, 0.8112190955144877E-01, 0.9742631623318954E+00, 0.1315663829494665E+01, 0.1544732645319877E+00, 0.1710650671895379E+00, 0.1146377683139938E+01, 0.8797682289623537E+00, 0.6964776521238716E-01, 0.4211038940233675E+00, 0.1311831861373520E+01, 0.1896031949674164E+00, 0.4911110065859084E+00, 0.1191897606932286E+01, 0.5147826357647675E+00, 0.5405288693957555E+00, 0.5388939477127089E+00, 0.4653859482687157E+00, 0.1688462397843937E+00, 0.5960928726645816E+00, 0.2380205454936569E+00, 0.1116684901463164E+00, 0.6803588803422466E+00, 0.3598944880993349E+00, 0.8083375797812383E+00, 0.5585061130503639E+00, 0.6439457065812542E+00, 0.9846976386969850E+00, 0.8958050967125053E+00, 0.4676068229793028E+00, 0.5200424313866183E+00, 0.1233621533334973E+01, 0.1331299890548604E+00, 0.1822471700779458E+00, 0.9512006336257413E+00, 0.3798865198782122E+00, 0.9904762202705547E+00, 0.7044652781161848E+00, 0.7143741900128092E+00, 0.1337385443370563E+01, 0.2263121215682984E+00, 0.3674861422265960E+00, 0.7024108348076278E+00, 0.4399391431629689E+00, 0.3651408483866702E+00, 0.4888049885267526E+00, 0.1965545752794010E+00, 0.5247059546398414E+00, 0.6759292101929097E-02, 0.3701064434002514E-01, 0.2911649714609818E-01, 0.3589696456182207E-01, 0.2137969692662087E-23, 0.5594353069081231E-15, -0.2117520252210367E-04, 0.1112732565966979E-09, 0.9895730475750664E-03, 0.6252320936560726E-03, 0.1037537856266618 - 321},
			i0:       1,
			n0:       21,
			pp:       1,
			tau:      1.1127325659669789E-010,
			sigma:    2.1175313795360271E-005,
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			tauOut:   1.1127325659669789E-010,
			sigmaOut: 2.1175313795360271E-005,
			dminOut:  -2.0808762284537102E-024,
			dmin1Out: 2.9139336744737766E-002,
			dmin2Out: 4.9426557292086552E-002,
			dnOut:    -2.0808762284537102E-024,
			dnm1Out:  2.9139336744737766E-002,
			dnm2Out:  0.15959234211062134,
		},
		{
			z:        []float64{0.2440445008292708E+01, 0.2398080345610006E+01, 0.2094976520226600E-01, 0.4236466279397526E-01, 0.1195399335050165E+01, 0.1206825371451915E+01, 0.4200549016048655E-02, 0.9523728911788614E-02, 0.5236691632680260E+00, 0.5272444803016919E+00, 0.9895328911616120E-03, 0.6252320936560726E-03, 0.8561706005512968E+00, 0.8287928057414093E+00, 0.3940429656773515E-01, 0.2836732781232222E-01, 0.1391235531991410E+01, 0.1189283688611819E+01, 0.1042571673718422E+00, 0.2413561400585997E+00, 0.8502943288943339E+00, 0.6009636865971842E+00, 0.9053227710395735E-01, 0.3535878097802652E+00, 0.2082981609510011E+00, 0.2177085286147829E+00, 0.5123864833424303E+00, 0.8112190955144877E-01, 0.9743424132304999E+00, 0.1315663829494665E+01, 0.1544607000116935E+00, 0.1710650671895379E+00, 0.1146411422862754E+01, 0.8797682289623537E+00, 0.6964571542795012E-01, 0.4211038940233675E+00, 0.1311855086360479E+01, 0.1896031949674164E+00, 0.4911023119923957E+00, 0.1191897606932286E+01, 0.5148125055608023E+00, 0.5405288693957555E+00, 0.5388626806938843E+00, 0.4653859482687157E+00, 0.1688986820057405E+00, 0.5960928726645816E+00, 0.2379466412690434E+00, 0.1116684901463164E+00, 0.6804539597693821E+00, 0.3598944880993349E+00, 0.8082246312519304E+00, 0.5585061130503639E+00, 0.6440798303130841E+00, 0.9846976386969850E+00, 0.8956185534970393E+00, 0.4676068229793028E+00, 0.5202501498046066E+00, 0.1233621533334973E+01, 0.1330768347199243E+00, 0.1822471700779458E+00, 0.9512749631631994E+00, 0.3798865198782122E+00, 0.9903988276741268E+00, 0.7044652781161848E+00, 0.7144727578117591E+00, 0.1337385443370563E+01, 0.2262808998212762E+00, 0.3674861422265960E+00, 0.7024632317571722E+00, 0.4399391431629689E+00, 0.3651136124179467E+00, 0.4888049885267526E+00, 0.1966029864506465E+00, 0.5247059546398414E+00, 0.6757627705811050E-02, 0.3701064434002514E-01, 0.2913933674473832E-01, 0.3589696456182207E-01, 0.2136293938333395E-23, 0.5594353069081231E-15, 0.0000000000000000E+00, 0.1112732565966979E-09, 0.9895328911616120E-03, 0.6252320936560726E-03},
			i0:       1,
			n0:       21,
			pp:       0,
			tau:      -0.0000000000000000,
			sigma:    2.1175425068616867E-005,
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 2.1175425068616867E-005,
			dminOut:  0.0000000000000000,
			dmin1Out: 2.7016889331018056E-002,
			dmin2Out: 5.3061698118516694E-002,
			dnOut:    0.0000000000000000,
			dnm1Out:  2.7016889331018056E-002,
			dnm2Out:  8.6018658784156071E-002,
		},
		{
			z:        []float64{0.2440445008292708E+01, 0.2461394773494974E+01, 0.2094976520226600E-01, 0.1017444891892999E-01, 0.1195399335050165E+01, 0.1189425435147283E+01, 0.4200549016048655E-02, 0.1849378635683999E-02, 0.5236691632680260E+00, 0.5228093175235037E+00, 0.9895328911616120E-03, 0.1620493249248586E-02, 0.8561706005512968E+00, 0.8939544038697832E+00, 0.3940429656773515E-01, 0.6132377362967349E-01, 0.1391235531991410E+01, 0.1434168925733579E+01, 0.1042571673718422E+00, 0.6181229879703373E-01, 0.8502943288943339E+00, 0.8790143072012576E+00, 0.9053227710395735E-01, 0.2145324219750511E-01, 0.2082981609510011E+00, 0.6992314020959263E+00, 0.5123864833424303E+00, 0.7139837844669097E+00, 0.9743424132304999E+00, 0.4148193287752837E+00, 0.1544607000116935E+00, 0.4268738185358478E+00, 0.1146411422862754E+01, 0.7891833197548568E+00, 0.6964571542795012E-01, 0.1157716892137957E+00, 0.1311855086360479E+01, 0.1687185709139079E+01, 0.4911023119923957E+00, 0.1498504938454686E+00, 0.5148125055608023E+00, 0.9038246924092180E+00, 0.5388626806938843E+00, 0.1006978425303630E+00, 0.1688986820057405E+00, 0.3061474807444209E+00, 0.2379466412690434E+00, 0.5288684194677825E+00, 0.6804539597693821E+00, 0.9598101715535300E+00, 0.8082246312519304E+00, 0.5423584775195998E+00, 0.6440798303130841E+00, 0.9973399062905237E+00, 0.8956185534970393E+00, 0.4671884516860899E+00, 0.5202501498046066E+00, 0.1861385328384410E+00, 0.1330768347199243E+00, 0.6800991665489665E+00, 0.9512749631631994E+00, 0.1261574624288360E+01, 0.9903988276741268E+00, 0.5608966509936130E+00, 0.7144727578117591E+00, 0.3798570066394223E+00, 0.2262808998212762E+00, 0.4184574968871406E+00, 0.7024632317571722E+00, 0.6491193472879784E+00, 0.3651136124179467E+00, 0.1105843276664904E+00, 0.1966029864506465E+00, 0.9277628648996712E-01, 0.6757627705811050E-02, 0.2122447413720272E-02, 0.2913933674473832E-01, 0.2701688933101806E-01, 0.2136293938333395E-23, 0.0000000000000000E+00, 0.2117542506861687E-04},
			i0:       1,
			n0:       20,
			pp:       1,
			tau:      -0.0000000000000000,
			sigma:    2.1175425068616867E-005,
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 2.1175425068616867E-005,
			dminOut:  2.5763383633962696E-002,
			dmin1Out: 4.3622798915905092E-002,
			dmin2Out: 7.4536672467372611E-002,
			dnOut:    2.5763383633962696E-002,
			dnm1Out:  4.3622798915905092E-002,
			dnm2Out:  9.8141518071882677E-002,
		},
		{
			z:        []float64{0.2471569222413904E+01, 0.2461394773494974E+01, 0.4896382518051712E-02, 0.1017444891892999E-01, 0.1186378431264915E+01, 0.1189425435147283E+01, 0.8149780515932184E-03, 0.1849378635683999E-02, 0.5236148327211592E+00, 0.5228093175235037E+00, 0.2766627272719901E-02, 0.1620493249248586E-02, 0.9525115502267366E+00, 0.8939544038697832E+00, 0.9233342160256496E-01, 0.6132377362967349E-01, 0.1403647802928048E+01, 0.1434168925733579E+01, 0.3870906568602875E-01, 0.6181229879703373E-01, 0.8617584837127339E+00, 0.8790143072012576E+00, 0.1740717486950262E-01, 0.2145324219750511E-01, 0.1395808011693333E+01, 0.6992314020959263E+00, 0.2121884039551361E+00, 0.7139837844669097E+00, 0.6295047433559955E+00, 0.4148193287752837E+00, 0.5351535485381410E+00, 0.4268738185358478E+00, 0.3698014604305115E+00, 0.7891833197548568E+00, 0.5281978587564573E+00, 0.1157716892137957E+00, 0.1308838344228090E+01, 0.1687185709139079E+01, 0.1034799882693896E+00, 0.1498504938454686E+00, 0.9010425466701916E+00, 0.9038246924092180E+00, 0.3421413441684364E-01, 0.1006978425303630E+00, 0.8008017657953598E+00, 0.3061474807444209E+00, 0.6338813300623194E+00, 0.5288684194677825E+00, 0.8682873190108105E+00, 0.9598101715535300E+00, 0.6229686202966810E+00, 0.5423584775195998E+00, 0.8415597376799326E+00, 0.9973399062905237E+00, 0.1033340463692495E+00, 0.4671884516860899E+00, 0.7629036530181579E+00, 0.1861385328384410E+00, 0.1124645093942705E+01, 0.6800991665489665E+00, 0.6978261813392677E+00, 0.1261574624288360E+01, 0.3053203341720497E+00, 0.5608966509936130E+00, 0.4929941693545132E+00, 0.3798570066394223E+00, 0.5509778292160957E+00, 0.4184574968871406E+00, 0.2087258457383731E+00, 0.6491193472879784E+00, 0.4915348757406203E-01, 0.1105843276664904E+00, 0.4574524632962537E-01, 0.9277628648996712E-01, 0.1253505697055357E-02, 0.2122447413720272E-02, 0.2576338363396270E-01, 0.2701688933101806E-01, 0.8149780515932184E-03, 0.1620493249248586E-02},
			i0:       1,
			n0:       20,
			pp:       0,
			tau:      2.0080554394878082E-002,
			sigma:    2.1175425068616867E-005,
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			tauOut:   2.0080554394878082E-002,
			sigmaOut: 2.1175425068616867E-005,
			dminOut:  -2.1187919252575148E-003,
			dmin1Out: -2.1187919252575148E-003,
			dmin2Out: 3.1777270007152948E-002,
			dnOut:    4.3005217031728403E-002,
			dnm1Out:  -2.1187919252575148E-003,
			dnm2Out:  3.1777270007152948E-002,
		},
		{
			z:        []float64{0.2471569222413904E+01, 0.2456385050537078E+01, 0.4896382518051712E-02, 0.2364842030515144E-02, 0.1186378431264915E+01, 0.1164748012891115E+01, 0.8149780515932184E-03, 0.3663750368606913E-03, 0.5236148327211592E+00, 0.5059345305621403E+00, 0.2766627272719901E-02, 0.5208666879309463E-02, 0.9525115502267366E+00, 0.1019555750555114E+01, 0.9233342160256496E-01, 0.1271177219084926E+00, 0.1403647802928048E+01, 0.1295158592310706E+01, 0.3870906568602875E-01, 0.2575581550365553E-01, 0.8617584837127339E+00, 0.8333292886837029E+00, 0.1740717486950262E-01, 0.2915663048658400E-01, 0.1395808011693333E+01, 0.1558759230767007E+01, 0.2121884039551361E+00, 0.8569226352498945E-01, 0.6295047433559955E+00, 0.1058885473974269E+01, 0.5351535485381410E+00, 0.1868951540728986E+00, 0.3698014604305115E+00, 0.6910236107191923E+00, 0.5281978587564573E+00, 0.1000437033056103E+01, 0.1308838344228090E+01, 0.3918007450464985E+00, 0.1034799882693896E+00, 0.2379777816619179E+00, 0.9010425466701916E+00, 0.6771983450302391E+00, 0.3421413441684364E-01, 0.4045895778871807E-01, 0.8008017657953598E+00, 0.1374143583674083E+01, 0.6338813300623194E+00, 0.4005339232303680E+00, 0.8682873190108105E+00, 0.1070641461682245E+01, 0.6229686202966810E+00, 0.4896740201485869E+00, 0.8415597376799326E+00, 0.4351392095057171E+00, 0.1033340463692495E+00, 0.1811694274708021E+00, 0.7629036530181579E+00, 0.1686298765095183E+01, 0.1124645093942705E+01, 0.4654019842229331E+00, 0.6978261813392677E+00, 0.5176639768935063E+00, 0.3053203341720497E+00, 0.2907699806261721E+00, 0.4929941693545132E+00, 0.7331214635495586E+00, 0.5509778292160957E+00, 0.1568680213363420E+00, 0.2087258457383731E+00, 0.8093075758121498E-01, 0.4915348757406203E-01, 0.2778348386000479E-01, 0.4574524632962537E-01, -0.8652862282021575E-03, 0.1253505697055357E-02, -0.3732238779264379E-01, 0.2576338363396270E-01, 0.4300521703172840E-01, 0.8149780515932184E-03, 0.3663750368606913E-03},
			i0:       1,
			n0:       20,
			pp:       0,
			tau:      5.0201385987195205E-003,
			sigma:    2.1175425068616867E-005,
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			tauOut:   5.0201385987195205E-003,
			sigmaOut: 2.1175425068616867E-005,
			dminOut:  1.8576185384092288E-002,
			dmin1Out: 1.8576185384092288E-002,
			dmin2Out: 5.2365600435162571E-002,
			dnOut:    1.9114649409197451E-002,
			dnm1Out:  1.8576185384092288E-002,
			dnm2Out:  5.2365600435162571E-002,
		},
		{
			z:        []float64{0.2471569222413904E+01, 0.2471445466333236E+01, 0.4896382518051712E-02, 0.2350431231346416E-02, 0.1186378431264915E+01, 0.1179822839486443E+01, 0.8149780515932184E-03, 0.3616937915375072E-03, 0.5236148327211592E+00, 0.5209996276036221E+00, 0.2766627272719901E-02, 0.5058054349403302E-02, 0.9525115502267366E+00, 0.1034766778881179E+01, 0.9233342160256496E-01, 0.1252490967185870E+00, 0.1403647802928048E+01, 0.1312087633296770E+01, 0.3870906568602875E-01, 0.2542350442532051E-01, 0.8617584837127339E+00, 0.8487220155581966E+00, 0.1740717486950262E-01, 0.2862783537884150E-01, 0.1395808011693333E+01, 0.1574348441670908E+01, 0.2121884039551361E+00, 0.8484373804386666E-01, 0.6295047433559955E+00, 0.1074794415251550E+01, 0.5351535485381410E+00, 0.1841287608083240E+00, 0.3698014604305115E+00, 0.7088504197799252E+00, 0.5281978587564573E+00, 0.9752771411128711E+00, 0.1308838344228090E+01, 0.4320210527858890E+00, 0.1034799882693896E+00, 0.2158225196628609E+00, 0.9010425466701916E+00, 0.7144140228254550E+00, 0.3421413441684364E-01, 0.3835134583138245E-01, 0.8008017657953598E+00, 0.1391311611427577E+01, 0.6338813300623194E+00, 0.3955915526975877E+00, 0.8682873190108105E+00, 0.1090644248011184E+01, 0.6229686202966810E+00, 0.4806932321292802E+00, 0.8415597376799326E+00, 0.4591804133211825E+00, 0.1033340463692495E+00, 0.1716839812178710E+00, 0.7629036530181579E+00, 0.1710844627144272E+01, 0.1124645093942705E+01, 0.4587247601659613E+00, 0.6978261813392677E+00, 0.5394016167466366E+00, 0.3053203341720497E+00, 0.2790520826393697E+00, 0.4929941693545132E+00, 0.7598997773325197E+00, 0.5509778292160957E+00, 0.1513401067044909E+00, 0.2087258457383731E+00, 0.1015190880092246E+00, 0.4915348757406203E-01, 0.2214892234681356E-01, 0.4574524632962537E-01, 0.1982969108114764E-01, 0.1253505697055357E-02, 0.1628595626045726E-02, 0.2576338363396270E-01, 0.1911464940919745E-01, 0.8149780515932184E-03, 0.3616937915375072E-03, 0.2117542506861687E-04},
			i0:       1,
			n0:       20,
			pp:       1,
			tau:      5.4769133315280185E-003,
			sigma:    5.0413140237881371E-003,
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			tauOut:   5.4769133315280185E-003,
			sigmaOut: 5.0413140237881371E-003,
			dminOut:  9.5622848228283271E-003,
			dmin1Out: 9.5622848228283271E-003,
			dmin2Out: 6.9533978479808370E-002,
			dnOut:    1.0856003705186750E-002,
			dnm1Out:  9.5622848228283271E-003,
			dnm2Out:  6.9533978479808370E-002,
		},
		{
			z:        []float64{0.2468318984233055E+01, 0.2471445466333236E+01, 0.1123474100024551E-02, 0.2350431231346416E-02, 0.1173584145846428E+01, 0.1179822839486443E+01, 0.1605699355811189E-03, 0.3616937915375072E-03, 0.5204201986859162E+00, 0.5209996276036221E+00, 0.1005707814522541E-01, 0.5058054349403302E-02, 0.1144481884123012E+01, 0.1034766778881179E+01, 0.1435914304680996E+00, 0.1252490967185870E+00, 0.1188442793922463E+01, 0.1312087633296770E+01, 0.1815610143690141E-01, 0.2542350442532051E-01, 0.8537168361686087E+00, 0.8487220155581966E+00, 0.5279290053521807E-01, 0.2862783537884150E-01, 0.1600922365848029E+01, 0.1574348441670908E+01, 0.5696064828871891E-01, 0.8484373804386666E-01, 0.1196485614439627E+01, 0.1074794415251550E+01, 0.1090859328498209E+00, 0.1841287608083240E+00, 0.1569564714711448E+01, 0.7088504197799252E+00, 0.2684440171930437E+00, 0.9752771411128711E+00, 0.3739226419241781E+00, 0.4320210527858890E+00, 0.4123490187575627E+00, 0.2158225196628609E+00, 0.3349394365677468E+00, 0.7144140228254550E+00, 0.1593084209965356E+00, 0.3835134583138245E-01, 0.1622117829797102E+01, 0.1391311611427577E+01, 0.2659792301064862E+00, 0.3955915526975877E+00, 0.1299881336702450E+01, 0.1090644248011184E+01, 0.1698038973078534E+00, 0.4806932321292802E+00, 0.4555835838996722E+00, 0.4591804133211825E+00, 0.6447216871142054E+00, 0.1716839812178710E+00, 0.1519370786864500E+01, 0.1710844627144272E+01, 0.1628548339973444E+00, 0.4587247601659613E+00, 0.6501219520571339E+00, 0.5394016167466366E+00, 0.3261720586281595E+00, 0.2790520826393697E+00, 0.5795909120773233E+00, 0.7598997773325197E+00, 0.2650819619788820E-01, 0.1513401067044909E+00, 0.9168290082662192E-01, 0.1015190880092246E+00, 0.4790492926791300E-02, 0.2214892234681356E-01, 0.1119088044887405E-01, 0.1982969108114764E-01, 0.2781732372482683E-02, 0.1628595626045726E-02, 0.1085600370518675E-01, 0.1911464940919745E-01, 0.1605699355811189E-03, 0.3616937915375072E-03},
			i0:       1,
			n0:       20,
			pp:       0,
			tau:      5.1216063611655054E-003,
			sigma:    1.0518227355316156E-002,
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			tauOut:   5.1216063611655054E-003,
			sigmaOut: 1.0518227355316156E-002,
			dminOut:  2.0601312480394186E-003,
			dmin1Out: 5.4371870398960158E-003,
			dmin2Out: 8.0023511442426670E-002,
			dnOut:    2.0601312480394186E-003,
			dnm1Out:  5.4371870398960158E-003,
			dnm2Out:  8.0023511442426670E-002,
		},
		{
			z:        []float64{0.2468318984233055E+01, 0.2464320851971913E+01, 0.1123474100024551E-02, 0.5350323562789559E-03, 0.1173584145846428E+01, 0.1168088077064565E+01, 0.1605699355811189E-03, 0.7153898701552432E-04, 0.5204201986859162E+00, 0.5252841314829605E+00, 0.1005707814522541E-01, 0.2191222436498315E-01, 0.1144481884123012E+01, 0.1261039483864963E+01, 0.1435914304680996E+00, 0.1353250258951489E+00, 0.1188442793922463E+01, 0.1066152263103050E+01, 0.1815610143690141E-01, 0.1453842008528346E-01, 0.8537168361686087E+00, 0.8868497102573779E+00, 0.5279290053521807E-01, 0.9530062900995111E-01, 0.1600922365848029E+01, 0.1557460778765631E+01, 0.5696064828871891E-01, 0.4375878814786067E-01, 0.1196485614439627E+01, 0.1256691152780422E+01, 0.1090859328498209E+00, 0.1362446379077657E+00, 0.1569564714711448E+01, 0.1696642487635560E+01, 0.2684440171930437E+00, 0.5916231430550117E-01, 0.3739226419241781E+00, 0.7219877400150740E+00, 0.4123490187575627E+00, 0.1912940350054112E+00, 0.3349394365677468E+00, 0.2978322161977056E+00, 0.1593084209965356E+00, 0.8676597630518320E+00, 0.1622117829797102E+01, 0.1015315690490590E+01, 0.2659792301064862E+00, 0.3405260456467969E+00, 0.1299881336702450E+01, 0.1124037582002341E+01, 0.1698038973078534E+00, 0.6882320425428856E-01, 0.4555835838996722E+00, 0.1026360460398424E+01, 0.6447216871142054E+00, 0.9544125430154021E+00, 0.1519370786864500E+01, 0.7226914714852769E+00, 0.1628548339973444E+00, 0.1465016632377001E+00, 0.6501219520571339E+00, 0.8246707410864278E+00, 0.3261720586281595E+00, 0.2292385937027206E+00, 0.5795909120773233E+00, 0.3717389082113253E+00, 0.2650819619788820E-01, 0.6537783023029759E-02, 0.9168290082662192E-01, 0.8481400436921797E-01, 0.4790492926791300E-02, 0.6320870478125323E-03, 0.1119088044887405E-01, 0.8218919412378699E-02, 0.2781732372482683E-02, 0.3674266095981827E-02, 0.1085600370518675E-01, 0.2060131248039419E-02, 0.1605699355811189E-03, 0.7153898701552432E-04, 0.2117542506861687E-04},
			i0:       1,
			n0:       20,
			pp:       1,
			tau:      1.2817878169164906E-003,
			sigma:    1.5639833716481661E-002,
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			tauOut:   1.2817878169164906E-003,
			sigmaOut: 1.5639833716481661E-002,
			dminOut:  6.0731361153288982E-005,
			dmin1Out: 6.8738708636769136E-003,
			dmin2Out: 8.1489512892123819E-002,
			dnOut:    6.0731361153288982E-005,
			dnm1Out:  6.8738708636769136E-003,
			dnm2Out:  8.1489512892123819E-002,
		},
		{
			z:        []float64{0.2463574096511276E+01, 0.2464320851971913E+01, 0.2536822079344948E-03, 0.5350323562789559E-03, 0.1166624146026729E+01, 0.1168088077064565E+01, 0.3221114082852138E-04, 0.7153898701552432E-04, 0.5458823568901986E+00, 0.5252841314829605E+00, 0.5061929508212644E-01, 0.2191222436498315E-01, 0.1344463426861069E+01, 0.1261039483864963E+01, 0.1073120173669855E+00, 0.1353250258951489E+00, 0.9720968780044319E+00, 0.1066152263103050E+01, 0.1326348631702415E-01, 0.1453842008528346E-01, 0.9676050651333883E+00, 0.8868497102573779E+00, 0.1533962535161303E+00, 0.9530062900995111E-01, 0.1446541525580445E+01, 0.1557460778765631E+01, 0.3801569533217738E-01, 0.4375878814786067E-01, 0.1353638307539094E+01, 0.1256691152780422E+01, 0.1707682473962209E+00, 0.1362446379077657E+00, 0.1583754766727924E+01, 0.1696642487635560E+01, 0.2697037855661164E-01, 0.5916231430550117E-01, 0.8850296086469572E+00, 0.7219877400150740E+00, 0.6437471225190403E-01, 0.1912940350054112E+00, 0.1099835479180717E+01, 0.2978322161977056E+00, 0.8009821360646626E+00, 0.8676597630518320E+00, 0.5535778122558079E+00, 0.1015315690490590E+01, 0.6914368034330997E+00, 0.3405260456467969E+00, 0.5001421950066134E+00, 0.1124037582002341E+01, 0.1412346654806686E+00, 0.6882320425428856E-01, 0.1838256550116241E+01, 0.1026360460398424E+01, 0.3752173792456719E+00, 0.9544125430154021E+00, 0.4926939676603885E+00, 0.7226914714852769E+00, 0.2452143584512202E+00, 0.1465016632377001E+00, 0.8074131885210117E+00, 0.8246707410864278E+00, 0.1055431169003394E+00, 0.2292385937027206E+00, 0.2714517865170992E+00, 0.3717389082113253E+00, 0.2042703660177667E-02, 0.6537783023029759E-02, 0.8212159993993635E-01, 0.8481400436921797E-01, 0.6326073178529442E-04, 0.6320870478125323E-03, 0.1054813695965874E-01, 0.8218919412378699E-02, 0.7176120699696391E-03, 0.3674266095981827E-02, 0.6073136115328898E-04, 0.2060131248039419E-02, 0.3221114082852138E-04, 0.7153898701552432E-04},
			i0:       1,
			n0:       20,
			pp:       0,
			tau:      5.6837241251038845E-005,
			sigma:    1.6921621533398150E-002,
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			tauOut:   5.6837241251038845E-005,
			sigmaOut: 1.6921621533398150E-002,
			dminOut:  3.1568086081919418E-009,
			dmin1Out: 1.0483100129151506E-002,
			dmin2Out: 8.1316774559040517E-002,
			dnOut:    3.1568086081919418E-009,
			dnm1Out:  1.0483100129151506E-002,
			dnm2Out:  8.1316774559040517E-002,
		},
		{
			z:        []float64{0.2463574096511276E+01, 0.2463770941477959E+01, 0.2536822079344948E-03, 0.1201214707955848E-03, 0.1166624146026729E+01, 0.1166479398455512E+01, 0.3221114082852138E-04, 0.1507398544447245E-04, 0.5458823568901986E+00, 0.5964297407456295E+00, 0.5061929508212644E-01, 0.1141052940222717E+00, 0.1344463426861069E+01, 0.1337613312964532E+01, 0.1073120173669855E+00, 0.7798791776646297E-01, 0.9720968780044319E+00, 0.9073156093137420E+00, 0.1326348631702415E-01, 0.1414482062243694E-01, 0.9676050651333883E+00, 0.1106799660785830E+01, 0.1533962535161303E+00, 0.2004825791345134E+00, 0.1446541525580445E+01, 0.1284017804536858E+01, 0.3801569533217738E-01, 0.4007693764646178E-01, 0.1353638307539094E+01, 0.1484272780047602E+01, 0.1707682473962209E+00, 0.1822138285193538E+00, 0.1583754766727924E+01, 0.1428454479523931E+01, 0.2697037855661164E-01, 0.1671007646458111E-01, 0.8850296086469572E+00, 0.9326374071930291E+00, 0.6437471225190403E-01, 0.7591545433480534E-01, 0.1099835479180717E+01, 0.1824845323669324E+01, 0.8009821360646626E+00, 0.2429827519008994E+00, 0.5535778122558079E+00, 0.1001975026546757E+01, 0.6914368034330997E+00, 0.3451350696526060E+00, 0.5001421950066134E+00, 0.2961849535934249E+00, 0.1412346654806686E+00, 0.8765656248686587E+00, 0.1838256550116241E+01, 0.1336851467252003E+01, 0.3752173792456719E+00, 0.1382856239786244E+00, 0.4926939676603885E+00, 0.5995658648917332E+00, 0.2452143584512202E+00, 0.3302211126778973E+00, 0.8074131885210117E+00, 0.5826783555022028E+00, 0.1055431169003394E+00, 0.4916926700063749E-01, 0.2714517865170992E+00, 0.2242683859353883E+00, 0.2042703660177667E-02, 0.7479881396448043E-03, 0.8212159993993635E-01, 0.8138003529082581E-01, 0.6326073178529442E-04, 0.8199589256196194E-05, 0.1054813695965874E-01, 0.1120071219912114E-01, 0.7176120699696391E-03, 0.3890963093641941E-05, 0.6073136115328898E-04, 0.3156808608191942E-08, 0.3221114082852138E-04, 0.1507398544447245E-04, 0.2117542506861687E-04},
			i0:       1,
			n0:       20,
			pp:       1,
			tau:      3.1557121791797713E-009,
			sigma:    1.6978458774649190E-002,
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			tauOut:   3.1557121791797713E-009,
			sigmaOut: 1.6978458774649190E-002,
			dminOut:  6.9684975813366743E-017,
			dmin1Out: 1.1199576261102989E-002,
			dmin2Out: 8.1067607231828140E-002,
			dnOut:    6.9684975813366743E-017,
			dnm1Out:  1.1199576261102989E-002,
			dnm2Out:  8.1067607231828140E-002,
		},
		{
			z:        []float64{0.2463891059793043E+01, 0.2463770941477959E+01, 0.5686908130061341E-04, 0.1201214707955848E-03, 0.1166437600203943E+01, 0.1166479398455512E+01, 0.7707718980490818E-05, 0.1507398544447245E-04, 0.7105273238932086E+00, 0.5964297407456295E+00, 0.2148105431436762E+00, 0.1141052940222717E+00, 0.1200790684431606E+01, 0.1337613312964532E+01, 0.5892755169139442E-01, 0.7798791776646297E-01, 0.8625328750890724E+00, 0.9073156093137420E+00, 0.1815059242254727E-01, 0.1414482062243694E-01, 0.1289131644342084E+01, 0.1106799660785830E+01, 0.1996872873596725E+00, 0.2004825791345134E+00, 0.1124407451667935E+01, 0.1284017804536858E+01, 0.5290351604133232E-01, 0.4007693764646178E-01, 0.1613583089369911E+01, 0.1484272780047602E+01, 0.1613081850537457E+00, 0.1822138285193538E+00, 0.1283856367779054E+01, 0.1428454479523931E+01, 0.1213877407087503E-01, 0.1671007646458111E-01, 0.9964140843012472E+00, 0.9326374071930291E+00, 0.1390325207358455E+00, 0.7591545433480534E-01, 0.1928795551678665E+01, 0.1824845323669324E+01, 0.1262252233392066E+00, 0.2429827519008994E+00, 0.1220884869704444E+01, 0.1001975026546757E+01, 0.8372928285471114E-01, 0.3451350696526060E+00, 0.1089021292451660E+01, 0.2961849535934249E+00, 0.1076046951396362E+01, 0.8765656248686587E+00, 0.3990901366785531E+00, 0.1336851467252003E+01, 0.2077509116934600E+00, 0.1382856239786244E+00, 0.7220360627204584E+00, 0.5995658648917332E+00, 0.2664862668525171E+00, 0.3302211126778973E+00, 0.3653613524946110E+00, 0.5826783555022028E+00, 0.3018138637972599E-01, 0.4916926700063749E-01, 0.1948349845395949E+00, 0.2242683859353883E+00, 0.3124249032854923E-03, 0.7479881396448043E-03, 0.8107580682108434E-01, 0.8138003529082581E-01, 0.1132782305976083E-05, 0.8199589256196194E-05, 0.1120346722419663E-01, 0.1120071219912114E-01, 0.1096359327194516E-11, 0.3890963093641941E-05, 0.6968497581336674E-16, 0.3156808608191942E-08, 0.7707718980490818E-05, 0.1507398544447245E-04},
			i0:       1,
			n0:       20,
			pp:       0,
			tau:      6.9684975806547287E-017,
			sigma:    1.6978461930361368E-002,
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			tauOut:   6.9684975806547287E-017,
			sigmaOut: 1.6978461930361368E-002,
			dminOut:  6.1629758220391547E-032,
			dmin1Out: 1.1203310405167735E-002,
			dmin2Out: 8.0927116373146771E-002,
			dnOut:    6.1629758220391547E-032,
			dnm1Out:  1.1203310405167735E-002,
			dnm2Out:  8.0927116373146771E-002,
		},
		{
			z:        []float64{0.2463891059793043E+01, 0.2463947928874343E+01, 0.5686908130061341E-04, 0.2692193042748079E-04, 0.1166437600203943E+01, 0.1166418385992496E+01, 0.7707718980490818E-05, 0.4695180568393632E-05, 0.7105273238932086E+00, 0.9253331718563164E+00, 0.2148105431436762E+00, 0.2787563517334627E+00, 0.1200790684431606E+01, 0.9809618843895378E+00, 0.5892755169139442E-01, 0.5181337969514327E-01, 0.8625328750890724E+00, 0.8288700878164763E+00, 0.1815059242254727E-01, 0.2822939734392020E-01, 0.1289131644342084E+01, 0.1460589534357837E+01, 0.1996872873596725E+00, 0.1537255119449346E+00, 0.1124407451667935E+01, 0.1023585455764333E+01, 0.5290351604133232E-01, 0.8339725654733963E-01, 0.1613583089369911E+01, 0.1691494017876317E+01, 0.1613081850537457E+00, 0.1224340957564512E+00, 0.1283856367779054E+01, 0.1173561046093478E+01, 0.1213877407087503E-01, 0.1030644761994533E-01, 0.9964140843012472E+00, 0.1125140157417147E+01, 0.1390325207358455E+00, 0.2383394688796517E+00, 0.1928795551678665E+01, 0.1816681306138221E+01, 0.1262252233392066E+00, 0.8482856339700598E-01, 0.1220884869704444E+01, 0.1219785589162149E+01, 0.8372928285471114E-01, 0.7475327847832687E-01, 0.1089021292451660E+01, 0.2090314965369696E+01, 0.1076046951396362E+01, 0.2054425921547012E+00, 0.3990901366785531E+00, 0.4013984562173118E+00, 0.2077509116934600E+00, 0.3737026089221466E+00, 0.7220360627204584E+00, 0.6148197206508288E+00, 0.2664862668525171E+00, 0.1583615157552351E+00, 0.3653613524946110E+00, 0.2371812231191019E+00, 0.3018138637972599E-01, 0.2479281399828426E-01, 0.1948349845395949E+00, 0.1703545954445960E+00, 0.3124249032854923E-03, 0.1486904479375115E-03, 0.8107580682108434E-01, 0.8092824915545274E-01, 0.1132782305976083E-05, 0.1568190288260776E-06, 0.1120346722419663E-01, 0.1120331040626409E-01, 0.1096359327194516E-11, 0.6819392699821255E-26, 0.6968497581336674E-16, 0.6162975822039155E-31, 0.7707718980490818E-05, 0.4695180568393632E-05, 0.2117542506861687E-04},
			i0:       1,
			n0:       20,
			pp:       1,
			tau:      6.1629758220391547E-032,
			sigma:    1.6978461930361441E-002,
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.6978461930361441E-002,
			dminOut:  6.1629758220391547E-032,
			dmin1Out: 1.1203288675083998E-002,
			dmin2Out: 8.0846453894262649E-002,
			dnOut:    6.1629758220391547E-032,
			dnm1Out:  1.1203288675083998E-002,
			dnm2Out:  8.0846453894262649E-002,
		},
		{
			z:        []float64{0.2463974850804771E+01, 0.2463947928874343E+01, 0.1274454348702788E-04, 0.2692193042748079E-04, 0.1166410336629578E+01, 0.1166418385992496E+01, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.1204085798823124E+01, 0.9253331718563164E+00, 0.2271012218143261E+00, 0.2787563517334627E+00, 0.8056740422703550E+00, 0.9809618843895378E+00, 0.5330513126246473E-01, 0.5181337969514327E-01, 0.8037943538979316E+00, 0.8288700878164763E+00, 0.5129615818002433E-01, 0.2822939734392020E-01, 0.1563018888122747E+01, 0.1460589534357837E+01, 0.1006713350698832E+00, 0.1537255119449346E+00, 0.1006311377241790E+01, 0.1023585455764333E+01, 0.1401812239704283E+00, 0.8339725654733963E-01, 0.1673746889662340E+01, 0.1691494017876317E+01, 0.8584564749956700E-01, 0.1224340957564512E+00, 0.1098021846213856E+01, 0.1173561046093478E+01, 0.1056099032774466E-01, 0.1030644761994533E-01, 0.1352918635969054E+01, 0.1125140157417147E+01, 0.3200390963041470E+00, 0.2383394688796517E+00, 0.1581470773231080E+01, 0.1816681306138221E+01, 0.6542811978092533E-01, 0.8482856339700598E-01, 0.1229110747859551E+01, 0.1219785589162149E+01, 0.1271308521106110E+00, 0.7475327847832687E-01, 0.2168626705413786E+01, 0.2090314965369696E+01, 0.3802606466401751E-01, 0.2054425921547012E+00, 0.7370750004754409E+00, 0.4013984562173118E+00, 0.3117182559112661E+00, 0.3737026089221466E+00, 0.4614629804947978E+00, 0.6148197206508288E+00, 0.8139413038408401E-01, 0.1583615157552351E+00, 0.1805799067333021E+00, 0.2371812231191019E+00, 0.2338892446571373E-01, 0.2479281399828426E-01, 0.1471143614268198E+00, 0.1703545954445960E+00, 0.8179526119010886E-04, 0.1486904479375115E-03, 0.8084661071329148E-01, 0.8092824915545274E-01, 0.2173118009582292E-07, 0.1568190288260776E-06, 0.1120328867508400E-01, 0.1120331040626409E-01, 0.3751376363572422E-55, 0.6819392699821255E-26},
			i0:       1,
			n0:       19,
			pp:       0,
			tau:      1.1203284285913290E-002,
			sigma:    1.6978461930361441E-002,
			i0Out:    1,
			n0Out:    19,
			ppOut:    0,
			tauOut:   1.1203284285913290E-002,
			sigmaOut: 1.6978461930361441E-002,
			dminOut:  8.9043085058082561E-010,
			dmin1Out: 6.9585227169791292E-002,
			dmin2Out: 0.11373841371027797,
			dnOut:    8.9043085058082561E-010,
			dnm1Out:  6.9585227169791292E-002,
			dnm2Out:  0.11373841371027797,
		},
		{
			z:        []float64{0.2463974850804771E+01, 0.2452784311062345E+01, 0.1274454348702788E-04, 0.6060609239813679E-05, 0.1166410336629578E+01, 0.1155204716501079E+01, 0.3724766654883956E-05, 0.3882375624867434E-05, 0.1204085798823124E+01, 0.1419979853975912E+01, 0.2271012218143261E+00, 0.1288536304732592E+00, 0.8056740422703550E+00, 0.7189222587736472E+00, 0.5330513126246473E-01, 0.5959804835594534E-01, 0.8037943538979316E+00, 0.7842891794360973E+00, 0.5129615818002433E-01, 0.1022287011292917E+00, 0.1563018888122747E+01, 0.1550258237777425E+01, 0.1006713350698832E+00, 0.6534828028921505E-01, 0.1006311377241790E+01, 0.1069941036637090E+01, 0.1401812239704283E+00, 0.2192904838448094E+00, 0.1673746889662340E+01, 0.1529098769031184E+01, 0.8584564749956700E-01, 0.6164441320989392E-01, 0.1098021846213856E+01, 0.1035735139045794E+01, 0.1056099032774466E-01, 0.1379518767882891E-01, 0.1352918635969054E+01, 0.1647959260308459E+01, 0.3200390963041470E+00, 0.3071268139247320E+00, 0.1581470773231080E+01, 0.1328568794801360E+01, 0.6542811978092533E-01, 0.6053010243026304E-01, 0.1229110747859551E+01, 0.1284508213253985E+01, 0.1271308521106110E+00, 0.2146341752620367E+00, 0.2168626705413786E+01, 0.1980815310529854E+01, 0.3802606466401751E-01, 0.1414976019284330E-01, 0.7370750004754409E+00, 0.1023440211907950E+01, 0.3117182559112661E+00, 0.1405518698344743E+00, 0.4614629804947978E+00, 0.3911019567584944E+00, 0.8139413038408401E-01, 0.3758136265851575E-01, 0.1805799067333021E+00, 0.1551841842545868E+00, 0.2338892446571373E-01, 0.2217266343062855E-01, 0.1471143614268198E+00, 0.1138202089714681E+00, 0.8179526119010886E-04, 0.5809925758690545E-04, 0.8084661071329148E-01, 0.6958524890097138E-01, 0.2173118009582292E-07, 0.3498739858072780E-08, 0.1120328867508400E-01, 0.8904308505808256E-09, 0.3751376363572422E-55, 0.3882375624867434E-05, 0.1697846193036144E-01},
			i0:       1,
			n0:       19,
			pp:       1,
			tau:      8.9043080564395014E-010,
			sigma:    2.8181746216274728E-002,
			i0Out:    1,
			n0Out:    19,
			ppOut:    1,
			tauOut:   8.9043080564395014E-010,
			sigmaOut: 2.8181746216274728E-002,
			dminOut:  1.3962105636374437E-019,
			dmin1Out: 6.9544122471813200E-002,
			dmin2Out: 9.8247027562672340E-002,
			dnOut:    1.3962105636374437E-019,
			dnm1Out:  6.9544122471813200E-002,
			dnm2Out:  9.8247027562672340E-002,
		},
		{
			z:        []float64{0.2452790370781154E+01, 0.2452784311062345E+01, 0.2854399814229969E-05, 0.6060609239813679E-05, 0.1155205743586459E+01, 0.1155204716501079E+01, 0.4772219324121025E-05, 0.3882375624867434E-05, 0.1548828711339416E+01, 0.1419979853975912E+01, 0.5981019230390531E-01, 0.1288536304732592E+00, 0.7187101139352565E+00, 0.7189222587736472E+00, 0.6503610222645050E-01, 0.5959804835594534E-01, 0.8214817774485077E+00, 0.7842891794360973E+00, 0.1929207566298143E+00, 0.1022287011292917E+00, 0.1422685760546395E+01, 0.1550258237777425E+01, 0.4914564318703864E-01, 0.6534828028921505E-01, 0.1240085876404429E+01, 0.1069941036637090E+01, 0.2703980549150243E+00, 0.2192904838448094E+00, 0.1320345126435623E+01, 0.1529098769031184E+01, 0.4835651195207321E-01, 0.6164441320989392E-01, 0.1001173813882118E+01, 0.1035735139045794E+01, 0.2270725319399537E-01, 0.1379518767882891E-01, 0.1932378820148765E+01, 0.1647959260308459E+01, 0.2111589595024383E+00, 0.3071268139247320E+00, 0.1177939936838754E+01, 0.1328568794801360E+01, 0.6600626338337756E-01, 0.6053010243026304E-01, 0.1433136124242214E+01, 0.1284508213253985E+01, 0.2966575563411978E+00, 0.2146341752620367E+00, 0.1698307513491068E+01, 0.1980815310529854E+01, 0.8526979628348903E-02, 0.1414976019284330E-01, 0.1155465101223645E+01, 0.1023440211907950E+01, 0.4757401262929915E-01, 0.1405518698344743E+00, 0.3811093058972801E+00, 0.3911019567584944E+00, 0.1530278326215801E-01, 0.3758136265851575E-01, 0.1620540635326265E+00, 0.1551841842545868E+00, 0.1557318051836492E-01, 0.2217266343062855E-01, 0.9830512682025924E-01, 0.1138202089714681E+00, 0.4112553872737890E-04, 0.5809925758690545E-04, 0.6954412597055305E-01, 0.6958524890097138E-01, 0.4479725446695418E-16, 0.3498739858072780E-08, 0.1396210563637444E-18, 0.8904308505808256E-09, 0.2854399814229969E-05, 0.3882375624867434E-05},
			i0:       1,
			n0:       19,
			pp:       0,
			tau:      1.3962105636374427E-019,
			sigma:    2.8181747106705537E-002,
			i0Out:    1,
			n0Out:    19,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 2.8181747106705537E-002,
			dminOut:  1.3962105636374430E-019,
			dmin1Out: 6.9512134440652351E-002,
			dmin2Out: 8.9358776708560295E-002,
			dnOut:    1.3962105636374430E-019,
			dnm1Out:  6.9512134440652351E-002,
			dnm2Out:  8.9358776708560295E-002,
		},
		{
			z:        []float64{0.2452790370781154E+01, 0.2452793225180968E+01, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1155205743586459E+01, 0.1155209171453138E+01, 0.4772219324121025E-05, 0.6398278760815103E-05, 0.1548828711339416E+01, 0.1608632505364561E+01, 0.5981019230390531E-01, 0.2672219415054498E-01, 0.7187101139352565E+00, 0.7570240220111620E+00, 0.6503610222645050E-01, 0.7057368234283014E-01, 0.8214817774485077E+00, 0.9438288517354918E+00, 0.1929207566298143E+00, 0.2908001941945216E+00, 0.1422685760546395E+01, 0.1181031209538913E+01, 0.4914564318703864E-01, 0.5160305461093759E-01, 0.1240085876404429E+01, 0.1458880876708516E+01, 0.2703980549150243E+00, 0.2447209773632920E+00, 0.1320345126435623E+01, 0.1123980661024405E+01, 0.4835651195207321E-01, 0.4307304847484577E-01, 0.1001173813882118E+01, 0.9808080186012680E+00, 0.2270725319399537E-01, 0.4473761868138884E-01, 0.1932378820148765E+01, 0.2098800160969815E+01, 0.2111589595024383E+00, 0.1185117935689049E+00, 0.1177939936838754E+01, 0.1125434406653226E+01, 0.6600626338337756E-01, 0.8405284210411720E-01, 0.1433136124242214E+01, 0.1645740838479294E+01, 0.2966575563411978E+00, 0.3061331074057168E+00, 0.1698307513491068E+01, 0.1400701385713701E+01, 0.8526979628348903E-02, 0.7034066989504621E-02, 0.1155465101223645E+01, 0.1196005046863439E+01, 0.4757401262929915E-01, 0.1515955052150448E-01, 0.3811093058972801E+00, 0.3812525386379337E+00, 0.1530278326215801E-01, 0.6504555274179689E-02, 0.1620540635326265E+00, 0.1711226887768117E+00, 0.1557318051836492E-01, 0.8946350111698947E-02, 0.9830512682025924E-01, 0.8939990224728768E-01, 0.4112553872737890E-04, 0.3199152990069927E-04, 0.6954412597055305E-01, 0.6951213444065239E-01, 0.4479725446695418E-16, 0.8997910999570377E-34, 0.2818174710670554E-01},
			i0:       1,
			n0:       18,
			pp:       1,
			tau:      6.8088572105595116E-002,
			sigma:    2.8181747106705537E-002,
			i0Out:    1,
			n0Out:    18,
			ppOut:    1,
			tauOut:   6.8088572105595116E-002,
			sigmaOut: 2.8181747106705537E-002,
			dminOut:  1.2643548659038301E-003,
			dmin1Out: 1.3935943135754067E-002,
			dmin2Out: 9.9495800084882416E-002,
			dnOut:    1.2643548659038301E-003,
			dnm1Out:  1.3935943135754067E-002,
			dnm2Out:  9.9495800084882416E-002,
		},
		{
			z:        []float64{0.2384705997428018E+01, 0.2452793225180968E+01, 0.6512368848977837E-06, 0.1344352644992036E-05, 0.1087126346389419E+01, 0.1155209171453138E+01, 0.9467601652019932E-05, 0.6398278760815103E-05, 0.1567256659807859E+01, 0.1608632505364561E+01, 0.1290748567965171E-01, 0.2672219415054498E-01, 0.7466016465687454E+00, 0.7570240220111620E+00, 0.8921689079377818E-01, 0.7057368234283014E-01, 0.1077323583030640E+01, 0.9438288517354918E+00, 0.3187938243378626E+00, 0.2908001941945216E+00, 0.8457518677063927E+00, 0.1181031209538913E+01, 0.8901276181133644E-01, 0.5160305461093759E-01, 0.1546500520154876E+01, 0.1458880876708516E+01, 0.1778606876095877E+00, 0.2447209773632920E+00, 0.9211044497840675E+00, 0.1123980661024405E+01, 0.4586493023634136E-01, 0.4307304847484577E-01, 0.9115921349407204E+00, 0.9808080186012680E+00, 0.1030014605117353E+00, 0.4473761868138884E-01, 0.2046221921921390E+01, 0.2098800160969815E+01, 0.6518220171905387E-01, 0.1185117935689049E+00, 0.1076216474932694E+01, 0.1125434406653226E+01, 0.1285328723941422E+00, 0.8405284210411720E-01, 0.1755252501385274E+01, 0.1645740838479294E+01, 0.2442959445536252E+00, 0.3061331074057168E+00, 0.1095350936043985E+01, 0.1400701385713701E+01, 0.7680442260639305E-02, 0.7034066989504621E-02, 0.1135395583018709E+01, 0.1196005046863439E+01, 0.5090399511302622E-02, 0.1515955052150448E-01, 0.3145781222952156E+00, 0.3812525386379337E+00, 0.3538316586334169E-02, 0.6504555274179689E-02, 0.1084421501965814E+00, 0.1711226887768117E+00, 0.7375387005938499E-02, 0.8946350111698947E-02, 0.1396793466565477E-01, 0.8939990224728768E-01, 0.1592074691534444E-03, 0.3199152990069927E-04, 0.1264354865903830E-02, 0.6951213444065239E-01, 0.6512368848977837E-06, 0.1344352644992036E-05},
			i0:       1,
			n0:       18,
			pp:       0,
			tau:      1.2463174543591322E-003,
			sigma:    9.6270319212300656E-002,
			i0Out:    1,
			n0Out:    18,
			ppOut:    0,
			tauOut:   1.2463174543591322E-003,
			sigmaOut: 9.6270319212300656E-002,
			dminOut:  1.2236274461701734E-006,
			dmin1Out: 1.1812799375517376E-002,
			dmin2Out: 0.10597947881780349,
			dnOut:    1.2236274461701734E-006,
			dnm1Out:  1.1812799375517376E-002,
			dnm2Out:  0.10597947881780349,
		},
		{
			z:        []float64{0.2384705997428018E+01, 0.2383460331210544E+01, 0.6512368848977837E-06, 0.2970373645586866E-06, 0.1087126346389419E+01, 0.1085889199499347E+01, 0.9467601652019932E-05, 0.1366452649899944E-04, 0.1567256659807859E+01, 0.1578904163506652E+01, 0.1290748567965171E-01, 0.6103442048115082E-02, 0.7466016465687454E+00, 0.8284687778600495E+00, 0.8921689079377818E-01, 0.1160157908485996E+00, 0.1077323583030640E+01, 0.1278855299065544E+01, 0.3187938243378626E+00, 0.2108295383723412E+00, 0.8457518677063927E+00, 0.7226887736910288E+00, 0.8901276181133644E-01, 0.1904807262171572E+00, 0.1546500520154876E+01, 0.1532634164092948E+01, 0.1778606876095877E+00, 0.1068932656188067E+00, 0.9211044497840675E+00, 0.8588297969472432E+00, 0.4586493023634136E-01, 0.4868264913684867E-01, 0.9115921349407204E+00, 0.9646646288612479E+00, 0.1030014605117353E+00, 0.2184840619043245E+00, 0.2046221921921390E+01, 0.1891673744281760E+01, 0.6518220171905387E-01, 0.3708364593761750E-01, 0.1076216474932694E+01, 0.1166419383934860E+01, 0.1285328723941422E+00, 0.1934189785315259E+00, 0.1755252501385274E+01, 0.1804883149953014E+01, 0.2442959445536252E+00, 0.1482587897978486E+00, 0.1095350936043985E+01, 0.9535262710524167E+00, 0.7680442260639305E-02, 0.9145359161143373E-02, 0.1135395583018709E+01, 0.1130094305914509E+01, 0.5090399511302622E-02, 0.1416986451145964E-02, 0.3145781222952156E+00, 0.3154531349760448E+00, 0.3538316586334169E-02, 0.1216353924418735E-02, 0.1084421501965814E+00, 0.1133548658237420E+00, 0.7375387005938499E-02, 0.9088178357782586E-03, 0.1396793466565477E-01, 0.1197200684467082E-01, 0.1592074691534444E-03, 0.1681378409852773E-04, 0.1264354865903830E-02, 0.1223627446170173E-05, 0.6512368848977837E-06, 0.2970373645586866E-06, 0.2818174710670554E-01},
			i0:       1,
			n0:       18,
			pp:       1,
			tau:      1.2218877946904154E-006,
			sigma:    9.7516636666659787E-002,
			i0Out:    1,
			n0Out:    18,
			ppOut:    1,
			tauOut:   1.2218877946904154E-006,
			sigmaOut: 9.7516636666659787E-002,
			dminOut:  9.5986727915820745E-012,
			dmin1Out: 1.1875197612370336E-002,
			dmin2Out: 0.11291768719739126,
			dnOut:    9.5986727915820745E-012,
			dnm1Out:  1.1875197612370336E-002,
			dnm2Out:  0.11291768719739126,
		},
		{
			z:        []float64{0.2383459406360114E+01, 0.2383460331210544E+01, 0.1353283656358167E-06, 0.2970373645586866E-06, 0.1085901506809686E+01, 0.1085889199499347E+01, 0.1986826396898849E-04, 0.1366452649899944E-04, 0.1584986515403004E+01, 0.1578904163506652E+01, 0.3190255011763210E-02, 0.6103442048115082E-02, 0.9412930918090910E+00, 0.8284687778600495E+00, 0.1576208411525267E+00, 0.1160157908485996E+00, 0.1332062774397564E+01, 0.1278855299065544E+01, 0.1143821023097510E+00, 0.2108295383723412E+00, 0.7987861757106404E+00, 0.7226887736910288E+00, 0.3654761154847582E+00, 0.1904807262171572E+00, 0.1274050092339202E+01, 0.1532634164092948E+01, 0.7205613198290631E-01, 0.1068932656188067E+00, 0.8354550922133908E+00, 0.8588297969472432E+00, 0.5621179414582513E-01, 0.4868264913684867E-01, 0.1126935674731953E+01, 0.9646646288612479E+00, 0.3667472533840468E+00, 0.2184840619043245E+00, 0.1562008914947536E+01, 0.1891673744281760E+01, 0.2769195683500124E-01, 0.3708364593761750E-01, 0.1332145183743590E+01, 0.1166419383934860E+01, 0.2620575140703801E+00, 0.1934189785315259E+00, 0.1691083203792688E+01, 0.1804883149953014E+01, 0.8359650824372877E-01, 0.1482587897978486E+00, 0.8790739000820365E+00, 0.9535262710524167E+00, 0.1175682535061811E-01, 0.9145359161143373E-02, 0.1119753245127243E+01, 0.1130094305914509E+01, 0.3991886785572840E-03, 0.1416986451145964E-02, 0.3162690783341116E+00, 0.3154531349760448E+00, 0.4359567385560517E-03, 0.1216353924418735E-02, 0.1138265050331695E+00, 0.1133548658237420E+00, 0.9558734450579545E-04, 0.9088178357782586E-03, 0.1189201139646886E-01, 0.1197200684467082E-01, 0.1730052806966466E-08, 0.1681378409852773E-04, 0.9598672791582074E-11, 0.1223627446170173E-05, 0.1353283656358167E-06, 0.2970373645586866E-06},
			i0:       1,
			n0:       18,
			pp:       0,
			tau:      9.5986713933289272E-012,
			sigma:    9.7517858554454467E-002,
			i0Out:    1,
			n0Out:    18,
			ppOut:    0,
			tauOut:   9.5986713933289272E-012,
			sigmaOut: 9.7517858554454467E-002,
			dminOut:  6.6174125886815435E-022,
			dmin1Out: 1.1882019541390456E-002,
			dmin2Out: 0.11366976186212303,
			dnOut:    6.6174125886815435E-022,
			dnm1Out:  1.1882019541390456E-002,
			dnm2Out:  0.11366976186212303,
		},
		{
			z:        []float64{0.2383459406360114E+01, 0.2383459541678881E+01, 0.1353283656358167E-06, 0.6165545233233256E-07, 0.1085901506809686E+01, 0.1085921313408604E+01, 0.1986826396898849E-04, 0.2899927470478232E-04, 0.1584986515403004E+01, 0.1588147771130463E+01, 0.3190255011763210E-02, 0.1890859942802736E-02, 0.9412930918090910E+00, 0.1097023073009216E+01, 0.1576208411525267E+00, 0.1913914667196325E+00, 0.1332062774397564E+01, 0.1255053409978084E+01, 0.1143821023097510E+00, 0.7279916643176543E-01, 0.7987861757106404E+00, 0.1091463124754034E+01, 0.3654761154847582E+00, 0.4266153094141972E+00, 0.1274050092339202E+01, 0.9194909148983121E+00, 0.7205613198290631E-01, 0.6547064404326042E-01, 0.8354550922133908E+00, 0.8261962423063568E+00, 0.5621179414582513E-01, 0.7667315937770838E-01, 0.1126935674731953E+01, 0.1417009768728692E+01, 0.3667472533840468E+00, 0.4042756034295816E+00, 0.1562008914947536E+01, 0.1185425268343357E+01, 0.2769195683500124E-01, 0.3111938635974581E-01, 0.1332145183743590E+01, 0.1563083311444625E+01, 0.2620575140703801E+00, 0.2835172362390010E+00, 0.1691083203792688E+01, 0.1491162475787817E+01, 0.8359650824372877E-01, 0.4928202642453804E-01, 0.8790739000820365E+00, 0.8415486989985177E+00, 0.1175682535061811E-01, 0.1564347179719430E-01, 0.1119753245127243E+01, 0.1104508961999007E+01, 0.3991886785572840E-03, 0.1143051254380294E-03, 0.3162690783341116E+00, 0.3165907299376309E+00, 0.4359567385560517E-03, 0.1567431614478116E-03, 0.1138265050331695E+00, 0.1137653492066288E+00, 0.9558734450579545E-04, 0.9991845479738400E-05, 0.1189201139646886E-01, 0.1188202127144326E-01, 0.1730052806966466E-08, 0.1397591405272086E-17, 0.9598672791582074E-11, 0.6617412588681544E-21, 0.1353283656358167E-06, 0.6165545233233256E-07, 0.2818174710670554E-01},
			i0:       1,
			n0:       18,
			pp:       1,
			tau:      6.6174125886815426E-022,
			sigma:    9.7517858564053150E-002,
			i0Out:    1,
			n0Out:    18,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 9.7517858564053150E-002,
			dminOut:  6.6174125886815426E-022,
			dmin1Out: 1.1880977265773463E-002,
			dmin2Out: 0.11370904624558997,
			dnOut:    6.6174125886815426E-022,
			dnm1Out:  1.1880977265773463E-002,
			dnm2Out:  0.11370904624558997,
		},
		{
			z:        []float64{0.2383459603334333E+01, 0.2383459541678881E+01, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.1085950284592642E+01, 0.1085921313408604E+01, 0.4240998334843304E-04, 0.2899927470478232E-04, 0.1589996221089918E+01, 0.1588147771130463E+01, 0.1304604978030436E-02, 0.1890859942802736E-02, 0.1287109934750819E+01, 0.1097023073009216E+01, 0.1866247058326724E+00, 0.1913914667196325E+00, 0.1141227870577177E+01, 0.1255053409978084E+01, 0.6962466280543776E-01, 0.7279916643176543E-01, 0.1448453771362794E+01, 0.1091463124754034E+01, 0.2708190685256155E+00, 0.4266153094141972E+00, 0.7141424904159570E+00, 0.9194909148983121E+00, 0.7574342769943966E-01, 0.6547064404326042E-01, 0.8271259739846255E+00, 0.8261962423063568E+00, 0.1313543755784947E+00, 0.7667315937770838E-01, 0.1689930996579779E+01, 0.1417009768728692E+01, 0.2835846650840220E+00, 0.4042756034295816E+00, 0.9329599896190804E+00, 0.1185425268343357E+01, 0.5213749144931325E-01, 0.3111938635974581E-01, 0.1794463056234313E+01, 0.1563083311444625E+01, 0.2355970842920850E+00, 0.2835172362390010E+00, 0.1304847417920270E+01, 0.1491162475787817E+01, 0.3178396542921673E-01, 0.4928202642453804E-01, 0.8254082053664953E+00, 0.8415486989985177E+00, 0.2093310277804656E-01, 0.1564347179719430E-01, 0.1083690164346398E+01, 0.1104508961999007E+01, 0.3339325601415243E-04, 0.1143051254380294E-03, 0.3167140798430647E+00, 0.3165907299376309E+00, 0.5630296103885627E-04, 0.1567431614478116E-03, 0.1137190380910697E+00, 0.1137653492066288E+00, 0.1044005669799533E-05, 0.9991845479738400E-05, 0.1188097726577346E-01, 0.1188202127144326E-01, 0.7784240935906335E-37, 0.1397591405272086E-17},
			i0:       1,
			n0:       17,
			pp:       0,
			tau:      1.1880832155707781E-002,
			sigma:    9.7517858564053150E-002,
			i0Out:    1,
			n0Out:    17,
			ppOut:    0,
			tauOut:   1.1880832155707781E-002,
			sigmaOut: 9.7517858564053150E-002,
			dminOut:  2.3287035572430725E-008,
			dmin1Out: 0.10181720511285566,
			dmin2Out: 0.30482311636216664,
			dnOut:    2.3287035572430725E-008,
			dnm1Out:  0.10181720511285566,
			dnm2Out:  0.30482311636216664,
		},
		{
			z:        []float64{0.2383459603334333E+01, 0.2371578799269292E+01, 0.2809066689524106E-07, 0.1286276792433928E-07, 0.1085950284592642E+01, 0.1074111849557515E+01, 0.4240998334843304E-04, 0.6277904232066118E-04, 0.1589996221089918E+01, 0.1579357214869920E+01, 0.1304604978030436E-02, 0.1063198377376995E-02, 0.1287109934750819E+01, 0.1460790610050406E+01, 0.1866247058326724E+00, 0.1457986614708343E+00, 0.1141227870577177E+01, 0.1053173039756072E+01, 0.6962466280543776E-01, 0.9575644420574685E-01, 0.1448453771362794E+01, 0.1611635563526955E+01, 0.2708190685256155E+00, 0.1200044280642223E+00, 0.7141424904159570E+00, 0.6580006578954666E+00, 0.7574342769943966E-01, 0.9521169265880262E-01, 0.8271259739846255E+00, 0.8513878247486099E+00, 0.1313543755784947E+00, 0.2607270439790754E+00, 0.1689930996579779E+01, 0.1700907785529018E+01, 0.2835846650840220E+00, 0.1555482010511415E+00, 0.9329599896190804E+00, 0.8176684478615445E+00, 0.5213749144931325E-01, 0.1144214412274440E+00, 0.1794463056234313E+01, 0.1903757867143247E+01, 0.2355970842920850E+00, 0.1614796988702029E+00, 0.1304847417920270E+01, 0.1163270852323576E+01, 0.3178396542921673E-01, 0.2255256874351997E-01, 0.8254082053664953E+00, 0.8119079072453143E+00, 0.2093310277804656E-01, 0.2794035799797570E-01, 0.1083690164346398E+01, 0.1043902367448729E+01, 0.3339325601415243E-04, 0.1013132519024162E-04, 0.3167140798430647E+00, 0.3048794193232055E+00, 0.5630296103885627E-04, 0.2100082250625822E-04, 0.1137190380910697E+00, 0.1018182491185255E+00, 0.1044005669799533E-05, 0.1218230301111122E-06, 0.1188097726577346E-01, 0.2328703557243073E-07, 0.7784240935906335E-37, 0.1286276792433928E-07, 0.9751785856405315E-01},
			i0:       1,
			n0:       17,
			pp:       1,
			tau:      2.3287007705477136E-008,
			sigma:    0.10939869071976092,
			i0Out:    1,
			n0Out:    17,
			ppOut:    1,
			tauOut:   2.3287007705477136E-008,
			sigmaOut: 0.10939869071976092,
			dminOut:  2.6961654281145418E-018,
			dmin1Out: 0.10181121275944585,
			dmin2Out: 0.30487633314418217,
			dnOut:    2.6961654281145418E-018,
			dnm1Out:  0.10181121275944585,
			dnm2Out:  0.30487633314418217,
		},
		{
			z:        []float64{0.2371578788845052E+01, 0.2371578799269292E+01, 0.5825676764620370E-08, 0.1286276792433928E-07, 0.1074174599487151E+01, 0.1074111849557515E+01, 0.9230392664199863E-04, 0.6277904232066118E-04, 0.1580328086033647E+01, 0.1579357214869920E+01, 0.9827770701659698E-03, 0.1063198377376995E-02, 0.1605606471164067E+01, 0.1460790610050406E+01, 0.9563440497488795E-01, 0.1457986614708343E+00, 0.1053295055699924E+01, 0.1053173039756072E+01, 0.1465159169633776E+00, 0.9575644420574685E-01, 0.1585124051340792E+01, 0.1611635563526955E+01, 0.4981502397231050E-01, 0.1200044280642223E+00, 0.7033973032949510E+00, 0.6580006578954666E+00, 0.1152436546510613E+00, 0.9521169265880262E-01, 0.9968711907896164E+00, 0.8513878247486099E+00, 0.4448645553200346E+00, 0.2607270439790754E+00, 0.1411591407973117E+01, 0.1700907785529018E+01, 0.9010174998427344E-01, 0.1555482010511415E+00, 0.8419881158177074E+00, 0.8176684478615445E+00, 0.2587099684834221E+00, 0.1144214412274440E+00, 0.1806527574243020E+01, 0.1903757867143247E+01, 0.1039810460775319E+00, 0.1614796988702029E+00, 0.1081842351702556E+01, 0.1163270852323576E+01, 0.1692539477932339E-01, 0.2255256874351997E-01, 0.8229228471769590E+00, 0.8119079072453143E+00, 0.3544318396494814E-01, 0.2794035799797570E-01, 0.1008469291521964E+01, 0.1043902367448729E+01, 0.3062892015595061E-05, 0.1013132519024162E-04, 0.3048973339666884E+00, 0.3048794193232055E+00, 0.7013072071892081E-05, 0.2100082250625822E-04, 0.1018113345824760E+00, 0.1018182491185255E+00, 0.2786425742647189E-13, 0.1218230301111122E-06, 0.2696165428114542E-17, 0.2328703557243073E-07, 0.5825676764620370E-08, 0.1286276792433928E-07},
			i0:       1,
			n0:       17,
			pp:       0,
			tau:      2.6961654281138038E-018,
			sigma:    0.10939871400676864,
			i0Out:    1,
			n0Out:    17,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.10939871400676864,
			dminOut:  2.6961654281138038E-018,
			dmin1Out: 0.10180899282350273,
			dmin2Out: 0.30489636738914000,
			dnOut:    2.6961654281138038E-018,
			dnm1Out:  0.10180899282350273,
			dnm2Out:  0.30489636738914000,
		},
		{
			z:        []float64{0.2371578788845052E+01, 0.2371578794670729E+01, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.1074174599487151E+01, 0.1074266900775131E+01, 0.9230392664199863E-04, 0.1357860766428602E-03, 0.1580328086033647E+01, 0.1581175077027170E+01, 0.9827770701659698E-03, 0.9979623676695663E-03, 0.1605606471164067E+01, 0.1700242913771285E+01, 0.9563440497488795E-01, 0.5924520849284005E-01, 0.1053295055699924E+01, 0.1140565764170461E+01, 0.1465159169633776E+00, 0.2036234219705987E+00, 0.1585124051340792E+01, 0.1431315653342504E+01, 0.4981502397231050E-01, 0.2448080089382756E-01, 0.7033973032949510E+00, 0.7941601570521848E+00, 0.1152436546510613E+00, 0.1446598374682775E+00, 0.9968711907896164E+00, 0.1297075908641373E+01, 0.4448645553200346E+00, 0.4841405038964208E+00, 0.1411591407973117E+01, 0.1017552654060970E+01, 0.9010174998427344E-01, 0.7455594793877945E-01, 0.8419881158177074E+00, 0.1026142136362350E+01, 0.2587099684834221E+00, 0.4554599945126984E+00, 0.1806527574243020E+01, 0.1455048625807853E+01, 0.1039810460775319E+00, 0.7731088667813635E-01, 0.1081842351702556E+01, 0.1021456859803743E+01, 0.1692539477932339E-01, 0.1363571444815687E-01, 0.8229228471769590E+00, 0.8447303166937503E+00, 0.3544318396494814E-01, 0.4231334180394086E-01, 0.1008469291521964E+01, 0.9661590126100381E+00, 0.3062892015595061E-05, 0.9665775484099522E-06, 0.3048973339666884E+00, 0.3049033804612119E+00, 0.7013072071892081E-05, 0.2341758973227438E-05, 0.1018113345824760E+00, 0.1018089928235306E+00, 0.2786425742647189E-13, 0.7379175991216932E-30, 0.1093987140067686E+00},
			i0:       1,
			n0:       16,
			pp:       1,
			tau:      0.10180650470263587,
			sigma:    0.10939871400676864,
			i0Out:    1,
			n0Out:    16,
			ppOut:    1,
			tauOut:   0.10180650470263587,
			sigmaOut: 0.10939871400676864,
			dminOut:  1.3142485785350155E-006,
			dmin1Out: 0.20309651253108729,
			dmin2Out: 0.57610166617362735,
			dnOut:    1.3142485785350155E-006,
			dnm1Out:  0.20309651253108729,
			dnm2Out:  0.81137144083676127,
		},
		{
			z:        []float64{0.2269772292606755E+01, 0.2371578794670729E+01, 0.1248859578385863E-08, 0.2638661645752538E-08, 0.9725961809002787E+00, 0.1074266900775131E+01, 0.2207509801202943E-03, 0.1357860766428602E-03, 0.1480145783712083E+01, 0.1581175077027170E+01, 0.1146359002276934E-02, 0.9979623676695663E-03, 0.1656535258559212E+01, 0.1700242913771285E+01, 0.4079180092843103E-01, 0.5924520849284005E-01, 0.1201590880509993E+01, 0.1140565764170461E+01, 0.2425529321011353E+00, 0.2036234219705987E+00, 0.1111437017432560E+01, 0.1431315653342504E+01, 0.1749237822536809E-01, 0.2448080089382756E-01, 0.8195211115924584E+00, 0.7941601570521848E+00, 0.2289566278084960E+00, 0.1446598374682775E+00, 0.1450453280026662E+01, 0.1297075908641373E+01, 0.3396444831847066E+00, 0.4841405038964208E+00, 0.6506576141124067E+00, 0.1017552654060970E+01, 0.1175810411452473E+00, 0.7455594793877945E-01, 0.1262214585027166E+01, 0.1026142136362350E+01, 0.5250426092262996E+00, 0.4554599945126984E+00, 0.9055103985570538E+00, 0.1455048625807853E+01, 0.8721019179982008E-01, 0.7731088667813635E-01, 0.8460758777494440E+00, 0.1021456859803743E+01, 0.1361402882064932E-01, 0.1363571444815687E-01, 0.7716231249744060E+00, 0.8447303166937503E+00, 0.5298106707064092E-01, 0.4231334180394086E-01, 0.8113724074143097E+00, 0.9661590126100381E+00, 0.3632274887524200E-06, 0.9665775484099522E-06, 0.2030988542900605E+00, 0.3049033804612119E+00, 0.1173872316183806E-05, 0.2341758973227438E-05, 0.1314248578535016E-05, 0.1018089928235306E+00, 0.1248859578385863E-08, 0.2638661645752538E-08},
			i0:       1,
			n0:       16,
			pp:       0,
			tau:      1.3142409824136984E-006,
			sigma:    0.21120521870940451,
			i0Out:    1,
			n0Out:    16,
			ppOut:    0,
			tauOut:   1.3142409824136984E-006,
			sigmaOut: 0.21120521870940451,
			dminOut:  8.4118841991824567E-018,
			dmin1Out: 0.20309744276965425,
			dmin2Out: 0.50018599148866161,
			dnOut:    8.4118841991824567E-018,
			dnm1Out:  0.20309744276965425,
			dnm2Out:  0.75834178007645303,
		},
		{
			z:        []float64{0.2269772292606755E+01, 0.2269770979614632E+01, 0.1248859578385863E-08, 0.5351359530665278E-09, 0.9725961809002787E+00, 0.9728156171042807E+00, 0.2207509801202943E-03, 0.3358741643642203E-03, 0.1480145783712083E+01, 0.1480954954309013E+01, 0.1146359002276934E-02, 0.1282269997958535E-02, 0.1656535258559212E+01, 0.1696043475248702E+01, 0.4079180092843103E-01, 0.2889964597634762E-01, 0.1201590880509993E+01, 0.1415242852393798E+01, 0.2425529321011353E+00, 0.1904848393814713E+00, 0.1111437017432560E+01, 0.9384432420354740E+00, 0.1749237822536809E-01, 0.1527569554079381E-01, 0.8195211115924584E+00, 0.1033200729619178E+01, 0.2289566278084960E+00, 0.3214195289148511E+00, 0.1450453280026662E+01, 0.1468676920055535E+01, 0.3396444831847066E+00, 0.1504703083827628E+00, 0.6506576141124067E+00, 0.6177670326339089E+00, 0.1175810411452473E+00, 0.2402402478867145E+00, 0.1262214585027166E+01, 0.1547015632125768E+01, 0.5250426092262996E+00, 0.3073217441808570E+00, 0.9055103985570538E+00, 0.6853975319350345E+00, 0.8721019179982008E-01, 0.1076549537133787E+00, 0.8460758777494440E+00, 0.7520336386157322E+00, 0.1361402882064932E-01, 0.1396865635082151E-01, 0.7716231249744060E+00, 0.8106342214532430E+00, 0.5298106707064092E-01, 0.5302931309687425E-01, 0.8113724074143097E+00, 0.7583421433039418E+00, 0.3632274887524200E-06, 0.9727942389020712E-07, 0.2030988542900605E+00, 0.2030986166419704E+00, 0.1173872316183806E-05, 0.7596112905317353E-11, 0.1314248578535016E-05, 0.8411884199182457E-17, 0.1248859578385863E-08, 0.5351359530665278E-09, 0.1093987140067686E+00},
			i0:       1,
			n0:       16,
			pp:       1,
			tau:      8.4118841988678429E-018,
			sigma:    0.21120653295038691,
			i0Out:    1,
			n0Out:    16,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.21120653295038691,
			dminOut:  8.4118841988678429E-018,
			dmin1Out: 0.20309858884628909,
			dmin2Out: 0.53283137411117854,
			dnOut:    8.4118841988678429E-018,
			dnm1Out:  0.20309858884628909,
			dnm2Out:  0.71080515929391042,
		},
		{
			z:        []float64{0.2269770980149768E+01, 0.2269770979614632E+01, 0.2293573303077261E-09, 0.5351359530665278E-09, 0.9731514910392876E+00, 0.9728156171042807E+00, 0.5111377954200868E-03, 0.3358741643642203E-03, 0.1481726086511552E+01, 0.1480954954309013E+01, 0.1467737987028945E-02, 0.1282269997958535E-02, 0.1723475383238021E+01, 0.1696043475248702E+01, 0.2373112943910766E-01, 0.2889964597634762E-01, 0.1581996562336162E+01, 0.1415242852393798E+01, 0.1129959536471923E+00, 0.1904848393814713E+00, 0.8407229839290754E+00, 0.9384432420354740E+00, 0.1877296098701645E-01, 0.1527569554079381E-01, 0.1335847297547013E+01, 0.1033200729619178E+01, 0.3533797947109679E+00, 0.3214195289148511E+00, 0.1265767433727330E+01, 0.1468676920055535E+01, 0.7343813202351114E-01, 0.1504703083827628E+00, 0.7845691484971122E+00, 0.6177670326339089E+00, 0.4737063898809231E+00, 0.2402402478867145E+00, 0.1380630986425702E+01, 0.1547015632125768E+01, 0.1525661578238559E+00, 0.3073217441808570E+00, 0.6404863278245572E+00, 0.6853975319350345E+00, 0.1264041760751794E+00, 0.1076549537133787E+00, 0.6395981188913744E+00, 0.7520336386157322E+00, 0.1770404028911661E-01, 0.1396865635082151E-01, 0.8459594942610007E+00, 0.8106342214532430E+00, 0.4753698401003136E-01, 0.5302931309687425E-01, 0.7108052565733343E+00, 0.7583421433039418E+00, 0.2779568135873871E-07, 0.9727942389020712E-07, 0.2030985888538852E+00, 0.2030986166419704E+00, 0.3146138162949754E-27, 0.7596112905317353E-11, 0.8411884198867843E-17, 0.8411884199182457E-17, 0.2293573303077261E-09, 0.5351359530665278E-09},
			i0:       1,
			n0:       16,
			pp:       0,
			tau:      8.4118841988678429E-018,
			sigma:    0.21120653295038691,
			i0Out:    1,
			n0Out:    16,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.21120653295038691,
			dminOut:  8.4118841988678429E-018,
			dmin1Out: 0.20309858045029386,
			dmin2Out: 0.51860540644834729,
			dnOut:    8.4118841988678429E-018,
			dnm1Out:  0.20309858045029386,
			dnm2Out:  0.67176795876146822,
		},
		{
			z:        []float64{0.2269770980149768E+01, 0.2269770980379126E+01, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.9731514910392876E+00, 0.9736626287363720E+00, 0.5111377954200868E-03, 0.7778528033461282E-03, 0.1481726086511552E+01, 0.1482415971695234E+01, 0.1467737987028945E-02, 0.1706410574351102E-02, 0.1723475383238021E+01, 0.1745500102102777E+01, 0.2373112943910766E-01, 0.2150819994097728E-01, 0.1581996562336162E+01, 0.1673484316042377E+01, 0.1129959536471923E+00, 0.5676676764251993E-01, 0.8407229839290754E+00, 0.8027291772735718E+00, 0.1877296098701645E-01, 0.3124068479314146E-01, 0.1335847297547013E+01, 0.1657986407464839E+01, 0.3533797947109679E+00, 0.2697830536296953E+00, 0.1265767433727330E+01, 0.1069422512121146E+01, 0.7343813202351114E-01, 0.5387701498318342E-01, 0.7845691484971122E+00, 0.1204398523394852E+01, 0.4737063898809231E+00, 0.5430210247136315E+00, 0.1380630986425702E+01, 0.9901761195359265E+00, 0.1525661578238559E+00, 0.9868601781741700E-01, 0.6404863278245572E+00, 0.6682044860823195E+00, 0.1264041760751794E+00, 0.1209927124430272E+00, 0.6395981188913744E+00, 0.5363094467374639E+00, 0.1770404028911661E-01, 0.2792585709699239E-01, 0.8459594942610007E+00, 0.8655706211740396E+00, 0.4753698401003136E-01, 0.3903729781186600E-01, 0.7108052565733343E+00, 0.6717679865571495E+00, 0.2779568135873871E-07, 0.8403591378512072E-08, 0.2030985888538852E+00, 0.2030985804502939E+00, 0.3146138162949754E-27, 0.1303059324279677E-43, 0.2112065329503869E+00},
			i0:       1,
			n0:       15,
			pp:       1,
			tau:      0.10154929022514693,
			sigma:    0.21120653295038691,
			i0Out:    1,
			n0Out:    15,
			ppOut:    1,
			tauOut:   0.10154929022514693,
			sigmaOut: 0.21120653295038691,
			dminOut:  0.10154928703201281,
			dmin1Out: 0.32391205918605420,
			dmin2Out: 0.32391205918605420,
			dnOut:    0.10154928703201281,
			dnm1Out:  0.53450854002684800,
			dnm2Out:  0.69531980574199759,
		},
		{
			z:        []float64{0.2168221690252314E+01, 0.2269770980379126E+01, 0.4415865678637858E-10, 0.9833566024906726E-10, 0.8728911912704126E+00, 0.9736626287363720E+00, 0.1321013925721922E-02, 0.7778528033461282E-03, 0.1381252078118717E+01, 0.1482415971695234E+01, 0.2156405683614187E-02, 0.1706410574351102E-02, 0.1663302606134993E+01, 0.1745500102102777E+01, 0.2163985983955575E-01, 0.2150819994097728E-01, 0.1607061933620194E+01, 0.1673484316042377E+01, 0.2835506195054301E-01, 0.5676676764251993E-01, 0.7040655098910235E+00, 0.8027291772735718E+00, 0.7356791380810462E-01, 0.3124068479314146E-01, 0.1752652257061283E+01, 0.1657986407464839E+01, 0.1646145547572216E+00, 0.2697830536296953E+00, 0.8571356821219610E+00, 0.1069422512121146E+01, 0.7570493055431493E-01, 0.5387701498318342E-01, 0.1570165327329021E+01, 0.1204398523394852E+01, 0.3424393862982660E+00, 0.5430210247136315E+00, 0.6448734608299306E+00, 0.9901761195359265E+00, 0.1022564019526126E+00, 0.9868601781741700E-01, 0.5853915063475871E+00, 0.6682044860823195E+00, 0.1108480973262629E+00, 0.1209927124430272E+00, 0.3518379162830466E+00, 0.5363094467374639E+00, 0.6870152520689508E-01, 0.2792585709699239E-01, 0.7343571035538636E+00, 0.8655706211740396E+00, 0.3571015630515456E-01, 0.3903729781186600E-01, 0.5345085484304394E+00, 0.6717679865571495E+00, 0.3193134112956561E-08, 0.8403591378512072E-08, 0.1015492870320128E+00, 0.2030985804502939E+00, 0.4415865678637858E-10, 0.9833566024906726E-10},
			i0:       1,
			n0:       15,
			pp:       0,
			tau:      0.10154100618138810,
			sigma:    0.31275582317553385,
			i0Out:    1,
			n0Out:    15,
			ppOut:    0,
			tauOut:   0.10154100618138810,
			sigmaOut: 0.31275582317553385,
			dminOut:  8.2800215414347011E-006,
			dmin1Out: 0.16866357962181588,
			dmin2Out: 0.16866357962181588,
			dnOut:    8.2800215414347011E-006,
			dnm1Out:  0.39110725789187228,
			dnm2Out:  0.42026820370851276,
		},
		{
			z:        []float64{0.2168221690252314E+01, 0.2066680684115085E+01, 0.4415865678637858E-10, 0.1865101988102620E-10, 0.8728911912704126E+00, 0.7726711989960955E+00, 0.1321013925721922E-02, 0.2361487308570419E-02, 0.1381252078118717E+01, 0.1279505990312372E+01, 0.2156405683614187E-02, 0.2803234389363145E-02, 0.1663302606134993E+01, 0.1580598225403798E+01, 0.2163985983955575E-01, 0.2200217261925751E-01, 0.1607061933620194E+01, 0.1511873816770092E+01, 0.2835506195054301E-01, 0.1320468740761088E-01, 0.7040655098910235E+00, 0.6628877301101292E+00, 0.7356791380810462E-01, 0.1945110224949301E+00, 0.1752652257061283E+01, 0.1621214783142186E+01, 0.1646145547572216E+00, 0.8703165684534678E-01, 0.8571356821219610E+00, 0.7442679496495411E+00, 0.7570493055431493E-01, 0.1597129865933493E+00, 0.1570165327329021E+01, 0.1651350720852550E+01, 0.3424393862982660E+00, 0.1337269360034139E+00, 0.6448734608299306E+00, 0.5118619205977412E+00, 0.1022564019526126E+00, 0.1169456581236225E+00, 0.5853915063475871E+00, 0.4777529393688394E+00, 0.1108480973262629E+00, 0.8163333047984263E-01, 0.3518379162830466E+00, 0.2373651048287110E+00, 0.6870152520689508E-01, 0.2125478936639627E+00, 0.7343571035538636E+00, 0.4559783600136673E+00, 0.3571015630515456E-01, 0.4186028435717903E-01, 0.5345085484304394E+00, 0.3911072610850064E+00, 0.3193134112956561E-08, 0.8290832843879624E-09, 0.1015492870320128E+00, 0.8280021541434701E-05, 0.4415865678637858E-10, 0.1865101988102620E-10, 0.2112065329503869E+00},
			i0:       1,
			n0:       15,
			pp:       1,
			tau:      8.2795951083136037E-006,
			sigma:    0.41429682935692197,
			i0Out:    1,
			n0Out:    15,
			ppOut:    1,
			tauOut:   8.2795951083136037E-006,
			sigmaOut: 0.41429682935692197,
			dminOut:  4.2641220534843968E-010,
			dmin1Out: 0.19554701532405336,
			dmin2Out: 0.19554701532405336,
			dnOut:    4.2641220534843968E-010,
			dnm1Out:  0.32821333105682254,
			dnm2Out:  0.21848306981706411,
		},
		{
			z:        []float64{0.2066672404538628E+01, 0.2066680684115085E+01, 0.6973096395115281E-11, 0.1865101988102620E-10, 0.7750244067025847E+00, 0.7726711989960955E+00, 0.3898634844569491E-02, 0.2361487308570419E-02, 0.1278402310262057E+01, 0.1279505990312372E+01, 0.3465878671879142E-02, 0.2803234389363145E-02, 0.1599126239756068E+01, 0.1580598225403798E+01, 0.2080167773382635E-01, 0.2200217261925751E-01, 0.1504268546848768E+01, 0.1511873816770092E+01, 0.5818924606767697E-02, 0.1320468740761088E-01, 0.8515715484031834E+00, 0.6628877301101292E+00, 0.3703084558710277E+00, 0.1945110224949301E+00, 0.1337929704521397E+01, 0.1621214783142186E+01, 0.4841425717359337E-01, 0.8703165684534678E-01, 0.8555583994741887E+00, 0.7442679496495411E+00, 0.3082690272254148E+00, 0.1597129865933493E+00, 0.1476800350035441E+01, 0.1651350720852550E+01, 0.4635002036444261E-01, 0.1337269360034139E+00, 0.5824492787618127E+00, 0.5118619205977412E+00, 0.9592445892242578E-01, 0.1169456581236225E+00, 0.4634535313311479E+00, 0.4777529393688394E+00, 0.4180980990954932E-01, 0.8163333047984263E-01, 0.4080949089880160E+00, 0.2373651048287110E+00, 0.2374870106014949E+00, 0.2125478936639627E+00, 0.2603433541742431E+00, 0.4559783600136673E+00, 0.6288565043307552E-01, 0.4186028435717903E-01, 0.3282133318859058E+00, 0.3911072610850064E+00, 0.2091574834858362E-13, 0.8290832843879624E-09, 0.4264122053484397E-09, 0.8280021541434701E-05, 0.6973096395115281E-11, 0.1865101988102620E-10},
			i0:       1,
			n0:       15,
			pp:       0,
			tau:      4.2641207498271701E-010,
			sigma:    0.41430510895203027,
			i0Out:    1,
			n0Out:    15,
			ppOut:    0,
			tauOut:   4.2641207498271701E-010,
			sigmaOut: 0.41430510895203027,
			dminOut:  1.3036568472020817E-016,
			dmin1Out: 0.15841174212052453,
			dmin2Out: 0.15841174212052453,
			dnOut:    1.3036568472020817E-016,
			dnm1Out:  0.23494558610432464,
			dnm2Out:  0.15841174212052453,
		},
		{
			z:        []float64{0.2066672404538628E+01, 0.2066672404119188E+01, 0.6973096395115281E-11, 0.2614986238618434E-11, 0.7750244067025847E+00, 0.7789230411181270E+00, 0.3898634844569491E-02, 0.6398608757305904E-02, 0.1278402310262057E+01, 0.1275469579750218E+01, 0.3465878671879142E-02, 0.4345362379476143E-02, 0.1599126239756068E+01, 0.1615582554684006E+01, 0.2080167773382635E-01, 0.1936843737632438E-01, 0.1504268546848768E+01, 0.1490719033652799E+01, 0.5818924606767697E-02, 0.3324054047451484E-02, 0.8515715484031834E+00, 0.1218555949800348E+01, 0.3703084558710277E+00, 0.4065850919906259E+00, 0.1337929704521397E+01, 0.9797588692779520E+00, 0.4841425717359337E-01, 0.4227695780870786E-01, 0.8555583994741887E+00, 0.1121550468464483E+01, 0.3082690272254148E+00, 0.4059129037098644E+00, 0.1476800350035441E+01, 0.1117237466263607E+01, 0.4635002036444261E-01, 0.2416365074306882E-01, 0.5824492787618127E+00, 0.6542100865147574E+00, 0.9592445892242578E-01, 0.6795451513972497E-01, 0.4634535313311479E+00, 0.4373088256745602E+00, 0.4180980990954932E-01, 0.3901675330591518E-01, 0.4080949089880160E+00, 0.6065651658571838E+00, 0.2374870106014949E+00, 0.1019316116273065E+00, 0.2603433541742431E+00, 0.2212973925536000E+00, 0.6288565043307552E-01, 0.9326774535516916E-01, 0.3282133318859058E+00, 0.2349455861043456E+00, 0.2091574834858362E-13, 0.3796083394336032E-22, 0.4264122053484397E-09, 0.1303656847202082E-15, 0.6973096395115281E-11, 0.2614986238618434E-11, 0.2112065329503869E+00},
			i0:       1,
			n0:       15,
			pp:       1,
			tau:      1.3036568471812905E-016,
			sigma:    0.41430510937844234,
			i0Out:    1,
			n0Out:    15,
			ppOut:    1,
			tauOut:   1.3036568471812905E-016,
			sigmaOut: 0.41430510937844234,
			dminOut:  2.0791168714198411E-027,
			dmin1Out: 0.15670572686712736,
			dmin2Out: 0.18680490956440032,
			dnOut:    2.0791168714198411E-027,
			dnm1Out:  0.15670572686712736,
			dnm2Out:  0.18680490956440032,
		},
		{
			z:        []float64{0.2066672404121803E+01, 0.2066672404119188E+01, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.7853216498744473E+00, 0.7789230411181270E+00, 0.1039221422709001E-01, 0.6398608757305904E-02, 0.1269422727902604E+01, 0.1275469579750218E+01, 0.5530302475095172E-02, 0.4345362379476143E-02, 0.1629420689585235E+01, 0.1615582554684006E+01, 0.1771973219288664E-01, 0.1936843737632438E-01, 0.1476323355507364E+01, 0.1490719033652799E+01, 0.2743671176012720E-02, 0.3324054047451484E-02, 0.1622397370614960E+01, 0.1218555949800348E+01, 0.2455350071499520E+00, 0.4065850919906259E+00, 0.7765008199367077E+00, 0.9797588692779520E+00, 0.6106335063429082E-01, 0.4227695780870786E-01, 0.1466400021540057E+01, 0.1121550468464483E+01, 0.3092615230516922E+00, 0.4059129037098644E+00, 0.8321395939549830E+00, 0.1117237466263607E+01, 0.1899693772291602E-01, 0.2416365074306882E-01, 0.7031676639315663E+00, 0.6542100865147574E+00, 0.4226176876348708E-01, 0.6795451513972497E-01, 0.4340638102169883E+00, 0.4373088256745602E+00, 0.5452240634477346E-01, 0.3901675330591518E-01, 0.6539743711397168E+00, 0.6065651658571838E+00, 0.3449248298919955E-01, 0.1019316116273065E+00, 0.2800726549195695E+00, 0.2212973925536000E+00, 0.7823985923721809E-01, 0.9326774535516916E-01, 0.1567057268671274E+00, 0.2349455861043456E+00, 0.3158014839988858E-37, 0.3796083394336032E-22},
			i0:       1,
			n0:       14,
			pp:       0,
			tau:      5.2226904068357879E-002,
			sigma:    0.41430510937844245,
			i0Out:    1,
			n0Out:    14,
			ppOut:    0,
			tauOut:   5.2226904068357879E-002,
			sigmaOut: 0.41430510937844245,
			dminOut:  6.1979733096699005E-002,
			dmin1Out: 0.21025182606004778,
			dmin2Out: 0.35454858327394784,
			dnOut:    6.1979733096699005E-002,
			dnm1Out:  0.21025182606004778,
			dnm2Out:  0.51458348500658069,
		},
		{
			z:        []float64{0.2066672404121803E+01, 0.2014445500054431E+01, 0.9855809897129084E-12, 0.3842238913414725E-12, 0.7853216498744473E+00, 0.7434869600327952E+00, 0.1039221422709001E-01, 0.1774357001838869E-01, 0.1269422727902604E+01, 0.1204982556290953E+01, 0.5530302475095172E-02, 0.7478273627729330E-02, 0.1629420689585235E+01, 0.1587435244082034E+01, 0.1771973219288664E-01, 0.1647944669694028E-01, 0.1476323355507364E+01, 0.1410360675918078E+01, 0.2743671176012720E-02, 0.3156160674217245E-02, 0.1622397370614960E+01, 0.1812549313022337E+01, 0.2455350071499520E+00, 0.1051878329628395E+00, 0.7765008199367077E+00, 0.6801494335398012E+00, 0.6106335063429082E-01, 0.1316523902981273E+00, 0.1466400021540057E+01, 0.1591782250225264E+01, 0.3092615230516922E+00, 0.1616733433116970E+00, 0.8321395939549830E+00, 0.6372362842978442E+00, 0.1899693772291602E-01, 0.2096244776016672E-01, 0.7031676639315663E+00, 0.6722400808665288E+00, 0.4226176876348708E-01, 0.2728832287468248E-01, 0.4340638102169883E+00, 0.4090709896187213E+00, 0.5452240634477346E-01, 0.8716398206477828E-01, 0.6539743711397168E+00, 0.5490759679957803E+00, 0.3449248298919955E-01, 0.1759392479116377E-01, 0.2800726549195695E+00, 0.2884916852972659E+00, 0.7823985923721809E-01, 0.4249908970207047E-01, 0.1567057268671274E+00, 0.6197973309669901E-01, 0.3158014839988858E-37, 0.3842238913414725E-12, 0.4143051093784424E+00},
			i0:       1,
			n0:       14,
			pp:       1,
			tau:      4.3992746597899146E-002,
			sigma:    0.46653201344680034,
			i0Out:    1,
			n0Out:    14,
			ppOut:    1,
			tauOut:   4.3992746597899146E-002,
			sigmaOut: 0.46653201344680034,
			dminOut:  8.3978432207528000E-003,
			dmin1Out: 0.23219513064786862,
			dmin2Out: 0.34734604130905122,
			dnOut:    8.3978432207528000E-003,
			dnm1Out:  0.23219513064786862,
			dnm2Out:  0.39493697562207530,
		},
		{
			z:        []float64{0.1970452753456916E+01, 0.2014445500054431E+01, 0.1449745254963753E-12, 0.3842238913414725E-12, 0.7172377834531399E+00, 0.7434869600327952E+00, 0.2980976860358389E-01, 0.1774357001838869E-01, 0.1138658314717199E+01, 0.1204982556290953E+01, 0.1042566937606312E-01, 0.7478273627729330E-02, 0.1549496274805013E+01, 0.1587435244082034E+01, 0.1499968987352189E-01, 0.1647944669694028E-01, 0.1354524400120875E+01, 0.1410360675918078E+01, 0.4223398900256122E-02, 0.3156160674217245E-02, 0.1869521000487022E+01, 0.1812549313022337E+01, 0.3826832915293116E-01, 0.1051878329628395E+00, 0.7295407480870981E+00, 0.6801494335398012E+00, 0.2872518617030954E+00, 0.1316523902981273E+00, 0.1422210985235967E+01, 0.1591782250225264E+01, 0.7243940711431240E-01, 0.1616733433116970E+00, 0.5417665783457993E+00, 0.6372362842978442E+00, 0.2601082853889214E-01, 0.2096244776016672E-01, 0.6295248286044199E+00, 0.6722400808665288E+00, 0.1773220171177086E-01, 0.2728832287468248E-01, 0.4345100233738295E+00, 0.4090709896187213E+00, 0.1101462457758059E+00, 0.8716398206477828E-01, 0.4125309004132391E+00, 0.5490759679957803E+00, 0.1230380805149813E-01, 0.1759392479116377E-01, 0.2746942203499391E+00, 0.2884916852972659E+00, 0.9589143278047064E-02, 0.4249908970207047E-01, 0.8397843220752800E-02, 0.6197973309669901E-01, 0.1449745254963753E-12, 0.3842238913414725E-12},
			i0:       1,
			n0:       14,
			pp:       0,
			tau:      8.0121275381568099E-003,
			sigma:    0.51052476004469949,
			i0Out:    1,
			n0Out:    14,
			ppOut:    0,
			tauOut:   8.0121275381568099E-003,
			sigmaOut: 0.51052476004469949,
			dminOut:  8.3015533671799979E-005,
			dmin1Out: 0.25644350373773794,
			dmin2Out: 0.31779880038403513,
			dnOut:    8.3015533671799979E-005,
			dnm1Out:  0.25644350373773794,
			dnm2Out:  0.31779880038403513,
		},
		{
			z:        []float64{0.1970452753456916E+01, 0.1962440625918904E+01, 0.1449745254963753E-12, 0.5298565773193886E-13, 0.7172377834531399E+00, 0.7390354245185139E+00, 0.2980976860358389E-01, 0.4592897681783073E-01, 0.1138658314717199E+01, 0.1095142879737275E+01, 0.1042566937606312E-01, 0.1475107600976594E-01, 0.1549496274805013E+01, 0.1541732761130612E+01, 0.1499968987352189E-01, 0.1317831886314191E-01, 0.1354524400120875E+01, 0.1337557352619832E+01, 0.4223398900256122E-02, 0.5903098601340335E-02, 0.1869521000487022E+01, 0.1893874103500456E+01, 0.3826832915293116E-01, 0.1474137347708128E-01, 0.7295407480870981E+00, 0.9940391087749554E+00, 0.2872518617030954E+00, 0.4109825756725979E+00, 0.1422210985235967E+01, 0.1075655689139524E+01, 0.7243940711431240E-01, 0.3648495529374628E-01, 0.5417665783457993E+00, 0.5232803240527883E+00, 0.2601082853889214E-01, 0.3129195122603765E-01, 0.6295248286044199E+00, 0.6079529515519964E+00, 0.1773220171177086E-01, 0.1267338099203561E-01, 0.4345100233738295E+00, 0.5239707606194429E+00, 0.1101462457758059E+00, 0.8671997249104711E-01, 0.4125309004132391E+00, 0.3301026084355332E+00, 0.1230380805149813E-01, 0.1023858907404432E-01, 0.2746942203499391E+00, 0.2660326470157850E+00, 0.9589143278047064E-02, 0.3027001489241909E-03, 0.8397843220752800E-02, 0.8301553367179998E-04, 0.1449745254963753E-12, 0.5298565773193886E-13, 0.4143051093784424E+00},
			i0:       1,
			n0:       14,
			pp:       1,
			tau:      8.2898056486573887E-005,
			sigma:    0.51853688758285632,
			i0Out:    1,
			n0Out:    14,
			ppOut:    1,
			tauOut:   8.2898056486573887E-005,
			sigmaOut: 0.51853688758285632,
			dminOut:  1.9676836973621424E-008,
			dmin1Out: 0.25663722751477031,
			dmin2Out: 0.28224921557690585,
			dnOut:    1.9676836973621424E-008,
			dnm1Out:  0.25663722751477031,
			dnm2Out:  0.28224921557690585,
		},
		{
			z:        []float64{0.1962357727862470E+01, 0.1962440625918904E+01, 0.1995470932711637E-13, 0.5298565773193886E-13, 0.7848815032798381E+00, 0.7390354245185139E+00, 0.6408456783027588E-01, 0.4592897681783073E-01, 0.1045726489860278E+01, 0.1095142879737275E+01, 0.2174776805091989E-01, 0.1475107600976594E-01, 0.1533080413886347E+01, 0.1541732761130612E+01, 0.1149760777771624E-01, 0.1317831886314191E-01, 0.1331879945386970E+01, 0.1337557352619832E+01, 0.8393943921304423E-02, 0.5903098601340335E-02, 0.1900138634999746E+01, 0.1893874103500456E+01, 0.7711806645770664E-02, 0.1474137347708128E-01, 0.1397226979745296E+01, 0.9940391087749554E+00, 0.3163950825942626E+00, 0.4109825756725979E+00, 0.7956626637825215E+00, 0.1075655689139524E+01, 0.2399491656225486E-01, 0.3648495529374628E-01, 0.5304944606600845E+00, 0.5232803240527883E+00, 0.3586094769777509E-01, 0.3129195122603765E-01, 0.5846824867897703E+00, 0.6079529515519964E+00, 0.1135741402906866E-01, 0.1267338099203561E-01, 0.5992504210249349E+00, 0.5239707606194429E+00, 0.4777049480214081E-01, 0.8671997249104711E-01, 0.2924878046509502E+00, 0.3301026084355332E+00, 0.9312521444528056E-02, 0.1023858907404432E-01, 0.2569399276636945E+00, 0.2660326470157850E+00, 0.9780034825247492E-07, 0.3027001489241909E-03, 0.1967683697362142E-07, 0.8301553367179998E-04, 0.1995470932711637E-13, 0.5298565773193886E-13},
			i0:       1,
			n0:       14,
			pp:       0,
			tau:      1.9676827722764296E-008,
			sigma:    0.51861978563934286,
			i0Out:    1,
			n0Out:    14,
			ppOut:    0,
			tauOut:   1.9676827722764296E-008,
			sigmaOut: 0.51861978563934286,
			dminOut:  1.5032959860012969E-015,
			dmin1Out: 0.24838793989138258,
			dmin2Out: 0.27047787997045275,
			dnOut:    1.5032959860012969E-015,
			dnm1Out:  0.24838793989138258,
			dnm2Out:  0.27047787997045275,
		},
		{
			z:        []float64{0.1962357727862470E+01, 0.1962357708185662E+01, 0.1995470932711637E-13, 0.7981257539768321E-14, 0.7848815032798381E+00, 0.8489660514332782E+00, 0.6408456783027588E-01, 0.7893711422056097E-01, 0.1045726489860278E+01, 0.9885371240138099E+00, 0.2174776805091989E-01, 0.3372769361380378E-01, 0.1533080413886347E+01, 0.1510850308373431E+01, 0.1149760777771624E-01, 0.1013563894066502E-01, 0.1331879945386970E+01, 0.1330138230690781E+01, 0.8393943921304423E-02, 0.1199097716077874E-01, 0.1900138634999746E+01, 0.1895859444807910E+01, 0.7711806645770664E-02, 0.5683514322519618E-02, 0.1397226979745296E+01, 0.1707938528340212E+01, 0.3163950825942626E+00, 0.1473962616612956E+00, 0.7956626637825215E+00, 0.6722612990066530E+00, 0.2399491656225486E-01, 0.1893485515094503E-01, 0.5304944606600845E+00, 0.5474205335300870E+00, 0.3586094769777509E-01, 0.3830193935796279E-01, 0.5846824867897703E+00, 0.5577379417840483E+00, 0.1135741402906866E-01, 0.1220274725600271E-01, 0.5992504210249349E+00, 0.6348181488942451E+00, 0.4777049480214081E-01, 0.2200990500366971E-01, 0.2924878046509502E+00, 0.2797904014149808E+00, 0.9312521444528056E-02, 0.8551968095484232E-02, 0.2569399276636945E+00, 0.2483880376917308E+00, 0.9780034825247492E-07, 0.7747561140265878E-14, 0.1967683697362142E-07, 0.1503295986001297E-14, 0.1995470932711637E-13, 0.7981257539768321E-14, 0.4143051093784424E+00},
			i0:       1,
			n0:       14,
			pp:       1,
			tau:      1.5032959860012393E-015,
			sigma:    0.51861980531617058,
			i0Out:    1,
			n0Out:    14,
			ppOut:    1,
			tauOut:   1.5032959860012393E-015,
			sigmaOut: 0.51861980531617058,
			dminOut:  9.2691156363468887E-030,
			dmin1Out: 0.24076768543540897,
			dmin2Out: 0.27020241256687788,
			dnOut:    9.2691156363468887E-030,
			dnm1Out:  0.24076768543540897,
			dnm2Out:  0.27020241256687788,
		},
		{
			z:        []float64{0.1962357708185669E+01, 0.1962357708185662E+01, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.9279031656538340E+00, 0.8489660514332782E+00, 0.8409527066820451E-01, 0.7893711422056097E-01, 0.9381695469594078E+00, 0.9885371240138099E+00, 0.5431587122209677E-01, 0.3372769361380378E-01, 0.1466670076091998E+01, 0.1510850308373431E+01, 0.9192115573380727E-02, 0.1013563894066502E-01, 0.1332937092278178E+01, 0.1330138230690781E+01, 0.1705497388769041E-01, 0.1199097716077874E-01, 0.1884487985242738E+01, 0.1895859444807910E+01, 0.5151050664063701E-02, 0.5683514322519618E-02, 0.1850183739337442E+01, 0.1707938528340212E+01, 0.5355619565040129E-01, 0.1473962616612956E+00, 0.6376399585071952E+00, 0.6722612990066530E+00, 0.1625577000116482E-01, 0.1893485515094503E-01, 0.5694667028868834E+00, 0.5474205335300870E+00, 0.3751307094084999E-01, 0.3830193935796279E-01, 0.5324276180991995E+00, 0.5577379417840483E+00, 0.1454944326918194E-01, 0.1220274725600271E-01, 0.6422786106287314E+00, 0.6348181488942451E+00, 0.9587988848101409E-02, 0.2200990500366971E-01, 0.2787543806623621E+00, 0.2797904014149808E+00, 0.7620352256320365E-02, 0.8551968095484232E-02, 0.2407676854354167E+00, 0.2483880376917308E+00, 0.4837392336267432E-28, 0.7747561140265878E-14},
			i0:       1,
			n0:       13,
			pp:       0,
			tau:      0.19361025134591583,
			sigma:    0.51861980531617213,
			i0Out:    1,
			n0Out:    13,
			ppOut:    0,
			tauOut:   0.19361025134591583,
			sigmaOut: 0.51861980531617213,
			dminOut:  2.5949209826162500E-002,
			dmin1Out: 7.8890170839591350E-002,
			dmin2Out: 0.28791033268116917,
			dnOut:    2.5949209826162500E-002,
			dnm1Out:  7.8890170839591350E-002,
			dnm2Out:  0.41777236268627749,
		},
		{
			z:        []float64{0.1962357708185669E+01, 0.1768747456839757E+01, 0.3452895805257589E-14, 0.1811424766142649E-14, 0.9279031656538340E+00, 0.8183881849761209E+00, 0.8409527066820451E-01, 0.9640366690596863E-01, 0.9381695469594078E+00, 0.7024714999296202E+00, 0.5431587122209677E-01, 0.1134045480653624E+00, 0.1466670076091998E+01, 0.1168847392254101E+01, 0.9192115573380727E-02, 0.1048255904531585E-01, 0.1332937092278178E+01, 0.1145899255774637E+01, 0.1705497388769041E-01, 0.2804774784346500E-01, 0.1884487985242738E+01, 0.1667981036717421E+01, 0.5151050664063701E-02, 0.5713728135608638E-02, 0.1850183739337442E+01, 0.1704415955506319E+01, 0.5355619565040129E-01, 0.2003593680404179E-01, 0.6376399585071952E+00, 0.4402495403584025E+00, 0.1625577000116482E-01, 0.2102698332839761E-01, 0.5694667028868834E+00, 0.3923425391534199E+00, 0.3751307094084999E-01, 0.5090703407211448E-01, 0.5324276180991995E+00, 0.3024597759503511E+00, 0.1454944326918194E-01, 0.3089599659653809E-01, 0.6422786106287314E+00, 0.4273603515343789E+00, 0.9587988848101409E-02, 0.6253958476854957E-02, 0.2787543806623621E+00, 0.8651052309591171E-01, 0.7620352256320365E-02, 0.2120822426333837E-01, 0.2407676854354167E+00, 0.2594920982616250E-01, 0.4837392336267432E-28, 0.1811424766142649E-14, 0.5186198053161721E+00},
			i0:       1,
			n0:       13,
			pp:       1,
			tau:      1.8780556192507153E-002,
			sigma:    0.71223005666208794,
			i0Out:    1,
			n0Out:    13,
			ppOut:    1,
			tauOut:   1.8780556192507153E-002,
			sigmaOut: 0.71223005666208794,
			dminOut:  8.7653100724300811E-004,
			dmin1Out: 6.6256164803006098E-002,
			dmin2Out: 0.24571773655185866,
			dnOut:    8.7653100724300811E-004,
			dnm1Out:  6.6256164803006098E-002,
			dnm2Out:  0.36084635920253871,
		},
		{
			z:        []float64{0.1749966900647251E+01, 0.1768747456839757E+01, 0.8471295234418270E-15, 0.1811424766142649E-14, 0.8960112956895816E+00, 0.8183881849761209E+00, 0.7558032897122403E-01, 0.9640366690596863E-01, 0.7215151628312513E+00, 0.7024714999296202E+00, 0.1837142406762631E+00, 0.1134045480653624E+00, 0.9768351544306461E+00, 0.1168847392254101E+01, 0.1229681032071614E-01, 0.1048255904531585E-01, 0.1142869637104878E+01, 0.1145899255774637E+01, 0.4093477506677201E-01, 0.2804774784346500E-01, 0.1613979433593750E+01, 0.1667981036717421E+01, 0.6033886923870183E-02, 0.5713728135608638E-02, 0.1699637449193983E+01, 0.1704415955506319E+01, 0.5189819730562233E-02, 0.2003593680404179E-01, 0.4373061477637308E+00, 0.4402495403584025E+00, 0.1886499897608887E-01, 0.2102698332839761E-01, 0.4056040180569384E+00, 0.3923425391534199E+00, 0.3796148320598534E-01, 0.5090703407211448E-01, 0.2766137331483968E+00, 0.3024597759503511E+00, 0.4773343613933298E-01, 0.3089599659653809E-01, 0.3671003176793937E+00, 0.4273603515343789E+00, 0.1473802100398464E-02, 0.6253958476854957E-02, 0.8746438906634448E-01, 0.8651052309591171E-01, 0.6292122626412339E-02, 0.2120822426333837E-01, 0.8765310072430081E-03, 0.2594920982616250E-01, 0.8471295234418270E-15, 0.1811424766142649E-14},
			i0:       1,
			n0:       13,
			pp:       0,
			tau:      8.1622622999092049E-004,
			sigma:    0.73101061285459512,
			i0Out:    1,
			n0Out:    13,
			ppOut:    0,
			tauOut:   8.1622622999092049E-004,
			sigmaOut: 0.73101061285459512,
			dminOut:  6.9556311345050133E-007,
			dmin1Out: 8.6231166987816729E-002,
			dmin2Out: 0.25114202935985525,
			dnOut:    6.9556311345050133E-007,
			dnm1Out:  8.6231166987816729E-002,
			dnm2Out:  0.30765445605457048,
		},
		{
			z:        []float64{0.1749966900647251E+01, 0.1749150674417261E+01, 0.8471295234418270E-15, 0.4339463906783712E-15, 0.8960112956895816E+00, 0.9707753984308143E+00, 0.7558032897122403E-01, 0.5617401661873561E-01, 0.7215151628312513E+00, 0.8482391606587878E+00, 0.1837142406762631E+00, 0.2115659556707206E+00, 0.9768351544306461E+00, 0.7767497828506508E+00, 0.1229681032071614E-01, 0.1809289356632693E-01, 0.1142869637104878E+01, 0.1164895292375332E+01, 0.4093477506677201E-01, 0.5671572845129930E-01, 0.1613979433593750E+01, 0.1562481365836330E+01, 0.6033886923870183E-02, 0.6563547191183529E-02, 0.1699637449193983E+01, 0.1697447495503371E+01, 0.5189819730562233E-02, 0.1337031089310571E-02, 0.4373061477637308E+00, 0.4540178894205181E+00, 0.1886499897608887E-01, 0.1685334336738995E-01, 0.4056040180569384E+00, 0.4258959316655428E+00, 0.3796148320598534E-01, 0.2465547755855056E-01, 0.2766137331483968E+00, 0.2988754654991882E+00, 0.4773343613933298E-01, 0.5862963539483231E-01, 0.3671003176793937E+00, 0.3091282581549689E+00, 0.1473802100398464E-02, 0.4169958485368410E-03, 0.8746438906634448E-01, 0.9252328961422907E-01, 0.6292122626412339E-02, 0.5960921413863723E-04, 0.8765310072430081E-03, 0.6955631134505013E-06, 0.8471295234418270E-15, 0.4339463906783712E-15, 0.5186198053161721E+00},
			i0:       1,
			n0:       13,
			pp:       1,
			tau:      6.9511331676175615E-007,
			sigma:    0.73182683908458601,
			i0Out:    1,
			n0Out:    13,
			ppOut:    1,
			tauOut:   6.9511331676175615E-007,
			sigmaOut: 0.73182683908458601,
			dminOut:  1.2278907249223888E-012,
			dmin1Out: 9.2372077429255559E-002,
			dmin2Out: 0.25591158693238830,
			dnOut:    1.2278907249223888E-012,
			dnm1Out:  9.2372077429255559E-002,
			dnm2Out:  0.25591158693238830,
		},
		{
			z:        []float64{0.1749149979303945E+01, 0.1749150674417261E+01, 0.2408395422307052E-15, 0.4339463906783712E-15, 0.1026948719936233E+01, 0.9707753984308143E+00, 0.4639861736277134E-01, 0.5617401661873561E-01, 0.1013405803853420E+01, 0.8482391606587878E+00, 0.1621599259654447E+00, 0.2115659556707206E+00, 0.6326820553382161E+00, 0.7767497828506508E+00, 0.3331266686486833E-01, 0.1809289356632693E-01, 0.1188297658848447E+01, 0.1164895292375332E+01, 0.7457497555020480E-01, 0.5671572845129930E-01, 0.1494469242363992E+01, 0.1562481365836330E+01, 0.7455005714048079E-02, 0.6563547191183529E-02, 0.1691328825765317E+01, 0.1697447495503371E+01, 0.3589107121045615E-03, 0.1337031089310571E-02, 0.4705116269624868E+00, 0.4540178894205181E+00, 0.1525524548983380E-01, 0.1685334336738995E-01, 0.4352954686209429E+00, 0.4258959316655428E+00, 0.1692854133253905E-01, 0.2465547755855056E-01, 0.3405758644481647E+00, 0.2988754654991882E+00, 0.5321597610926388E-01, 0.5862963539483231E-01, 0.2563285827809251E+00, 0.3091282581549689E+00, 0.1505170716567330E-03, 0.4169958485368410E-03, 0.9243168664339420E-01, 0.9252328961422907E-01, 0.4485687980202113E-09, 0.5960921413863723E-04, 0.1227890724922389E-11, 0.6955631134505013E-06, 0.2408395422307052E-15, 0.4339463906783712E-15},
			i0:       1,
			n0:       13,
			pp:       0,
			tau:      1.2278907189544363E-012,
			sigma:    0.73182753419790281,
			i0Out:    1,
			n0Out:    13,
			ppOut:    0,
			tauOut:   1.2278907189544363E-012,
			sigmaOut: 0.73182753419790281,
			dminOut:  4.9598525010503808E-024,
			dmin1Out: 9.2368632133320736E-002,
			dmin2Out: 0.22049265431403467,
			dnOut:    4.9598525010503808E-024,
			dnm1Out:  9.2368632133320736E-002,
			dnm2Out:  0.22049265431403467,
		},
		{
			z:        []float64{0.1749149979303945E+01, 0.1749149979302717E+01, 0.2408395422307052E-15, 0.1414000300319855E-15, 0.1026948719936233E+01, 0.1073347337297776E+01, 0.4639861736277134E-01, 0.4380746706334982E-01, 0.1013405803853420E+01, 0.1131758262754287E+01, 0.1621599259654447E+00, 0.9065158049178278E-01, 0.6326820553382161E+00, 0.5753431417100738E+00, 0.3331266686486833E-01, 0.6880305191066147E-01, 0.1188297658848447E+01, 0.1194069582486762E+01, 0.7457497555020480E-01, 0.9333627524262254E-01, 0.1494469242363992E+01, 0.1408587972834190E+01, 0.7455005714048079E-02, 0.8951422490882578E-02, 0.1691328825765317E+01, 0.1682736313985311E+01, 0.3589107121045615E-03, 0.1003553923945662E-03, 0.4705116269624868E+00, 0.4856665170586981E+00, 0.1525524548983380E-01, 0.1367304312976171E-01, 0.4352954686209429E+00, 0.4385509668224923E+00, 0.1692854133253905E-01, 0.1314659648329904E-01, 0.3405758644481647E+00, 0.3806452440729016E+00, 0.5321597610926388E-01, 0.3583592846566260E-01, 0.2563285827809251E+00, 0.2206431713856914E+00, 0.1505170716567330E-03, 0.6305450884558253E-04, 0.9243168664339420E-01, 0.9236863258188953E-01, 0.4485687980202113E-09, 0.5962992535266723E-20, 0.1227890724922389E-11, 0.4959852501050381E-23, 0.2408395422307052E-15, 0.1414000300319855E-15, 0.5186198053161721E+00},
			i0:       1,
			n0:       13,
			pp:       1,
			tau:      4.9598525010503808E-024,
			sigma:    0.73182753419913071,
			i0Out:    1,
			n0Out:    13,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.73182753419913071,
			dminOut:  4.9598525010503808E-024,
			dmin1Out: 9.2339683204333278E-002,
			dmin2Out: 0.20112464802017624,
			dnOut:    4.9598525010503808E-024,
			dnm1Out:  9.2339683204333278E-002,
			dnm2Out:  0.20112464802017624,
		},
		{
			z:        []float64{0.1749149979302717E+01, 0.1749149979302717E+01, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.1117154804361126E+01, 0.1073347337297776E+01, 0.4438011869593646E-01, 0.4380746706334982E-01, 0.1178029724550133E+01, 0.1131758262754287E+01, 0.4427372589519608E-01, 0.9065158049178278E-01, 0.5998724677255393E+00, 0.5753431417100738E+00, 0.1369551627869799E+00, 0.6880305191066147E-01, 0.1150450694942405E+01, 0.1194069582486762E+01, 0.1142789997988411E+00, 0.9333627524262254E-01, 0.1303260395526232E+01, 0.1408587972834190E+01, 0.1155784656615063E-01, 0.8951422490882578E-02, 0.1671278822811555E+01, 0.1682736313985311E+01, 0.2916285016424428E-04, 0.1003553923945662E-03, 0.4993103973382956E+00, 0.4856665170586981E+00, 0.1200921574220688E-01, 0.1367304312976171E-01, 0.4396883475635844E+00, 0.4385509668224923E+00, 0.1138121911768345E-01, 0.1314659648329904E-01, 0.4050999534208807E+00, 0.3806452440729016E+00, 0.1951852336551518E-01, 0.3583592846566260E-01, 0.2011877025290218E+00, 0.2206431713856914E+00, 0.2894937755625969E-04, 0.6305450884558253E-04, 0.9233968320433328E-01, 0.9236863258188953E-01, 0.3202909346606844E-42, 0.5962992535266723E-20},
			i0:       1,
			n0:       12,
			pp:       0,
			tau:      9.2159326345418235E-002,
			sigma:    0.73182753419913071,
			i0Out:    1,
			n0Out:    12,
			ppOut:    0,
			tauOut:   9.2159326345418235E-002,
			sigmaOut: 0.73182753419913071,
			dminOut:  1.5272797578758102E-004,
			dmin1Out: 9.6724015270930774E-002,
			dmin2Out: 0.29962742013672317,
			dnOut:    1.5272797578758102E-004,
			dnm1Out:  9.6724015270930774E-002,
			dnm2Out:  0.29962742013672317,
		},
		{
			z:        []float64{0.1749149979302717E+01, 0.1656990652957299E+01, 0.8676862906242004E-16, 0.5850002270797901E-16, 0.1117154804361126E+01, 0.1069375596711644E+01, 0.4438011869593646E-01, 0.4888936980013561E-01, 0.1178029724550133E+01, 0.1081254754299776E+01, 0.4427372589519608E-01, 0.2456274906772994E-01, 0.5998724677255393E+00, 0.6201055550993710E+00, 0.1369551627869799E+00, 0.2540860356894923E+00, 0.1150450694942405E+01, 0.9184843327063353E+00, 0.1142789997988411E+00, 0.1621533314992303E+00, 0.1303260395526232E+01, 0.1060505584247734E+01, 0.1155784656615063E-01, 0.1821431635083262E-01, 0.1671278822811555E+01, 0.1560934342965469E+01, 0.2916285016424428E-04, 0.9328588590960435E-05, 0.4993103973382956E+00, 0.4191509581464933E+00, 0.1200921574220688E-01, 0.1259763844648080E-01, 0.4396883475635844E+00, 0.3463126018893689E+00, 0.1138121911768345E-01, 0.1331320693873929E-01, 0.4050999534208807E+00, 0.3191459435022383E+00, 0.1951852336551518E-01, 0.1230436091267282E-01, 0.2011877025290218E+00, 0.9675296464848704E-01, 0.2894937755625969E-04, 0.2762888312745317E-04, 0.9233968320433328E-01, 0.1527279757875810E-03, 0.3202909346606844E-42, 0.5850002270797901E-16, 0.7318275341991307E+00},
			i0:       1,
			n0:       12,
			pp:       1,
			tau:      1.5267965277267402E-004,
			sigma:    0.82398686054454895,
			i0Out:    1,
			n0Out:    12,
			ppOut:    1,
			tauOut:   1.5267965277267402E-004,
			sigmaOut: 0.82398686054454895,
			dminOut:  2.8998364833248812E-009,
			dmin1Out: 9.2869952038417761E-002,
			dmin2Out: 0.30683160095299705,
			dnOut:    2.8998364833248812E-009,
			dnm1Out:  9.2869952038417761E-002,
			dnm2Out:  0.30683160095299705,
		},
		{
			z:        []float64{0.1656837973304527E+01, 0.1656990652957299E+01, 0.3775776370348287E-16, 0.5850002270797901E-16, 0.1118112286859007E+01, 0.1069375596711644E+01, 0.4727777715386321E-01, 0.4888936980013561E-01, 0.1058387046560870E+01, 0.1081254754299776E+01, 0.1439123541326830E-01, 0.2456274906772994E-01, 0.8596476757228223E+00, 0.6201055550993710E+00, 0.2714763844897646E+00, 0.2540860356894923E+00, 0.8090086000630283E+00, 0.9184843327063353E+00, 0.2125620340079330E+00, 0.1621533314992303E+00, 0.8660051869378607E+00, 0.1060505584247734E+01, 0.3283046378299830E-01, 0.1821431635083262E-01, 0.1527960528118289E+01, 0.1560934342965469E+01, 0.2559023465659061E-05, 0.9328588590960435E-05, 0.4315933579167358E+00, 0.4191509581464933E+00, 0.1010840613748274E-01, 0.1259763844648080E-01, 0.3493647230378528E+00, 0.3463126018893689E+00, 0.1216166289646864E-01, 0.1331320693873929E-01, 0.3191359618656699E+00, 0.3191459435022383E+00, 0.3730332957296599E-02, 0.1230436091267282E-01, 0.9289758092154521E-01, 0.9675296464848704E-01, 0.4542317842367971E-07, 0.2762888312745317E-04, 0.2899836483324881E-08, 0.1527279757875810E-03, 0.3775776370348287E-16, 0.5850002270797901E-16},
			i0:       1,
			n0:       12,
			pp:       0,
			tau:      2.8998350258011044E-009,
			sigma:    0.82413954019732161,
			i0Out:    1,
			n0Out:    12,
			ppOut:    0,
			tauOut:   2.8998350258011044E-009,
			sigmaOut: 0.82413954019732161,
			dminOut:  2.2456987483859239E-017,
			dmin1Out: 9.1786476970678488E-002,
			dmin2Out: 0.30815750788579249,
			dnOut:    2.2456987483859239E-017,
			dnm1Out:  9.1786476970678488E-002,
			dnm2Out:  0.30815750788579249,
		},
		{
			z:        []float64{0.1656837973304527E+01, 0.1656837970404692E+01, 0.3775776370348287E-16, 0.2548071704976161E-16, 0.1118112286859007E+01, 0.1165390061113035E+01, 0.4727777715386321E-01, 0.4293685745187325E-01, 0.1058387046560870E+01, 0.1029841421622430E+01, 0.1439123541326830E-01, 0.1201290976848257E-01, 0.8596476757228223E+00, 0.1119111147544269E+01, 0.2714763844897646E+00, 0.1962510428460807E+00, 0.8090086000630283E+00, 0.8253195883250457E+00, 0.2125620340079330E+00, 0.2230406579474441E+00, 0.8660051869378607E+00, 0.6757949898735797E+00, 0.3283046378299830E-01, 0.7422909836846008E-01, 0.1527960528118289E+01, 0.1453733985873459E+01, 0.2559023465659061E-05, 0.7597383986781573E-06, 0.4315933579167358E+00, 0.4417010014159849E+00, 0.1010840613748274E-01, 0.7995273950601426E-02, 0.3493647230378528E+00, 0.3535311090838850E+00, 0.1216166289646864E-01, 0.1097845108004235E-01, 0.3191359618656699E+00, 0.3118878408430891E+00, 0.3730332957296599E-02, 0.1111101051031713E-02, 0.9289758092154521E-01, 0.9178652239385691E-01, 0.4542317842367971E-07, 0.1435066789177947E-14, 0.2899836483324881E-08, 0.2245698748385924E-16, 0.3775776370348287E-16, 0.2548071704976161E-16, 0.7318275341991307E+00},
			i0:       1,
			n0:       12,
			pp:       1,
			tau:      2.2456987483858885E-017,
			sigma:    0.82413954309715665,
			i0Out:    1,
			n0Out:    12,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.82413954309715665,
			dminOut:  2.2456987483858888E-017,
			dmin1Out: 9.1450429866798411E-002,
			dmin2Out: 0.30232944966555197,
			dnOut:    2.2456987483858888E-017,
			dnm1Out:  9.1450429866798411E-002,
			dnm2Out:  0.30232944966555197,
		},
		{
			z:        []float64{0.1656837970404692E+01, 0.1656837970404692E+01, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.1208326918564908E+01, 0.1165390061113035E+01, 0.3659452888027460E-01, 0.4293685745187325E-01, 0.1005259802510638E+01, 0.1029841421622430E+01, 0.1337343958524596E-01, 0.1201290976848257E-01, 0.1301988750805104E+01, 0.1119111147544269E+01, 0.1244018658302015E+00, 0.1962510428460807E+00, 0.9239583804422884E+00, 0.8253195883250457E+00, 0.1631347930486186E+00, 0.2230406579474441E+00, 0.5868892951934211E+00, 0.6757949898735797E+00, 0.1838666404767374E+00, 0.7422909836846008E-01, 0.1269868105135121E+01, 0.1453733985873459E+01, 0.2642614694812039E-06, 0.7597383986781573E-06, 0.4496960111051168E+00, 0.4417010014159849E+00, 0.6285530663790794E-02, 0.7995273950601426E-02, 0.3582240295001366E+00, 0.3535311090838850E+00, 0.9558391177537082E-02, 0.1097845108004235E-01, 0.3034405507165837E+00, 0.3118878408430891E+00, 0.3360925270585024E-03, 0.1111101051031713E-02, 0.9145042986679984E-01, 0.9178652239385691E-01, 0.3524015903480299E-30, 0.1435066789177947E-14},
			i0:       1,
			n0:       11,
			pp:       0,
			tau:      9.1173077708044642E-002,
			sigma:    0.82413954309715665,
			i0Out:    1,
			n0Out:    11,
			ppOut:    0,
			tauOut:   9.1173077708044642E-002,
			sigmaOut: 0.82413954309715665,
			dminOut:  1.2510327751137640E-004,
			dmin1Out: 0.20154260650453434,
			dmin2Out: 0.26087886618071221,
			dnOut:    1.2510327751137640E-004,
			dnm1Out:  0.20154260650453434,
			dnm2Out:  0.26087886618071221,
		},
		{
			z:        []float64{0.1656837970404692E+01, 0.1565664892696647E+01, 0.1792267857826344E-16, 0.1383211380667930E-16, 0.1208326918564908E+01, 0.1153748369737138E+01, 0.3659452888027460E-01, 0.3188477647299814E-01, 0.1005259802510638E+01, 0.8955753879148407E+00, 0.1337343958524596E-01, 0.1944232516271161E-01, 0.1301988750805104E+01, 0.1315775213764550E+01, 0.1244018658302015E+00, 0.8735697805677015E-01, 0.9239583804422884E+00, 0.9085631177260922E+00, 0.1631347930486186E+00, 0.1053774491236745E+00, 0.5868892951934211E+00, 0.5742054088384394E+00, 0.1838666404767374E+00, 0.4066251880351924E+00, 0.1269868105135121E+01, 0.7720701036533529E+00, 0.2642614694812039E-06, 0.1539203864417861E-06, 0.4496960111051168E+00, 0.3648083101404764E+00, 0.6285530663790794E-02, 0.6172085611379779E-02, 0.3582240295001366E+00, 0.2704372573582493E+00, 0.9558391177537082E-02, 0.1072486650400470E-01, 0.3034405507165837E+00, 0.2018786990315928E+00, 0.3360925270585024E-03, 0.1522488812438262E-03, 0.9145042986679984E-01, 0.1251032775113764E-03, 0.3524015903480299E-30, 0.1383211380667930E-16, 0.8241395430971566E+00},
			i0:       1,
			n0:       11,
			pp:       1,
			tau:      1.2498058228587147E-004,
			sigma:    0.91531262080520126,
			i0Out:    1,
			n0Out:    11,
			ppOut:    1,
			tauOut:   1.2498058228587147E-004,
			sigmaOut: 0.91531262080520126,
			dminOut:  2.4554386425680716E-008,
			dmin1Out: 0.19392428663386141,
			dmin2Out: 0.26581143244488259,
			dnOut:    2.4554386425680716E-008,
			dnm1Out:  0.19392428663386141,
			dnm2Out:  0.26581143244488259,
		},
		{
			z:        []float64{0.1565539912114361E+01, 0.1565664892696647E+01, 0.1019378594629470E-16, 0.1383211380667930E-16, 0.1185508165627851E+01, 0.1153748369737138E+01, 0.2408690373149840E-01, 0.3188477647299814E-01, 0.8908058287637680E+00, 0.8955753879148407E+00, 0.2871751477260568E-01, 0.1944232516271161E-01, 0.1374289696466428E+01, 0.1315775213764550E+01, 0.5775298217141787E-01, 0.8735697805677015E-01, 0.9560626040960629E+00, 0.9085631177260922E+00, 0.6328905763825028E-01, 0.1053774491236745E+00, 0.9174165586530958E+00, 0.5742054088384394E+00, 0.3422034931823232E+00, 0.4066251880351924E+00, 0.4297417838091302E+00, 0.7720701036533529E+00, 0.1306631986684747E-06, 0.1539203864417861E-06, 0.3708552845063717E+00, 0.3648083101404764E+00, 0.4500844331080801E-02, 0.6172085611379779E-02, 0.2765362989488873E+00, 0.2704372573582493E+00, 0.7829431815445537E-02, 0.1072486650400470E-01, 0.1940765355151052E+00, 0.2018786990315928E+00, 0.9814083907923291E-07, 0.1522488812438262E-03, 0.2455438642568072E-07, 0.1251032775113764E-03, 0.1019378594629470E-16, 0.1383211380667930E-16},
			i0:       1,
			n0:       11,
			pp:       0,
			tau:      2.4554370888251911E-008,
			sigma:    0.91543760138748709,
			i0Out:    1,
			n0Out:    11,
			ppOut:    0,
			tauOut:   2.4554370888251911E-008,
			sigmaOut: 0.91543760138748709,
			dminOut:  2.7649302792337775E-015,
			dmin1Out: 0.18866996369915248,
			dmin2Out: 0.27322036360404983,
			dnOut:    2.7649302792337775E-015,
			dnm1Out:  0.18866996369915248,
			dnm2Out:  0.27322036360404983,
		},
		{
			z:        []float64{0.1565539912114361E+01, 0.1565539887559990E+01, 0.1019378594629470E-16, 0.7719264500395369E-17, 0.1185508165627851E+01, 0.1209595044804978E+01, 0.2408690373149840E-01, 0.1773879145177052E-01, 0.8908058287637680E+00, 0.9017845275302323E+00, 0.2871751477260568E-01, 0.4376453959373496E-01, 0.1374289696466428E+01, 0.1388278114489740E+01, 0.5775298217141787E-01, 0.3977262621431848E-01, 0.9560626040960629E+00, 0.9795790109656241E+00, 0.6328905763825028E-01, 0.5927283946360358E-01, 0.9174165586530958E+00, 0.1200347187817445E+01, 0.3422034931823232E+00, 0.1225138368952072E+00, 0.4297417838091302E+00, 0.3072280530227507E+00, 0.1306631986684747E-06, 0.1577236754259593E-06, 0.3708552845063717E+00, 0.3753559465594062E+00, 0.4500844331080801E-02, 0.3315910790466669E-02, 0.2765362989488873E+00, 0.2810497954194954E+00, 0.7829431815445537E-02, 0.5406547261581845E-02, 0.1940765355151052E+00, 0.1886700618399915E+00, 0.9814083907923291E-07, 0.1277249852674422E-13, 0.2455438642568072E-07, 0.2764930279233778E-14, 0.1019378594629470E-16, 0.7719264500395369E-17, 0.8241395430971566E+00},
			i0:       1,
			n0:       11,
			pp:       1,
			tau:      2.7649302792335523E-015,
			sigma:    0.91543762594185796,
			i0Out:    1,
			n0Out:    11,
			ppOut:    1,
			tauOut:   2.7649302792335523E-015,
			sigmaOut: 0.91543762594185796,
			dminOut:  3.4709879829724519E-029,
			dmin1Out: 0.18507826404867261,
			dmin2Out: 0.27717487004787600,
			dnOut:    3.4709879829724519E-029,
			dnm1Out:  0.18507826404867261,
			dnm2Out:  0.27858872904535659,
		},
		{
			z:        []float64{0.1565539887559988E+01, 0.1565539887559990E+01, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.1227333836256746E+01, 0.1209595044804978E+01, 0.1303359134714337E-01, 0.1773879145177052E-01, 0.9325154757768211E+00, 0.9017845275302323E+00, 0.6515425651042277E-01, 0.4376453959373496E-01, 0.1362896484193633E+01, 0.1388278114489740E+01, 0.2858649229958118E-01, 0.3977262621431848E-01, 0.1010265358129644E+01, 0.9795790109656241E+00, 0.7042504782685149E-01, 0.5927283946360358E-01, 0.1252435976885798E+01, 0.1200347187817445E+01, 0.3005318297487194E-01, 0.1225138368952072E+00, 0.2771750277715515E+00, 0.3072280530227507E+00, 0.2135925446109612E-06, 0.1577236754259593E-06, 0.3786716437573254E+00, 0.3753559465594062E+00, 0.2461066374135985E-02, 0.3315910790466669E-02, 0.2839952763069384E+00, 0.2810497954194954E+00, 0.3591797791316158E-02, 0.5406547261581845E-02, 0.1850782640486854E+00, 0.1886700618399915E+00, 0.1908115364037247E-27, 0.1277249852674422E-13},
			i0:       1,
			n0:       10,
			pp:       0,
			tau:      9.2539132024336307E-002,
			sigma:    0.91543762594186073,
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			tauOut:   9.2539132024336307E-002,
			sigmaOut: 0.91543762594186073,
			dminOut:  8.9088074504151571E-002,
			dmin1Out: 0.17705256406949207,
			dmin2Out: 0.17705256406949207,
			dnOut:    8.9088074504151571E-002,
			dnm1Out:  0.18903428744201686,
			dnm2Out:  0.28613205491191551,
		},
		{
			z:        []float64{0.1565539887559988E+01, 0.1473000755535651E+01, 0.5964194309842799E-17, 0.4969486576955697E-17, 0.1227333836256746E+01, 0.1147828295579553E+01, 0.1303359134714337E-01, 0.1058871408116432E-01, 0.9325154757768211E+00, 0.8945418861817434E+00, 0.6515425651042277E-01, 0.9926701980086396E-01, 0.1362896484193633E+01, 0.1199676824668014E+01, 0.2858649229958118E-01, 0.2407310226126826E-01, 0.1010265358129644E+01, 0.9640781716708908E+00, 0.7042504782685149E-01, 0.9148932748822959E-01, 0.1252435976885798E+01, 0.1098460700348104E+01, 0.3005318297487194E-01, 0.7583331677723075E-02, 0.2771750277715515E+00, 0.1770527776620367E+00, 0.2135925446109612E-06, 0.4568210735249311E-06, 0.3786716437573254E+00, 0.2885931212860515E+00, 0.2461066374135985E-02, 0.2421856840585221E-02, 0.2839952763069384E+00, 0.1926260852333330E+00, 0.3591797791316158E-02, 0.3451057520197492E-02, 0.1850782640486854E+00, 0.8908807450415157E-01, 0.1908115364037247E-27, 0.4969486576955697E-17, 0.9154376259418607E+00},
			i0:       1,
			n0:       10,
			pp:       1,
			tau:      7.5387921832037685E-002,
			sigma:    1.0079767579661971,
			i0Out:    1,
			n0Out:    10,
			ppOut:    1,
			tauOut:   7.5387921832037685E-002,
			sigmaOut: 1.0079767579661971,
			dminOut:  1.1106216559126303E-002,
			dmin1Out: 0.10021441433509834,
			dmin2Out: 0.10021441433509834,
			dnOut:    1.1106216559126303E-002,
			dnm1Out:  0.11507463320059727,
			dnm2Out:  0.21320388392650824,
		},
		{
			z:        []float64{0.1397612833703614E+01, 0.1473000755535651E+01, 0.4081328655531061E-17, 0.4969486576955697E-17, 0.1083029087828680E+01, 0.1147828295579553E+01, 0.8745885380967966E-02, 0.1058871408116432E-01, 0.9096750987696016E+00, 0.8945418861817434E+00, 0.1309130515609722E+00, 0.9926701980086396E-01, 0.1017448953536272E+01, 0.1199676824668014E+01, 0.2281033592282574E-01, 0.2407310226126826E-01, 0.9573692414042568E+00, 0.9640781716708908E+00, 0.1049724875218357E+00, 0.9148932748822959E-01, 0.9256836226719531E+00, 0.1098460700348104E+01, 0.1450441494900679E-02, 0.7583331677723075E-02, 0.1002148711561719E+00, 0.1770527776620367E+00, 0.1315527505616969E-05, 0.4568210735249311E-06, 0.2156257407670935E+00, 0.2885931212860515E+00, 0.2163530200698068E-02, 0.2421856840585221E-02, 0.1185256907207948E+00, 0.1926260852333330E+00, 0.2593936112987583E-02, 0.3451057520197492E-02, 0.1110621655912630E-01, 0.8908807450415157E-01, 0.4081328655531061E-17, 0.4969486576955697E-17},
			i0:       1,
			n0:       10,
			pp:       0,
			tau:      9.2006637361187298E-003,
			sigma:    1.0833646797982348,
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			tauOut:   9.2006637361187298E-003,
			sigmaOut: 1.0833646797982348,
			dminOut:  1.6452860548183366E-003,
			dmin1Out: 9.0837695870445614E-002,
			dmin2Out: 9.0837695870445614E-002,
			dnOut:    1.6452860548183366E-003,
			dnm1Out:  0.10809563217808144,
			dnm2Out:  0.20642195434625446,
		},
		{
			z:        []float64{0.1397612833703614E+01, 0.1388412169967495E+01, 0.4081328655531061E-17, 0.3183635051997810E-17, 0.1083029087828680E+01, 0.1082574309473529E+01, 0.8745885380967966E-02, 0.7349069784991225E-02, 0.9096750987696016E+00, 0.1024038416809464E+01, 0.1309130515609722E+00, 0.1300706547025319E+00, 0.1017448953536272E+01, 0.9009879710204475E+00, 0.2281033592282574E-01, 0.2423774201322422E-01, 0.9573692414042568E+00, 0.1028903323176749E+01, 0.1049724875218357E+00, 0.9444163542020824E-01, 0.9256836226719531E+00, 0.8234917650105269E+00, 0.1450441494900679E-02, 0.1765115496075164E-03, 0.1002148711561719E+00, 0.9083901139795122E-01, 0.1315527505616969E-05, 0.3122684720284652E-05, 0.2156257407670935E+00, 0.2085854845469525E+00, 0.2163530200698068E-02, 0.1229394806594584E-02, 0.1185256907207948E+00, 0.1106895682910690E+00, 0.2593936112987583E-02, 0.2602667681892373E-03, 0.1110621655912630E-01, 0.1645286054818337E-02, 0.4081328655531061E-17, 0.3183635051997810E-17, 0.9154376259418607E+00},
			i0:       1,
			n0:       10,
			pp:       1,
			tau:      1.5594219393746818E-003,
			sigma:    1.0925653435343534,
			i0Out:    1,
			n0Out:    10,
			ppOut:    1,
			tauOut:   1.5594219393746818E-003,
			sigmaOut: 1.0925653435343534,
			dminOut:  8.1926049685687600E-005,
			dmin1Out: 8.9258234858565516E-002,
			dmin2Out: 8.9258234858565516E-002,
			dnOut:    8.1926049685687600E-005,
			dnm1Out:  0.10847668953277810,
			dnm2Out:  0.20701876553386761,
		},
		{
			z:        []float64{0.1386852748028120E+01, 0.1388412169967495E+01, 0.2485138759635906E-17, 0.3183635051997810E-17, 0.1088363957319145E+01, 0.1082574309473529E+01, 0.6914717946174950E-02, 0.7349069784991225E-02, 0.1145634931626446E+01, 0.1024038416809464E+01, 0.1022944500333619E+00, 0.1300706547025319E+00, 0.8213718410609353E+00, 0.9009879710204475E+00, 0.3036175828902931E-01, 0.2423774201322422E-01, 0.1091423778368554E+01, 0.1028903323176749E+01, 0.7125729765473905E-01, 0.9444163542020824E-01, 0.7508515569660207E+00, 0.8234917650105269E+00, 0.2135460001102832E-04, 0.1765115496075164E-03, 0.8926135754328580E-01, 0.9083901139795122E-01, 0.7297073710223142E-05, 0.3122684720284652E-05, 0.2082481603404622E+00, 0.2085854845469525E+00, 0.6534568189162364E-03, 0.1229394806594584E-02, 0.1087369563009673E+00, 0.1106895682910690E+00, 0.3938065757966957E-05, 0.2602667681892373E-03, 0.8192604968568760E-04, 0.1645286054818337E-02, 0.2485138759635906E-17, 0.3183635051997810E-17},
			i0:       1,
			n0:       10,
			pp:       0,
			tau:      8.1416944590412474E-005,
			sigma:    1.0941247654737283,
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			tauOut:   8.1416944590412474E-005,
			sigmaOut: 1.0941247654737283,
			dminOut:  5.0612658152462498E-007,
			dmin1Out: 8.9177229293753768E-002,
			dmin2Out: 8.9177229293753768E-002,
			dnOut:    5.0612658152462498E-007,
			dnm1Out:  0.10831524323681536,
			dnm2Out:  0.20814970454128662,
		},
		{
			z:        []float64{0.1386852748028120E+01, 0.1386771331083530E+01, 0.2485138759635906E-17, 0.1950383162890474E-17, 0.1088363957319145E+01, 0.1095197258320730E+01, 0.6914717946174950E-02, 0.7233164949324962E-02, 0.1145634931626446E+01, 0.1240614799765893E+01, 0.1022944500333619E+00, 0.6772592167212049E-01, 0.8213718410609353E+00, 0.7839262607332537E+00, 0.3036175828902931E-01, 0.4227125255215914E-01, 0.1091423778368554E+01, 0.1120328406526543E+01, 0.7125729765473905E-01, 0.4775711530437247E-01, 0.7508515569660207E+00, 0.7030343793170689E+00, 0.2135460001102832E-04, 0.2711304941630149E-05, 0.8926135754328580E-01, 0.8918452636746399E-01, 0.7297073710223142E-05, 0.1703885458517271E-04, 0.2082481603404622E+00, 0.2088031613602029E+00, 0.6534568189162364E-03, 0.3402961195615630E-03, 0.1087369563009673E+00, 0.1083191813025733E+00, 0.3938065757966957E-05, 0.2978513750500819E-08, 0.8192604968568760E-04, 0.5061265815246250E-06, 0.2485138759635906E-17, 0.1950383162890474E-17, 0.9154376259418607E+00},
			i0:       1,
			n0:       10,
			pp:       1,
			tau:      5.0604049633765406E-007,
			sigma:    1.0942061824183187,
			i0Out:    1,
			n0Out:    10,
			ppOut:    1,
			tauOut:   5.0604049633765406E-007,
			sigmaOut: 1.0942061824183187,
			dminOut:  8.6071246971392626E-011,
			dmin1Out: 8.9183660885577137E-002,
			dmin2Out: 8.9183660885577137E-002,
			dnOut:    8.6071246971392626E-011,
			dnm1Out:  0.10814239569959758,
			dnm2Out:  0.20876277035564564,
		},
		{
			z:        []float64{0.1386770825043033E+01, 0.1386771331083530E+01, 0.1540308069724697E-17, 0.1950383162890474E-17, 0.1102429917229558E+01, 0.1095197258320730E+01, 0.8139811288713328E-02, 0.7233164949324962E-02, 0.1300200404108803E+01, 0.1240614799765893E+01, 0.4083380405309871E-01, 0.6772592167212049E-01, 0.7853632031918177E+00, 0.7839262607332537E+00, 0.6030036143936175E-01, 0.4227125255215914E-01, 0.1107784654351057E+01, 0.1120328406526543E+01, 0.3030814137396716E-01, 0.4775711530437247E-01, 0.6727284432075471E+00, 0.7030343793170689E+00, 0.3594413905024206E-06, 0.2711304941630149E-05, 0.8920069974016231E-01, 0.8918452636746399E-01, 0.3988496406087022E-04, 0.1703885458517271E-04, 0.2091030664752072E+00, 0.2088031613602029E+00, 0.1762795624794033E-03, 0.3402961195615630E-03, 0.1081423986781113E+00, 0.1083191813025733E+00, 0.1393999949133917E-13, 0.2978513750500819E-08, 0.8607124697139263E-10, 0.5061265815246250E-06, 0.1540308069724697E-17, 0.1950383162890474E-17},
			i0:       1,
			n0:       10,
			pp:       0,
			tau:      8.6071215292546838E-011,
			sigma:    1.0942066884588149,
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			tauOut:   8.6071215292546838E-011,
			sigmaOut: 1.0942066884588149,
			dminOut:  3.1678834699165494E-017,
			dmin1Out: 8.9200650586607991E-002,
			dmin2Out: 8.9200650586607991E-002,
			dnOut:    3.1678834699165494E-017,
			dnm1Out:  0.10805126770967630,
			dnm2Out:  0.20900961033533805,
		},
		{
			z:        []float64{0.1386770825043033E+01, 0.1386770824956962E+01, 0.1540308069724697E-17, 0.1224486171222500E-17, 0.1102429917229558E+01, 0.1110569728432201E+01, 0.8139811288713328E-02, 0.9529690622753703E-02, 0.1300200404108803E+01, 0.1331504517453077E+01, 0.4083380405309871E-01, 0.2408506071837550E-01, 0.7853632031918177E+00, 0.8215785038267327E+00, 0.6030036143936175E-01, 0.8130667336500198E-01, 0.1107784654351057E+01, 0.1056786122273951E+01, 0.3030814137396716E-01, 0.1929354325655847E-01, 0.6727284432075471E+00, 0.6534352593063080E+00, 0.3594413905024206E-06, 0.4906748310831983E-07, 0.8920069974016231E-01, 0.8924053555066887E-01, 0.3988496406087022E-04, 0.9345605379795243E-04, 0.2091030664752072E+00, 0.2091858898978174E+00, 0.1762795624794033E-03, 0.9113088236382798E-04, 0.1081423986781113E+00, 0.1080512677096902E+00, 0.1393999949133917E-13, 0.1110429488179469E-22, 0.8607124697139263E-10, 0.3167883469916549E-16, 0.1540308069724697E-17, 0.1224486171222500E-17, 0.9154376259418607E+00},
			i0:       1,
			n0:       10,
			pp:       1,
			tau:      3.1678834698836348E-017,
			sigma:    1.0942066885448862,
			i0Out:    1,
			n0Out:    10,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.0942066885448862,
			dminOut:  3.1678834699165494E-017,
			dmin1Out: 8.9240528714785800E-002,
			dmin2Out: 8.9240528714785800E-002,
			dnOut:    3.1678834699165494E-017,
			dnm1Out:  0.10800416691102718,
			dnm2Out:  0.20896705170504334,
		},
		{
			z:        []float64{0.1386770824956962E+01, 0.1386770824956962E+01, 0.9806070694382835E-18, 0.1224486171222500E-17, 0.1120099419054954E+01, 0.1110569728432201E+01, 0.1132830345080667E-01, 0.9529690622753703E-02, 0.1344261274720646E+01, 0.1331504517453077E+01, 0.1472018016266302E-01, 0.2408506071837550E-01, 0.8881649970290716E+00, 0.8215785038267327E+00, 0.9674301998819117E-01, 0.8130667336500198E-01, 0.9793366455423188E+00, 0.1056786122273951E+01, 0.1287308250760436E-01, 0.1929354325655847E-01, 0.6405622258661866E+00, 0.6534352593063080E+00, 0.6835883063177366E-08, 0.4906748310831983E-07, 0.8933398476858376E-01, 0.8924053555066887E-01, 0.2188381927740679E-03, 0.9345605379795243E-04, 0.2090581825874072E+00, 0.2091858898978174E+00, 0.4710079866305792E-04, 0.9113088236382798E-04, 0.1080041669110272E+00, 0.1080512677096902E+00, 0.3257014354834561E-38, 0.1110429488179469E-22},
			i0:       1,
			n0:       9,
			pp:       0,
			tau:      2.2310132178696450E-002,
			sigma:    1.0942066885448862,
			i0Out:    1,
			n0Out:    9,
			ppOut:    0,
			tauOut:   2.2310132178696450E-002,
			sigmaOut: 1.0942066885448862,
			dminOut:  6.7023851586769906E-002,
			dmin1Out: 6.7023851586769906E-002,
			dmin2Out: 6.7023851586769906E-002,
			dnOut:    8.5666701693601133E-002,
			dnm1Out:  0.18606768031923254,
			dnm2Out:  6.7023851586769906E-002,
		},
		{
			z:        []float64{0.1386770824956962E+01, 0.1364460692778266E+01, 0.9806070694382835E-18, 0.8049901434408684E-18, 0.1120099419054954E+01, 0.1109117590327065E+01, 0.1132830345080667E-01, 0.1373001363427395E-01, 0.1344261274720646E+01, 0.1322941309070339E+01, 0.1472018016266302E-01, 0.9882485852396858E-02, 0.8881649970290716E+00, 0.9527153989861693E+00, 0.9674301998819117E-01, 0.9944626147083412E-01, 0.9793366455423188E+00, 0.8704533344003925E+00, 0.1287308250760436E-01, 0.9473236598617136E-02, 0.6405622258661866E+00, 0.6087788639247561E+00, 0.6835883063177366E-08, 0.1003117403762534E-08, 0.8933398476858376E-01, 0.6724268977954398E-01, 0.2188381927740679E-03, 0.6803700894781421E-03, 0.2090581825874072E+00, 0.1861147811178956E+00, 0.4710079866305792E-04, 0.2733303872960605E-04, 0.1080041669110272E+00, 0.8566670169360113E-01, 0.3257014354834561E-38, 0.8049901434408684E-18, 0.1094206688544886E+01},
			i0:       1,
			n0:       9,
			pp:       1,
			tau:      6.4730147312741043E-002,
			sigma:    1.1165168207235825,
			i0Out:    1,
			n0Out:    9,
			ppOut:    1,
			tauOut:   6.4730147312741043E-002,
			sigmaOut: 1.1165168207235825,
			dminOut:  2.5125423409859404E-003,
			dmin1Out: 2.5125423409859404E-003,
			dmin2Out: 2.5125423409859404E-003,
			dnOut:    2.0907912921905053E-002,
			dnm1Out:  8.1725879285544201E-002,
			dnm2Out:  2.5125423409859404E-003,
		},
		{
			z:        []float64{0.1299730545465525E+01, 0.1364460692778266E+01, 0.6869337119490330E-18, 0.8049901434408684E-18, 0.1058117456648598E+01, 0.1109117590327065E+01, 0.1716633828952343E-01, 0.1373001363427395E-01, 0.1250927309320471E+01, 0.1322941309070339E+01, 0.7526573591998700E-02, 0.9882485852396858E-02, 0.9799049395522637E+00, 0.9527153989861693E+00, 0.8833849733474472E-01, 0.9944626147083412E-01, 0.7268579263515238E+00, 0.8704533344003925E+00, 0.7934296380510910E-02, 0.9473236598617136E-02, 0.5361144212346215E+00, 0.6087788639247561E+00, 0.1258170079408221E-09, 0.1003117403762534E-08, 0.3192912430464083E-02, 0.6724268977954398E-01, 0.3965875451961033E-01, 0.6803700894781421E-03, 0.8175321232427381E-01, 0.1861147811178956E+00, 0.2864145895504550E-04, 0.2733303872960605E-04, 0.2090791292190505E-01, 0.8566670169360113E-01, 0.6869337119490330E-18, 0.8049901434408684E-18},
			i0:       1,
			n0:       9,
			pp:       0,
			tau:      2.3478378904869292E-003,
			sigma:    1.1812469680363236,
			i0Out:    1,
			n0Out:    9,
			ppOut:    0,
			tauOut:   2.3478378904869292E-003,
			sigmaOut: 1.1812469680363236,
			dminOut:  -6.4213352031532861E-004,
			dmin1Out: -6.4213352031532861E-004,
			dmin2Out: 8.4507453921550072E-004,
			dnOut:    1.9536180783236046E-002,
			dnm1Out:  -6.4213352031532861E-004,
			dnm2Out:  8.4507453921550072E-004,
		},
		{
			z:        []float64{0.1299730545465525E+01, 0.1297382707575038E+01, 0.6869337119490330E-18, 0.5602483738451179E-18, 0.1058117456648598E+01, 0.1072935957047634E+01, 0.1716633828952343E-01, 0.2001409424891253E-01, 0.1250927309320471E+01, 0.1236091950773070E+01, 0.7526573591998700E-02, 0.5966648869520194E-02, 0.9799049395522637E+00, 0.1059928950127001E+01, 0.8833849733474472E-01, 0.6057909540261972E-01, 0.7268579263515238E+00, 0.6718652894389281E+00, 0.7934296380510910E-02, 0.6331166051893822E-02, 0.5361144212346215E+00, 0.5274354174180577E+00, 0.1258170079408221E-09, 0.7616528495272331E-12, 0.3192912430464083E-02, 0.4050382905882583E-01, 0.3965875451961033E-01, 0.8004750795410222E-01, 0.8175321232427381E-01, -0.6134920613602832E-03, 0.2864145895504550E-04, -0.9761057518179271E-03, 0.2090791292190505E-01, 0.1953618078323605E-01, 0.6869337119490330E-18, 0.5602483738451179E-18},
			i0:       1,
			n0:       9,
			pp:       0,
			tau:      5.8695947262173229E-004,
			sigma:    1.1812469680363236,
			i0Out:    1,
			n0Out:    9,
			ppOut:    0,
			tauOut:   5.8695947262173229E-004,
			sigmaOut: 1.1812469680363236,
			dminOut:  2.6059529570832572E-003,
			dmin1Out: 2.6059529570832572E-003,
			dmin2Out: 2.6059529570832572E-003,
			dnOut:    2.0187357299561493E-002,
			dnm1Out:  4.4537716273149721E-003,
			dnm2Out:  2.6059529570832572E-003,
		},
		{
			z:        []float64{0.1299730545465525E+01, 0.1299143585992903E+01, 0.6869337119490330E-18, 0.5594890049187079E-18, 0.1058117456648598E+01, 0.1074696835465499E+01, 0.1716633828952343E-01, 0.1998130138542489E-01, 0.1250927309320471E+01, 0.1237885622054423E+01, 0.7526573591998700E-02, 0.5958003315736791E-02, 0.9799049395522637E+00, 0.1061698474098650E+01, 0.8833849733474472E-01, 0.6047812873071532E-01, 0.7268579263515238E+00, 0.6737271345286977E+00, 0.7934296380510910E-02, 0.6313669873066943E-02, 0.5361144212346215E+00, 0.5292137920147498E+00, 0.1258170079408221E-09, 0.7590933847144573E-12, 0.3192912430464083E-02, 0.4226470747669359E-01, 0.3965875451961033E-01, 0.7671248122433710E-01, 0.8175321232427381E-01, 0.4482413086270017E-02, 0.2864145895504550E-04, 0.1335961497218277E-03, 0.2090791292190505E-01, 0.2018735729956149E-01, 0.6869337119490330E-18, 0.5594890049187079E-18, 0.1094206688544886E+01},
			i0:       1,
			n0:       9,
			pp:       1,
			tau:      4.9324499328963489E-004,
			sigma:    1.1818339275089453,
			i0Out:    1,
			n0Out:    9,
			ppOut:    1,
			tauOut:   4.9324499328963489E-004,
			sigmaOut: 1.1818339275089453,
			dminOut:  1.0870277779300500E-003,
			dmin1Out: 1.0870277779300500E-003,
			dmin2Out: 4.1771462483342674E-002,
			dnOut:    1.7484624889890960E-002,
			dnm1Out:  1.0870277779300500E-003,
			dnm2Out:  4.1771462483342674E-002,
		},
		{
			z:        []float64{0.1298650340999613E+01, 0.1299143585992903E+01, 0.4630045856693428E-18, 0.5594890049187079E-18, 0.1094184891857634E+01, 0.1074696835465499E+01, 0.2260547177996662E-01, 0.1998130138542489E-01, 0.1220744908596903E+01, 0.1237885622054423E+01, 0.5181756634367578E-02, 0.5958003315736791E-02, 0.1116501601201708E+01, 0.1061698474098650E+01, 0.3649413160495897E-01, 0.6047812873071532E-01, 0.6430534278035160E+00, 0.6737271345286977E+00, 0.5195962000339362E-02, 0.6313669873066943E-02, 0.5235245850218799E+00, 0.5292137920147498E+00, 0.6128243213469890E-13, 0.7590933847144573E-12, 0.1184839437076798E+00, 0.4226470747669359E-01, 0.2902140315050332E-02, 0.7671248122433710E-01, 0.1220623927651878E-02, 0.4482413086270017E-02, 0.2209487416380896E-02, 0.1335961497218277E-03, 0.1748462488989096E-01, 0.2018735729956149E-01, 0.4630045856693428E-18, 0.5594890049187079E-18},
			i0:       1,
			n0:       9,
			pp:       0,
			tau:      8.6238530354903250E-004,
			sigma:    1.1823271725022351,
			i0Out:    1,
			n0Out:    9,
			ppOut:    0,
			tauOut:   8.6238530354903250E-004,
			sigmaOut: 1.1823271725022351,
			dminOut:  3.2884671240747138E-004,
			dmin1Out: 3.2884671240747138E-004,
			dmin2Out: 0.11762155840411674,
			dnOut:    1.4027859154437344E-003,
			dnm1Out:  3.2884671240747138E-004,
			dnm2Out:  0.11762155840411674,
		},
		{
			z:        []float64{0.1298650340999613E+01, 0.1297787955696064E+01, 0.4630045856693428E-18, 0.3903662538064461E-18, 0.1094184891857634E+01, 0.1115927978334052E+01, 0.2260547177996662E-01, 0.2472875948770642E-01, 0.1220744908596903E+01, 0.1200335520440015E+01, 0.5181756634367578E-02, 0.4819852017032840E-02, 0.1116501601201708E+01, 0.1147313495486085E+01, 0.3649413160495897E-01, 0.2045445862496274E-01, 0.6430534278035160E+00, 0.6269325458753436E+00, 0.5195962000339362E-02, 0.4338925882718484E-02, 0.5235245850218799E+00, 0.5183232738356738E+00, 0.6128243213469890E-13, 0.1400860159256393E-13, 0.1184839437076798E+00, 0.1205236987191671E+00, 0.2902140315050332E-02, 0.2939191169537379E-04, 0.1220623927651878E-02, 0.2538334128788367E-02, 0.2209487416380896E-02, 0.1521945367089819E-01, 0.1748462488989096E-01, 0.1402785915443734E-02, 0.4630045856693428E-18, 0.3903662538064461E-18, 0.1094206688544886E+01},
			i0:       1,
			n0:       9,
			pp:       1,
			tau:      1.8442717265434725E-004,
			sigma:    1.1831895578057841,
			i0Out:    1,
			n0Out:    9,
			ppOut:    1,
			tauOut:   1.8442717265434725E-004,
			sigmaOut: 1.1831895578057841,
			dminOut:  3.4295817345512611E-006,
			dmin1Out: 2.3532871395603098E-003,
			dmin2Out: 0.12033927154650945,
			dnOut:    3.4295817345512611E-006,
			dnm1Out:  2.3532871395603098E-003,
			dnm2Out:  0.12033927154650945,
		},
		{
			z:        []float64{0.1297603528523410E+01, 0.1297787955696064E+01, 0.3357116521683426E-18, 0.3903662538064461E-18, 0.1140472310649104E+01, 0.1115927978334052E+01, 0.2602676813137004E-01, 0.2472875948770642E-01, 0.1178944177153024E+01, 0.1200335520440015E+01, 0.4690536984322235E-02, 0.4819852017032840E-02, 0.1162892989954071E+01, 0.1147313495486085E+01, 0.1102729651913737E-01, 0.2045445862496274E-01, 0.6200597480662705E+00, 0.6269325458753436E+00, 0.3627015421456806E-02, 0.4338925882718484E-02, 0.5145118312415766E+00, 0.5183232738356738E+00, 0.3281495925457712E-14, 0.1400860159256393E-13, 0.1203686634582048E+00, 0.1205236987191671E+00, 0.6198165737098726E-06, 0.2939191169537379E-04, 0.1757274081045850E-01, 0.2538334128788367E-02, 0.1214929161054836E-02, 0.1521945367089819E-01, 0.3429581734551261E-05, 0.1402785915443734E-02, 0.3357116521683426E-18, 0.3903662538064461E-18},
			i0:       1,
			n0:       9,
			pp:       0,
			tau:      3.2077610710809750E-006,
			sigma:    1.1833739849784384,
			i0Out:    1,
			n0Out:    9,
			ppOut:    0,
			tauOut:   3.2077610710809750E-006,
			sigmaOut: 1.1833739849784384,
			dminOut:  3.3505760549073344E-012,
			dmin1Out: 1.7569442559804263E-002,
			dmin2Out: 0.12036545569713296,
			dnOut:    3.3505760549073344E-012,
			dnm1Out:  1.7569442559804263E-002,
			dnm2Out:  0.12036545569713296,
		},
		{
			z:        []float64{0.1297603528523410E+01, 0.1297600320762339E+01, 0.3357116521683426E-18, 0.2950599175525191E-18, 0.1140472310649104E+01, 0.1166495871019403E+01, 0.2602676813137004E-01, 0.2630451380147252E-01, 0.1178944177153024E+01, 0.1157326992574803E+01, 0.4690536984322235E-02, 0.4713095445958055E-02, 0.1162892989954071E+01, 0.1169203983266179E+01, 0.1102729651913737E-01, 0.5848066547299594E-02, 0.6200597480662705E+00, 0.6178354891793567E+00, 0.3627015421456806E-02, 0.3020451850239120E-02, 0.5145118312415766E+00, 0.5114881716302697E+00, 0.3281495925457712E-14, 0.7722354114894537E-15, 0.1203686634582048E+00, 0.1203660755137067E+00, 0.6198165737098726E-06, 0.9048958315991394E-07, 0.1757274081045850E-01, 0.1878437172085910E-01, 0.1214929161054836E-02, 0.2218173128942313E-06, 0.3429581734551261E-05, 0.3350576054907334E-11, 0.3357116521683426E-18, 0.2950599175525191E-18, 0.1094206688544886E+01},
			i0:       1,
			n0:       9,
			pp:       1,
			tau:      3.3505364896797715E-012,
			sigma:    1.1833771927395094,
			i0Out:    1,
			n0Out:    9,
			ppOut:    1,
			tauOut:   3.3505364896797715E-012,
			sigmaOut: 1.1833771927395094,
			dminOut:  2.0335799151090034E-023,
			dmin1Out: 1.8784357595683275E-002,
			dmin2Out: 0.12036607551035594,
			dnOut:    2.0335799151090034E-023,
			dnm1Out:  1.8784357595683275E-002,
			dnm2Out:  0.12036607551035594,
		},
		{
			z:        []float64{0.1297600320758988E+01, 0.1297600320762339E+01, 0.2652482201353177E-18, 0.2950599175525191E-18, 0.1192800384817525E+01, 0.1166495871019403E+01, 0.2552222839336001E-01, 0.2630451380147252E-01, 0.1136517859624050E+01, 0.1157326992574803E+01, 0.4848643531876122E-02, 0.4713095445958055E-02, 0.1170203406278252E+01, 0.1169203983266179E+01, 0.3087619670750762E-02, 0.5848066547299594E-02, 0.6177683213554945E+00, 0.6178354891793567E+00, 0.2500816796475141E-02, 0.3020451850239120E-02, 0.5089873548304448E+00, 0.5114881716302697E+00, 0.1826193617023393E-15, 0.7722354114894537E-15, 0.1203661659999391E+00, 0.1203660755137067E+00, 0.1412182528886294E-07, 0.9048958315991394E-07, 0.1878457941299617E-01, 0.1878437172085910E-01, 0.3956520722700361E-16, 0.2218173128942313E-06, 0.2033579915109003E-22, 0.3350576054907334E-11, 0.2652482201353177E-18, 0.2950599175525191E-18},
			i0:       1,
			n0:       9,
			pp:       0,
			tau:      2.0335799151089990E-023,
			sigma:    1.1833771927428600,
			i0Out:    1,
			n0Out:    9,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.1833771927428600,
			dminOut:  2.0335799151089993E-023,
			dmin1Out: 1.8784577209116738E-002,
			dmin2Out: 0.12036616599993906,
			dnOut:    2.0335799151089993E-023,
			dnm1Out:  1.8784577209116738E-002,
			dnm2Out:  0.12036616599993906,
		},
		{
			z:        []float64{0.1297600320758988E+01, 0.1297600320758988E+01, 0.2652482201353177E-18, 0.2438256017573345E-18, 0.1192800384817525E+01, 0.1218322613210885E+01, 0.2552222839336001E-01, 0.2380852827644004E-01, 0.1136517859624050E+01, 0.1117557974879486E+01, 0.4848643531876122E-02, 0.5077051306839188E-02, 0.1170203406278252E+01, 0.1168213974642164E+01, 0.3087619670750762E-02, 0.1632777609571200E-02, 0.6177683213554945E+00, 0.6186363605423986E+00, 0.2500816796475141E-02, 0.2057564358223963E-02, 0.5089873548304448E+00, 0.5069297904722210E+00, 0.1826193617023393E-15, 0.4336141378669131E-16, 0.1203661659999391E+00, 0.1203661801217644E+00, 0.1412182528886294E-07, 0.2203879431304952E-08, 0.1878457941299617E-01, 0.1878457720911678E-01, 0.3956520722700361E-16, 0.4283248425464200E-37, 0.1183377192742860E+01},
			i0:       1,
			n0:       8,
			pp:       1,
			tau:      1.8784576682472597E-002,
			sigma:    1.1833771927428600,
			i0Out:    1,
			n0Out:    8,
			ppOut:    1,
			tauOut:   1.8784576682472597E-002,
			sigmaOut: 1.1833771927428600,
			dminOut:  1.1910047947871760E-010,
			dmin1Out: 0.10158160343929173,
			dmin2Out: 0.48640978493259379,
			dnOut:    1.1910047947871760E-010,
			dnm1Out:  0.10158160343929173,
			dnm2Out:  0.48640978493259379,
		},
		{
			z:        []float64{0.1278815744076516E+01, 0.1297600320758988E+01, 0.2322916695987584E-18, 0.2438256017573345E-18, 0.1223346564804853E+01, 0.1218322613210885E+01, 0.2174969171530204E-01, 0.2380852827644004E-01, 0.1082100757788551E+01, 0.1117557974879486E+01, 0.5481081353963683E-02, 0.5077051306839188E-02, 0.1145581094215299E+01, 0.1168213974642164E+01, 0.8817320773368219E-03, 0.1632777609571200E-02, 0.6010276161408131E+00, 0.6186363605423986E+00, 0.1735428857154564E-02, 0.2057564358223963E-02, 0.4864097849325938E+00, 0.5069297904722210E+00, 0.1073014545319338E-16, 0.4336141378669131E-16, 0.1015816056431712E+00, 0.1203661801217644E+00, 0.4075436992240071E-09, 0.2203879431304952E-08, 0.1191004794787176E-09, 0.1878457720911678E-01, 0.2322916695987584E-18, 0.2438256017573345E-18},
			i0:       1,
			n0:       8,
			pp:       0,
			tau:      1.1910047900088848E-010,
			sigma:    1.2021617694253326,
			i0Out:    1,
			n0Out:    8,
			ppOut:    0,
			tauOut:   1.1910047900088848E-010,
			sigmaOut: 1.2021617694253326,
			dminOut:  0.0000000000000000,
			dmin1Out: 0.10158160552407068,
			dmin2Out: 0.48500827096381138,
			dnOut:    0.0000000000000000,
			dnm1Out:  0.10158160552407068,
			dnm2Out:  0.48500827096381138,
		},
		{
			z:        []float64{0.1278815744076516E+01, 0.1278815743957415E+01, 0.2322916695987584E-18, 0.2222159192042978E-18, 0.1223346564804853E+01, 0.1245096256401054E+01, 0.2174969171530204E-01, 0.1890244048666934E-01, 0.1082100757788551E+01, 0.1068679398536745E+01, 0.5481081353963683E-02, 0.5875497537946494E-02, 0.1145581094215299E+01, 0.1140587328635589E+01, 0.8817320773368219E-03, 0.4646249482278370E-03, 0.6010276161408131E+00, 0.6022984199306394E+00, 0.1735428857154564E-02, 0.1401513849681988E-02, 0.4864097849325938E+00, 0.4850082709638114E+00, 0.1073014545319338E-16, 0.2247354260070927E-17, 0.1015816056431712E+00, 0.1015816059316144E+00, 0.4075436992240071E-09, 0.4778291260603437E-18, 0.1191004794787176E-09, 0.0000000000000000E+00, 0.2322916695987584E-18, 0.2222159192042978E-18, 0.1183377192742860E+01},
			i0:       1,
			n0:       8,
			pp:       1,
			tau:      -0.0000000000000000,
			sigma:    1.2021617695444331,
			i0Out:    1,
			n0Out:    8,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.2021617695444331,
			dminOut:  0.0000000000000000,
			dmin1Out: 0.10158160593161437,
			dmin2Out: 0.48388184445926297,
			dnOut:    0.0000000000000000,
			dnm1Out:  0.10158160593161437,
			dnm2Out:  0.48388184445926297,
		},
		{
			z:        []float64{0.1278815743957415E+01, 0.1278815743957415E+01, 0.2163565864913247E-18, 0.2222159192042978E-18, 0.1263998696887723E+01, 0.1245096256401054E+01, 0.1598154237018549E-01, 0.1890244048666934E-01, 0.1058573353704506E+01, 0.1068679398536745E+01, 0.6330707284251229E-02, 0.5875497537946494E-02, 0.1134721246299565E+01, 0.1140587328635589E+01, 0.2466181655543824E-03, 0.4646249482278370E-03, 0.6034533156147670E+00, 0.6022984199306394E+00, 0.1126426504548419E-02, 0.1401513849681988E-02, 0.4838818444592630E+00, 0.4850082709638114E+00, 0.4717884282068346E-18, 0.2247354260070927E-17, 0.1015816059316144E+00, 0.1015816059316144E+00, 0.0000000000000000E+00, 0.4778291260603437E-18},
			i0:       1,
			n0:       7,
			pp:       0,
			tau:      -0.0000000000000000,
			sigma:    1.2021617695444331,
			i0Out:    1,
			n0Out:    7,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.2021617695444331,
			dminOut:  0.10158160593161437,
			dmin1Out: 0.48298010023310722,
			dmin2Out: 0.60332139679048247,
			dnOut:    0.10158160593161437,
			dnm1Out:  0.48298010023310722,
			dnm2Out:  0.60332139679048247,
		},
		{
			z:        []float64{0.1278815743957415E+01, 0.1278815743957415E+01, 0.2163565864913247E-18, 0.2138497627045302E-18, 0.1263998696887723E+01, 0.1279980239257909E+01, 0.1598154237018549E-01, 0.1321710631563048E-01, 0.1058573353704506E+01, 0.1051686954673127E+01, 0.6330707284251229E-02, 0.6830538334266978E-02, 0.1134721246299565E+01, 0.1128137326130852E+01, 0.2466181655543824E-03, 0.1319188242844840E-03, 0.6034533156147670E+00, 0.6044478232950309E+00, 0.1126426504548419E-02, 0.9017442261557238E-03, 0.4838818444592630E+00, 0.4829801002331072E+00, 0.4717884282068346E-18, 0.9922774494036444E-19, 0.1015816059316144E+00, 0.1015816059316144E+00, 0.0000000000000000E+00, 0.2138497627045302E-18, 0.1202161769544433E+01},
			i0:       1,
			n0:       7,
			pp:       1,
			tau:      0.10158160593161437,
			sigma:    1.2021617695444331,
			i0Out:    1,
			n0Out:    7,
			ppOut:    1,
			tauOut:   0.10158160593161437,
			sigmaOut: 1.2021617695444331,
			dminOut:  0.0000000000000000,
			dmin1Out: 0.38053382595775820,
			dmin2Out: 0.50278793042091730,
			dnOut:    0.0000000000000000,
			dnm1Out:  0.38053382595775820,
			dnm2Out:  0.50278793042091730,
		},
		{
			z:        []float64{0.1177234138025801E+01, 0.1278815743957415E+01, 0.2325140442247288E-18, 0.2138497627045302E-18, 0.1191615739641925E+01, 0.1279980239257909E+01, 0.1166505092896250E-01, 0.1321710631563048E-01, 0.9452708361468171E+00, 0.1051686954673127E+01, 0.8151933771558135E-02, 0.6830538334266978E-02, 0.1018535705251965E+01, 0.1128137326130852E+01, 0.7828694249915420E-04, 0.1319188242844840E-03, 0.5036896746470730E+00, 0.6044478232950309E+00, 0.8646683437346259E-03, 0.9017442261557238E-03, 0.3805338259577582E+00, 0.4829801002331072E+00, 0.2648835135390502E-19, 0.9922774494036444E-19, 0.0000000000000000E+00, 0.1015816059316144E+00, 0.2325140442247288E-18, 0.2138497627045302E-18},
			i0:       1,
			n0:       7,
			pp:       0,
			tau:      -0.0000000000000000,
			sigma:    1.3037433754760475,
			i0Out:    1,
			n0Out:    7,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.3037433754760475,
			dminOut:  0.0000000000000000,
			dmin1Out: 0.37988164443454930,
			dmin2Out: 0.50365062581387521,
			dnOut:    0.0000000000000000,
			dnm1Out:  0.37988164443454930,
			dnm2Out:  0.50365062581387521,
		},
		{
			z:        []float64{0.1177234138025801E+01, 0.1177234138025801E+01, 0.2325140442247288E-18, 0.2353545363971710E-18, 0.1191615739641925E+01, 0.1203280790570888E+01, 0.1166505092896250E-01, 0.9163806595868684E-02, 0.9452708361468171E+00, 0.9442589633225065E+00, 0.8151933771558135E-02, 0.8793176380307672E-02, 0.1018535705251965E+01, 0.1009820815814156E+01, 0.7828694249915420E-04, 0.3904883319791864E-04, 0.5036896746470730E+00, 0.5045152941576099E+00, 0.8646683437346259E-03, 0.6521815232088979E-03, 0.3805338259577582E+00, 0.3798816444345493E+00, 0.2648835135390502E-19, 0.0000000000000000E+00, 0.1303743375476047E+01},
			i0:       1,
			n0:       6,
			pp:       1,
			tau:      -0.0000000000000000,
			sigma:    1.3037433754760475,
			i0Out:    1,
			n0Out:    6,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.3037433754760475,
			dminOut:  0.37939119035907048,
			dmin1Out: 0.50449560273072791,
			dmin2Out: 0.93712213699945368,
			dnOut:    0.37939119035907048,
			dnm1Out:  0.50449560273072791,
			dnm2Out:  1.0004335774214996,
		},
		{
			z:        []float64{0.1177234138025801E+01, 0.1177234138025801E+01, 0.2405618249359890E-18, 0.2353545363971710E-18, 0.1212444597166756E+01, 0.1203280790570888E+01, 0.7136826323052847E-02, 0.9163806595868684E-02, 0.9459153133797613E+00, 0.9442589633225065E+00, 0.9387238392656353E-02, 0.8793176380307672E-02, 0.1000472626254697E+01, 0.1009820815814156E+01, 0.1969142688202247E-04, 0.3904883319791864E-04, 0.5051477842539368E+00, 0.5045152941576099E+00, 0.4904540754787929E-03, 0.6521815232088979E-03, 0.3793911903590705E+00, 0.3798816444345493E+00, 0.2405618249359890E-18, 0.2353545363971710E-18},
			i0:       1,
			n0:       6,
			pp:       0,
			tau:      0.37791671367087804,
			sigma:    1.3037433754760475,
			i0Out:    1,
			n0Out:    6,
			ppOut:    0,
			tauOut:   0.37791671367087804,
			sigmaOut: 1.3037433754760475,
			dminOut:  1.7417124602014944E-005,
			dmin1Out: 0.12721465844156904,
			dmin2Out: 0.55997778901175921,
			dnOut:    1.7417124602014944E-005,
			dnm1Out:  0.12721465844156904,
			dnm2Out:  0.60606091447901145,
		},
		{
			z:        []float64{0.1177234138025801E+01, 0.7993174243549228E+00, 0.2405618249359890E-18, 0.3648961927279404E-18, 0.1212444597166756E+01, 0.8416647098189309E+00, 0.7136826323052847E-02, 0.8020810697124018E-02, 0.9459153133797613E+00, 0.5693650274044155E+00, 0.9387238392656353E-02, 0.1649499810480807E-01, 0.1000472626254697E+01, 0.6060806059058935E+00, 0.1969142688202247E-04, 0.1641214148963635E-04, 0.5051477842539368E+00, 0.1277051125170478E+00, 0.4904540754787929E-03, 0.1457059563590431E-02, 0.3793911903590705E+00, 0.1741712460201494E-04, 0.2405618249359890E-18, 0.3648961927279404E-18, 0.1303743375476047E+01},
			i0:       1,
			n0:       6,
			pp:       1,
			tau:      1.7220607103034587E-005,
			sigma:    1.6816600891469256,
			i0Out:    1,
			n0Out:    6,
			ppOut:    1,
			tauOut:   1.7220607103034587E-005,
			sigmaOut: 1.6816600891469256,
			dminOut:  5.8496704963934779E-012,
			dmin1Out: 0.12768433261716639,
			dmin2Out: 0.56397303984076830,
			dnOut:    5.8496704963934779E-012,
			dnm1Out:  0.12768433261716639,
			dnm2Out:  0.58884056200405466,
		},
		{
			z:        []float64{0.7993002037478197E+00, 0.7993174243549228E+00, 0.3842364192156412E-18, 0.3648961927279404E-18, 0.8496682999089519E+00, 0.8416647098189309E+00, 0.5374766956544110E-02, 0.8020810697124018E-02, 0.5804680379455763E+00, 0.5693650274044155E+00, 0.1722282329473577E-01, 0.1649499810480807E-01, 0.5888569741455443E+00, 0.6060806059058935E+00, 0.3559292778388127E-05, 0.1641214148963635E-04, 0.1291413921807568E+00, 0.1277051125170478E+00, 0.1965116493098606E-06, 0.1457059563590431E-02, 0.5849670496393478E-11, 0.1741712460201494E-04, 0.3842364192156412E-18, 0.3648961927279404E-18},
			i0:       1,
			n0:       6,
			pp:       0,
			tau:      5.8496615949827599E-012,
			sigma:    1.6816773097540285,
			i0Out:    1,
			n0Out:    6,
			ppOut:    0,
			tauOut:   5.8496615949827599E-012,
			sigmaOut: 1.6816773097540285,
			dminOut:  5.2413685591311719E-023,
			dmin1Out: 0.12914058828956335,
			dmin2Out: 0.57178448001987381,
			dnOut:    5.2413685591311719E-023,
			dnm1Out:  0.12914058828956335,
			dnm2Out:  0.57178448001987381,
		},
		{
			z:        []float64{0.7993002037478197E+00, 0.7993002037419701E+00, 0.3842364192156412E-18, 0.4084491703488284E-18, 0.8496682999089519E+00, 0.8550430668596465E+00, 0.5374766956544110E-02, 0.3648799166500929E-02, 0.5804680379455763E+00, 0.5940420620679615E+00, 0.1722282329473577E-01, 0.1707249411982080E-01, 0.5888569741455443E+00, 0.5717880393126522E+00, 0.3559292778388127E-05, 0.8038853438286423E-06, 0.1291413921807568E+00, 0.1291407848012127E+00, 0.1965116493098606E-06, 0.8901358303923820E-17, 0.5849670496393478E-11, 0.5241368559131172E-22, 0.3842364192156412E-18, 0.4084491703488284E-18, 0.1303743375476047E+01},
			i0:       1,
			n0:       6,
			pp:       1,
			tau:      5.2413685591311714E-023,
			sigma:    1.6816773097598783,
			i0Out:    1,
			n0Out:    6,
			ppOut:    1,
			tauOut:   0.0000000000000000,
			sigmaOut: 1.6816773097598783,
			dminOut:  5.2413685591311719E-023,
			dmin1Out: 0.12914059800027977,
			dmin2Out: 0.55574794205624078,
			dnOut:    5.2413685591311719E-023,
			dnm1Out:  0.12914059800027977,
			dnm2Out:  0.55574794205624078,
		},
		{
			z:        []float64{0.7993002037419701E+00, 0.7993002037419701E+00, 0.4369342452764876E-18, 0.4084491703488284E-18, 0.8586918660261474E+00, 0.8550430668596465E+00, 0.2524235138002424E-02, 0.3648799166500929E-02, 0.6085903210497798E+00, 0.5940420620679615E+00, 0.1604009725641142E-01, 0.1707249411982080E-01, 0.5557487459415846E+00, 0.5717880393126522E+00, 0.1868009328861996E-06, 0.8038853438286423E-06, 0.1291405980002798E+00, 0.1291407848012127E+00, 0.3612752323451872E-38, 0.8901358303923820E-17},
			i0:       1,
			n0:       5,
			pp:       0,
			tau:      0.12914051019182277,
			sigma:    1.6816773097598783,
			i0Out:    1,
			n0Out:    5,
			ppOut:    0,
			tauOut:   0.12914051019182277,
			sigmaOut: 1.6816773097598783,
			dminOut:  2.8760335918365243E-008,
			dmin1Out: 0.40854091047624980,
			dmin2Out: 0.47735135942481804,
			dnOut:    2.8760335918365243E-008,
			dnm1Out:  0.40854091047624980,
			dnm2Out:  0.47735135942481804,
		},
		{
			z:        []float64{0.7993002037419701E+00, 0.6701596935501473E+00, 0.4369342452764876E-18, 0.5598544436768911E-18, 0.8586918660261474E+00, 0.7320755909723271E+00, 0.2524235138002424E-02, 0.2098451433139096E-02, 0.6085903210497798E+00, 0.4933914566812295E+00, 0.1604009725641142E-01, 0.1806732527351204E-01, 0.5557487459415846E+00, 0.4085410972771827E+00, 0.1868009328861996E-06, 0.5904812108429529E-07, 0.1291405980002798E+00, 0.2876033591836524E-07, 0.3612752323451872E-38, 0.5598544436768911E-18, 0.1681677309759878E+01},
			i0:       1,
			n0:       5,
			pp:       1,
			tau:      2.8760330654564486E-008,
			sigma:    1.8108178199517009,
			i0Out:    1,
			n0Out:    5,
			ppOut:    1,
			tauOut:   2.8760330654564486E-008,
			sigmaOut: 1.8108178199517009,
			dminOut:  9.5429721930147733E-016,
			dmin1Out: 0.39406941605409296,
			dmin2Out: 0.49198119275848701,
			dnOut:    9.5429721930147733E-016,
			dnm1Out:  0.39406941605409296,
			dnm2Out:  0.49198119275848701,
		},
		{
			z:        []float64{0.6701596647898166E+00, 0.6701596935501473E+00, 0.6115792910959321E-18, 0.5598544436768911E-18, 0.7341740136451356E+00, 0.7320755909723271E+00, 0.1410235162411720E-02, 0.2098451433139096E-02, 0.5100485180319990E+00, 0.4933914566812295E+00, 0.1447165246275905E-01, 0.1806732527351204E-01, 0.3940694751022140E+00, 0.4085410972771827E+00, 0.4309503539425748E-14, 0.5904812108429529E-07, 0.9542972193014773E-15, 0.2876033591836524E-07, 0.6115792910959321E-18, 0.5598544436768911E-18},
			i0:       1,
			n0:       5,
			pp:       0,
			tau:      9.5429721930146451E-016,
			sigma:    1.8108178487120317,
			i0Out:    1,
			n0Out:    5,
			ppOut:    0,
			tauOut:   9.5429721930146451E-016,
			sigmaOut: 1.8108178487120317,
			dminOut:  1.9721522630525295E-030,
			dmin1Out: 0.38317668529976806,
			dmin2Out: 0.50907067170667131,
			dnOut:    1.9721522630525295E-030,
			dnm1Out:  0.38317668529976806,
			dnm2Out:  0.50907067170667131,
		},
		{
			z:        []float64{0.6701596647898166E+00, 0.6701596647898156E+00, 0.6115792910959321E-18, 0.6699979816704878E-18, 0.7341740136451356E+00, 0.7355842488075462E+00, 0.1410235162411720E-02, 0.9778463253267723E-03, 0.5100485180319990E+00, 0.5235423241694304E+00, 0.1447165246275905E-01, 0.1089278980244500E-01, 0.3940694751022140E+00, 0.3831766852997724E+00, 0.4309503539425748E-14, 0.1073277003016631E-28, 0.1810817848712033E+01},
			i0:       1,
			n0:       4,
			pp:       1,
			tau:      0.31895966905528556,
			sigma:    1.8108178487120326,
			i0Out:    1,
			n0Out:    4,
			ppOut:    1,
			tauOut:   0.31895966905528556,
			sigmaOut: 1.8108178487120326,
			dminOut:  4.4735699115826311E-002,
			dmin1Out: 0.20335674284478694,
			dmin2Out: 0.35119999573453003,
			dnOut:    4.4735699115826311E-002,
			dnm1Out:  0.20335674284478694,
			dnm2Out:  0.41662457975226058,
		},
		{
			z:        []float64{0.3511999957345300E+00, 0.6701596647898156E+00, 0.1403302870260262E-17, 0.6699979816704878E-18, 0.4176024260775874E+00, 0.7355842488075462E+00, 0.1225912269357841E-02, 0.9778463253267723E-03, 0.2142495326472319E+00, 0.5235423241694304E+00, 0.1948131712866055E-01, 0.1089278980244500E-01, 0.4473569911582631E-01, 0.3831766852997724E+00, 0.1403302870260262E-17, 0.6699979816704878E-18},
			i0:       1,
			n0:       4,
			pp:       0,
			tau:      3.9897804510299034E-002,
			sigma:    2.1297775177673182,
			i0Out:    1,
			n0Out:    4,
			ppOut:    0,
			tauOut:   3.9897804510299034E-002,
			sigmaOut: 2.1297775177673182,
			dminOut:  3.2556800493375310E-004,
			dmin1Out: 0.17365859019389857,
			dmin2Out: 0.31130219122423097,
			dnOut:    3.2556800493375310E-004,
			dnm1Out:  0.17365859019389857,
			dnm2Out:  0.37770462156728835,
		},
		{
			z:        []float64{0.3511999957345300E+00, 0.3113021912242310E+00, 0.1403302870260262E-17, 0.1882488140663987E-17, 0.4176024260775874E+00, 0.3789305338366462E+00, 0.1225912269357841E-02, 0.6931379430343076E-03, 0.2142495326472319E+00, 0.1931399073225591E+00, 0.1948131712866055E-01, 0.4512326600593524E-02, 0.4473569911582631E-01, 0.3255680049337531E-03, 0.1403302870260262E-17, 0.1882488140663987E-17, 0.1810817848712033E+01},
			i0:       1,
			n0:       4,
			pp:       1,
			tau:      3.1797943619548145E-004,
			sigma:    2.1696753222776173,
			i0Out:    1,
			n0Out:    4,
			ppOut:    1,
			tauOut:   3.1797943619548145E-004,
			sigmaOut: 2.1696753222776173,
			dminOut:  1.3065740500455263E-007,
			dmin1Out: 0.19246898670783216,
			dmin2Out: 0.31098421178803548,
			dnOut:    1.3065740500455263E-007,
			dnm1Out:  0.19246898670783216,
			dnm2Out:  0.37861255440045072,
		},
		{
			z:        []float64{0.3109842117880355E+00, 0.3113021912242310E+00, 0.2293789231233262E-17, 0.1882488140663987E-17, 0.3793056923434850E+00, 0.3789305338366462E+00, 0.3529411785314453E-03, 0.6931379430343076E-03, 0.1969813133084257E+00, 0.1931399073225591E+00, 0.7457911333267062E-05, 0.4512326600593524E-02, 0.1306574050045526E-06, 0.3255680049337531E-03, 0.2293789231233262E-17, 0.1882488140663987E-17},
			i0:       1,
			n0:       4,
			pp:       0,
			tau:      1.3065240973281283E-007,
			sigma:    2.1699933017138124,
			i0Out:    1,
			n0Out:    4,
			ppOut:    0,
			tauOut:   1.3065240973281283E-007,
			sigmaOut: 2.1699933017138124,
			dminOut:  4.4031878668096113E-014,
			dmin1Out: 0.19679806330970273,
			dmin2Out: 0.31098408113562576,
			dnOut:    4.4031878668096113E-014,
			dnm1Out:  0.19679806330970273,
			dnm2Out:  0.37930556169107532,
		},
		{
			z:        []float64{0.3109842117880355E+00, 0.3109840811356258E+00, 0.2293789231233262E-17, 0.2797722987188916E-17, 0.3793056923434850E+00, 0.3796585028696068E+00, 0.3529411785314453E-03, 0.1831193463132450E-03, 0.1969813133084257E+00, 0.1968055212210360E+00, 0.7457911333267062E-05, 0.4951239861123181E-11, 0.1306574050045526E-06, 0.4403187866809611E-13, 0.2293789231233262E-17, 0.2797722987188916E-17, 0.1810817848712033E+01},
			i0:       1,
			n0:       4,
			pp:       1,
			tau:      4.4031878666982759E-014,
			sigma:    2.1699934323662222,
			i0Out:    1,
			n0Out:    4,
			ppOut:    1,
			tauOut:   4.4031878666982759E-014,
			sigmaOut: 2.1699934323662222,
			dminOut:  5.0676424551397798E-027,
			dmin1Out: 0.19671064247077341,
			dmin2Out: 0.31098408113558174,
			dnOut:    5.0676424551397798E-027,
			dnm1Out:  0.19671064247077341,
			dnm2Out:  0.37965850286956282,
		},
		{
			z:        []float64{0.3109840811355817E+00, 0.3109840811356258E+00, 0.3415542419024794E-17, 0.2797722987188916E-17, 0.3798416222158761E+00, 0.3796585028696068E+00, 0.9487875021856221E-04, 0.1831193463132450E-03, 0.1967106424757246E+00, 0.1968055212210360E+00, 0.1108289770587888E-23, 0.4951239861123181E-11, 0.5067642455139780E-26, 0.4403187866809611E-13, 0.3415542419024794E-17, 0.2797722987188916E-17},
			i0:       1,
			n0:       4,
			pp:       0,
			tau:      5.0676424551397798E-027,
			sigma:    2.1699934323662662,
			i0Out:    1,
			n0Out:    4,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 2.1699934323662662,
			dminOut:  5.0676424551397798E-027,
			dmin1Out: 0.19666151937261311,
			dmin2Out: 0.31098408113558174,
			dnOut:    5.0676424551397798E-027,
			dnm1Out:  0.19666151937261311,
			dnm2Out:  0.37984162221587608,
		},
		{
			z:        []float64{0.3109840811355817E+00, 0.3109840811355817E+00, 0.3415542419024794E-17, 0.4171805735046273E-17, 0.3798416222158761E+00, 0.3799365009660947E+00, 0.9487875021856221E-04, 0.4912310311151891E-04, 0.1967106424757246E+00, 0.1966615193726131E+00, 0.1108289770587888E-23, 0.2855879641297252E-49, 0.2169993432366266E+01},
			i0:       1,
			n0:       3,
			pp:       1,
			tau:      9.8330759686306557E-002,
			sigma:    2.1699934323662662,
			i0Out:    1,
			n0Out:    3,
			ppOut:    1,
			tauOut:   9.8330759686306557E-002,
			sigmaOut: 2.1699934323662662,
			dminOut:  9.8296460175394978E-002,
			dmin1Out: 0.21265332144927518,
			dmin2Out: 0.21265332144927518,
			dnOut:    9.8296460175394978E-002,
			dnm1Out:  0.28160574127978810,
			dnm2Out:  0.21265332144927518,
		},
		{
			z:        []float64{0.2126533214492752E+00, 0.3109840811355817E+00, 0.7453545812882342E-17, 0.4171805735046273E-17, 0.2816548643828996E+00, 0.3799365009660947E+00, 0.3429951091160839E-04, 0.4912310311151891E-04, 0.9829646017539498E-01, 0.1966615193726131E+00, 0.3799365009660947E+00, 0.2855879641297252E-49},
			i0:       1,
			n0:       3,
			pp:       0,
			tau:      9.7172513485343004E-002,
			sigma:    2.2683241920525727,
			i0Out:    1,
			n0Out:    3,
			ppOut:    0,
			tauOut:   9.7172513485343004E-002,
			sigmaOut: 2.2683241920525727,
			dminOut:  1.1056745151512559E-003,
			dmin1Out: 0.11548080796393217,
			dmin2Out: 0.11548080796393217,
			dnOut:    1.1056745151512559E-003,
			dnm1Out:  0.18448235089755655,
			dnm2Out:  0.11548080796393217,
		},
		{
			z:        []float64{0.2126533214492752E+00, 0.1154808079639322E+00, 0.7453545812882342E-17, 0.1817901582187390E-16, 0.2816548643828996E+00, 0.1845166504084682E+00, 0.3429951091160839E-04, 0.1827217490071947E-04, 0.9829646017539498E-01, 0.1105674515151256E-02, 0.3799365009660947E+00, 0.2816548643828996E+00, 0.2169993432366266E+01},
			i0:       1,
			n0:       3,
			pp:       1,
			tau:      1.0942861809085330E-003,
			sigma:    2.3654967055379159,
			i0Out:    1,
			n0Out:    3,
			ppOut:    1,
			tauOut:   1.0942861809085330E-003,
			sigmaOut: 2.3654967055379159,
			dminOut:  1.1278200095129901E-005,
			dmin1Out: 0.11438652178302365,
			dmin2Out: 0.11438652178302365,
			dnOut:    1.1278200095129901E-005,
			dnm1Out:  0.18342236422755959,
			dnm2Out:  0.11438652178302365,
		},
		{
			z:        []float64{0.1143865217830237E+00, 0.1154808079639322E+00, 0.2932453102768040E-16, 0.1817901582187390E-16, 0.1834406364024603E+00, 0.1845166504084682E+00, 0.1101341475930520E-06, 0.1827217490071947E-04, 0.1127820009512990E-04, 0.1105674515151256E-02, 0.1845166504084682E+00, 0.2816548643828996E+00},
			i0:       1,
			n0:       3,
			pp:       0,
			tau:      1.1269238360546607E-005,
			sigma:    2.3665909917188244,
			i0Out:    1,
			n0Out:    3,
			ppOut:    0,
			tauOut:   1.1269238360546607E-005,
			sigmaOut: 2.3665909917188244,
			dminOut:  8.9549629620034135E-009,
			dmin1Out: 0.11437525254466312,
			dmin2Out: 0.11437525254466312,
			dnOut:    8.9549629620034135E-009,
			dnm1Out:  0.18342936716409974,
			dnm2Out:  0.11437525254466312,
		},
		{
			z:        []float64{0.1143865217830237E+00, 0.1143752525446631E+00, 0.2932453102768040E-16, 0.4703212027287794E-16, 0.1834406364024603E+00, 0.1834294772982473E+00, 0.1101341475930520E-06, 0.6771621290952006E-11, 0.1127820009512990E-04, 0.8954962962003413E-08, 0.1845166504084682E+00, 0.1834406364024603E+00, 0.2169993432366266E+01},
			i0:       1,
			n0:       3,
			pp:       1,
			tau:      8.9549072084035346E-009,
			sigma:    2.3666022609571851,
			i0Out:    1,
			n0Out:    3,
			ppOut:    1,
			tauOut:   8.9549072084035346E-009,
			sigmaOut: 2.3666022609571851,
			dminOut:  5.5753269291151117E-014,
			dmin1Out: 0.11437524358975594,
			dmin2Out: 0.11437524358975594,
			dnOut:    5.5753269291151117E-014,
			dnm1Out:  0.18342946834334006,
			dnm2Out:  0.11437524358975594,
		},
		{
			z:        []float64{0.1143752435897560E+00, 0.1143752525446631E+00, 0.7542783706608855E-16, 0.4703212027287794E-16, 0.1834294683501117E+00, 0.1834294772982473E+00, 0.3305882004599510E-18, 0.6771621290952006E-11, 0.5575326929115112E-13, 0.8954962962003413E-08, 0.1834294772982473E+00, 0.1834406364024603E+00},
			i0:       1,
			n0:       3,
			pp:       0,
			tau:      5.5753269214454873E-014,
			sigma:    2.3666022699120921,
			i0Out:    1,
			n0Out:    3,
			ppOut:    0,
			tauOut:   5.5753269214454873E-014,
			sigmaOut: 2.3666022699120921,
			dminOut:  7.6696244203643861E-023,
			dmin1Out: 0.11437524358970023,
			dmin2Out: 0.11437524358970023,
			dnOut:    7.6696244203643861E-023,
			dnm1Out:  0.18342946835005580,
			dnm2Out:  0.11437524358970023,
		},
		{
			z:        []float64{0.1648283185136998E+01, 0.1396221235720571E+01, 0.1712714336271993E+00, 0.2520619494164272E+00, 0.1510753432847732E+01, 0.1119977945086946E+01, 0.3792800633372563E+00, 0.5620469213879850E+00, 0.1328441621586708E+01, 0.1019485448443405E+01, 0.1388998759717073E+00, 0.6882362364805590E+00, 0.4466938077796418E+00, 0.2681061628164644E+00, 0.7093206047768255E+00, 0.3174875209348847E+00, 0.6843112870203156E+00, 0.9979892152967577E+00, 0.6490582981441884E+00, 0.3956426765003833E+00, 0.5085572738629487E+00, 0.1122623886995757E+01, 0.9750235054014829E-02, 0.3499168501137979E-01, 0.1364886053450573E+00, 0.1417066070690837E+00, 0.2301225778544498E-01, 0.4532233329988395E-02, 0.1646009972289452E+01, 0.6930161671496210E+00, 0.2362515608142310E+00, 0.9760060629252760E+00, 0.5818602562677768E+00, 0.3984323866837953E+00, 0.1797665269485310E-01, 0.4196794303982125E+00, 0.5600419521166516E+00, 0.2492354636952108E-01, 0.2195137569256029E+00, 0.5530950584419837E+00, 0.4184071984843414E+00, 0.2222708575473020E+00, 0.2727864547293006E+00, 0.4156500978626423E+00, 0.6774373914466536E-01, 0.2745959086613283E+00, 0.1050967099374242E+00, 0.6593428521263771E-01, 0.2040338718098096E+00, 0.1079809097801335E+00, 0.1271971985482246E+00, 0.2011496719671002E+00, 0.4444741998443960E-01, 0.1290210252363728E+00, 0.5776327498150620E+00, 0.4262359329629137E-01, 0.3402556968467140E+00, 0.6023491555328507E+00, 0.1086565805630269E+00, 0.3155392911289253E+00, 0.9382999256694983E+00, 0.1171677238805356E+00, 0.7901625299559836E+00, 0.9297887823519896E+00, 0.6095853796269167E+00, 0.7973955560628040E+00, 0.8018125008387630E+00, 0.6023523535200964E+00, 0.2443177602187348E-01, 0.8114406374558937E+00, 0.2277830378453201E+00, 0.1480363940474286E-01, 0.6969081780841352E+00, 0.3759308106650992E+00, 0.8012406122589412E-01, 0.5487604052643561E+00, 0.9781022865072954E-01, 0.1017549972519246E+00, 0.2961248981181939E-03, 0.7617929262469909E-01, 0.8408398800007548E-04, 0.3802088861182694E-03, 0.0000000000000000E+00, 0.0000000000000000E+00},
			i0:       1,
			n0:       21,
			pp:       0,
			tau:      0.0000000000000000,
			sigma:    0.0000000000000000,
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			tauOut:   0.0000000000000000,
			sigmaOut: 0.0000000000000000,
			dminOut:  7.8766827084626452E-005,
			dmin1Out: 1.2363512593342330E-003,
			dmin2Out: 1.2363512593342330E-003,
			dnOut:    7.8766827084626452E-005,
			dnm1Out:  4.3867054272813191E-003,
			dnm2Out:  3.7622286348031123E-003,
		},
	} {
		// TODO(btracey): Check other outputs
		i0 := test.i0 - 1
		n0 := test.n0 - 1
		z := make([]float64, len(test.z))
		copy(z, test.z)

		i0Out, n0Out, ppOut, tauOut, sigmaOut, dminOut, dmin1Out, dmin2Out, dnOut, dnm1Out, dnm2Out := impl.Dlasq5(i0, n0, z, test.pp, test.tau, test.sigma)

		if i0Out != test.i0Out-1 {
			t.Errorf("Wrong i0. Want %v, got %v", test.n0Out, n0Out)
		}
		if n0Out != test.n0Out-1 {
			t.Errorf("Wrong n0. Want %v, got %v", test.n0Out, n0Out)
		}
		if ppOut != test.ppOut {
			t.Errorf("Wrong pp. Want %v, got %v", test.ppOut, ppOut)
		}
		if !floats.EqualWithinAbsOrRel(tauOut, test.tauOut, dTol, dTol) {
			t.Errorf("Wrong tau. Want %v, got %v", test.tauOut, tauOut)
		}
		if !floats.EqualWithinAbsOrRel(sigmaOut, test.sigmaOut, dTol, dTol) {
			t.Errorf("Wrong tau. Want %v, got %v", test.sigmaOut, sigmaOut)
		}

		if !floats.EqualWithinAbsOrRel(dminOut, test.dminOut, dTol, dTol) {
			t.Errorf("Wrong dmin. Want %v, got %v", test.dminOut, dminOut)
		}
		if !floats.EqualWithinAbsOrRel(dmin1Out, test.dmin1Out, dTol, dTol) {
			t.Errorf("Wrong dmin1. Want %v, got %v", test.dmin1Out, dmin1Out)
		}
		if !floats.EqualWithinAbsOrRel(dmin2Out, test.dmin2Out, dTol, dTol) {
			t.Errorf("Wrong dmin2. Want %v, got %v", test.dmin2Out, dmin2Out)
		}
		if !floats.EqualWithinAbsOrRel(dnOut, test.dnOut, dTol, dTol) {
			t.Errorf("Wrong dn. Want %v, got %v", test.dnOut, dnOut)
		}
		if !floats.EqualWithinAbsOrRel(dnm1Out, test.dnm1Out, dTol, dTol) {
			t.Errorf("Wrong dnm1. Want %v, got %v", test.dnm1Out, dnm1Out)
		}
		if !floats.EqualWithinAbsOrRel(dnm2Out, test.dnm2Out, dTol, dTol) {
			t.Errorf("Wrong dnm2. Want %v, got %v", test.dnm2Out, dnm2Out)
		}
	}
}
