/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.prestosql.sql.planner.assertions;

import io.prestosql.Session;
import io.prestosql.cost.StatsProvider;
import io.prestosql.metadata.Metadata;
import io.prestosql.sql.planner.Symbol;
import io.prestosql.sql.planner.assertions.PlanMatchPattern.Ordering;
import io.prestosql.sql.planner.plan.ExchangeNode;
import io.prestosql.sql.planner.plan.PlanNode;

import java.util.List;
import java.util.Set;

import static com.google.common.base.MoreObjects.toStringHelper;
import static com.google.common.base.Preconditions.checkState;
import static io.prestosql.sql.planner.assertions.MatchResult.NO_MATCH;
import static io.prestosql.sql.planner.assertions.Util.orderingSchemeMatches;
import static java.util.Objects.requireNonNull;

final class ExchangeMatcher
        implements Matcher
{
    private final ExchangeNode.Scope scope;
    private final ExchangeNode.Type type;
    private final List<Ordering> orderBy;
    private final Set<String> partitionedBy;

    public ExchangeMatcher(ExchangeNode.Scope scope, ExchangeNode.Type type, List<Ordering> orderBy, Set<String> partitionedBy)
    {
        this.scope = scope;
        this.type = type;
        this.orderBy = requireNonNull(orderBy, "orderBy is null");
        this.partitionedBy = requireNonNull(partitionedBy, "partitionedBy is null");
    }

    @Override
    public boolean shapeMatches(PlanNode node)
    {
        if (!(node instanceof ExchangeNode)) {
            return false;
        }

        ExchangeNode exchangeNode = (ExchangeNode) node;
        return exchangeNode.getScope() == scope && exchangeNode.getType() == type;
    }

    @Override
    public MatchResult detailMatches(PlanNode node, StatsProvider stats, Session session, Metadata metadata, SymbolAliases symbolAliases)
    {
        checkState(shapeMatches(node), "Plan testing framework error: shapeMatches returned false in detailMatches in %s", this.getClass().getName());
        ExchangeNode exchangeNode = (ExchangeNode) node;

        if (!orderBy.isEmpty()) {
            if (!exchangeNode.getOrderingScheme().isPresent()) {
                return NO_MATCH;
            }

            if (!orderingSchemeMatches(orderBy, exchangeNode.getOrderingScheme().get(), symbolAliases)) {
                return NO_MATCH;
            }
        }

        if (!partitionedBy.isEmpty()) {
            Set<Symbol> partitionedColumns = exchangeNode.getPartitioningScheme().getPartitioning().getColumns();
            if (!partitionedBy.stream()
                    .map(symbolAliases::get)
                    .map(Symbol::from)
                    .allMatch(partitionedColumns::contains)) {
                return NO_MATCH;
            }
        }

        return MatchResult.match();
    }

    @Override
    public String toString()
    {
        return toStringHelper(this)
                .add("scope", scope)
                .add("type", type)
                .add("orderBy", orderBy)
                .add("partitionedBy", partitionedBy)
                .toString();
    }
}
