"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FormFactory = exports.ListFormFactory = exports.InlineFormFactory = exports.validateForm = exports.getFormElement = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _classnames = _interopRequireDefault(require("classnames"));

var _patternflyReact = require("patternfly-react");

var _lodash = require("lodash");

var _ = require(".");

var _constants = require("../../constants");

var _validations = require("../../utils/validations");

var _strings = require("../Wizard/CreateVmWizard/strings");

var _selectors = require("../../k8s/selectors");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const getFormElement = props => {
  const type = props.type,
        id = props.id,
        value = props.value,
        title = props.title,
        onChange = props.onChange,
        onBlur = props.onBlur,
        choices = props.choices,
        defaultValue = props.defaultValue,
        isControlled = props.isControlled,
        disabled = props.disabled,
        className = props.className,
        LoadingComponent = props.LoadingComponent;

  switch (type) {
    case 'textarea':
      return _react.default.createElement(_.TextArea, {
        id: id,
        key: id,
        value: isControlled ? value || '' : undefined,
        defaultValue: isControlled ? undefined : defaultValue,
        onBlur: onBlur,
        className: className,
        onChange: onChange,
        disabled: disabled
      });

    case 'dropdown':
      return _react.default.createElement(_.Dropdown, {
        id: id,
        value: value || defaultValue,
        onChange: onChange,
        choices: choices,
        disabled: disabled
      });

    case 'checkbox':
      return _react.default.createElement(_.Checkbox, {
        id: id,
        key: id,
        title: title,
        checked: isControlled ? value || false : undefined,
        onBlur: onBlur,
        onChange: onChange,
        disabled: disabled
      });

    case 'positive-number':
      return _react.default.createElement(_.Integer, {
        id: id,
        key: id,
        value: isControlled ? value || '' : undefined,
        defaultValue: isControlled ? undefined : defaultValue,
        onBlur: onBlur,
        onChange: onChange,
        positive: true,
        disabled: disabled
      });

    case 'label':
      return _react.default.createElement("div", {
        className: className,
        key: id
      }, value);

    case 'loading':
      return _react.default.createElement(LoadingComponent, null);

    default:
      return _react.default.createElement(_.Text, {
        id: id,
        key: id,
        value: isControlled ? value || '' : undefined,
        defaultValue: isControlled ? undefined : defaultValue,
        onBlur: onBlur,
        onChange: onChange,
        disabled: disabled
      });
  }
};

exports.getFormElement = getFormElement;

const validateForm = (formFields, formValues) => {
  let formValid = true;
  const visibleFieldKeys = Object.keys(formFields).filter(key => formFields[key] && (formFields[key].isVisible ? formFields[key].isVisible(formValues) : true)); // check if all required fields are defined

  const requiredKeys = visibleFieldKeys.filter(key => formFields[key].required);
  formValid = requiredKeys.every(key => (0, _selectors.settingsValue)(formValues, key));

  if (formValid) {
    // check if all fields are valid
    formValid = visibleFieldKeys.every(key => (0, _lodash.get)(formValues[key], 'validation.type') !== _constants.VALIDATION_ERROR_TYPE);
  }

  return formValid;
};

exports.validateForm = validateForm;

const _onChange = (formFields, formValues, value, key, onFormChange) => {
  let validation;
  const changedField = formFields[key];

  if (changedField.required && String(value).trim().length === 0) {
    validation = (0, _validations.getValidationObject)(_strings.ERROR_IS_REQUIRED);
  } else if (changedField.validate) {
    validation = changedField.validate(value);
  }

  if (validation) {
    validation.message = `${changedField.title} ${validation.message}`;
  }

  const newFormValues = _objectSpread({}, formValues, {
    [key]: {
      value,
      validation
    }
  });

  const formValid = validateForm(formFields, newFormValues);
  onFormChange({
    value,
    validation
  }, key, formValid);
};

const getFormGroups = ({
  fields,
  fieldsValues,
  onFormChange,
  textPosition,
  labelSize,
  controlSize,
  horizontal
}) => Object.keys(fields).filter(key => fields[key] && (!fields[key].isVisible || fields[key].isVisible(fieldsValues))).map(key => {
  const field = fields[key];
  const values = fieldsValues[key];
  const validation = (0, _lodash.get)(values, 'validation');
  const value = (0, _lodash.get)(values, 'value');
  const hasValidationMessage = !!(0, _lodash.get)(validation, 'message');
  const child = getFormElement(_objectSpread({}, field, {
    value,
    isControlled: true,
    onChange: newValue => _onChange(fields, fieldsValues, newValue, key, onFormChange)
  }));

  const label = horizontal && field.title && _react.default.createElement(_patternflyReact.Col, {
    sm: labelSize,
    className: textPosition
  }, field.type !== 'checkbox' && _react.default.createElement(_react.default.Fragment, null, _react.default.createElement(_patternflyReact.ControlLabel, {
    className: field.required ? 'required-pf' : null
  }, field.title), field.help && _react.default.createElement(_patternflyReact.FieldLevelHelp, {
    className: "kubevirt-form-group__field-help",
    placement: "right",
    content: field.help
  })));

  return _react.default.createElement(_patternflyReact.FormGroup, {
    key: key,
    validationState: validation && validation.type !== _constants.VALIDATION_INFO_TYPE ? validation.type : null,
    className: (0, _classnames.default)('kubevirt-form-group', {
      'kubevirt-form-group--no-bottom': horizontal && field.noBottom,
      'kubevirt-form-group--help': !horizontal && hasValidationMessage,
      'kubevirt-form-group--no-help': !horizontal && !hasValidationMessage
    })
  }, label, _react.default.createElement(_patternflyReact.Col, {
    sm: controlSize
  }, child, _react.default.createElement(_patternflyReact.HelpBlock, null, (0, _lodash.get)(validation, 'message'))));
});

const InlineFormFactory = ({
  fields,
  fieldsValues,
  onFormChange
}) => getFormGroups({
  fields,
  fieldsValues,
  onFormChange,
  horizontal: true
});

exports.InlineFormFactory = InlineFormFactory;

const ListFormFactory = ({
  fields,
  fieldsValues,
  onFormChange,
  actions,
  columnSizes
}) => {
  const formGroups = getFormGroups({
    fields,
    fieldsValues,
    onFormChange
  });
  const form = formGroups.map((formGroup, index) => _react.default.createElement(_patternflyReact.Col, _extends({
    key: `col-${index}`
  }, columnSizes, {
    className: (0, _classnames.default)('kubevirt-list-form-factory__column', {
      'kubevirt-list-form-factory__column--last': index === formGroups.length - 1
    })
  }), formGroup));
  return _react.default.createElement(_react.default.Fragment, null, form, _react.default.createElement("div", {
    className: "kubevirt-list-form-factory__actions"
  }, actions));
};

exports.ListFormFactory = ListFormFactory;
ListFormFactory.propTypes = {
  fields: _propTypes.default.object.isRequired,
  fieldsValues: _propTypes.default.object.isRequired,
  onFormChange: _propTypes.default.func.isRequired,
  actions: _propTypes.default.object.isRequired,
  columnSizes: _propTypes.default.object.isRequired
};

const FormFactory = ({
  fields,
  fieldsValues,
  onFormChange,
  textPosition,
  labelSize,
  controlSize,
  formClassName
}) => {
  const formGroups = getFormGroups({
    fields,
    fieldsValues,
    onFormChange,
    textPosition,
    labelSize,
    controlSize,
    horizontal: true
  });
  return _react.default.createElement(_patternflyReact.Form, {
    horizontal: true,
    className: formClassName
  }, formGroups);
};

exports.FormFactory = FormFactory;
FormFactory.defaultProps = {
  textPosition: 'text-right',
  labelSize: 3,
  controlSize: 5,
  formClassName: undefined
};
FormFactory.propTypes = {
  fields: _propTypes.default.object.isRequired,
  fieldsValues: _propTypes.default.object.isRequired,
  onFormChange: _propTypes.default.func.isRequired,
  textPosition: _propTypes.default.string,
  labelSize: _propTypes.default.number,
  controlSize: _propTypes.default.number,
  formClassName: _propTypes.default.string
};