"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SerialConsoleConnector = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _reactConsole = require("@patternfly/react-console");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const _AccessConsoles$const = _reactConsole.AccessConsoles.constants,
      CONNECTED = _AccessConsoles$const.CONNECTED,
      DISCONNECTED = _AccessConsoles$const.DISCONNECTED,
      LOADING = _AccessConsoles$const.LOADING;
const _console = console,
      debug = _console.debug,
      info = _console.info,
      error = _console.error;
/**
 * Kubevirt serial console is accessed via WebSocket proxy in k8s API.
 * Protocol used is "plain.kubevirt.io", means binary and single channel - forwarding of unix socket only (see subresource.go in kubevirt).
 */

class SerialConsoleConnector extends _react.default.Component {
  constructor(...args) {
    super(...args);

    _defineProperty(this, "state", {
      status: LOADING,
      passKeys: false
    });

    _defineProperty(this, "onBackendDisconnected", event => {
      debug('Backend has disconnected');

      if (this.childSerialconsole) {
        this.childSerialconsole.onConnectionClosed('Reason for disconnect provided by backend.');
      }

      if (event) {
        info('Serial console connection closed, reason: ', event.reason);
      }

      this.ws && this.ws.destroy && this.ws.destroy();
      this.setState({
        passKeys: false,
        status: DISCONNECTED // will close the terminal window

      });
    });

    _defineProperty(this, "onConnect", () => {
      debug('SerialConsoleConnector.onConnect(), ', this.state);
      const _this$props = this.props,
            vmi = _this$props.vmi,
            host = _this$props.host,
            path = _this$props.path,
            WSFactory = _this$props.WSFactory;

      if (this.ws) {
        this.ws.destroy();
        this.setState({
          status: LOADING
        });
      }

      const options = {
        host,
        path,
        reconnect: false,
        jsonParse: false,
        subprotocols: ['plain.kubevirt.io']
      };
      this.ws = new WSFactory(`${vmi.metadata.name}-serial`, options).onmessage(this.onDataFromBackend).onopen(this.setConnected).onclose(this.onBackendDisconnected).onerror(event => {
        error('WS error received: ', event);
      });
    });

    _defineProperty(this, "onData", data => {
      debug('UI terminal component produced data, i.e. a key was pressed, pass it to backend. [', data, ']');
      this.ws && this.ws.send(new Blob([data])); // data are resent back from backend so _will_ pass through onDataFromBackend
    });

    _defineProperty(this, "onDataFromBackend", data => {
      // plain.kubevirt.io is binary and single-channel protocol
      debug('Backend sent data, pass them to the UI component. [', data, ']');

      if (this.childSerialconsole) {
        const reader = new FileReader();
        reader.addEventListener('loadend', e => {
          // Blob to text transformation ...
          const target = e.target || e.srcElement;
          const text = target.result;
          this.childSerialconsole.onDataReceived(text);
        });
        reader.readAsText(data);
      }
    });

    _defineProperty(this, "onDisconnect", () => {
      // disconnection initiated by UI component
      this.onBackendDisconnected();
    });

    _defineProperty(this, "onResize", (rows, cols) => {
      debug('UI has been resized, pass this info to backend. [', rows, ', ', cols, ']. Ignoring since recently not supported by backend.', this);
    });

    _defineProperty(this, "setConnected", () => {
      this.setState({
        status: CONNECTED,
        passKeys: true
      });
    });
  }

  /*
        autoFit={this.props.autoFit}
        cols={this.props.cols}
        rows={this.props.rows}
  */
  render() {
    return _react.default.createElement(_reactConsole.SerialConsole, {
      onConnect: this.onConnect,
      onDisconnect: this.onDisconnect,
      onResize: this.onResize,
      onData: this.onData,
      id: "serial-console-todo",
      status: this.state.status,
      ref: c => {
        this.childSerialconsole = c;
      }
    });
  }

}

exports.SerialConsoleConnector = SerialConsoleConnector;
SerialConsoleConnector.propTypes = {
  vmi: _propTypes.default.object.isRequired,
  host: _propTypes.default.string.isRequired,
  path: _propTypes.default.string.isRequired,
  WSFactory: _propTypes.default.func.isRequired // reference to OKD utility

};