"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isVmStarting = exports.isVmiRunning = exports.getVmStatus = exports.getVmStatusDetail = exports.isBeingMigrated = void 0;

var _lodash = require("lodash");

var _constants = require("../../constants");

const NOT_HANDLED = null;
const failingContainerStatus = ['ImagePullBackOff', 'ErrImagePull', 'CrashLoopBackOff'];

const isMigrationStatus = (migration, status) => {
  const phase = (0, _lodash.get)(migration, 'status.phase');
  return phase && phase.toLowerCase() === status.toLowerCase();
};

const getConditionOfType = (pod, type) => (0, _lodash.get)(pod, 'status.conditions', []).find(condition => condition.type === type);

const getNotRedyConditionMessage = pod => {
  const notReadyCondition = (0, _lodash.get)(pod, 'status.conditions', []).find(condition => condition.status !== 'True');

  if (notReadyCondition) {
    // at least one pod condition not met. This can be just tentative, let the user analyze progress via Pod events
    return notReadyCondition.message || `Step: ${notReadyCondition.type}`;
  }

  return undefined;
};

const getFailingContainerStatus = pod => (0, _lodash.get)(pod, 'status.containerStatuses', []).find(container => !container.ready && (0, _lodash.includes)(failingContainerStatus, (0, _lodash.get)(container, 'state.waiting.reason')));

const getContainerStatusReason = containerStatus => {
  const status = Object.getOwnPropertyNames(containerStatus.state).find(pn => !!containerStatus.state[pn].reason);
  return status ? containerStatus.state[status].message : undefined;
};

const isSchedulable = pod => {
  const podScheduledCond = getConditionOfType(pod, 'PodScheduled');
  return !(podScheduledCond && podScheduledCond.status !== 'True' && podScheduledCond.reason === 'Unschedulable');
};

const isRunning = vm => {
  if (!(0, _lodash.get)(vm, 'spec.running', false)) {
    return {
      status: _constants.VM_STATUS_OFF
    };
  } // spec.running === true


  return NOT_HANDLED;
};

const isReady = vm => {
  if ((0, _lodash.get)(vm, 'status.ready', false)) {
    // we are all set
    return {
      status: _constants.VM_STATUS_RUNNING
    };
  }

  return NOT_HANDLED;
};

const isCreated = (vm, launcherPod = null) => {
  if ((0, _lodash.get)(vm, 'status.created', false)) {
    // created but not yet ready
    let message;

    if (launcherPod) {
      // pod created, so check for it's potential error
      if (!isSchedulable(launcherPod)) {
        return {
          status: _constants.VM_STATUS_POD_ERROR,
          message: 'Pod scheduling failed.'
        };
      }

      const failingContainer = getFailingContainerStatus(launcherPod);

      if (failingContainer) {
        return {
          status: _constants.VM_STATUS_POD_ERROR,
          message: getContainerStatusReason(failingContainer)
        };
      }

      message = getNotRedyConditionMessage(launcherPod);
    }

    return {
      status: _constants.VM_STATUS_STARTING,
      message
    };
  }

  return NOT_HANDLED;
};

const isVmError = vm => {
  // is an issue with the VM definition?
  const condition = (0, _lodash.get)(vm, 'status.conditions[0]');

  if (condition) {
    // Do we need to analyze additional conditions in the array? Probably not.
    if (condition.type === 'Failure') {
      return {
        status: _constants.VM_STATUS_ERROR,
        message: condition.message
      };
    }
  }

  return NOT_HANDLED;
};

const isBeingImported = (vm, importerPods) => {
  if (importerPods && importerPods.length > 0 && !(0, _lodash.get)(vm, 'status.created', false)) {
    const importerPodsStatuses = importerPods.map(pod => {
      if (!isSchedulable(pod)) {
        return {
          status: _constants.VM_STATUS_IMPORT_ERROR,
          message: 'Importer pod scheduling failed.',
          pod
        };
      }

      const failingContainer = getFailingContainerStatus(pod);

      if (failingContainer) {
        return {
          status: _constants.VM_STATUS_IMPORT_ERROR,
          message: getContainerStatusReason(failingContainer),
          pod
        };
      }

      return {
        status: _constants.VM_STATUS_IMPORTING,
        message: getNotRedyConditionMessage(pod),
        pod
      };
    });
    const importErrorStatus = importerPodsStatuses.find(status => status.status === _constants.VM_STATUS_IMPORT_ERROR);
    const message = importerPodsStatuses.map(podStatus => `${podStatus.pod.metadata.name}: ${podStatus.message}`).join('\n\n');
    return {
      status: importErrorStatus ? importErrorStatus.status : _constants.VM_STATUS_IMPORTING,
      message,
      pod: importErrorStatus ? importErrorStatus.pod : importerPods[0],
      importerPodsStatuses
    };
  }

  return NOT_HANDLED;
};

const isBeingMigrated = (vm, migration) => {
  if (migration) {
    if (!isMigrationStatus(migration, 'succeeded') && !isMigrationStatus(migration, 'failed')) {
      return {
        status: _constants.VM_STATUS_MIGRATING,
        message: (0, _lodash.get)(migration, 'status.phase')
      };
    }
  }

  return NOT_HANDLED;
};

exports.isBeingMigrated = isBeingMigrated;

const isWaitingForVmi = vm => {
  // assumption: spec.running === true
  if (!(0, _lodash.get)(vm, 'status.created', false)) {
    return {
      status: _constants.VM_STATUS_VMI_WAITING
    };
  }

  return NOT_HANDLED;
};

const getVmStatusDetail = (vm, launcherPod, importerPods, migration) => isBeingMigrated(vm, migration) || // must be precceding isRunning() since vm.status.ready is true for a migrating VM
isRunning(vm) || isReady(vm) || isVmError(vm) || isCreated(vm, launcherPod) || isBeingImported(vm, importerPods) || isWaitingForVmi(vm) || {
  status: _constants.VM_STATUS_UNKNOWN
};

exports.getVmStatusDetail = getVmStatusDetail;

const getVmStatus = (vm, launcherPod, importerPods, migration) => {
  const vmStatus = getVmStatusDetail(vm, launcherPod, importerPods, migration).status;
  return vmStatus === _constants.VM_STATUS_OFF || vmStatus === _constants.VM_STATUS_RUNNING ? vmStatus : _constants.VM_STATUS_OTHER;
};

exports.getVmStatus = getVmStatus;

const isVmiRunning = vmi => (0, _lodash.get)(vmi, 'status.phase') === 'Running';

exports.isVmiRunning = isVmiRunning;

const isVmStarting = (vm, vmi) => (0, _lodash.get)(vm, 'spec.running') && !isVmiRunning(vmi);

exports.isVmStarting = isVmStarting;