"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateContainer = exports.validateURL = exports.validateDNS1123SubdomainValue = exports.getValidationObject = exports.isPositiveNumber = void 0;

var _lodash = require("lodash");

var _strings = require("./strings");

var _constants = require("../constants");

/* eslint-disable no-new */
const isPositiveNumber = value => value && value.match(/^[1-9]\d*$/);

exports.isPositiveNumber = isPositiveNumber;
const alphanumberincRegex = '[a-zA-Z0-9]';

const getValidationObject = (message, type = _constants.VALIDATION_ERROR_TYPE) => ({
  message,
  type
}); // DNS-1123 subdomain


exports.getValidationObject = getValidationObject;

const validateDNS1123SubdomainValue = value => {
  if (!value) {
    return getValidationObject(_strings.EMPTY_ERROR);
  }

  if (value.toLowerCase() !== value) {
    return getValidationObject(_strings.DNS1123_UPPERCASE_ERROR);
  }

  if (value.length > 253) {
    return getValidationObject(_strings.DNS1123_TOO_LONG_ERROR);
  }

  if (!value.charAt(0).match(alphanumberincRegex)) {
    return getValidationObject(_strings.DNS1123_START_ERROR);
  }

  if (!value.charAt(value.length - 1).match(alphanumberincRegex)) {
    return getValidationObject(_strings.DNS1123_END_ERROR);
  }

  for (let i = 1; i < value.length - 1; i++) {
    if (!value.charAt(i).match('[-a-zA-Z0-9.]')) {
      return getValidationObject(`${_strings.DNS1123_CONTAINS_ERROR} ${value.charAt(i)}`);
    }
  }

  return null;
};

exports.validateDNS1123SubdomainValue = validateDNS1123SubdomainValue;

const validateURL = value => {
  if (!value) {
    return getValidationObject(_strings.EMPTY_ERROR);
  }

  if ((0, _lodash.trimStart)(value).length !== value.length) {
    return getValidationObject(_strings.START_WHITESPACE_ERROR);
  }

  if ((0, _lodash.trimEnd)(value).length !== value.length) {
    return getValidationObject(_strings.END_WHITESPACE_ERROR);
  }

  try {
    new URL(value);
    return null;
  } catch (_unused) {
    return getValidationObject(_strings.URL_INVALID_ERROR);
  }
};

exports.validateURL = validateURL;

const validateContainer = value => {
  if (!value) {
    return getValidationObject(_strings.EMPTY_ERROR);
  }

  if ((0, _lodash.trimStart)(value).length !== value.length) {
    return getValidationObject(_strings.START_WHITESPACE_ERROR);
  }

  if ((0, _lodash.trimEnd)(value).length !== value.length) {
    return getValidationObject(_strings.END_WHITESPACE_ERROR);
  }

  return null;
};

exports.validateContainer = validateContainer;