"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createVm = exports.createVmTemplate = void 0;

var _lodash = require("lodash");

var _templates = require("../utils/templates");

var _selectors = require("../utils/selectors");

var _models = require("../models");

var _constants = require("../constants");

var _constants2 = require("../components/Wizard/CreateVmWizard/constants");

var _selectors2 = require("./selectors");

var _cloudInit = require("./request/cloudInit");

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const IS_TEMPLATE = 'isTemplate';
const FALLBACK_DISK = {
  disk: {
    bus: 'virtio'
  }
};

const createVmTemplate = (k8sCreate, templates, basicSettings, networks, storage) => {
  const getSetting = param => {
    switch (param) {
      case _constants2.NAME_KEY:
        return `\${${_constants.TEMPLATE_PARAM_VM_NAME}}`;

      case _constants2.DESCRIPTION_KEY:
        return undefined;

      case IS_TEMPLATE:
        return true;

      default:
        return (0, _selectors2.settingsValue)(basicSettings, param);
    }
  };

  const template = getModifiedVmTemplate(templates, basicSettings, getSetting, networks, storage);
  const vmTemplate = createTemplateObject(_selectors2.settingsValue.bind(undefined, basicSettings), (0, _selectors2.selectVm)(template.objects), template);
  return k8sCreate(_models.TemplateModel, vmTemplate);
};

exports.createVmTemplate = createVmTemplate;

const createVm = (k8sCreate, templates, basicSettings, networks, storage, persistentVolumeClaims) => {
  const getSetting = _selectors2.settingsValue.bind(undefined, basicSettings);

  const template = getModifiedVmTemplate(templates, basicSettings, getSetting, networks, storage); // ProcessedTemplates endpoit will reject the request if user cannot post to the namespace
  // common-templates are stored in openshift namespace, default user can read but cannot post

  const postTemplate = (0, _lodash.cloneDeep)(template);
  postTemplate.metadata.namespace = (0, _selectors2.settingsValue)(basicSettings, _constants2.NAMESPACE_KEY);
  return k8sCreate(_models.ProcessedTemplatesModel, postTemplate).then(({
    objects
  }) => {
    const vm = (0, _selectors2.selectVm)(objects);
    addMetadata(vm, template, getSetting);
    const namespace = getSetting(_constants2.NAMESPACE_KEY);

    if (namespace) {
      vm.metadata.namespace = namespace;
    }

    const promises = getPvcClones(storage, getSetting, persistentVolumeClaims).map(pvcClone => k8sCreate(_models.PersistentVolumeClaimModel, pvcClone));
    const vmCreatePromise = k8sCreate(_models.VirtualMachineModel, vm);
    promises.push(vmCreatePromise);
    return Promise.all(promises);
  });
};

exports.createVm = createVm;

const getPvcClones = (storage, getSetting, persistentVolumeClaims) => storage.filter(s => s.storageType === _constants2.STORAGE_TYPE_PVC).map(pvcStorage => {
  const pvc = persistentVolumeClaims.find(p => (0, _selectors.getName)(p) === pvcStorage.name && (0, _selectors.getNamespace)(p) === getSetting(_constants2.NAMESPACE_KEY));
  const pvcClone = {
    kind: _models.PersistentVolumeClaimModel.kind,
    apiVersion: _models.PersistentVolumeClaimModel.apiVersion,
    metadata: {
      name: `${(0, _selectors.getName)(pvc)}-${getSetting(_constants2.NAME_KEY)}`,
      namespace: getSetting(_constants2.NAMESPACE_KEY),
      annotations: {
        [_constants.ANNOTATION_CLONE_REQUEST]: `${(0, _selectors.getNamespace)(pvc)}/${(0, _selectors.getName)(pvc)}`
      },
      labels: {
        app: _constants.LABEL_CLONE_APP
      }
    },
    spec: _objectSpread({}, pvc.spec)
  }; // delete bound pv name

  delete pvcClone.spec.volumeName;
  return pvcClone;
});

const getModifiedVmTemplate = (templates, basicSettings, getSetting, networks, storage) => {
  const template = resolveTemplate(templates, basicSettings, getSetting);
  const vm = (0, _selectors2.selectVm)(template.objects);
  modifyVmObject(vm, template, getSetting, networks, storage);
  return template;
};

const createTemplateObject = (getSetting, vm, template) => {
  const vmTemplate = {
    apiVersion: `${_models.TemplateModel.apiGroup}/${_models.TemplateModel.apiVersion}`,
    kind: _models.TemplateModel.kind,
    metadata: {
      name: getSetting(_constants2.NAME_KEY),
      namespace: getSetting(_constants2.NAMESPACE_KEY)
    },
    objects: [vm],
    parameters: [{
      name: _constants.TEMPLATE_PARAM_VM_NAME,
      description: _constants.TEMPLATE_PARAM_VM_NAME_DESC
    }]
  };
  const description = getSetting(_constants2.DESCRIPTION_KEY);

  if (description) {
    addAnnotation(vmTemplate, 'description', description);
  }

  addMetadata(vmTemplate, template, getSetting);
  addLabel(vmTemplate, [_constants.TEMPLATE_TYPE_LABEL], _constants.TEMPLATE_TYPE_VM);
  return vmTemplate;
};

const resolveTemplate = (templates, basicSettings, getSetting) => {
  let chosenTemplate;

  if (getSetting(_constants2.USER_TEMPLATE_KEY)) {
    chosenTemplate = templates.find(template => template.metadata.name === getSetting(_constants2.USER_TEMPLATE_KEY));

    if (!chosenTemplate) {
      return null;
    }

    chosenTemplate = (0, _lodash.cloneDeep)(chosenTemplate);
  } else {
    const baseTemplates = (0, _templates.getTemplatesWithLabels)((0, _templates.getTemplate)(templates, _constants.TEMPLATE_TYPE_BASE), [(0, _selectors2.getOsLabel)(basicSettings), (0, _selectors2.getWorkloadLabel)(basicSettings), (0, _selectors2.getFlavorLabel)(basicSettings)]);

    if (baseTemplates.length === 0) {
      return null;
    }

    chosenTemplate = (0, _lodash.cloneDeep)(baseTemplates[0]);
  }

  setParameterValue(chosenTemplate, _constants.TEMPLATE_PARAM_VM_NAME, getSetting(_constants2.NAME_KEY)); // no more required parameters

  chosenTemplate.parameters.forEach(param => {
    if (param.name !== _constants.TEMPLATE_PARAM_VM_NAME && param.required) {
      delete param.required;
    }
  }); // make sure api version is correct

  chosenTemplate.apiVersion = _constants.TEMPLATE_API_VERSION;
  return chosenTemplate;
};

const addMetadata = (vm, template, getSetting) => {
  const flavor = getSetting(_constants2.FLAVOR_KEY);
  addLabel(vm, `${_constants.TEMPLATE_FLAVOR_LABEL}/${flavor}`, 'true');
  const os = getSetting(_constants2.OPERATING_SYSTEM_KEY);
  addLabel(vm, `${_constants.TEMPLATE_OS_LABEL}/${os.id}`, 'true');
  const workload = getSetting(_constants2.WORKLOAD_PROFILE_KEY);
  addLabel(vm, `${_constants.TEMPLATE_WORKLOAD_LABEL}/${workload}`, 'true');
  addLabel(vm, _constants.ANNOTATION_USED_TEMPLATE, `${template.metadata.namespace}_${template.metadata.name}`);
  addTemplateLabel(vm, _constants.TEMPLATE_VM_NAME_LABEL, vm.metadata.name); // for pairing service-vm (like for RDP)

  addAnnotation(vm, `${_constants.TEMPLATE_OS_NAME_ANNOTATION}/${os.id}`, os.name);
};

const modifyVmObject = (vm, template, getSetting, networks, storages) => {
  setFlavor(vm, getSetting);
  addNetworks(vm, template, getSetting, networks);
  vm.spec.running = getSetting(_constants2.START_VM_KEY, false);
  const description = getSetting(_constants2.DESCRIPTION_KEY);

  if (description) {
    addAnnotation(vm, 'description', description);
  }

  addStorages(vm, template, storages, getSetting);
};

const setFlavor = (vm, getSetting) => {
  if (getSetting(_constants2.FLAVOR_KEY) === _constants.CUSTOM_FLAVOR) {
    vm.spec.template.spec.domain.cpu.cores = parseInt(getSetting(_constants2.CPU_KEY), 10);
    vm.spec.template.spec.domain.resources.requests.memory = `${getSetting(_constants2.MEMORY_KEY)}G`;
  }
};

const setParameterValue = (template, paramName, paramValue) => {
  const parameter = template.parameters.find(param => param.name === paramName);
  parameter.value = paramValue;
};

const addNetworks = (vm, template, getSetting, networks) => {
  const defaultInterface = getDefaultInterface(vm, template) || {
    bridge: {}
  };
  removeInterfacesAndNetworks(vm);

  if (!networks.find(network => network.network === _constants.POD_NETWORK)) {
    getDevices(vm).autoattachPodInterface = false;
  }

  networks.forEach(network => {
    addInterface(vm, defaultInterface, network);
    addNetwork(vm, network);
  });

  if (getSetting(_constants2.PROVISION_SOURCE_TYPE_KEY) === _constants.PROVISION_SOURCE_PXE) {
    addAnnotation(vm, _constants.ANNOTATION_PXE_INTERFACE, networks.find(network => network.isBootable).name);
    const startOnCreation = getSetting(_constants2.START_VM_KEY, false);
    addAnnotation(vm, _constants.ANNOTATION_FIRST_BOOT, `${!startOnCreation}`);
  }
};

const addCloudInit = (vm, defaultDisk, getSetting) => {
  if (getSetting(_constants2.CLOUD_INIT_KEY)) {
    const existingCloudInitVolume = (0, _selectors.getCloudInitVolume)(vm);
    const cloudInit = new _cloudInit.CloudInit({
      volume: existingCloudInitVolume
    });

    if (getSetting(_constants2.USE_CLOUD_INIT_CUSTOM_SCRIPT_KEY)) {
      cloudInit.setUserData(getSetting(_constants2.CLOUD_INIT_CUSTOM_SCRIPT_KEY));
    } else {
      cloudInit.setPredefinedUserData({
        hostname: getSetting(_constants2.HOST_NAME_KEY),
        sshAuthorizedKeys: getSetting(_constants2.AUTHKEYS_KEY)
      });
    }

    const _cloudInit$build = cloudInit.build(),
          volume = _cloudInit$build.volume,
          disk = _cloudInit$build.disk;

    if (volume !== existingCloudInitVolume) {
      addDisk(vm, defaultDisk, disk, getSetting);
      addVolume(vm, volume);
    }
  }
};

const addStorages = (vm, template, storages, getSetting) => {
  const defaultDisk = getDefaultDisk(vm, template);
  removeDisksAndVolumes(vm);

  if (storages) {
    storages.forEach(storage => {
      switch (storage.storageType) {
        case _constants2.STORAGE_TYPE_PVC:
          addPvcVolume(vm, storage, getSetting);
          break;

        case _constants2.STORAGE_TYPE_DATAVOLUME:
          addDataVolumeTemplate(vm, storage, getSetting);
          addDataVolume(vm, storage, getSetting);
          break;

        case _constants2.STORAGE_TYPE_CONTAINER:
          addContainerVolume(vm, storage, getSetting);
          break;

        default:
          if (storage.templateStorage) {
            addVolume(vm, storage.templateStorage.volume);
          }

      }

      addDisk(vm, defaultDisk, storage, getSetting);
    });
  }

  addCloudInit(vm, defaultDisk, getSetting);
};

const getDefaultDisk = (vm, template) => {
  const defaultDiskName = (0, _selectors2.getTemplateAnnotations)(template, _constants.ANNOTATION_DEFAULT_DISK);
  return getDevice(vm, 'disks', defaultDiskName) || FALLBACK_DISK;
};

const getDefaultInterface = (vm, template) => {
  const defaultInterfaceName = (0, _selectors2.getTemplateAnnotations)(template, _constants.ANNOTATION_DEFAULT_NETWORK);
  return getDevice(vm, 'interfaces', defaultInterfaceName);
};

const getDevices = vm => {
  const domain = getDomain(vm);

  if (!domain.devices) {
    domain.devices = {};
  }

  return domain.devices;
};

const getDisks = vm => {
  const devices = getDevices(vm);

  if (!devices.disks) {
    devices.disks = [];
  }

  return devices.disks;
};

const getDomain = vm => {
  const spec = getSpec(vm);

  if (!spec.domain) {
    spec.domain = {};
  }

  return spec.domain;
};

const getInterfaces = vm => {
  const devices = getDevices(vm);

  if (!devices.interfaces) {
    devices.interfaces = [];
  }

  return devices.interfaces;
};

const getVolumes = vm => {
  const spec = getSpec(vm);

  if (!spec.volumes) {
    spec.volumes = [];
  }

  return spec.volumes;
};

const getVmSpec = vm => {
  if (!vm.spec) {
    vm.spec = {};
  }

  return vm.spec;
};

const getSpec = vm => {
  if (!vm.spec.template.spec) {
    vm.spec.template.spec = {};
  }

  return vm.spec.template.spec;
};

const getDevice = (vm, deviceType, deviceName) => (0, _lodash.get)(getDevices(vm), deviceType, []).find(device => device.name === deviceName);

const getDataVolumeTemplates = vm => {
  const spec = getVmSpec(vm);

  if (!spec.dataVolumeTemplates) {
    spec.dataVolumeTemplates = [];
  }

  return spec.dataVolumeTemplates;
};

const getNetworks = vm => {
  const spec = getSpec(vm);

  if (!spec.networks) {
    spec.networks = [];
  }

  return spec.networks;
};

const getAnnotations = vm => {
  if (!vm.metadata.annotations) {
    vm.metadata.annotations = {};
  }

  return vm.metadata.annotations;
};

const getLabels = vm => {
  if (!vm.metadata.labels) {
    vm.metadata.labels = {};
  }

  return vm.metadata.labels;
};

const getTemplateLabels = vm => {
  if (!vm.spec) {
    vm.spec = {};
  }

  if (!vm.spec.template) {
    vm.spec.template = {};
  }

  if (!vm.spec.template.metadata) {
    vm.spec.template.metadata = {};
  }

  if (!vm.spec.template.metadata.labels) {
    vm.spec.template.metadata.labels = {};
  }

  return vm.spec.template.metadata.labels;
};

const addDisk = (vm, defaultDisk, storage, getSetting) => {
  const diskSpec = _objectSpread({}, storage.templateStorage ? storage.templateStorage.disk : defaultDisk, {
    name: storage.name
  });

  if (storage.isBootable) {
    const imageSource = getSetting(_constants2.PROVISION_SOURCE_TYPE_KEY);
    diskSpec.bootOrder = imageSource === _constants.PROVISION_SOURCE_PXE ? _constants.BOOT_ORDER_SECOND : _constants.BOOT_ORDER_FIRST;
  } else {
    delete diskSpec.bootOrder;
  }

  const disks = getDisks(vm);
  disks.push(diskSpec);
};

const addPvcVolume = (vm, storage, getSetting) => {
  const claimName = getSetting(IS_TEMPLATE) ? storage.name : `${storage.name}-${getSetting(_constants2.NAME_KEY)}`;

  const volume = _objectSpread({}, storage.templateStorage ? storage.templateStorage.volume : {}, {
    name: storage.name,
    persistentVolumeClaim: _objectSpread({}, storage.templateStorage ? storage.templateStorage.volume.persistentVolumeClaim : {}, {
      claimName
    })
  });

  addVolume(vm, volume);
};

const addContainerVolume = (vm, storage, getSetting) => {
  const volume = _objectSpread({}, storage.templateStorage ? storage.templateStorage.volume : {}, {
    name: storage.name,
    containerDisk: _objectSpread({}, storage.templateStorage ? storage.templateStorage.volume.containerDisk : {})
  });

  if (storage.rootStorage) {
    volume.containerDisk.image = getSetting(_constants2.CONTAINER_IMAGE_KEY);
  }

  addVolume(vm, volume);
};

const addDataVolume = (vm, storage, getSetting) => {
  const volume = _objectSpread({}, storage.templateStorage ? storage.templateStorage.volume : {}, {
    name: storage.name,
    dataVolume: _objectSpread({}, storage.templateStorage ? storage.templateStorage.volume.dataVolume : {}, {
      name: `${storage.name}-${getSetting(_constants2.NAME_KEY)}`
    })
  });

  addVolume(vm, volume);
};

const addVolume = (vm, volumeSpec) => {
  const volumes = getVolumes(vm);
  volumes.push(volumeSpec);
}; // TODO datavolumetemplate should have unique name bcs its created as object in kubernetes
// TOTO add ie vm name


const addDataVolumeTemplate = (vm, storage, getSetting) => {
  const urlSource = {
    http: {
      url: getSetting(_constants2.IMAGE_URL_KEY)
    }
  };
  const blankSource = {
    blank: {}
  };
  const source = getSetting(_constants2.PROVISION_SOURCE_TYPE_KEY) === _constants.PROVISION_SOURCE_URL && storage.isBootable ? urlSource : blankSource;

  const dataVolume = _objectSpread({}, storage.templateStorage ? storage.templateStorage.dataVolume : {}, {
    metadata: _objectSpread({}, storage.templateStorage ? storage.templateStorage.dataVolume.metadata : {}, {
      name: `${storage.name}-${getSetting(_constants2.NAME_KEY)}`
    }),
    spec: _objectSpread({}, storage.templateStorage ? storage.templateStorage.dataVolume.spec : {}, {
      pvc: _objectSpread({}, storage.templateStorage ? storage.templateStorage.dataVolume.spec.pvc : {}, {
        accessModes: [...(storage.templateStorage ? storage.templateStorage.dataVolume.spec.pvc.accessModes : [_constants.PVC_ACCESSMODE_RWO])],
        resources: _objectSpread({}, storage.templateStorage ? storage.templateStorage.dataVolume.spec.pvc.resources : {}, {
          requests: _objectSpread({}, storage.templateStorage ? storage.templateStorage.dataVolume.spec.pvc.resources.requests : {}, {
            storage: `${storage.size}Gi`
          })
        }),
        storageClassName: storage.storageClass
      }),
      source: _objectSpread({}, storage.templateStorage ? storage.templateStorage.dataVolume.spec.source : source)
    })
  });

  const dataVolumes = getDataVolumeTemplates(vm);
  dataVolumes.push(dataVolume);
};

const addInterface = (vm, defaultInterface, network) => {
  const interfaceSpec = _objectSpread({}, network.templateNetwork ? network.templateNetwork.interface : defaultInterface, {
    name: network.name
  });

  if (network.mac) {
    interfaceSpec.macAddress = network.mac;
  }

  if (network.isBootable) {
    interfaceSpec.bootOrder = _constants.BOOT_ORDER_FIRST;
  } else {
    delete interfaceSpec.bootOrder;
  }

  const interfaces = getInterfaces(vm);
  interfaces.push(interfaceSpec);
};

const addNetwork = (vm, network) => {
  const networkSpec = _objectSpread({}, network.templateNetwork ? network.templateNetwork.network : {}, {
    name: network.name
  }); // TODO support unknown types


  switch (network.networkType) {
    case _constants2.NETWORK_TYPE_MULTUS:
      networkSpec.multus = _objectSpread({}, network.templateNetwork ? network.templateNetwork.network.multus : {}, {
        networkName: network.network
      });
      break;

    case _constants2.NETWORK_TYPE_POD:
    default:
      networkSpec.pod = {};
      break;
  }

  const networks = getNetworks(vm);
  networks.push(networkSpec);
};

const addAnnotation = (vm, key, value) => {
  const annotations = getAnnotations(vm);
  annotations[key] = value;
};

const addLabel = (vm, key, value) => {
  const labels = getLabels(vm);
  labels[key] = value;
};

const addTemplateLabel = (vm, key, value) => {
  const labels = getTemplateLabels(vm);
  labels[key] = value;
};

const removeDisksAndVolumes = vm => {
  delete vm.spec.template.spec.domain.devices.disks;
  delete vm.spec.template.spec.volumes;
  delete vm.spec.dataVolumeTemplates;
};

const removeInterfacesAndNetworks = vm => {
  delete vm.spec.template.spec.domain.devices.interfaces;
  delete vm.spec.template.spec.networks;
};