"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.prefixedId = prefixedId;
exports.getAddNicPatch = exports.getUpdateFlavorPatch = exports.getUpdateDescriptionPatch = exports.getAddDiskPatch = exports.getPxeBootPatch = exports.getBootDeviceIndex = exports.setNativeValue = exports.getSequence = void 0;

var _lodash = require("lodash");

var _selectors = require("./selectors");

var _constants = require("../constants");

var _constants2 = require("../components/Wizard/CreateVmWizard/constants");

function prefixedId(idPrefix, id) {
  return idPrefix && id ? `${idPrefix}-${id}` : null;
}

const getSequence = (from, to) => Array.from({
  length: to - from + 1
}, (v, i) => i + from);

exports.getSequence = getSequence;

const setNativeValue = (element, value) => {
  const valueSetter = Object.getOwnPropertyDescriptor(element, 'value').set;
  const prototype = Object.getPrototypeOf(element);
  const prototypeValueSetter = Object.getOwnPropertyDescriptor(prototype, 'value').set;

  if (valueSetter && valueSetter !== prototypeValueSetter) {
    prototypeValueSetter.call(element, value);
  } else {
    valueSetter.call(element, value);
  }
};

exports.setNativeValue = setNativeValue;

const getBootDeviceIndex = (devices, bootOrder) => devices.findIndex(device => device.bootOrder === bootOrder);

exports.getBootDeviceIndex = getBootDeviceIndex;

const getPxeBootPatch = vm => {
  const patches = [];
  const annotations = (0, _lodash.get)(vm, 'metadata.annotations', {});

  if (annotations[_constants.ANNOTATION_FIRST_BOOT]) {
    if (annotations[_constants.ANNOTATION_FIRST_BOOT] === 'true') {
      patches.push({
        op: 'replace',
        path: `/metadata/annotations/${_constants.ANNOTATION_FIRST_BOOT}`,
        value: 'false'
      });
    } else {
      // find bootable disk and change boot order
      const bootableDiskIndex = getBootDeviceIndex((0, _selectors.getDisks)(vm), _constants.BOOT_ORDER_SECOND);
      const bootableInterfaceIndex = getBootDeviceIndex((0, _selectors.getInterfaces)(vm), _constants.BOOT_ORDER_FIRST);

      if (bootableDiskIndex !== -1 && bootableInterfaceIndex !== -1) {
        patches.push({
          op: 'replace',
          path: `/spec/template/spec/domain/devices/disks/${bootableDiskIndex}/bootOrder`,
          value: _constants.BOOT_ORDER_FIRST
        }, {
          op: 'remove',
          path: `/spec/template/spec/domain/devices/interfaces/${bootableInterfaceIndex}/bootOrder`
        });
      }
    }
  }

  return patches;
};

exports.getPxeBootPatch = getPxeBootPatch;

const getAddDiskPatch = (vm, storage) => {
  const disk = {
    name: storage.name
  };

  if (storage.bus) {
    disk.disk = {
      bus: storage.bus
    };
  }

  const volume = {
    name: storage.name,
    dataVolume: {
      name: `${storage.name}-${(0, _selectors.getName)(vm)}`
    }
  };
  const dataVolumeTemplate = {
    metadata: {
      name: `${storage.name}-${(0, _selectors.getName)(vm)}`
    },
    spec: {
      pvc: {
        accessModes: [_constants.PVC_ACCESSMODE_RWO],
        resources: {
          requests: {
            storage: `${storage.size}Gi`
          }
        }
      },
      source: {
        blank: {}
      }
    }
  };

  if (storage.storageClass) {
    dataVolumeTemplate.spec.pvc.storageClassName = storage.storageClass;
  }

  const patch = [];
  const hasDisk = (0, _lodash.get)(vm, 'spec.template.spec.domain.devices.disks', false);

  if (hasDisk) {
    patch.push({
      op: 'add',
      path: '/spec/template/spec/domain/devices/disks/0',
      value: disk
    });
  } else {
    patch.push({
      op: 'add',
      path: '/spec/template/spec/domain/devices/disks',
      value: [disk]
    });
  }

  const hasVolume = (0, _lodash.get)(vm, 'spec.template.spec.volumes', false);

  if (hasVolume) {
    patch.push({
      op: 'add',
      path: '/spec/template/spec/volumes/0',
      value: volume
    });
  } else {
    patch.push({
      op: 'add',
      path: '/spec/template/spec/volumes',
      value: [volume]
    });
  }

  const hasDataVolume = (0, _lodash.get)(vm, 'spec.dataVolumeTemplates', false);

  if (hasDataVolume) {
    patch.push({
      op: 'add',
      path: '/spec/dataVolumeTemplates/0',
      value: dataVolumeTemplate
    });
  } else {
    patch.push({
      op: 'add',
      path: '/spec/dataVolumeTemplates',
      value: [dataVolumeTemplate]
    });
  }

  return patch;
};

exports.getAddDiskPatch = getAddDiskPatch;

const getUpdateDescriptionPatch = (vm, description) => {
  const patch = [];

  if (description !== (0, _selectors.getDescription)(vm)) {
    if (!description && (0, _lodash.has)(vm.metadata, 'annotations.description')) {
      patch.push({
        op: 'remove',
        path: '/metadata/annotations/description'
      });
    } else if (!(0, _lodash.has)(vm.metadata, 'annotations')) {
      patch.push({
        op: 'add',
        path: '/metadata/annotations',
        value: {
          description
        }
      });
    } else {
      patch.push({
        op: (0, _lodash.has)(vm.metadata, 'annotations.description') ? 'replace' : 'add',
        path: '/metadata/annotations/description',
        value: description
      });
    }
  }

  return patch;
};

exports.getUpdateDescriptionPatch = getUpdateDescriptionPatch;

const getDomainPatch = vm => {
  let patch;

  if (!(0, _lodash.has)(vm, 'spec')) {
    patch = {
      op: 'add',
      path: '/spec',
      value: {
        template: {
          spec: {
            domain: {}
          }
        }
      }
    };
  } else if (!(0, _lodash.has)(vm.spec, 'template')) {
    patch = {
      op: 'add',
      path: '/spec/template',
      value: {
        spec: {
          domain: {}
        }
      }
    };
  } else if (!(0, _lodash.has)(vm.spec.template, 'spec')) {
    patch = {
      op: 'add',
      path: '/spec/template/spec',
      value: {
        domain: {}
      }
    };
  } else if (!(0, _lodash.has)(vm.spec.template.spec, 'domain')) {
    patch = {
      op: 'add',
      path: '/spec/template/spec/domain',
      value: {}
    };
  }

  return patch;
};

const getLabelsPatch = vm => {
  if (!(0, _lodash.has)(vm.metadata, 'labels')) {
    return {
      op: 'add',
      path: '/metadata/labels',
      value: {}
    };
  }

  return null;
};

const getCpuPatch = (vm, cpu) => {
  if (!(0, _lodash.has)(vm.spec, 'template.spec.domain.cpu')) {
    return {
      op: 'add',
      path: '/spec/template/spec/domain/cpu',
      value: {
        cores: parseInt(cpu, 10)
      }
    };
  }

  return {
    op: (0, _lodash.has)(vm.spec, 'template.spec.domain.cpu.cores') ? 'replace' : 'add',
    path: '/spec/template/spec/domain/cpu/cores',
    value: parseInt(cpu, 10)
  };
};

const getMemoryPatch = (vm, memory) => {
  if (!(0, _lodash.has)(vm.spec, 'template.spec.domain.resources')) {
    return {
      op: 'add',
      path: '/spec/template/spec/domain/resources',
      value: {
        requests: {
          memory
        }
      }
    };
  }

  if (!(0, _lodash.has)(vm.spec, 'template.spec.domain.resources.requests')) {
    return {
      op: 'add',
      path: '/spec/template/spec/domain/resources/requests',
      value: {
        memory
      }
    };
  }

  return {
    op: (0, _lodash.has)(vm.spec, 'template.spec.domain.resources.requests.memory') ? 'replace' : 'add',
    path: '/spec/template/spec/domain/resources/requests/memory',
    value: memory
  };
};

const getUpdateFlavorPatch = (vm, flavor, cpu, memory) => {
  const patch = [];

  if (flavor !== (0, _selectors.getFlavor)(vm)) {
    const labelKey = `${_constants.TEMPLATE_FLAVOR_LABEL}/${flavor}`.replace('~', '~0').replace('/', '~1');
    const labelPatch = getLabelsPatch(vm);

    if (labelPatch) {
      patch.push(labelPatch);
    }

    const flavorLabel = Object.keys(vm.metadata.labels || {}).find(key => key.startsWith(_constants.TEMPLATE_FLAVOR_LABEL));

    if (flavorLabel) {
      const flavorParts = flavorLabel.split('/');

      if (flavorParts[flavorParts.length - 1] !== flavor) {
        const escapedLabel = flavorLabel.replace('~', '~0').replace('/', '~1');
        patch.push({
          op: 'remove',
          path: `/metadata/labels/${escapedLabel}`
        });
      }
    }

    patch.push({
      op: 'add',
      path: `/metadata/labels/${labelKey}`,
      value: 'true'
    });
  }

  const vmCpu = (0, _selectors.getCpu)(vm);
  const vmMemory = (0, _selectors.getMemory)(vm);

  if (parseInt(cpu, 10) !== vmCpu || memory !== vmMemory) {
    const domainPatch = getDomainPatch(vm);

    if (domainPatch) {
      patch.push(domainPatch);
    }
  }

  if (parseInt(cpu, 10) !== vmCpu) {
    patch.push(getCpuPatch(vm, cpu));
  }

  if (memory !== vmMemory) {
    patch.push(getMemoryPatch(vm, memory));
  }

  return patch;
};

exports.getUpdateFlavorPatch = getUpdateFlavorPatch;

const getAddNicPatch = (vm, nic) => {
  const i = {
    name: nic.name,
    model: nic.model,
    bridge: {}
  };

  if (nic.mac) {
    i.macAddress = nic.mac;
  }

  const network = {
    name: nic.name
  };

  if ((0, _lodash.get)(nic.network, 'networkType') === _constants2.NETWORK_TYPE_POD) {
    network.pod = {};
  } else {
    network.multus = {
      networkName: (0, _lodash.get)(nic.network, 'name')
    };
  }

  const patch = [];
  const hasInterfaces = (0, _lodash.get)(vm, 'spec.template.spec.domain.devices.interfaces', false);

  if (hasInterfaces) {
    patch.push({
      op: 'add',
      path: '/spec/template/spec/domain/devices/interfaces/0',
      value: i
    });
  } else {
    patch.push({
      op: 'add',
      path: '/spec/template/spec/domain/devices/interfaces',
      value: [i]
    });
  }

  const hasNetworks = (0, _lodash.get)(vm, 'spec.template.spec.networks', false);

  if (hasNetworks) {
    patch.push({
      op: 'add',
      path: '/spec/template/spec/networks/0',
      value: network
    });
  } else {
    patch.push({
      op: 'add',
      path: '/spec/template/spec/networks',
      value: [network]
    });
  }

  return patch;
};

exports.getAddNicPatch = getAddNicPatch;