"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Url = void 0;

var _propTypes = _interopRequireDefault(require("prop-types"));

var _react = _interopRequireDefault(require("react"));

var _utils = require("../../../utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const ELIPSIS = '…';

const elipsizeLeft = word => `${ELIPSIS}${word}`;

const resolveOrigin = ({
  hostname,
  origin,
  port
}, maxHostnameParts) => {
  const hostnameParts = hostname.split('.');

  if (hostnameParts.length <= maxHostnameParts) {
    return origin;
  }

  const resolvedHostname = hostnameParts.slice(hostnameParts.length - maxHostnameParts).join('.');
  const resolvedPort = port ? `:${port}` : '';
  return `${elipsizeLeft(resolvedHostname)}${resolvedPort}`;
};

const resolvePathname = ({
  pathname
}, maxPathnameParts) => {
  const pathnameParts = pathname.split('/').filter(part => part);

  if (pathnameParts.length <= maxPathnameParts) {
    return pathname;
  }

  const resolvedPathname = pathnameParts.slice(pathnameParts.length - maxPathnameParts).join('/');
  return `/${elipsizeLeft(`/${resolvedPathname}`)}`;
};

const resolveUrl = ({
  urlObj,
  maxHostnameParts,
  maxPathnameParts
}) => {
  const search = urlObj.search,
        hash = urlObj.hash;
  const resolvedOrigin = resolveOrigin(urlObj, maxHostnameParts);
  const resolvedPathname = resolvePathname(urlObj, maxPathnameParts);
  const resolvedSearchHash = search.length > 0 || hash.length > 0 ? ELIPSIS : '';
  return `${resolvedOrigin}${resolvedPathname}${resolvedSearchHash}`;
};

const Url = ({
  url,
  short,
  maxHostnameParts,
  maxPathnameParts
}) => {
  const urlObj = short ? (0, _utils.parseUrl)(url) : undefined;
  const resolvedUrl = urlObj ? resolveUrl({
    urlObj,
    maxHostnameParts,
    maxPathnameParts
  }) : url;
  const resolvedTitle = resolvedUrl === url ? undefined : url;
  return _react.default.createElement("a", {
    href: url,
    title: resolvedTitle
  }, resolvedUrl);
};

exports.Url = Url;
Url.propTypes = {
  url: _propTypes.default.string.isRequired,
  short: _propTypes.default.bool,
  maxHostnameParts: _propTypes.default.number,
  maxPathnameParts: _propTypes.default.number
};
Url.defaultProps = {
  short: false,
  maxHostnameParts: 3,
  maxPathnameParts: 1
};