"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NetworksTab = exports.isBootableNetwork = exports.hasError = exports.validateNetworksNamespace = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _Form = require("../../Form");

var _TableFactory = require("../../Table/TableFactory");

var _constants = require("../../Table/constants");

var _constants2 = require("../../../constants");

var _validations = require("../../../utils/validations");

var _constants3 = require("./constants");

var _strings = require("./strings");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

function _slicedToArray(arr, i) { return _arrayWithHoles(arr) || _iterableToArrayLimit(arr, i) || _nonIterableRest(); }

function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance"); }

function _iterableToArrayLimit(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"] != null) _i["return"](); } finally { if (_d) throw _e; } } return _arr; }

function _arrayWithHoles(arr) { if (Array.isArray(arr)) return arr; }

const validateNetwork = network => {
  const errors = Array(4).fill(null);

  if (!network || network.id == null) {
    errors[0] = _strings.ERROR_EMPTY_ENTITY; // row error on index 0
  }

  if (!network.name) {
    errors[1] = _strings.ERROR_EMPTY_NAME;
  }

  if (network.id !== 1 && !network.network) {
    errors[3] = _strings.ERROR_NETWORK_NOT_SELECTED;
  }

  return errors;
};

const validateNetworksNamespace = (networkConfigs, namespace, networks) => {
  const availableNetworkConfigs = networkConfigs.filter(nc => nc.metadata.namespace === namespace);
  networks.filter(network => isBootableNetwork(network)).forEach(network => {
    if (!network.errors) {
      network.errors = Array(4).fill(null);
    }

    network.errors[3] = availableNetworkConfigs.some(nc => nc.metadata.name === network.network) ? null : _strings.ERROR_NETWORK_NOT_FOUND;
  });
};

exports.validateNetworksNamespace = validateNetworksNamespace;

const hasError = network => network.errors ? network.errors.some(error => !!error) : false;

exports.hasError = hasError;

const isBootableNetwork = network => network.network !== _constants2.POD_NETWORK && network.network !== '';

exports.isBootableNetwork = isBootableNetwork;

const resolveBootableNetwork = (sourceType, rows) => {
  if (sourceType === _constants2.PROVISION_SOURCE_PXE && !rows.some(row => row.isBootable && !hasError(row))) {
    const bootableNetworks = rows.filter(row => isBootableNetwork(row));

    if (bootableNetworks.length > 0) {
      let bootableNetwork;
      bootableNetworks.filter(n => n.templateNetwork && n.templateNetwork.interface.bootOrder).forEach(n => {
        const bootOrder = n.templateNetwork.interface.bootOrder;

        if (!bootableNetwork || bootOrder < bootableNetwork.templateNetwork.interface.bootOrder) {
          bootableNetwork = n;
        }
      });

      if (!bootableNetwork) {
        var _bootableNetworks = _slicedToArray(bootableNetworks, 1);

        bootableNetwork = _bootableNetworks[0];
      }

      bootableNetwork.isBootable = true;
    }
  }
};

const resolveInitialNetworks = (networks, networkConfigs, namespace, sourceType) => {
  let nextId = Math.max(...networks.map(network => network.id || 0), 0) + 1;
  const initialNetworks = networks.map(network => {
    if (network.templateNetwork) {
      const templateNetwork = network.templateNetwork;
      let networkType;

      if (templateNetwork.network.pod) {
        networkType = _constants3.NETWORK_TYPE_POD;
      } else if (templateNetwork.network.multus) {
        networkType = _constants3.NETWORK_TYPE_MULTUS;
      }

      return {
        templateNetwork,
        name: templateNetwork.interface.name,
        mac: templateNetwork.interface.macAddress,
        network: templateNetwork.network.pod ? _constants2.POD_NETWORK : templateNetwork.network.multus.networkName,
        // TODO support others even if unknown
        id: nextId++,
        editable: true,
        edit: false,
        networkType
      };
    }

    return _objectSpread({}, network, {
      id: nextId++,
      editable: true,
      edit: false
    });
  });
  validateNetworksNamespace(networkConfigs, namespace, initialNetworks);
  resolveBootableNetwork(sourceType, initialNetworks);
  return initialNetworks;
};

class NetworksTab extends _react.default.Component {
  constructor(props) {
    super(props);

    _defineProperty(this, "publishResults", rows => {
      let valid = this.props.sourceType === _constants2.PROVISION_SOURCE_PXE ? rows.some(row => row.isBootable) : true;
      const nics = rows.map(({
        templateNetwork,
        rootNetwork,
        id,
        isBootable,
        name,
        mac,
        network,
        errors,
        networkType
      }) => {
        const result = {
          id,
          isBootable,
          name,
          mac,
          network,
          errors,
          networkType,
          rootNetwork
        };

        if (templateNetwork) {
          result.templateNetwork = templateNetwork;
        }

        if (valid && errors) {
          for (const error of errors) {
            if (error) {
              valid = false;
              break;
            }
          }
        }

        return result;
      });
      this.props.onChange(nics, valid);
    });

    _defineProperty(this, "onRowActivate", rows => {
      this.setState({
        rows,
        editing: true
      });
    });

    _defineProperty(this, "onRowUpdate", (rows, updatedRowId, editing) => {
      const updatedRow = rows.find(r => r.id === updatedRowId);

      if (updatedRow.isBootable && updatedRow.network === _constants2.POD_NETWORK) {
        updatedRow.isBootable = false;
      }

      if (updatedRow.network === _constants2.POD_NETWORK) {
        updatedRow.networkType = _constants3.NETWORK_TYPE_POD;
      } else {
        updatedRow.networkType = _constants3.NETWORK_TYPE_MULTUS;
      }

      updatedRow.errors = validateNetwork(updatedRow);
      this.rowsChanged(rows, editing);
    });

    _defineProperty(this, "rowsChanged", (rows, editing) => {
      resolveBootableNetwork(this.props.sourceType, rows);
      this.publishResults(rows);
      this.setState({
        rows,
        editing
      });
    });

    _defineProperty(this, "createNic", () => {
      this.setState(state => ({
        nextId: state.nextId + 1,
        rows: [...state.rows, {
          id: state.nextId,
          isBootable: false,
          editable: true,
          edit: true,
          // trigger immediate edit,
          name: `nic${state.nextId - 1}`,
          mac: '',
          network: ''
        }]
      }));
    });

    _defineProperty(this, "getColumns", () => [{
      header: {
        label: _strings.HEADER_NAME,
        props: {
          style: {
            width: '32%'
          }
        }
      },
      property: 'name',
      renderConfig: () => ({
        id: 'name-edit',
        type: _Form.TEXT
      })
    }, {
      header: {
        label: _strings.HEADER_MAC,
        props: {
          style: {
            width: '32%'
          }
        }
      },
      property: 'mac',
      renderConfig: row => row.networkType === _constants3.NETWORK_TYPE_POD ? null : {
        id: 'mac-edit',
        type: _Form.TEXT
      }
    }, {
      header: {
        label: _strings.HEADER_NETWORK,
        props: {
          style: {
            width: '32%'
          }
        }
      },
      property: 'network',
      renderConfig: () => ({
        id: 'network-edit',
        type: _Form.DROPDOWN,
        choices: this.props.networkConfigs.filter(networkConfig => networkConfig.metadata.namespace === this.props.namespace).map(networkConfig => networkConfig.metadata.name).concat(_constants2.POD_NETWORK).filter(networkConfig => !this.state.rows.some(r => r.network === networkConfig)),
        initialValue: _strings.SELECT_NETWORK
      })
    }, {
      header: {
        props: {
          style: {
            width: '4%'
          }
        }
      },
      type: _constants.ACTIONS_TYPE,
      renderConfig: () => ({
        id: 'actions',
        actions: [{
          actionType: _constants.DELETE_ACTION,
          text: _strings.REMOVE_NIC_BUTTON
        }],
        visibleOnEdit: false
      })
    }]);

    _defineProperty(this, "getActionButtons", () => [{
      className: 'kubevirt-create-vm-wizard__button-create-network',
      onClick: this.createNic,
      id: 'create-network-btn',
      text: _strings.CREATE_NIC_BUTTON,
      disabled: this.state.editing
    }]);

    _defineProperty(this, "getFormFields", pxeNetworks => ({
      pxeNetwork: {
        id: 'pxe-nic-dropdown',
        title: _strings.PXE_NIC,
        type: 'dropdown',
        defaultValue: _strings.SELECT_PXE_NIC,
        choices: pxeNetworks.filter(n => !hasError(n)).map(n => ({
          name: n.name,
          id: n.id
        })),
        required: true,
        help: _strings.PXE_INFO
      }
    }));

    _defineProperty(this, "onFormChange", newValue => {
      this.setState(state => {
        state.rows.forEach(row => {
          row.isBootable = row.id === newValue.value.id;
        });
        this.publishResults(state.rows);
        return state.rows;
      });
    });

    const _rows = resolveInitialNetworks(props.networks, props.networkConfigs, props.namespace, props.sourceType);

    this.publishResults(_rows);
    this.state = {
      // eslint-disable-next-line
      nextId: Math.max(..._rows.map(network => network.id || 0), 0) + 1,
      editing: false,
      rows: _rows
    };
  }

  render() {
    const columns = this.getColumns();
    const actionButtons = this.getActionButtons();
    let pxeForm;

    if (this.props.sourceType === _constants2.PROVISION_SOURCE_PXE) {
      const pxeNetworks = this.state.rows.filter(row => isBootableNetwork(row));
      const bootableNetwork = pxeNetworks.find(row => row.isBootable);
      const values = {
        pxeNetwork: {
          value: bootableNetwork ? bootableNetwork.name : undefined,
          validation: pxeNetworks.length === 0 ? (0, _validations.getValidationObject)(_strings.PXE_NIC_NOT_FOUND_ERROR) : undefined
        }
      };
      pxeForm = _react.default.createElement(_Form.FormFactory, {
        fields: this.getFormFields(pxeNetworks),
        fieldsValues: values,
        onFormChange: this.onFormChange,
        textPosition: "text-left",
        labelSize: 2,
        controlSize: 10,
        formClassName: "kubevirt-create-vm-wizard__pxe-form"
      });
    }

    return _react.default.createElement(_react.default.Fragment, null, _react.default.createElement(_TableFactory.TableFactory, {
      actionButtons: actionButtons,
      columns: columns,
      rows: this.state.rows,
      onRowUpdate: this.onRowUpdate,
      onRowDeleteOrMove: this.rowsChanged,
      onRowActivate: this.onRowActivate
    }), pxeForm);
  }

}

exports.NetworksTab = NetworksTab;
NetworksTab.propTypes = {
  onChange: _propTypes.default.func.isRequired,
  networks: _propTypes.default.array.isRequired,
  sourceType: _propTypes.default.string.isRequired,
  networkConfigs: _propTypes.default.array.isRequired,
  namespace: _propTypes.default.string.isRequired
};